/*
 * cnic_if.h: QLogic CNIC core network driver.
 * Copyright (c)   2015-2017 QLogic Corporation
 * Copyright (c)   2016-2017 Cavium Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation.
 *
 */


#ifndef CNIC_IF_H
#define CNIC_IF_H

#include <vmkapi.h>

#include "qfle3_ecore_mfw_req.h"

enum {
	QFLE3_OOO_ETH_CL_ID_IDX,
	QFLE3_ISCSI_ETH_CL_ID_IDX,
	QFLE3_FCOE_ETH_CL_ID_IDX,
	QFLE3_MAX_CNIC_ETH_CL_ID_IDX,
};

#define CNIC_ULP_RDMA		0
#define CNIC_ULP_ISCSI		1
#define CNIC_ULP_FCOE		2
#define CNIC_ULP_L4		3
#define MAX_CNIC_ULP_TYPE_EXT	3
#define MAX_CNIC_ULP_TYPE	4

/* Use CPU native page size up to 16K for cnic ring sizes. */
#if (VMK_PAGE_SHIFT > 14)
#define CNIC_PAGE_BITS 14
#else
#define CNIC_PAGE_BITS VMK_PAGE_SHIFT
#endif
#define CNIC_PAGE_SIZE (1 << (CNIC_PAGE_BITS))
#define CNIC_PAGE_ALIGN(addr) ALIGN(addr, CNIC_PAGE_SIZE)
#define CNIC_PAGE_MASK (~((CNIC_PAGE_SIZE) - 1))

#ifndef VLAN_TAG_PRESENT
#define VLAN_TAG_PRESENT	0x1000
#endif

#define IPv4_ADDRLEN	4	//IPv4 address size
#define IPv6_ADDRLEN	16	//IPv6 address size


#define QFLE3_FCOE_NUM_CONNECTIONS 128

struct kwqe {
	vmk_uint32 kwqe_op_flag;

#define KWQE_QID_SHIFT		8
#define KWQE_OPCODE_MASK	0x00ff0000
#define KWQE_OPCODE_SHIFT	16
#define KWQE_FLAGS_LAYER_SHIFT	28
#define KWQE_OPCODE(x)		((x & KWQE_OPCODE_MASK) >> KWQE_OPCODE_SHIFT)
#define KWQE_LAYER_MASK		0x70000000
#define KWQE_LAYER_SHIFT	28
#define KWQE_FLAGS_LAYER_MASK_L2	(2<<28)
#define KWQE_FLAGS_LAYER_MASK_L3	(3<<28)
#define KWQE_FLAGS_LAYER_MASK_L4	(4<<28)
#define KWQE_FLAGS_LAYER_MASK_L5_RDMA	(5<<28)
#define KWQE_FLAGS_LAYER_MASK_L5_ISCSI	(6<<28)
#define KWQE_FLAGS_LAYER_MASK_L5_FCOE	(7<<28)

	vmk_uint32 kwqe_info0;
	vmk_uint32 kwqe_info1;
	vmk_uint32 kwqe_info2;
	vmk_uint32 kwqe_info3;
	vmk_uint32 kwqe_info4;
	vmk_uint32 kwqe_info5;
	vmk_uint32 kwqe_info6;
};

struct kwqe_16 {
	vmk_uint32 kwqe_info0;
	vmk_uint32 kwqe_info1;
	vmk_uint32 kwqe_info2;
	vmk_uint32 kwqe_info3;
};

struct kcqe {
	vmk_uint32 kcqe_info0;
	vmk_uint32 kcqe_info1;
	vmk_uint32 kcqe_info2;
	vmk_uint32 kcqe_info3;
	vmk_uint32 kcqe_info4;
	vmk_uint32 kcqe_info5;
	vmk_uint32 kcqe_info6;
	vmk_uint32 kcqe_op_flag;
		#define KCQE_RAMROD_COMPLETION		(0x1<<27) /* Everest */
		#define KCQE_FLAGS_LAYER_MASK		(0x7<<28)
		#define KCQE_FLAGS_LAYER_MASK_MISC	(0<<28)
		#define KCQE_FLAGS_LAYER_MASK_L2	(2<<28)
		#define KCQE_FLAGS_LAYER_MASK_L3	(3<<28)
		#define KCQE_FLAGS_LAYER_MASK_L4	(4<<28)
		#define KCQE_FLAGS_LAYER_MASK_L5_RDMA	(5<<28)
		#define KCQE_FLAGS_LAYER_MASK_L5_ISCSI	(6<<28)
		#define KCQE_FLAGS_LAYER_MASK_L5_FCOE	(7<<28)
		#define KCQE_FLAGS_NEXT 		(1<<31)
		#define KCQE_FLAGS_OPCODE_MASK		(0xff<<16)
		#define KCQE_FLAGS_OPCODE_SHIFT		(16)
		#define KCQE_OPCODE(op)			\
		(((op) & KCQE_FLAGS_OPCODE_MASK) >> KCQE_FLAGS_OPCODE_SHIFT)
};

#define MAX_CNIC_CTL_DATA	64
#define MAX_DRV_CTL_DATA	64

#define CNIC_CTL_STOP_CMD			1
#define CNIC_CTL_START_CMD			2
#define CNIC_CTL_COMPLETION_CMD			3
#define CNIC_CTL_STOP_ISCSI_CMD			4
#define CNIC_CTL_FCOE_STATS_GET_CMD		5
#define CNIC_CTL_ISCSI_STATS_GET_CMD		6
#define CNIC_CTL_NIC_RECOVERY_CMD         7
#define CNIC_CTL_NIC_RECOVERY_DONE_CMD         8

#define DRV_CTL_IO_WR_CMD		0x101
#define DRV_CTL_IO_RD_CMD		0x102
#define DRV_CTL_CTX_WR_CMD		0x103
#define DRV_CTL_CTXTBL_WR_CMD		0x104
#define DRV_CTL_RET_L5_SPQ_CREDIT_CMD	0x105
#define DRV_CTL_START_L2_CMD		0x106
#define DRV_CTL_STOP_L2_CMD		0x107
#define DRV_CTL_GET_OOO_CQE		0x108
#define DRV_CTL_SEND_OOO_PKT		0x109
#define DRV_CTL_COMP_OOO_TX_PKTS	0x10a
#define DRV_CTL_REUSE_OOO_PKT		0x10b
#define DRV_CTL_RET_L2_SPQ_CREDIT_CMD	0x10c
#define DRV_CTL_ISCSI_STOPPED_CMD	0x10d
#define DRV_CTL_ULP_REGISTER_CMD	0x10e
#define DRV_CTL_ULP_UNREGISTER_CMD	0x10f
#define DRV_CTL_START_NPAR_CMD		0x110
#define DRV_CTL_STOP_NPAR_CMD		0x111

struct cnic_ctl_completion {
	vmk_uint32	cid;
	vmk_uint8	opcode;
	vmk_uint8	error;
};

struct drv_ctl_spq_credit {
	vmk_uint32	credit_count;
};

struct cnic_ctl_info {
	int	cmd;
	union {
		struct cnic_ctl_completion comp;
		char bytes[MAX_CNIC_CTL_DATA];
	} data;
};

struct drv_ctl_io {
	vmk_uint32		cid_addr;
	vmk_uint32		offset;
	vmk_uint32		data;
	vmk_IOA	dma_addr;
};

struct drv_ctl_l2_ring {
	vmk_uint32		client_id;
	vmk_uint32		cid;
};

enum {
	OOO_BD_CQE,
	OOO_RAMROD_CQE
};

enum {
	OOO_OPCODE_ADD_PEN,
	OOO_OPCODE_ADD_NEW,
	OOO_OPCODE_ADD_RIGHT,
	OOO_OPCODE_ADD_LEFT,
	OOO_OPCODE_JOIN,
	OOO_OPCODE_NOP
};

struct cnic_ooo_cqe {
	vmk_uint32 cqe_type; /* OOO_BD_CQE or OOO_RAMROD_CQE */
	union {
		struct {
                        vmk_uint32		raw_data[4]; /* iSCSI CQE data */
			vmk_PktHandle 	*pkt_desc;
		} cqe;

		struct {
			struct {
				vmk_uint32	lo;
				vmk_uint32	hi;
			} data;
		} ramrod_data;
	} u;
};


struct drv_ctl_ooo_cqe {
	struct cnic_ooo_cqe *cqe;
};

struct drv_ctl_send_ooo_pkt {
	vmk_PktHandle *skb;
};

struct drv_ctl_ooo_pkt {
	vmk_PktHandle *skb;
};

struct drv_ctl_register_data {
	int ulp_type;
	struct fcoe_capabilities fcoe_features;
};

struct drv_ctl_info {
	int	cmd;
	int     drv_state;
#define DRV_NOP		0
#define DRV_ACTIVE	1
#define DRV_INACTIVE	2
#define DRV_UNLOADED	3
	union {
		struct drv_ctl_spq_credit credit;
		struct drv_ctl_io io;
		struct drv_ctl_l2_ring ring;
		int ulp_type;
		char bytes[MAX_DRV_CTL_DATA];
		struct drv_ctl_ooo_cqe ooo_cqe;
		struct drv_ctl_ooo_pkt pkt_desc;
		struct drv_ctl_register_data register_data;
	} data;
};

typedef struct cnicOps {
	/*
	 * XXX: Old driver: Calls to these functions are protected by RCU.
	 * When unregistering, we wait for any calls to complete before
	 * continuing.
	 */
	int		(*cnicIRQHandler) (void *, void *);
	int		(*cnicNotify) (void *, struct cnic_ctl_info *);
	int		(*cnicLinkUpdate) (void *, vmk_uint8, vmk_uint16);
	int		(*cnicLL2Rx) (void *, vmk_PktHandle *);
} cnic_ops;

#define CNIC_RX_IDX			0

struct cnic_q {
	vmk_IntrCookie	cnic_intr_cookie;

	void			*status_blk;
	vmk_IOA			status_block_ioa;

	vmk_uint32		status_blk_num;
	vmk_uint32		status_blk_num2;
	vmk_uint32		irq_flags;
#define CNIC_IRQ_FL_MSIX		0x00000001
};

/* SFP Diagnostic param Descriptor */
struct sfp_diagnostic {
	vmk_uint32 temperature;
	vmk_uint32 vcc;
	vmk_uint32 tx_bias;
	vmk_uint32 tx_power;
	vmk_uint32 rx_power;
	vmk_uint32 speed_cap;
	vmk_uint32 opr_speed;
};

/* Link Error Status Descriptor */
struct link_error_status {
	vmk_uint32 link_fail_cnt;
	vmk_uint32 vlink_fail_cnt;
	vmk_uint32 sym_err_dcc;
	vmk_uint32 err_blk_cnt;
	vmk_uint32 frame_chk_seq;
};

//SV: TODO: Use same as from qfle3f_npiv.c
#define MAX_NPIV_ENTRIES 64
#define FC_NPIV_WWN_SIZE 8

struct qcnic_fc_npiv_tbl {
	vmk_uint8 wwpn[MAX_NPIV_ENTRIES][FC_NPIV_WWN_SIZE];
	vmk_uint8 wwnn[MAX_NPIV_ENTRIES][FC_NPIV_WWN_SIZE];
	vmk_uint32 count;
};

typedef struct cnic_eth_dev {
	vmk_revnum	 		apiRevision;
#define CNIC_ETH_DEV_VER 0x00000004 /* Change this when the structure changes */
	vmk_ModuleID		moduleID;
	vmk_AddrCookie		driverData;
	vmk_uint32			driverState; // holding interrupt mode
#define CNIC_DRV_STATE_REGD		0x00000001
#define CNIC_DRV_STATE_USING_MSIX	0x00000002
#define CNIC_DRV_STATE_NO_ISCSI_OOO	0x00000004
#define CNIC_DRV_STATE_NO_ISCSI		0x00000008
#define CNIC_DRV_STATE_NO_FCOE		0x00000010
#define CNIC_DRV_STATE_HANDLES_IRQ	0x00000020
	vmk_uint32			chip_id;
	vmk_uint32			link_supported_speeds;
	vmk_uint32			mf_mode;
	vmk_uint32			max_kwqe_pending;
	vmk_PCIDevice 		pdev;
	vmk_PCIDeviceID 	pdev_id;
	vmk_PCIDeviceAddr 	pdev_addr;
	vmk_Name 			pdev_name;
	vmk_Device 			device;       // QFLE3 device
	vmk_uint8 			*reg_base;
	vmk_uint8 			*db_base;
	vmk_uint16 			db_size;
	vmk_uint32			mtu;
	struct cnic_q		irq_arr[2];

	vmk_uint32			ctx_tbl_offset;
	vmk_uint32			ctx_tbl_len;
	vmk_int32			ctx_blk_size;
	vmk_uint32			starting_cid;

	vmk_uint32			nof_ofld_conns;

	/* SV: XXX: to be reviewed. */
	vmk_uint32		mf_sub_mode;
	vmk_uint32		e1hov_tag;
	struct vlan_group	**cna_vlgrp;
	//======
	vmk_uint32		max_iscsi_conn;
	vmk_uint32		max_fcoe_conn;
	vmk_uint32		max_fcoe_exchanges;
	vmk_uint32		max_rdma_conn;
	//======
	vmk_uint32		fcoe_init_cid;
	vmk_uint32		fcoe_wwn_port_name_hi;
	vmk_uint32		fcoe_wwn_port_name_lo;
	vmk_uint32		fcoe_wwn_node_name_hi;
	vmk_uint32		fcoe_wwn_node_name_lo;
	//======
	vmk_uint32		iscsi_l2_client_id;
	vmk_uint32		iscsi_l2_cid;
	//======
	union drv_info_to_mcp	*addr_drv_info_to_mcp;
	//======

	VMK_ReturnStatus (*drv_get_sfp_diagnostic)(vmk_Device, struct sfp_diagnostic *);
	VMK_ReturnStatus (*drv_get_link_error)(vmk_Device, struct link_error_status *);
	VMK_ReturnStatus (*cnicRegister)(vmk_Device, struct cnicOps *, void *);
	VMK_ReturnStatus (*cnicUnregister)(vmk_Device);
	VMK_ReturnStatus (*cnicSlowPathCmd)(vmk_Device, struct kwqe_16 *[], vmk_uint32);
	VMK_ReturnStatus (*cnicDriverCtl)(vmk_Device, struct drv_ctl_info *);
	VMK_ReturnStatus (*cnicGetFcNpivTbl)(vmk_Device, struct qcnic_fc_npiv_tbl *);
} cnic_eth_dev_t;

#ifdef CNIC_STORAGE_IF
struct cnic_sockaddr {
	union {
		vmk_SocketIPAddress v4;
		vmk_SocketIPv6Address v6;
	} local;
	union {
		vmk_SocketIPAddress v4;
		vmk_SocketIPv6Address v6;
	} remote;
};

struct cnic_sock {
	struct cnic_dev *dev;
	void	*context;
	vmk_uint32	src_ip[4];
	vmk_uint32	dst_ip[4];
	vmk_uint16	src_port;
	vmk_uint16	dst_port;
	vmk_uint16	vlan_id;
	unsigned char old_ha[VMK_ETH_ADDR_LENGTH];
	vmk_EthAddress ha;
	vmk_uint16	pmtu;
	struct dst_entry *dst;
	vmk_uint32	cid;
	vmk_uint32	l5_cid;
	vmk_uint32	pg_cid;
	int	ulp_type;

	vmk_uint32	ka_timeout;
	vmk_uint32	ka_interval;
	vmk_uint8	ka_max_probe_count;
	vmk_uint8	tos;
	vmk_uint8	ttl;
	vmk_uint8	snd_seq_scale;
	vmk_uint32	rcv_buf;
	vmk_uint32	snd_buf;
	vmk_uint32	seed;

	unsigned long	tcp_flags;
#define SK_TCP_NO_DELAY_ACK	0x1
#define SK_TCP_KEEP_ALIVE	0x2
#define SK_TCP_NAGLE		0x4
#define SK_TCP_TIMESTAMP	0x8
#define SK_TCP_SACK		0x10
#define SK_TCP_SEG_SCALING	0x20

	vmk_BitVector	*flags;
#define SK_F_INUSE		0
#define SK_F_OFFLD_COMPLETE	1
#define SK_F_OFFLD_SCHED	2
#define SK_F_PG_OFFLD_COMPLETE	3
#define SK_F_CONNECT_START	4
#define SK_F_IPV6		5
#define SK_F_NDISC_WAITING	6
#define SK_F_CLOSING		7
#define SK_F_HW_ERR		8

	vmk_atomic64 ref_count;
	vmk_uint32 state;
	struct kwqe kwqe1;
	struct kwqe kwqe2;
	struct kwqe kwqe3;
};

struct cnic_ulp_ops {
	vmk_uint32             version;
	/*
	 * Change when cnic-n-storage interface changes.
	 * Used to compare ulp interface compatibility.
	 */
	#define CNIC_ULP_OPS_VER 0x07000004

	/*
	 * Calls to these functions are protected by RCU.  When
	 * unregistering, we wait for any calls to complete before
	 * continuing.
	 */
	void (*cnic_init)(struct cnic_dev *dev);
	void (*cnic_exit)(struct cnic_dev *dev);
	void (*cnic_start)(void *ulp_ctx);
	void (*cnic_stop)(void *ulp_ctx);
	void (*cnic_fw_recovery_notify)(void *ulp_ctx, int event);
	void (*indicate_kcqes)(void *ulp_ctx, struct kcqe *cqes[],
			vmk_uint32 num_cqes);
	void (*indicate_netevent)(void *ulp_ctx, unsigned long event);
	void (*indicate_inetevent)(void *ulp_ctx, unsigned long event);
	void (*cm_connect_complete)(struct cnic_sock *);
	void (*cm_close_complete)(struct cnic_sock *);
	void (*cm_abort_complete)(struct cnic_sock *);
	void (*cm_remote_close)(struct cnic_sock *);
	void (*cm_remote_abort)(struct cnic_sock *);
	int (*cnic_get_stats)(void *ulp_ctx);
	struct module *owner;
	vmk_atomic64 ref_count;
};

struct cnic_ll2_cb_ops {
	VMK_ReturnStatus (*rx_cb) (struct cnic_dev *, vmk_PktHandle *);
	VMK_ReturnStatus (*tx_cb) (struct cnic_dev *, vmk_PktHandle *);
};

struct cnic_ll2_cb_info {
	struct cnic_ll2_cb_ops *ll2_cb_ops;
	vmk_uint64 cookie;
	vmk_Lock ll2_cb_lck;
};

struct cnic_ll2_ops {
	VMK_ReturnStatus (*ll2_send) (struct cnic_dev *, vmk_PktHandle *);
	VMK_ReturnStatus (*register_cb_ops) (struct cnic_dev *,
		struct cnic_ll2_cb_ops *, vmk_uint64 cookie);
	VMK_ReturnStatus (*unregister_cb_ops) (struct cnic_dev *);
};

typedef struct cnic_dev {
	vmk_ListLinks	list;
	vmk_uint32		version;
#define CNIC_DEV_VER 0x07800009 /* Change this when the structure changes */

	vmk_Device		netdev; /* Physical: Parent NIC device */
	vmk_Device		ldev; /* Logical: cnic device */
	vmk_Name		name;
	vmk_PCIDevice	pdev;
	vmk_uint32		mtu;
	vmk_uint32		chip_id;
	vmk_uint32		link_supported_speeds;
	vmk_LinkStatus	uplinkLinkState;
	vmk_Name		uplinkName;
	vmk_UplinkSharedData *uplinkSharedData;
	void 			*regview;
	void 			*doorbells;

	vmk_SgOpsHandle	SgOps_handle;
	vmk_DMAEngine	dmaEngineCoherent;

	int (*register_device)(struct cnic_dev *dev, int ulp_type,
		void *ulp_ctx, struct cnic_ulp_ops *ulp_ops);
	int (*unregister_device)(struct cnic_dev *dev, int ulp_type);
	int (*submit_kwqes)(struct cnic_dev *dev, struct kwqe *wqes[],
		vmk_uint32 num_wqes);
	int (*submit_kwqes_16)(struct cnic_dev *dev, struct kwqe_16 *wqes[],
		vmk_uint32 num_wqes);

	int (*cm_get_uplinkName)(struct cnic_dev *, vmk_Name *);
	int (*cm_acquire)(struct cnic_dev *, vmk_uint8, vmk_uint32 *);
	int (*cm_create)(struct cnic_dev *, int, vmk_uint32, vmk_uint32,
		struct cnic_sock **, void *);
	int (*cm_destroy)(struct cnic_sock *);
	int (*cm_connect)(struct cnic_sock *, struct cnic_sockaddr *);
	int (*cm_abort)(struct cnic_sock *);
	int (*cm_close)(struct cnic_sock *);
	struct cnic_dev *(*cm_select_dev)(vmk_IscsiNetHandle iscsiNetHandle,
		vmk_SocketIPAddress *, int ulp_type);
	VMK_ReturnStatus (*drv_get_sfp_diagnostic)(struct cnic_dev *,
				struct sfp_diagnostic *);
	VMK_ReturnStatus (*drv_get_link_error)(struct cnic_dev *,
				struct link_error_status *);
	int (*cm_get_fc_npiv_tbl)(struct cnic_dev *, struct qcnic_fc_npiv_tbl *);
	void (*dump_queues)(struct cnic_dev *dev);

	vmk_BitVector	*flags;
#define CNIC_F_IF_UP		0
#define CNIC_F_CNIC_UP		1
#define CNIC_F_IF_GOING_DOWN	2
#define CNIC_F_QFLE3_LEGACY_CLASS	3
#define CNIC_F_QFLE3_CLASS	4
#define CNIC_F_ISCSI_OOO_ENABLE	8
	vmk_atomic64	ref_count;
	int			use_count;
	vmk_uint8	is_ipv6;
	vmk_uint8	reserved;
	vmk_SocketIPv6AddressAddr	ipv6_addr;
	char		dev_name[16];
	vmk_uint8	ipv4_addr[IPv4_ADDRLEN];
	vmk_uint8	subnet_mask[IPv4_ADDRLEN];
	vmk_uint8	subnet_mask_ipv6[IPv6_ADDRLEN];
	vmk_uint32	srcfamily;

	vmk_uint8	pf_id;
	vmk_uint8	pf_num;

	vmk_EthAddress	nic_mac;
	vmk_EthAddress	vmk_mac;
	vmk_EthAddress	iscsi_mac;
	vmk_EthAddress	fip_mac;

	vmk_uint16	vlan_id;
	int		max_iscsi_conn;
	int		max_fcoe_conn;
	int		max_rdma_conn;

	union drv_info_to_mcp	*stats_addr;
	struct fcoe_capabilities	*fcoe_cap;

	void		*cnic_priv;
	unsigned long	ooo_tx_count;
	unsigned long        isr_count;
	unsigned long        isr_rdma_count;
	unsigned long        isr_iscsi_count;
	unsigned long        isr_fcoe_count;
	unsigned long        isr_l4_count;
	unsigned long        isr_l2_count;
	unsigned long        isr_unknown_count;

	vmk_uint64		fcoe_wwnn;
	vmk_uint64		fcoe_wwpn;

	vmk_uint32		pmtu_fails;
	vmk_uint32		mf_mode;
	vmk_uint32		mf_sub_mode;
	vmk_uint32		e1hov_tag;

#if (VMWARE_ESX_DDK_VERSION >= 50000)
	struct vlan_group	**cna_vlgrp;
#endif

	vmk_Device		iSCSIDevice;
	vmk_Device		FCoEDevice;

	/* FCoE LL2-rx handling */
	vmk_PktList fcoe_rx_list;
	vmk_Lock fcoe_rx_list_lck;
	ql_vmk_tasklet_t fcoe_rx_dpc;
	vmk_WorldID fcoe_rx_world_id;

	struct cnic_ll2_ops ll2_ops;
	struct cnic_ll2_cb_info ll2_cb_info;

	/* Per-cdev cnic_ulp_ops */
	struct cnic_ulp_ops *cnic_ulp_tbl[MAX_CNIC_ULP_TYPE];
	vmk_Bool fw_recov_in_progress;
} cnic_dev_t;
#endif

#define CNIC_WR(dev, off, val)		WRT_REG_DWORD(val, dev->regview + off)
#define CNIC_WR16(dev, off, val)	WRT_REG_WORD(val, dev->regview + off)
#define CNIC_WR8(dev, off, val)		WRT_REG_BYTE(val, dev->regview + off)
#define CNIC_RD(dev, off)		RD_REG_DWORD(dev->regview + off)
#define CNIC_RD16(dev, off)		RD_REG_WORD(dev->regview + off)

/* For qfle3-N-cnic interface. */
typedef struct nicOps {
	struct cnic_eth_dev *(*cnicProbeNic) (vmk_AddrCookie data);
	VMK_ReturnStatus (*cnicLL2Tx) (vmk_AddrCookie data, vmk_PktHandle *);
} nic_ops;

#endif
