/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

#include "qfle3f.h"
#include <vmkapi.h>

extern struct cnic_ulp_ops qfle3f_cnic_cb;

static vmk_DMAConstraints qfle3f_dmaConstraint = {
   .sgMaxEntries     = 255,
   /* Required scatter/gather element alignment */
   .sgElemAlignment  = 0,
   /* If non-zero, scatter/gather element length multiple */
   .sgElemSizeMult   = 0,
   .sgElemMaxSize    = 0xffffffff,
   /* Max # of blocks per i/o */
   .maxTransfer      = 0xffffffff,
   /* Address mask to get boundary limit for scatter/gather element */
   .sgElemStraddle   = VMK_ADDRESS_MASK_32BIT+1,
   .addressMask      = VMK_ADDRESS_MASK_64BIT,
};

/* TODO -- Implement this */
static VMK_ReturnStatus
qfle3f_scsiDumpCommand(qfle3fHBA_t *hba, vmk_ScsiCommand *vmkCmd,
   void *deviceData)
{
   return VMK_NOT_SUPPORTED;
}

extern void qfle3f_restart_fcoe_disc(struct qfle3fHBA *hba);
static VMK_ReturnStatus
qfle3f_procInfo(void *clientData,
   char *buf, vmk_ByteCountSmall offset,
   vmk_ByteCountSmall count, vmk_ByteCountSmall * nbytes, int isWrite)
{
  vmk_WarningMessage("qfle3f: proc info is not supported.");
  return VMK_NOT_SUPPORTED;
}

/* TODO -- Implement this */
static void
qfle3f_scsiDumpQueue(qfle3fHBA_t *hba)
{

}

/* TODO -- Implement this */
static void
qfle3f_scsiDumpPollHandler(qfle3fHBA_t *hba)
{

}

static VMK_ReturnStatus
qfle3f_ioctl(void *client_data, void *deviceData,
   vmk_uint32 fileFlags, vmk_uint32 cmd, vmk_VA userArgsPtr,
   vmk_IoctlCallerSize callerSize, vmk_int32 *drvErr)
{
  vmk_LogMessage("qfle3f: ioctl is not supported.");
  return VMK_NOT_SUPPORTED;
}

struct qfle3f_rport *
qfle3f_findTargetByConnectionID(struct qfle3fHBA *hba,
								int conn_id,
								vmk_Bool checkIfTargetIsOnline)
{
	struct qfle3f_rport *target = NULL;
    VMK_ReturnStatus status;

	/* vmk_SemaLock(&hba->hbaMutex); */
    status = vmk_HashKeyFind(hba->connIdHashTable, (vmk_HashKey)&conn_id,
                                    (vmk_HashValue *)&target);
    if(status != VMK_OK) {
        qfle3f_log(hba, LOG_SESS, "Failed to find hash entry for = %d, status = %s",
                    conn_id, vmk_StatusToString(status));
        return NULL;
    }

    if(conn_id != target->fcoe_conn_id) {
        qfle3f_warning(hba, "ALERT: conn_id(%d) != target->fcoe_conn_id(%d)",
                    conn_id, target->fcoe_conn_id);
    }

	qfle3f_log(hba, LOG_INFO, "targetID = 0x%x", target->vmk_targetID);
    return target;
}

struct qfle3f_rport *
qfle3f_findTargetByTargetID(struct qfle3fHBA *hba,
								int targetID,
								vmk_Bool checkIfTargetIsOnline)
{
	struct qfle3f_rport *target = NULL;

    if(targetID >= QFLE3_FCOE_NUM_CONNECTIONS) {
        return NULL;
	}

	vmk_SemaLock(&hba->hbaMutex);

    target = hba->targetOffloadList[targetID];

    if((target) && (checkIfTargetIsOnline == VMK_TRUE)) {
        if((vmk_BitVectorTest(target->flags, QFLE3F_FLAG_OFFLOADED)) ||
	           (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_SESSION_READY))) {
	    	vmk_SemaUnlock(&hba->hbaMutex);
		    qfle3f_log(hba, LOG_INFO, "RPORT %p ONLINE", target);
    		return target;
	   	} else {
	   		qfle3f_err(hba, "rport 0x%x "
	    		   "is in DELETED state",
		    	   target->targetPortID);
	    	vmk_SemaUnlock(&hba->hbaMutex);
		    return NULL;
        }
	}

	vmk_SemaUnlock(&hba->hbaMutex);
	return target;
}

struct qfle3fFCLun *
qfle3f_getFCLun(struct qfle3fHBA *hba, struct qfle3f_rport *rport,
	int lun_id)
{
	struct qfle3fFCLun *fclun = NULL;
	vmk_ListLinks *current;

	/* find existing */
	if (vmk_ListIsEmpty(&rport->fcluns) == VMK_FALSE) {
		VMK_LIST_FORALL(&rport->fcluns, current) {
			fclun = VMK_LIST_ENTRY(current, struct qfle3fFCLun, list);
			if (!fclun)
				continue;
			if (lun_id == fclun->vmkLunId)
				return fclun;
		}
	}

	return NULL;
}

struct qfle3fFCLun *
qfle3f_allocateFCLun(struct qfle3fHBA *hba, struct qfle3f_rport *target,
	int channel, int targetID, int lun_id)
{
	struct qfle3fFCLun *fclun = NULL;

	fclun = qfle3f_alloc(sizeof(struct qfle3fFCLun));

	if (fclun) {
		fclun->target = target;
		fclun->vmkChannel = channel;
		fclun->vmkTargetId = targetID;
		fclun->vmkLunId = lun_id;
		fclun->qDepth = qfle3f_queue_depth;
		vmk_SpinlockLock(target->fclunsListLock);
		vmk_ListInsert(&fclun->list, vmk_ListAtRear(&target->fcluns));
		vmk_SpinlockUnlock(target->fclunsListLock);
		qfle3f_notice(hba, "FCLUN Allocated %d:%d:%d on RPORT %p",
			channel, targetID, lun_id, target);
	}

	return fclun;
}

const char createPathString[20] = "CREATE PATH ";
const char configPathString[20] = "CONFIGURE PATH ";
const char destroyPathString[20] = "DESTROY PATH ";
const char unknownAction[20] = "UNKNOWN ACTION ";

const char * actionToString(vmk_ScanAction action) {
    switch (action) {
        case VMK_SCSI_SCAN_CREATE_PATH: return createPathString;
        case VMK_SCSI_SCAN_CONFIGURE_PATH: return configPathString;
        case VMK_SCSI_SCAN_DESTROY_PATH: return destroyPathString;
    }
    return unknownAction;
}

static VMK_ReturnStatus
qfle3fDiscover(void *clientData, vmk_ScanAction action,
   int channel, int targetID, int lun_id, void **deviceData)
{
	struct qfle3fHBA *hba = (struct qfle3fHBA *)clientData;
	struct qfle3f_rport *target = NULL;
	struct qfle3fFCLun *fclun = NULL;

    qfle3f_log(hba, LOG_SESS, "Action: %s path %s:%d:%d:%d",
				actionToString(action),
                vmk_ScsiGetAdapterName(hba->scsiAdapter),
                channel, targetID, lun_id);

	switch (action) {
		case VMK_SCSI_SCAN_CREATE_PATH:
			target = qfle3f_findTargetByTargetID(hba, targetID,
										VMK_TRUE);

			if (!target) {
				qfle3f_notice(hba, "Target not exist %s:%d:%d:%d",
						vmk_ScsiGetAdapterName(hba->scsiAdapter), channel,
						targetID, lun_id);
				return VMK_NO_CONNECT;
			}

			qfle3f_log(hba, LOG_SESS, "Found RPORT TGT ID=%d-%d",
                        target->fcoe_conn_id, target->vmk_targetID);

			/* See if exists */
			vmk_SpinlockLock(target->fclunsListLock);
			if (qfle3f_getFCLun(hba, target, lun_id)) {
				vmk_SpinlockUnlock(target->fclunsListLock);

                qfle3f_notice(hba, "Path exists %s:%d:%d:%d",
					vmk_ScsiGetAdapterName(hba->scsiAdapter), channel,
					targetID, lun_id);
				return VMK_EXISTS;
			}
			vmk_SpinlockUnlock(target->fclunsListLock);
			target->vmk_channel = channel;

			fclun = qfle3f_allocateFCLun(hba, target, channel,
				targetID, lun_id);

			if (!fclun)
				return VMK_NO_MEMORY;

			/* if (vmk_BitVectorAtomicTestAndSet(target->localFlags, RPORT_FLG_OS_ACTIVE) == 0) { */
            ql_vmk_ref_get(&target->refcount);

			qfle3f_log(hba, LOG_SESS, "creating path %s:%d:%d:%d ref=%ld",
				vmk_ScsiGetAdapterName(hba->scsiAdapter), channel,
				targetID, lun_id, vmk_AtomicRead64(&target->refcount));

			break;

		case VMK_SCSI_SCAN_CONFIGURE_PATH:
			qfle3f_log(hba, LOG_SESS, "Keeping path %s:%d:%d:%d",
				vmk_ScsiGetAdapterName(hba->scsiAdapter), channel,
				targetID, lun_id);
			/*
			 * Nothing to do at this moment; return, don't break or we'll
			 * inadvertently overwrite device_data.
			 */
			return VMK_OK;

		case VMK_SCSI_SCAN_DESTROY_PATH:
			qfle3f_log(hba, LOG_SESS, "Destroying path %s:%d:%d:%d",
				vmk_ScsiGetAdapterName(hba->scsiAdapter), channel,
				targetID, lun_id);

			target = qfle3f_findTargetByTargetID(hba, targetID,
										VMK_FALSE);

			if (!target) {
				qfle3f_notice(hba, "Target not exist %s:%d:%d:%d",
						vmk_ScsiGetAdapterName(hba->scsiAdapter), channel,
						targetID, lun_id);
				return VMK_FAILURE;
			}

			vmk_SpinlockLock(target->fclunsListLock);
			fclun = qfle3f_getFCLun(hba, target, lun_id);
			if (fclun) {
				vmk_ListRemove(&fclun->list);
				qfle3f_free(fclun);
				fclun = NULL;
				vmk_SpinlockUnlock(target->fclunsListLock);
			}
			else {
				vmk_SpinlockUnlock(target->fclunsListLock);
				return VMK_FAILURE;
			}

            ql_vmk_ref_put(&target->refcount, qfle3f_freeTarget, target);

			break;

		default:
			VMK_ASSERT(VMK_FALSE);
			return VMK_NOT_IMPLEMENTED;
	}
	*deviceData = (void *)fclun;

	return VMK_OK;
}

static int
qfle3f_modifyQueueDepth(void *clientData, int qDepth, void *deviceData)
{
    qfle3fHBA_t *hba = (qfle3fHBA_t *)clientData;
    struct qfle3fFCLun *fclun = (struct qfle3fFCLun *)deviceData;
    VMK_ReturnStatus status;

    qfle3f_log(hba, LOG_SESS, "[%d:%d:%d], current qDepth = %d new qDepth = %d",
                    fclun->vmkChannel, fclun->vmkTargetId,
                    fclun->vmkLunId, hba->qDepth, qDepth);

    status = vmk_ScsiModifyQueueDepth(hba->scsiAdapter, fclun->vmkChannel,
        fclun->vmkTargetId, fclun->vmkLunId, qDepth);
    if (status == VMK_OK) {
        qfle3f_warning(hba, "%d:%d:%d: Queue depth adjusted to %d.",
                        fclun->vmkChannel, fclun->vmkTargetId,
                        fclun->vmkLunId, qDepth);
        fclun->qDepth = qDepth;
        return qDepth;
    }
	return 0;
}

static int
qfle3f_queryDeviceQueueDepth(void *clientData, void *deviceData)
{
	qfle3fHBA_t *hba = (qfle3fHBA_t *)clientData;
    struct qfle3fFCLun *fclun = (struct qfle3fFCLun *)deviceData;

    qfle3f_warning(hba, "qDepth = %d", fclun->qDepth);
	return fclun->qDepth;
}

static VMK_ReturnStatus
qfle3f_checkTarget(void *clientData, int channel, int targetID)
{
	struct qfle3fHBA *hba = (struct qfle3fHBA *)clientData;
	struct qfle3f_rport *target = NULL;

	target = qfle3f_findTargetByTargetID(hba, targetID, VMK_TRUE);
	if(target)
		return VMK_OK;

	return VMK_FAILURE;
}

static void
qfle3f_ScsiNotifyIOAllowed(vmk_Device device, vmk_Bool isAllowed)
{
	vmk_ScsiAdapter *scsiAdapter = NULL;
	qfle3fHBA_t *hba = NULL;
	VMK_ReturnStatus status = VMK_OK;

	status = vmk_DeviceGetRegistrationData(device,
			(vmk_AddrCookie *)&scsiAdapter);
	if ((status != VMK_OK) || (scsiAdapter == NULL)) {
		vmk_WarningMessage("Can't get scsi hba %s.",
				vmk_StatusToString(status));
		return;
	}

	hba = (qfle3fHBA_t *)scsiAdapter->clientData;
	hba->ioAllowed = isAllowed;

	qfle3f_notice(hba, "Called with isAllowed = %d.\n", isAllowed);

	if (isAllowed == VMK_FALSE) {
		if (hba->mgmtKeyAdded == VMK_TRUE) {
			vmk_MgmtDestroy(hba->kvMgmtHandle);
			hba->mgmtKeyAdded = VMK_FALSE;
		}

		vmk_BitVectorAtomicTestAndSet(hba->flags, QFLE3F_FLAG_UNLOADING);
		if (hba->type == HBA_PHYSICAL) {
			qfle3f_notice(hba, "Set flag for QFLE3F_FLAG_UNLOADING = %d.\n", isAllowed);
			vmk_AtomicWrite64(&hba->qlfcoeAdapter->unload_flags, VMK_TRUE);
		}
		return;
	}

	if((qfle3f_create_vmkMgmt_Entry == 1) && (hba->mgmtKeyAdded == VMK_FALSE)) {
		qfle3f_notice(hba, "Calling qfle3fKeyValueInit");
		qfle3fKeyValueInitPerHBA(hba);
		hba->mgmtKeyAdded = VMK_TRUE;
	}

	if (isAllowed == VMK_TRUE) {
		status = hba->cnic->cm_get_uplinkName(hba->cnic, &hba->vmnicName);
		if (status) {
			qfle3f_err(hba, "cm_get_uplinkName failed\n");
		}

		if (!hba->scsiAdapter->mgmtAdapter.t.fcoe) {
			vmk_FcoeAdapter *qlaFcoeAdapter = qfle3f_alloc(sizeof(vmk_FcoeAdapter));
			if(qlaFcoeAdapter) {
				qfle3fInitFcoeAdapter(qlaFcoeAdapter);
				hba->scsiAdapter->mgmtAdapter.t.fcoe = qlaFcoeAdapter;
			}
		}

		hba->scsiAdapter->mgmtAdapter.transport = VMK_STORAGE_ADAPTER_FCOE;

		if(!(vmk_ListIsEmpty(&hba->scsiScanList))) {
			qfle3f_log(hba, LOG_SESS, "Waking scsiScanWorldID = 0x%x", hba->scsiScanWorldID);
			vmk_AtomicWrite64(&hba->wakeupPending, VMK_TRUE);
			status = vmk_WorldForceWakeup(hba->scsiScanWorldID);
			qfle3f_log(hba, LOG_SESS, "scsiScanWorldID status = %s", vmk_StatusToString(status));
		}

	}

	qfle3f_log(hba, LOG_SESS, "Done: %d.", hba->ioAllowed);
	return;
}

#define QFLE3F_MAX_TARGET   QFLE3_FCOE_NUM_CONNECTIONS
#define QFLE3F_MAX_LUNS     0xFFFF

VMK_ReturnStatus
qfle3f_scsiAdapterSetup(struct qfle3fHBA *hba)
{
	VMK_ReturnStatus status = VMK_OK;
	vmk_DMAEngineProps dmaProps;
#if (VMWARE_ESX_DDK_VERSION >= 60000)
	vmk_ScsiAdapterCapabilities scsiadaptercap;
#endif

	hba->scsiAdapter->hostMaxSectors         = 512;
	hba->scsiAdapter->qDepthPtr              = &hba->qDepth;
	hba->scsiAdapter->flags                  = VMK_SCSI_ADAPTER_FLAG_REGISTER_WITHOUT_SCAN;
	if (hba->type == HBA_VIRTUAL) {
		hba->scsiAdapter->flags             |= VMK_SCSI_ADAPTER_FLAG_NPIV_VPORT;
	}
	hba->scsiAdapter->command                = (vmk_ScsiAdapterCommand)qfle3f_scsiCommand;
	hba->scsiAdapter->taskMgmt               = (vmk_ScsiAdapterTaskMgmt)qfle3f_scsiTaskMgmt;
	hba->scsiAdapter->dumpCommand            = (vmk_ScsiAdapterDumpCommand)qfle3f_scsiDumpCommand;
	hba->scsiAdapter->procInfo               = qfle3f_procInfo;
	hba->scsiAdapter->dumpQueue              = (vmk_ScsiAdapterDumpQueue)qfle3f_scsiDumpQueue;
	hba->scsiAdapter->dumpPollHandler        = (vmk_ScsiAdapterDumpPollHandler)qfle3f_scsiDumpPollHandler;
	hba->scsiAdapter->dumpPollHandlerData    = hba;
	hba->scsiAdapter->ioctl                  = qfle3f_ioctl;
	hba->scsiAdapter->discover               = qfle3fDiscover;
#if (VMWARE_ESX_DDK_VERSION <= 65000)
	hba->scsiAdapter->vportop                = NULL;
	hba->scsiAdapter->vportDiscover          = NULL;
#endif
	hba->scsiAdapter->modifyDeviceQueueDepth = qfle3f_modifyQueueDepth;
	hba->scsiAdapter->queryDeviceQueueDepth  = qfle3f_queryDeviceQueueDepth;
	hba->scsiAdapter->checkTarget            = qfle3f_checkTarget;
	hba->scsiAdapter->targetId               = -1;
	hba->scsiAdapter->moduleID               = vmk_ModuleCurrentID;
	hba->scsiAdapter->clientData             = hba;
	hba->scsiAdapter->channels               = 1;
	hba->scsiAdapter->maxTargets             = QFLE3F_MAX_TARGET;
	hba->scsiAdapter->maxLUNs                = QFLE3F_MAX_LUNS;
	hba->scsiAdapter->paeCapable             = VMK_TRUE;
	hba->scsiAdapter->maxCmdLen              = QFLE3F_MAX_CMD_LEN;


#if (VMWARE_ESX_DDK_VERSION >= 60000)
	/*
	 * Set second level addressing capability support. If there is a failure
	 * just issue a warning as we do not want to disable the complete adapter.
	 */
	scsiadaptercap = VMK_SCSI_ADAPTER_CAP_SECONDLEVEL_ADDRESSING;

	status = vmk_ScsiAdapterSetCapabilities(hba->scsiAdapter, scsiadaptercap);
	if (status != VMK_OK) {
		qfle3f_warning(hba, "Unable to set adapter capabilities. Reason: %s",
				vmk_StatusToString(status));
		status = VMK_OK;
	}
#endif

	/* Attaching scsi adapter with the physical device */
	hba->scsiAdapter->device = hba->nicDevice;

	hba->scsiAdapter->notifyIOAllowed = qfle3f_ScsiNotifyIOAllowed;

	/* NPIV: Share same dma-engine as of phba. */
	if (hba->type == HBA_VIRTUAL) {
		hba->ioDMAEngine = hba->phba->ioDMAEngine;
	} else {
		/* Create a DMA engine for mapping DMA buffers. */
		status = vmk_NameFormat(&dmaProps.name, "qfle3f_DMAEngine-%d", hba->instance);
		dmaProps.module = vmk_ModuleCurrentID;
		dmaProps.flags = VMK_DMA_ENGINE_FLAGS_NONE;
		dmaProps.device = hba->nicDevice;
		dmaProps.constraints = &qfle3f_dmaConstraint;
		dmaProps.bounce = NULL;
		status = vmk_DMAEngineCreate(&dmaProps, &hba->ioDMAEngine);
		if (status != VMK_OK) {
			vmk_WarningMessage("Failed to create DMA engine: %s",
					vmk_StatusToString(status));
			return VMK_FAILURE;
		}
	}

	vmk_NameInitialize(&hba->scsiAdapter->driverName, QFLE3F_DRIVER_NAME);
	hba->scsiAdapter->engine = hba->ioDMAEngine;

	vmk_LogMessage("qfle3f: scsi hba setup is done.");
	return status;
}

void
qfle3f_scsiAdapterDestroy(struct qfle3fHBA *hba)
{
    VMK_ReturnStatus status;

    status = vmk_DMAEngineDestroy(hba->ioDMAEngine);
    vmk_LogMessage("qfle3f: scsi hba destroy is done: %s.",
                    vmk_StatusToString(status));
}
