/*
 * Copyright (C) 2005-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* This is an implementation of the Intel UHCI. */

#define DEBUG_CONTROL_FLOW	0

#define USB_NUM_PORTS	2		/* UHCI root hub has 2 ports */

#define USB_SENDBUFFER_SIZE 0x500

#ifdef STATE

struct {
	/*
	 * Config Space
	 */
	uint8_t io_enable;
	uint8_t master_enable;
	uint8_t master_latency_timer;
	uint32_t base; 
	uint8_t interrupt_line;

	/*
	 * Process
	 */
	struct process process;

	/*
	 * I/O Space
	 */
	/* USB Command Register (16 Bit, R/W, WORD writeable only) */
	uint16_t usbcmd;

	/* USB Status (16 Bit, R/WC) */
	uint16_t usbsts;

	/* USB Interrupt Enable (16 Bit, R/W) */
	uint16_t usbintr;

	/* Frame Number (16 Bit, R/W, WORD writeable only, 15:11 reserved) */
	uint16_t frnum;

	/* Frame List Base Address (32 Bit, R/W, 11:0 reserved) */
	uint32_t flbaseadd;

	/* Start Of Frame Modify (8 Bit, R/W, 7 reserved) */
	uint8_t sofmod;

	/* Port Status And Control (16 Bit, R/W, WORD writeable only) */
	uint16_t portsc[USB_NUM_PORTS];

	unsigned char global_reset;	/* currently in global reset mode? */

	struct {
		int speed;
		char reset;
	} portstate[USB_NUM_PORTS];

	/* schedule state */
	char working;	/* currently working on a schedule */
	char cont;

	char timeout_interrupt;	/* trigger interrupt on timeout/crc error */
	char ioc;	/* trigger interrupt on completion at frame end */
	char spd;	/* trigger interrupt on short packet detection at frame end */
	char babble_interrupt, stalled_interrupt, data_buffer_error_interrupt,
	     bit_stuff_error_interrupt, hc_process_error_interrupt,
	     host_system_error_interrupt;	/* non-maskable interrupts */

	/* schedule list traversal state, cf. UHCI, Revision 1.1, chapter 3.4.2 "Transfer Queueing" */
	/* are we in a queue context? */
	char q_context;

	/* Pointer to QH */
	uint32_t qhp;

	/* QH */
	uint32_t qh[2];

	/* Pointer to TD */
	uint32_t tdp;

	/* TD */
	uint32_t td[4];	/* content to avoid re-read */
	unsigned char td_c_err, td_actlen, td_endpt, td_addr, td_pid;
	unsigned int td_maxlen;

	unsigned tds_sent;

	unsigned char sendbuffer[USB_SENDBUFFER_SIZE];

	/*
	 * count queue headers: bandwidth reclamation leads to the last QH
	 * pointing back in the queue, in effect creating an endless loop; real
	 * hardware terminates the frame if the 1ms time frame is over, we try
	 * to emulate this by counting QHs.
	 */
	unsigned long qh_count;
} NAME;

#endif /* STATE */
#ifdef BEHAVIOR

#include "std-pci.h"
#include "lib_usb.h" /* FIXME */

/*----------------------------------------------------------------------------*/

/* The following definitions are adopted from the chipset manual */

/* Possible values for USBCMD */
#define USBCMD_RESERVED			0xFF00
#define USBCMD_MAX_PACKET		(1<<7)
#define USBCMD_CONFIGURE_FLAG		(1<<6)
#define USBCMD_SOFTWARE_DEBUG		(1<<5)
#define USBCMD_FORCE_GLOBAL_RESUME	(1<<4)
#define USBCMD_ENTER_GLOBAL_SUSPEND	(1<<3)
#define USBCMD_GLOBAL_RESET		(1<<2)
#define USBCMD_HOST_CONTROLLER_RESET	(1<<1)
#define USBCMD_RUN_STOP			(1<<0)

/* Possible values for USBSTS */
#define USBSTS_RESERVED			0xFFC0
#define USBSTS_HC_HALTED		(1<<5)
#define USBSTS_HC_PROCESS_ERROR		(1<<4)
#define USBSTS_HOST_SYSTEM_ERROR	(1<<3)
#define USBSTS_RESUME_DETECT		(1<<2)
#define USBSTS_ERROR_INTERRUPT		(1<<1)
#define USBSTS_INTERRUPT		(1<<0)

/* Possible values for USBINTR */
#define USBINTR_RESERVED		0xFFF0
#define USBINTR_SPIE			(1<<3)
#define USBINTR_IOC			(1<<2)
#define USBINTR_RIE			(1<<1)
#define USBINTR_TIE			(1<<0)

/* Possible values for FRNUM */
#define FRNUM_RESERVED			0xF800
#define FRNUM_FRNUM			(~FRNUM_RESERVED)
#define FRNUM_FL_INDEX			0x03FF

/* Possible values for FLBASEADD */
#define FLBASEADD_FLBASEADD		0xFFFFF000UL
#define FLBASEADD_RESERVED		(~FLBASEADD_FLBASEADD)

/* Possible values for SOFMOD */
#define SOFMOD_RESERVED			(1<<7)
#define SOFMOD_SOF_TIMING		0x7F

/* Possible values for PORTSC0/1 */
#define PORTSC_RESERVED_CLEAR		0xE000	/* write as 0 */
#define PORTSC_SUSPEND			(1<<12)	/* R/W */
#define PORTSC_OVERCURRENT_INDICATOR_CHANGE	(1<<11)	/* R/WC */
#define PORTSC_OVERCURRENT_INDICATOR	(1<<10)	/* RO */
#define PORTSC_PORT_RESET		(1<<9)	/* R/W */
#define PORTSC_LOWSPEED_DEVICE_ATTACHED	(1<<8)	/* RO */
#define PORTSC_RESERVED_SET		(1<<7)	/* RO, read as 1 */
#define PORTSC_RESUME_DETECT		(1<<6)	/* R/W */
#define PORTSC_LINE_STATUS_DMINUS	(1<<5)	/* RO */
#define PORTSC_LINE_STATUS_DPLUS	(1<<4)	/* RO */
#define PORTSC_PORT_ENABLE_DISABLE_CHANGE	(1<<3)	/* R/WC */
#define PORTSC_PORT_ENABLE		(1<<2)	/* R/W */
#define PORTSC_CONNECT_STATUS_CHANGE	(1<<1)	/* R/WC */
#define PORTSC_CURRENT_CONNECT_STATUS	(1<<0)	/* RO */

#define SZ_USB_IOSPACE		32	/* I/O space size */

/* in-memory data structures (cf. UHCI, Rev. 1.1, Section 3 "Data Structures") */

/* Transfer Descriptor */
/* TD Control and Status */
#define USB_MEM_TD_1_RESERVED	0xC001F800UL	/* bits 31,30,16,15:11 */
#define USB_MEM_TD_1_SPD	(1<<29)		/* Short Packet Detect (SPD) */
#define USB_MEM_TD_1_C_ERR	(1<<28|1<<27)	/* Error Counter (ERR_C) */
#define USB_MEM_TD_1_C_ERR_SHIFT	27
#define USB_MEM_TD_1_LS		(1<<26)		/* Low Speed Device (LS) */
#define USB_MEM_TD_1_IOS	(1<<25)		/* Isochronous Select (IOS) */
#define USB_MEM_TD_1_IOC	(1<<24)		/* Interrupt on Complete (IOC) */
#define USB_MEM_TD_1_STATUS	0x00FF0000	/* Status */
#define USB_MEM_TD_1_STATUS_ACTIVE		(1<<23)	/* Status Active */
#define USB_MEM_TD_1_STATUS_STALLED		(1<<22)	/* Status Stalled */
#define USB_MEM_TD_1_STATUS_DATA_BUFFER_ERROR	(1<<21)	/* Status Data Buffer Error */
#define USB_MEM_TD_1_STATUS_BABBLE_DETECTED	(1<<20)	/* Status Babble Detected */
#define USB_MEM_TD_1_STATUS_NAK_RECEIVED	(1<<19)	/* Status NAK Received */
#define USB_MEM_TD_1_STATUS_CRC_TIMEOUT_ERROR	(1<<18)	/* Status CRC/Time Out Error */
#define USB_MEM_TD_1_STATUS_BITSTUFF_ERROR	(1<<17)	/* Status Bitstuff Error */
#define USB_MEM_TD_1_ACTLEN		0x000007FF	/* Actual Length (ActLen) */
/* TD Token */
#define USB_MEM_TD_2_MAXLEN		0xFFE00000	/* Maximum Length (MaxLen) (31:21) */
#define USB_MEM_TD_2_MAXLEN_SHIFT	21
#define USB_MEM_TD_2_MAXLEN_WRAP	0x7FF		/* wrap values at this maximum value */
#define USB_MEM_TD_2_MAXLEN_ILLEGAL	0x500		/* everything >= 0x500 is illegal */
#define USB_MEM_TD_2_RESERVED	(1<<20)		/* Reserved */
#define USB_MEM_TD_2_D		(1<<19)		/* Data Toggle (D) */
#define USB_MEM_TD_2_ENDPT	0x00078000UL	/* Endpoint (EndPt) (18:15) */
#define USB_MEM_TD_2_ENDPT_SHIFT	15
#define USB_MEM_TD_2_DEV_ADDR	0x00007F00UL	/* Device Address (14:8) */
#define USB_MEM_TD_2_DEV_ADDR_SHIFT	8
#define USB_MEM_TD_2_PID	0x000000FFUL	/* Packet Identification (PID) (7:0) */
/* TD Buffer Pointer -- extends over whole DWORD 3 */
/* DWORDS 4-7 are reserved for use by software */

/* parameter 2 of NAME_(advance_queue)() */
#define USB_QUEUE_NO_ADVANCE	0
#define USB_QUEUE_ADVANCE	1

/* see arch_usb_controller.h -> qh_count */
#define USB_MAX_QH_PER_FRAME	30

/* forward declarations */
static void
NAME_(connect_device)(struct cpssp *cpssp, int usb_portnr);
static void 
NAME_(packet_timeout)(struct cpssp *cpssp);

static void
NAME_(set_irq)(struct cpssp *cpssp, int val)
{
	NAME_(irqrq_out_set)(cpssp, val);
}

static void
NAME_(soft_reset)(struct cpssp *cpssp)
{
	int i;

	/* Initialize I/O Space Registers */
	cpssp->NAME.usbcmd		= 0x0;
	cpssp->NAME.usbsts		= 0x0;
	cpssp->NAME.usbintr		= 0x0;
	cpssp->NAME.frnum		= 0x0;
	cpssp->NAME.flbaseadd		= 0x0;
	cpssp->NAME.sofmod		= 0x40;
	for (i = 0; i < USB_NUM_PORTS; i++) {
		cpssp->NAME.portsc[i]	= 0x00;
	}

	cpssp->NAME.global_reset = 0;
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	/* Initialize Config Space Registers */
	cpssp->NAME.io_enable = 0;
	cpssp->NAME.master_enable = 0;
	cpssp->NAME.master_latency_timer = 0x00;
	cpssp->NAME.base = 0x00000000;
	cpssp->NAME.interrupt_line = 0;

	NAME_(soft_reset)(cpssp);
}

/*----------------------------I/O interface-----------------------------------*/

static int
NAME_(ior)(struct cpssp *cpssp, uint32_t port, unsigned int bs, uint32_t *valp)
{
	if (! cpssp->NAME.io_enable
	 || cpssp->NAME.base != (port & ~0x1f)) {
		return 1;
	}

	port &= 0x1f;

	switch (port) {
	case 0x00: /* USBCMD/USBSTS */
		*valp = (cpssp->NAME.usbcmd << 0)
			| (cpssp->NAME.usbsts << 16);
		break;

	case 0x04: /* USBINTR/FRNUM */
		*valp = (cpssp->NAME.usbintr << 0)
			| (cpssp->NAME.frnum << 16);
		break;

	case 0x08: /* FLBASEADD */
		*valp = cpssp->NAME.flbaseadd;
		break;

	case 0x0c: /* SOFMOD */
		*valp = cpssp->NAME.sofmod;
		break;

	case 0x10: /* PORTSC0/1 */
		*valp = ((cpssp->NAME.portsc[0] | 0x80) << 0)
			| ((cpssp->NAME.portsc[1] | 0x80) << 16);
		break;

	default:
		*valp = 0xff7f;	/* bit 7 = 0 makes linux kernel stop scanning for more ports */
		break;
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s port=0x%04x bs=0x%x *valp=0x%08x\n",
				__FUNCTION__, port, bs, *valp);
	}

	return 0;
}

static int
NAME_(iow)(struct cpssp *cpssp, uint32_t port, unsigned int bs, uint32_t val)
{
	int i;

	if (! cpssp->NAME.io_enable
	 || cpssp->NAME.base != (port & ~0x1f)) {
		return 1;
	}

	port &= 0x1f;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s port=0x%04x bs=0x%x val=0x%08x\n",
				__FUNCTION__, port, bs, val);
	}

	switch (port) {
	case 0x00: /* USBCMD/USBSTS */
		/*
		 * USBSTS
		 */
		if ((bs >> 3) & 1) {
			/* Bit 15-8: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Bit 7-6: Reserved */

			cpssp->NAME.usbsts &= ~(val >> 16);

			/* FIXME: Linux Kernel sources tell this:
			 * 'Contrary to the UHCI specification, the "HC Halted" status
			 * bit is persistent: it is RO, not R/WC.'
			 */

			/* are all interrupt reasons cleared? */
			if ((cpssp->NAME.usbsts & 0x1F) == 0) {
				NAME_(set_irq)(cpssp, 0);
			}
		}

		/*
		 * USBCMD
		 */
		if ((bs >> 1) & 1) {
			/* 15-8: Reserved */
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.usbcmd = val;

			/* HCReset */
			if (val & USBCMD_HOST_CONTROLLER_RESET) {
				/* this implicitly clears the HCReset bit */
				NAME_(soft_reset)(cpssp);

				/* TODO: terminate transactions in progress? */

				/* connect status change  and  port enable/disable
				 * change  of every port is set now */
				for (i = 0; i < USB_NUM_PORTS; i++) {
					cpssp->NAME.portsc[i] |=
							PORTSC_PORT_ENABLE_DISABLE_CHANGE
							| PORTSC_CONNECT_STATUS_CHANGE;
					/* FIXME: reconnect devices "after 64 bit-times"? */
					NAME_(connect_device)(cpssp, i);
				}
			}

			/* GReset... */
			if (val & USBCMD_GLOBAL_RESET) {		/* ... was SET */
				cpssp->NAME.global_reset = 1;
			} else if (cpssp->NAME.global_reset) {	/* ... was CLEARED */
				cpssp->NAME.global_reset = 0;
				/* reset after the guaranteed 10ms minimum global_reset delay */
				NAME_(soft_reset)(cpssp);

				/* reset everyone downstream */
				for (i = 0; i < USB_NUM_PORTS; i++) {
					if (i == 0) {
						NAME_(0_reset_set)(cpssp, 1);
					} else { assert(i == 1);
						NAME_(1_reset_set)(cpssp, 1);
					}
					NAME_(connect_device)(cpssp, i);
				}
			}

			/* Run/Stop */
			if (val & USBCMD_RUN_STOP) {			/* run */
				/* clear corresponding status bit */
				cpssp->NAME.usbsts &= ~USBSTS_HC_HALTED;
				/* TODO: maybe immediately start a new frame?
				 * problem: locking... */
			} else {					/* stop */
				/* set corresponding status bit */
				cpssp->NAME.usbsts |= USBSTS_HC_HALTED;

				if (0 < cpssp->NAME.tds_sent) {
					NAME_(packet_timeout)(cpssp);
					cpssp->NAME.tds_sent = 0;
				}
			}
		}
		break;

	case 0x04: /* USBINTR/FRNUM */
		/*
		 * FRNUM
		 */
		/*
		 * "This register cannot be written unless the Host Controller
		 * is in the STOPPED state as indicated by the HCHalted bit
		 * (USBSTS register). A write to this register while the
		 * Run/Stop bit is set (USBCMD register) is ignored."
		 */
		if (! (cpssp->NAME.usbcmd & USBCMD_RUN_STOP)
		 && cpssp->NAME.usbsts & USBSTS_HC_HALTED) {
			if ((bs >> 3) & 1) {
				/* Bit 15-11: Reserved */
				cpssp->NAME.frnum &= 0x00ff;
				cpssp->NAME.frnum |= (val >> 16) & 0x0700;
			}
			if ((bs >> 2) & 1) {
				cpssp->NAME.frnum &= 0xff00;
				cpssp->NAME.frnum |= (val >> 16) & 0x00ff;
			}
		}

		/*
		 * USBINTR
		 */
		if ((bs >> 1) & 1) {
			/* Bit 15-8: Reserved */
		}
		if ((bs >> 0) & 1) {
			/* Bit 7-4: Reserved */
			cpssp->NAME.usbintr = (val >> 0) & 0xf;
			/* Update IRQ? FIXME */
		}
		break;

	case 0x08: /* FLBASEADD */
		if ((bs >> 3) & 1) {
			cpssp->NAME.flbaseadd &= 0x00fff000;
			cpssp->NAME.flbaseadd |= (val & 0xff000000);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.flbaseadd &= 0xff00f000;
			cpssp->NAME.flbaseadd |= (val & 0x00ff0000);
		}
		if ((bs >> 1) & 1) {
			/* Bit 11-8: Hardwired to 0 */
			cpssp->NAME.flbaseadd &= 0xffff0000;
			cpssp->NAME.flbaseadd |= (val & 0x0000f000);
		}
		if ((bs >> 0) & 1) {
			/* Bit 7-0: Hardwired to 0 */
		}

	case 0x0c: /* SOFMOD */
		if ((bs >> 3) & 1) {
			/* Bit 15-8: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Bit 7-0: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Bit 15-8: Reserved */
		}
		if ((bs >> 0) & 1) {
			/* Bit 7: Reserved */
			cpssp->NAME.sofmod = val & 0x7f;
		}
		break;

	case 0x10: /* PORTSC0/PORTSC1 */
		for (i = 0; i < 2; i++) {
			uint16_t new;
			uint16_t mask;

			if ((bs >> 1) & 1) {
				/* Bit 15-13: Reserved */
				new = cpssp->NAME.portsc[i];

				/* R/W Bits */
				mask = PORTSC_SUSPEND | PORTSC_PORT_RESET;
				new &= ~mask;
				new |= val & mask;

				/* R/WC Bits */
				mask = PORTSC_OVERCURRENT_INDICATOR_CHANGE;
				new &= ~(val & mask);

				cpssp->NAME.portsc[i] = new;

				if (new & PORTSC_PORT_RESET) {
					/* Reset set. */
					cpssp->NAME.portstate[i].reset = 1;
					cpssp->NAME.portsc[i]
						&= ~(PORTSC_LOWSPEED_DEVICE_ATTACHED
						| PORTSC_LINE_STATUS_DMINUS
						| PORTSC_LINE_STATUS_DPLUS
						| PORTSC_PORT_ENABLE_DISABLE_CHANGE
						| PORTSC_PORT_ENABLE
						| PORTSC_CONNECT_STATUS_CHANGE
						| PORTSC_CURRENT_CONNECT_STATUS);

				} else if (cpssp->NAME.portstate[i].reset) {
					/* Reset cleared. */
					cpssp->NAME.portstate[i].reset = 0;

					/* reset everyone on this port */
					if (i == 0) {
						NAME_(0_reset_set)(cpssp, 1);
					} else { assert(i == 1);
						NAME_(1_reset_set)(cpssp, 1);
					}
					NAME_(connect_device)(cpssp, i);
				}
			}
			if ((bs >> 0) & 1) {
				new = cpssp->NAME.portsc[i];

				/* R/W Bits */
				mask = PORTSC_RESUME_DETECT | PORTSC_PORT_ENABLE;
				new &= ~mask;
				new |= val & mask;

				/* R/WC Bits */
				mask = PORTSC_PORT_ENABLE_DISABLE_CHANGE | PORTSC_CONNECT_STATUS_CHANGE;
				new &= ~(val & mask);

				cpssp->NAME.portsc[i] = new;
			}

			val >>= 16;
			bs >>= 2;
		}
		break;

	case 0x14:
	case 0x18:
	case 0x1c:
		/* Reserved */
		break;

	default:
		assert(0); /* Mustn't happen. */
	}

	return 0;
}

static int
NAME_(ior_info)(
	struct cpssp *cpssp,
	uint32_t port,
	unsigned int bs,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t *),
	void **csp
)
{
	return 1; /* FIXME */
}

static int
NAME_(iow_info)(
	struct cpssp *cpssp,
	uint32_t port,
	unsigned int bs,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t),
	void **csp
)
{
	return 1; /* FIXME */
}

/* ---------------------Configuration Interface (PCI bus)---------------*/

static void
NAME_(c0r)(struct cpssp *cpssp, uint8_t addr, unsigned int bs, uint32_t *valp)
{
	*valp = 0x00000000;
	switch (addr) {
	case 0x00:
		/* Vendor Identification Register */
		/* 6.1.1 */
		*valp |= 0x8086 << 0;

		/* Device Identification Register */
		/* 6.1.2 */
		*valp |= DEVICE_ID << 16;
		break;

	case 0x04:
		/* PCI Command Register */
		/* 6.1.3 */
		*valp |= cpssp->NAME.io_enable << 0;
		*valp |= 0b0 << 1; /* Memory Space Enable */
		*valp |= cpssp->NAME.master_enable << 2;
		*valp |= 0b0 << 3; /* Special Cycle Enable */
		*valp |= 0b0 << 4; /* Memory Write and Invalidate Enable */
		*valp |= 0b0000 << 5; /* Reserved */
		*valp |= 0b0 << 9; /* Fast Back to Back Enable */
		*valp |= 0b000000 << 10; /* Reserved */

		/* PCI Status Register */
		/* 6.1.4 */
		*valp |= 0b0000000 << (0 + 16); /* Reserved */
		*valp |= 0b1 << (7 + 16); /* Fast Back to Back Capable */
		*valp |= 0b0 << (8 + 16); /* Data Parity Detected */
		*valp |= 0b01 << (9 + 16); /* DEVSEL Timing Status */
		*valp |= 0b0 << (11 + 16); /* Signaled Target-Abort Status */
		*valp |= 0b0 << (12 + 16); /* Received Target-Abort Status */
		*valp |= 0b0 << (13 + 16); /* Master-Abort Status */
		*valp |= 0b0 << (14 + 16); /* SERR# Status */
		*valp |= 0b0 << (15 + 16); /* Detected Parity */
		break;

	case 0x08:
		/* Revision Identification Register */
		/* 6.1.5 */
		*valp |= REVISION_ID << 0;

		/* Class Code Register */
		/* 6.1.6 */
		*valp |= 0x00 << (0 + 8); /* UHCI */
		*valp |= 0x03 << (8 + 8); /* USB Controller */
		*valp |= 0x0c << (16 + 8); /* Serial Bus Controller */
		break;

	case 0x0c:
		*valp |= 0x00 << 0; /* Reserved */

		/* Master Latency Timer Register */
		/* 6.1.7 */
		*valp |= cpssp->NAME.master_latency_timer << 8;

		/* Header Type Register */
		/* 6.1.8 */
		*valp |= 0b0000000 << (0 + 16);
		*valp |= MULTIFUNC << (7 + 16);

		*valp |= 0x00 << 24; /* Reserved */
		break;

	case 0x10:
	case 0x14:
	case 0x18:
	case 0x1c:
		goto reserved;

	case 0x20:
		/* USB I/O Space Base Address Register */
		/* 6.1.13 */
		*valp |= 0b1 << 0;
		*valp |= 0b0000 << 1; /* Reserved */
		*valp |= cpssp->NAME.base << (5-5);

		*valp |= 0x0000 << 16; /* Reserved */
		break;

	case 0x24:
	case 0x28:
	case 0x2c:
	case 0x30:
	case 0x34:
	case 0x38:
		goto reserved;

	case 0x3c:
		/* Interrupt Line Register */
		/* 6.1.9 */
		*valp |= cpssp->NAME.interrupt_line << 0;

		/* Interrupt Pin Register */
		/* 6.1.10 */
		*valp |= INTERRUPT_PIN << 8;

		*valp |= 0x0000 << 16; /* Reserved */
		break;

	case 0x40:
	case 0x44:
	case 0x48:
	case 0x4c:
	case 0x50:
	case 0x54:
	case 0x58:
	case 0x5c:
		goto reserved;

	case 0x60:
		/* Serial Bus Release Number Register */
		/* 6.1.11 */
		*valp |= 0x10 << 0; /* Version 1.0 */

		*valp |= 0x000000 << 8; /* Reserved */
		break;

	case 0x64: case 0x68: case 0x6c:
	case 0x70: case 0x74: case 0x78: case 0x7c:
	case 0x80: case 0x84: case 0x88: case 0x8c:
	case 0x90: case 0x94: case 0x98: case 0x9c:
	case 0xa0: case 0xa4: case 0xa8: case 0xac:
	case 0xb0: case 0xb4: case 0xb8: case 0xbc:
		goto reserved;

	case 0xc0:
		/* Legacy Support Register */
		/* 6.1.12 */
		/* FIXME */
		fprintf(stderr, "WARNING: %s: reading legacy support register.\n",
				__FUNCTION__);

		*valp |= 0x0000 << 16; /* Reserved */
		break;

	case 0xc4: case 0xc8: case 0xcc:
	case 0xd0: case 0xd4: case 0xd8: case 0xdc:
	case 0xe0: case 0xe4: case 0xe8: case 0xec:
	case 0xf0: case 0xf4: case 0xf8: case 0xfc:
		/* MISCSUP Register must be handled elsewhere! */
		goto reserved;

	reserved:
		*valp |= 0x00000000 << 0; /* Reserved */
		break;

	default:
		assert(0); /* Cannot happen. */
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s addr=0x%08x bs=0x%x *valp=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(c0w)(struct cpssp *cpssp, uint8_t addr, unsigned int bs, uint32_t val)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s addr=0x%08x bs=0x%x vap=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x00:
		/* Vendor Identification Register */
		/* 6.1.1 */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}
		if ((bs >> 1) & 1) {
			/* Read-only */
		}

		/* Device Identification Register */
		/* 6.1.2 */
		if ((bs >> 2) & 1) {
			/* Read-only */
		}
		if ((bs >> 3) & 1) {
			/* Read-only */
		}
		break;

	case 0x04:
		/* PCI Command Register */
		/* 6.1.3 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.io_enable = (val >> 0) & 1;
			/* Bit 1: Hardwired to '0'. */
			cpssp->NAME.master_enable = (val >> 2) & 1;
			/* Bit 3-7: Hardwired to '0'. */
		}
		if ((bs >> 1) & 1) {
			/* Hardwired to '0'. */
		}

		/* PCI Status Register */
		/* 6.1.4 */
		if ((bs >> 2) & 1) {
			/* Read-only */
		}
		if ((bs >> 3) & 1) {
			/* Read-only */
		}
		break;

	case 0x08:
		/* Revision Identification Register */
		/* 6.1.5 */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}

		/* Class Code Register */
		/* 6.1.6 */
		if ((bs >> 1) & 1) {
			/* Read-only */
		}
		if ((bs >> 2) & 1) {
			/* Read-only */
		}
		if ((bs >> 3) & 1) {
			/* Read-only */
		}
		break;

	case 0x0c:
		if ((bs >> 0) & 1) {
			/* Reserved */
		}

		/* Master Latency Timer Register */
		/* 6.1.7 */
		if ((bs >> 1) & 1) {
			cpssp->NAME.master_latency_timer = (val >> 8) & 0xf0;
		}

		/* Header Type Register */
		/* 6.1.8 */
		if ((bs >> 2) & 1) {
			/* Read-only */
		}

		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x10:
	case 0x14:
	case 0x18:
	case 0x1c:
		goto reserved;

	case 0x20:
		/* USB I/O Space Base Address Register */
		/* 6.1.13 */
		if ((bs >> 0) & 1) {
			/* Bit 0-4: Read-only */
			cpssp->NAME.base &= ~(0xe0 << 0);
			cpssp->NAME.base |= val & (0xe0 << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.base &= ~(0xff << 8);
			cpssp->NAME.base |= val & (0xff << 8);
		}

		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x24:
	case 0x28:
	case 0x2c:
	case 0x30:
	case 0x34:
	case 0x38:
		goto reserved;

	case 0x3c:
		/* Interrupt Line Register */
		/* 6.1.9 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.interrupt_line = (val >> 0) & 0xff;
		}

		/* Interrupt Pin Register */
		/* 6.1.10 */
		if ((bs >> 1) & 1) {
			/* Read-only */
		}

		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x40: case 0x44: case 0x48: case 0x4c:
	case 0x50: case 0x54: case 0x58: case 0x5c:
		goto reserved;

	case 0x60:
		/* Serial Bus Release Number Register */
		/* 6.1.11 */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}

		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x64: case 0x68: case 0x6c:
	case 0x70: case 0x74: case 0x78: case 0x7c:
	case 0x80: case 0x84: case 0x88: case 0x8c:
	case 0x90: case 0x94: case 0x98: case 0x9c:
	case 0xa0: case 0xa4: case 0xa8: case 0xac:
	case 0xb0: case 0xb4: case 0xb8: case 0xbc:
		goto reserved;

	case 0xc0:
		/* Legacy Support Register */
		/* 6.1.12 */
		fprintf(stderr, "WARNING: %s: writing legacy support register.\n",
				__FUNCTION__);
		if ((bs >> 0) & 1) {
			/* FIXME */
		}
		if ((bs >> 1) & 1) {
			/* FIXME */
		}

		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0xc4: case 0xc8: case 0xcc:
	case 0xd0: case 0xd4: case 0xd8: case 0xdc:
	case 0xe0: case 0xe4: case 0xe8: case 0xec:
	case 0xf0: case 0xf4: case 0xf8: case 0xfc:
		/* MISCSUP Register must be handled elsewhere! */
		goto reserved;

	reserved:
		if ((bs >> 0) & 1) {
			/* Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	default:
		assert(0); /* Cannot happen. */
	}
}

/*----------------------------Auxiliary Functions-----------------------------*/

/* set I/O registers accoording to device attached or not */
static void
NAME_(connect_device)(struct cpssp *cpssp, int usb_portnr)
{
	int current_connect_status = cpssp->NAME.portsc[usb_portnr] & PORTSC_CURRENT_CONNECT_STATUS;
	char signal_global_resume = 0;

	if (cpssp->NAME.portstate[usb_portnr].speed != USB_SPEED_UNCONNECTED) {
		if (! current_connect_status) {
			cpssp->NAME.portsc[usb_portnr] |=
				  PORTSC_CONNECT_STATUS_CHANGE
				| PORTSC_CURRENT_CONNECT_STATUS;
			signal_global_resume = 1;
		}

		if (cpssp->NAME.portstate[usb_portnr].speed == USB_SPEED_FULL) { /* full speed device */
			/* cf. USB Spec. 1.1, chapter 7.1.5, "Device Speed Identification"  */
			cpssp->NAME.portsc[usb_portnr] |=
				  PORTSC_LINE_STATUS_DPLUS;
			cpssp->NAME.portsc[usb_portnr] &=
				~(PORTSC_LINE_STATUS_DMINUS
				| PORTSC_LOWSPEED_DEVICE_ATTACHED);
		} else { /* low speed device */
			cpssp->NAME.portsc[usb_portnr] |=
				  PORTSC_LOWSPEED_DEVICE_ATTACHED
				| PORTSC_LINE_STATUS_DMINUS;
			cpssp->NAME.portsc[usb_portnr] &=
				~PORTSC_LINE_STATUS_DPLUS;
		}
	} else { /* no device attached */
		if (current_connect_status) {
			cpssp->NAME.portsc[usb_portnr] |=
				PORTSC_CONNECT_STATUS_CHANGE;
			signal_global_resume = 1;
		}
		cpssp->NAME.portsc[usb_portnr] &=
			~(PORTSC_LOWSPEED_DEVICE_ATTACHED
			| PORTSC_LINE_STATUS_DMINUS
			| PORTSC_LINE_STATUS_DPLUS
			| PORTSC_PORT_ENABLE_DISABLE_CHANGE
			| PORTSC_PORT_ENABLE
			| PORTSC_CURRENT_CONNECT_STATUS);
	}

	/* if Resume Interrupt Enabled and in Global Suspend Mode
	 * and bits 1, 3 or 6 of PORTSC were set, signal interrupt */
	if (cpssp->NAME.usbcmd & USBCMD_ENTER_GLOBAL_SUSPEND
	 && signal_global_resume) {
		cpssp->NAME.usbcmd |= USBCMD_FORCE_GLOBAL_RESUME;
		cpssp->NAME.usbsts |= USBSTS_RESUME_DETECT;

		if (cpssp->NAME.usbintr & USBINTR_RIE) {
			NAME_(set_irq)(cpssp, 1);
		}
	}
}

/* send a USB token packet over CIM */
static void
NAME_(send_usb_token)(struct cpssp *cpssp, unsigned char pid, unsigned char addr, unsigned char endp)
{
	int usb_portnr;

	for (usb_portnr = 0; usb_portnr < USB_NUM_PORTS; usb_portnr++) {
		/* port disabled or in selective suspend mode? */
		if (! (cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE)
		 || (cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE
		  && cpssp->NAME.portsc[usb_portnr] & PORTSC_SUSPEND)) {
			continue;
		}

		if (usb_portnr == 0) {
			NAME_(0_send_token)(cpssp, pid, addr, endp);
		} else { assert(usb_portnr == 1);
			NAME_(1_send_token)(cpssp, pid, addr, endp);
		}
	}
}

/* send a USB start-of-frame packet over CIM */
static __attribute__ ((unused)) void
NAME_(send_usb_sof)(struct cpssp *cpssp, unsigned short frame_num)
{
	int usb_portnr;

	for (usb_portnr = 0; usb_portnr < USB_NUM_PORTS; usb_portnr++) {
		/* port disabled or in selective suspend mode? */
		if (!(cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE)
		 || (cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE
		  && cpssp->NAME.portsc[usb_portnr] & PORTSC_SUSPEND)) {
			continue;
		}

		if (usb_portnr == 0) {
			NAME_(0_send_sof)(cpssp, frame_num);
		} else { assert(usb_portnr == 1);
			NAME_(1_send_sof)(cpssp, frame_num);
		}
	}
}

/* send a USB data packet over CIM */
static void
NAME_(send_usb_data)(struct cpssp *cpssp, unsigned char pid, unsigned char *data, unsigned int length)
{
	int usb_portnr;

	for (usb_portnr = 0; usb_portnr < USB_NUM_PORTS; usb_portnr++) {
		/* port disabled or in selective suspend mode? */
		if (!(cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE)
		 || (cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE
		  && cpssp->NAME.portsc[usb_portnr] & PORTSC_SUSPEND)) {
			continue;
		}

		if (usb_portnr == 0) {
			NAME_(0_send_data)(cpssp, pid, data, length, 0);
		} else { assert(usb_portnr == 1);
			NAME_(1_send_data)(cpssp, pid, data, length, 0);
		}
	}
}

/* send a USB handshake packet over CIM */
static void
NAME_(send_usb_handshake)(struct cpssp *cpssp, unsigned char pid)
{
	int usb_portnr;

	for (usb_portnr = 0; usb_portnr < USB_NUM_PORTS; usb_portnr++) {
		/* port disabled or in selective suspend mode? */
		if (!(cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE)
		 || (cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE
		  && cpssp->NAME.portsc[usb_portnr] & PORTSC_SUSPEND)) {
			continue;
		}

		if (usb_portnr == 0) {
			NAME_(0_send_handshake)(cpssp, pid);
		} else { assert(usb_portnr == 1);
			NAME_(1_send_handshake)(cpssp, pid);
		}
	}
}

/*----------------------In-memory data structure related----------------------*/

/* some forward declarations */
static void
NAME_(work_frame)(struct cpssp *cpssp);
static void
NAME_(finish_frame)(struct cpssp *cpssp);

static void
NAME_(read_td)(struct cpssp *cpssp)
{
	uint32_t addr = cpssp->NAME.tdp;

	/* fetch TD from memory (4 DWORDs) */
	NAME_(mem_read)(cpssp, addr +  0, 0xf, &cpssp->NAME.td[0]);
	NAME_(mem_read)(cpssp, addr +  4, 0xf, &cpssp->NAME.td[1]);
	NAME_(mem_read)(cpssp, addr +  8, 0xf, &cpssp->NAME.td[2]);
	NAME_(mem_read)(cpssp, addr + 12, 0xf, &cpssp->NAME.td[3]);

	/* TD inactive? */
	if (!(cpssp->NAME.td[1] & USB_MEM_TD_1_STATUS_ACTIVE)) {
		return;
	}

	cpssp->NAME.td_c_err = (cpssp->NAME.td[1] & USB_MEM_TD_1_C_ERR)
			>> USB_MEM_TD_1_C_ERR_SHIFT;
	cpssp->NAME.td_actlen = (cpssp->NAME.td[1] + 1) & USB_MEM_TD_1_ACTLEN;
	cpssp->NAME.td_maxlen = (((cpssp->NAME.td[2] & USB_MEM_TD_2_MAXLEN)
		>> USB_MEM_TD_2_MAXLEN_SHIFT) + 1) & USB_MEM_TD_2_MAXLEN_WRAP;

	cpssp->NAME.td_endpt = (cpssp->NAME.td[2] & USB_MEM_TD_2_ENDPT)
		>> USB_MEM_TD_2_ENDPT_SHIFT;
	cpssp->NAME.td_addr = (cpssp->NAME.td[2] & USB_MEM_TD_2_DEV_ADDR)
		>> USB_MEM_TD_2_DEV_ADDR_SHIFT;
	cpssp->NAME.td_pid = cpssp->NAME.td[2] & USB_MEM_TD_2_PID;
}

static void
NAME_(read_qh)(struct cpssp *cpssp)
{
	if (50 <= cpssp->NAME.qh_count) {
		NAME_(finish_frame)(cpssp);
		return;
	}

	cpssp->NAME.qh_count++;

	NAME_(mem_read)(cpssp, cpssp->NAME.qhp + 0, 0xf, &cpssp->NAME.qh[0]);
	NAME_(mem_read)(cpssp, cpssp->NAME.qhp + 4, 0xf, &cpssp->NAME.qh[1]);

	cpssp->NAME.q_context = 1;

	/*
	 * make all further code independent of whether we are executing the
	 * first or any subsequent TD in this queue; needed for executing
	 * multiple transactions at once
	 */
	cpssp->NAME.td[0] = cpssp->NAME.qh[1];
}

static void
NAME_(advance_queue)(struct cpssp *cpssp, int advance)
{
	assert(cpssp->NAME.tds_sent == 0);

	if (cpssp->NAME.q_context) {
		if (advance == USB_QUEUE_ADVANCE) {
			/* tdlp -> qh[1] (in memory, too) */
			cpssp->NAME.qh[1] = cpssp->NAME.td[0];

			NAME_(mem_write)(cpssp, cpssp->NAME.qhp + 4,
				0xf, cpssp->NAME.td[0]);

		}

		/*
		 * We are ignoring td_vf completely here! FIXME
		 */
		/* go to the right */
		if (cpssp->NAME.qh[0] & 0x1) {
			/* nothing comes next */
			/* finish frame */
			NAME_(finish_frame)(cpssp);

		} else if (cpssp->NAME.qh[0] & 0x2) {
			/* QH comes next */
			cpssp->NAME.qhp = cpssp->NAME.qh[0];
			NAME_(read_qh)(cpssp);

		} else {
			/* TD comes next */
			cpssp->NAME.td[0] = cpssp->NAME.qh[0];
			/* this is missing in the specs! */
			cpssp->NAME.q_context = 0;
		}

	} else { /* not a queue context */
		if (cpssp->NAME.td[0] & 0x1) {
			/* nothing comes next */
			/* finish frame */
			NAME_(finish_frame)(cpssp);

		} else if (cpssp->NAME.td[0] & 0x2) {
			/* QH is next */
			cpssp->NAME.qhp = cpssp->NAME.td[0] & ~0xf;
			NAME_(read_qh)(cpssp);

		} else {
			/* TD is next */
		}
	}
}

/* handle packet timeouts that may occur on device disconnection */
static void
NAME_(packet_timeout)(struct cpssp *cpssp)
{
	uint32_t *tdctrl;

	tdctrl = &cpssp->NAME.td[1];

	/*
	 * - signal a timeout error in the TD
	 * - mark TD inactive
	 * - signal STALLED
	 * - set error count to zero (we do not retry even if error counter was >1)
	 */
	*tdctrl &= ~(USB_MEM_TD_1_STATUS | USB_MEM_TD_1_C_ERR);
	*tdctrl |= USB_MEM_TD_1_STATUS_STALLED | USB_MEM_TD_1_STATUS_CRC_TIMEOUT_ERROR;

	/* write TD control and status back to memory */
	NAME_(mem_write)(cpssp, cpssp->NAME.tdp + 4, 0xf, *tdctrl);

	/* generate interrupt if ordered to do so */
	cpssp->NAME.timeout_interrupt = 1;

	/* we are not waiting for an answer anymore */
	cpssp->NAME.tds_sent = 0;

	/* do not advance the current queue */
	NAME_(advance_queue)(cpssp, USB_QUEUE_NO_ADVANCE);
}

/* handle incoming USB linestate packet */
static void
NAME_(handle_packet_linestate)(struct cpssp *cpssp, int usb_portnr, int usb_speed)
{
	cpssp->NAME.portstate[usb_portnr].speed = usb_speed;
	NAME_(connect_device)(cpssp, usb_portnr);
}

/* handle incoming USB data packet */
static void
NAME_(handle_packet_usb_data)(
	struct cpssp *cpssp,
	int usb_portnr,
	unsigned char pid,
	unsigned int length,    /* not contained in real packets */
	unsigned char *data,
	unsigned short crc16
)
{
	uint32_t *tdctrl;
	int advance_queue;

	/* only process USB token packets if powered and port enabled */
	if (! cpssp->state_power
	 || ! (cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE)) {
		return;
	}

	tdctrl = &cpssp->NAME.td[1];

	if (cpssp->NAME.td_pid != USB_PID_IN) {
		return;
	}

	/* c_err: must only be decremented for errors we do not simulate */
	/* data toggle correct? */
	if (((cpssp->NAME.td[2] & USB_MEM_TD_2_D)
	   && pid != USB_PID_DATA1)
	 || (!(cpssp->NAME.td[2] & USB_MEM_TD_2_D)
	   && pid != USB_PID_DATA0)) {
		/*
		 * Data toggle incorrect.
		 */
		advance_queue = USB_QUEUE_NO_ADVANCE;

		NAME_(send_usb_handshake)(cpssp, USB_PID_ACK);

	} else {
		/*
		 * Data toggle correct.
		 */
		uint32_t addr;
		unsigned int i;

		if (cpssp->NAME.td_maxlen < length) {
			/* FIXME: is this the correct behaviour? */
			/* this would be "babble" on real hardware */
			length = cpssp->NAME.td_maxlen;
		}

		/* short packet detected? */
		if (*tdctrl & USB_MEM_TD_1_SPD
		 && cpssp->NAME.q_context
		 && length < cpssp->NAME.td_maxlen) {
			cpssp->NAME.spd = 1;
			advance_queue = USB_QUEUE_NO_ADVANCE;
		} else {
			advance_queue = USB_QUEUE_ADVANCE;
		}

		/* IOC? */
		if (*tdctrl & USB_MEM_TD_1_IOC) {
			cpssp->NAME.ioc = 1;
		}

		/* put data in memory */
		addr = cpssp->NAME.td[3];
		i = 0;
		while (i < length) {
			unsigned int count;
			unsigned int bs;
			uint32_t val;

			count = length - i;
			if (4 < (addr & 3) + count) {
				count = 4 - (addr & 3);
			}
			bs = ((1 << count) - 1) << (addr & 3);

			val = 0;
			if ((bs >> 0) & 1) {
				val |= data[i++] << 0;
			}
			if ((bs >> 1) & 1) {
				val |= data[i++] << 8;
			}
			if ((bs >> 2) & 1) {
				val |= data[i++] << 16;
			}
			if ((bs >> 3) & 1) {
				val |= data[i++] << 24;
			}

			NAME_(mem_write)(cpssp, addr & ~3, bs, val);

			addr += count;
		}

		*tdctrl &= ~(USB_MEM_TD_1_STATUS | USB_MEM_TD_1_ACTLEN);

		/*
		 * store data length:
		 * this also takes care of the special 0 encoding
		 */
		*tdctrl |= (length - 1) & USB_MEM_TD_1_ACTLEN;

		/* write TD control and status back to memory */
		NAME_(mem_write)(cpssp, cpssp->NAME.tdp + 4, 0xf, *tdctrl);

		NAME_(send_usb_handshake)(cpssp, USB_PID_ACK);
	}

	/* one answer less pending */
	cpssp->NAME.tds_sent--;
	assert(cpssp->NAME.tds_sent == 0);

	NAME_(advance_queue)(cpssp, advance_queue);

	NAME_(work_frame)(cpssp);
}

/* handle incoming USB handshake packet */
static void
NAME_(handle_packet_usb_handshake)(struct cpssp *cpssp, int usb_portnr, unsigned char pid)
{
	uint32_t *tdctrl;

	int advance_queue;

	/* only process USB token packets if powered and port enabled */
	if (! cpssp->state_power
	 || ! (cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE)) {
		return;
	}

	tdctrl = &cpssp->NAME.td[1];

	/* c_err: must only be decremented for errors we do not simulate */
	/* maybe timeout? */
	if (pid == USB_PID_ACK) {
		/* FIXME: correct behaviour? */

		advance_queue = USB_QUEUE_ADVANCE;
		*tdctrl &= ~(USB_MEM_TD_1_STATUS | USB_MEM_TD_1_ACTLEN);
		*tdctrl |= (cpssp->NAME.td_maxlen - 1) & USB_MEM_TD_1_ACTLEN;

		if (*tdctrl & USB_MEM_TD_1_IOC) {
			cpssp->NAME.ioc = 1;
		}

	} else if (pid == USB_PID_NAK) {
		/* FIXME: correct behaviour? */

		advance_queue = USB_QUEUE_NO_ADVANCE;
		/* *tdctrl &= ~(USB_MEM_TD_1_STATUS | USB_MEM_TD_1_ACTLEN); */
		*tdctrl &= ~USB_MEM_TD_1_ACTLEN;
		*tdctrl |= USB_MEM_TD_1_STATUS_NAK_RECEIVED;

		/* if this was an IN transaction, we didn't receive anything */
		if (cpssp->NAME.td_pid == USB_PID_IN) {
			*tdctrl |= USB_MEM_TD_1_ACTLEN;		/* encoded 0 */
		} else {
			*tdctrl |= (cpssp->NAME.td_maxlen - 1) & USB_MEM_TD_1_ACTLEN;
		}

		/*
		 * If a NAK handshake is received from a SETUP transaction, a
		 * Time Out Error will also be reported.
		 */
		if (cpssp->NAME.td_pid == USB_PID_SETUP) {
			*tdctrl |= USB_MEM_TD_1_STATUS_CRC_TIMEOUT_ERROR;
		}
	} else if (pid == USB_PID_STALL) {
		/* FIXME: correct behaviour? */

		advance_queue = USB_QUEUE_NO_ADVANCE;
		*tdctrl &= ~(USB_MEM_TD_1_STATUS | USB_MEM_TD_1_ACTLEN);
		*tdctrl |= USB_MEM_TD_1_STATUS_STALLED;

		/* if this was an IN transaction, we didn't receive anything */
		if (cpssp->NAME.td_pid == USB_PID_IN) {
			*tdctrl |= USB_MEM_TD_1_ACTLEN;		/* encoded 0 */
		} else {
			*tdctrl |= (cpssp->NAME.td_maxlen - 1) & USB_MEM_TD_1_ACTLEN;
		}

		/*
		 * If a STALL handshake is received from a SETUP transaction, a
		 * Time Out Error will also be reported.
		 */
		if (cpssp->NAME.td_pid == USB_PID_SETUP) {
			*tdctrl |= USB_MEM_TD_1_STATUS_CRC_TIMEOUT_ERROR;
		}

		cpssp->NAME.stalled_interrupt = 1;
	} else {
		/* there are no other handshake types, programming error! */
		return;
	}

	/* write TD control and status back to memory */
	NAME_(mem_write)(cpssp, cpssp->NAME.tdp + 4, 0xf, *tdctrl);

	/* one answer less pending */
	cpssp->NAME.tds_sent--;

	NAME_(advance_queue)(cpssp, advance_queue);

	NAME_(work_frame)(cpssp);
}

static void
NAME_(usb0_speed_set)(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(handle_packet_linestate)(cpssp, 0, val);
}

static void
NAME_(usb0_recv_token)(void *_cpssp, int pid, int addr, int endp)
{
	fixme();
}

static void
NAME_(usb0_recv_sof)(void *_cpssp, int frame_num)
{
	fixme();
}

static void
NAME_(usb0_recv_data)(void *_cpssp, int pid, unsigned int length, uint8_t *data, uint16_t crc16)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(handle_packet_usb_data)(cpssp, 0, pid, length, data, crc16);
}

static void
NAME_(usb0_recv_handshake)(void *_cpssp, int pid)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(handle_packet_usb_handshake)(cpssp, 0, pid);
}

static void
NAME_(usb1_speed_set)(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(handle_packet_linestate)(cpssp, 1, val);
}

static void
NAME_(usb1_recv_token)(void *_cpssp, int pid, int addr, int endp)
{
	fixme();
}

static void
NAME_(usb1_recv_sof)(void *_cpssp, int frame_num)
{
	fixme();
}

static void
NAME_(usb1_recv_data)(void *_cpssp, int pid, unsigned int length, uint8_t *data, uint16_t crc16)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(handle_packet_usb_data)(cpssp, 1, pid, length, data, crc16);
}

static void
NAME_(usb1_recv_handshake)(void *_cpssp, int pid)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(handle_packet_usb_handshake)(cpssp, 1, pid);
}

static void
NAME_(execute_td)(struct cpssp *cpssp)
{
	/* TD inactive? */
	if (!(cpssp->NAME.td[1] & USB_MEM_TD_1_STATUS_ACTIVE)) {
		/* IOC set? */
		if (cpssp->NAME.td[1] & USB_MEM_TD_1_IOC) {
			cpssp->NAME.ioc = 1;
		}

		NAME_(advance_queue)(cpssp, USB_QUEUE_NO_ADVANCE);
		return;
	}

	if (USB_MEM_TD_2_MAXLEN_ILLEGAL <= cpssp->NAME.td_maxlen) {
		cpssp->NAME.hc_process_error_interrupt = 1;
		NAME_(finish_frame)(cpssp);
		return;
	}

	/* check PID validity */
	if (cpssp->NAME.td_pid != USB_PID_SETUP
	 && cpssp->NAME.td_pid != USB_PID_IN
	 && cpssp->NAME.td_pid != USB_PID_OUT) {
		cpssp->NAME.hc_process_error_interrupt = 1;
		NAME_(finish_frame)(cpssp);
		return;
	}

	/* now we are sure we'll really execute this TD */

	cpssp->NAME.tds_sent++;

	/*
	 * In IN transfers the function is the sender, in
	 * OUT and SETUP transfers the host sends.
	 */
	if (cpssp->NAME.td_pid == USB_PID_SETUP
	 || cpssp->NAME.td_pid == USB_PID_OUT) {
		/* fetch data from memory */
		uint32_t addr;
		unsigned int i;

		addr = cpssp->NAME.td[3];
		i = 0;
		while (i < cpssp->NAME.td_maxlen) {
			unsigned int count;
			unsigned int bs;
			uint32_t val;

			count = cpssp->NAME.td_maxlen - i;
			if (4 < (addr & 3) + count) {
				count = 4 - (addr & 3);
			}
			bs = ((1 << count) - 1) << (addr & 3);

			NAME_(mem_read)(cpssp, addr & ~3, bs, &val);

			if ((bs >> 0) & 1) {
				cpssp->NAME.sendbuffer[i++] = val >> 0;
			}
			if ((bs >> 1) & 1) {
				cpssp->NAME.sendbuffer[i++] = val >> 8;
			}
			if ((bs >> 2) & 1) {
				cpssp->NAME.sendbuffer[i++] = val >> 16;
			}
			if ((bs >> 3) & 1) {
				cpssp->NAME.sendbuffer[i++] = val >> 24;
			}

			addr += count;
		}

		/* Send Token */
		NAME_(send_usb_token)(cpssp, cpssp->NAME.td_pid,
				cpssp->NAME.td_addr, cpssp->NAME.td_endpt);

		/* Send Data Packet */
		NAME_(send_usb_data)(cpssp,
				cpssp->NAME.td[2] & USB_MEM_TD_2_D ?
						USB_PID_DATA1 : USB_PID_DATA0,
				cpssp->NAME.sendbuffer,
				cpssp->NAME.td_maxlen);
	} else {
		/* Send Token */
		NAME_(send_usb_token)(cpssp, cpssp->NAME.td_pid,
				cpssp->NAME.td_addr, cpssp->NAME.td_endpt);
	}

	/* isochroneous TD and SETUP/OUT? */
	if (cpssp->NAME.td[1] & USB_MEM_TD_1_IOS
	 && (cpssp->NAME.td_pid == USB_PID_SETUP
	  || cpssp->NAME.td_pid == USB_PID_OUT)) {
		/* mark inactive, do not wait for an answer */
		NAME_(handle_packet_usb_handshake)(cpssp, 0, USB_PID_ACK);
	}
}

static void
NAME_(start_frame)(struct cpssp *cpssp)
{
	uint32_t flp;

	cpssp->NAME.working = 1;
	cpssp->NAME.timeout_interrupt = 0;
	cpssp->NAME.ioc = 0;
	cpssp->NAME.spd = 0;
	cpssp->NAME.babble_interrupt = 0;
	cpssp->NAME.stalled_interrupt = 0;
	cpssp->NAME.data_buffer_error_interrupt = 0;
	cpssp->NAME.bit_stuff_error_interrupt = 0;
	cpssp->NAME.hc_process_error_interrupt = 0;
	cpssp->NAME.host_system_error_interrupt = 0;

	cpssp->NAME.qh_count = 0;

	/* TODO: issue SOF? */

	/* read frame list pointer from frame list */
	NAME_(mem_read)(cpssp, cpssp->NAME.flbaseadd +
			((cpssp->NAME.frnum & FRNUM_FL_INDEX) << 2),
			0xf, &flp);

	/* initialize frame list traversal state */
	if (flp & 0x1) {
		NAME_(finish_frame)(cpssp);
		return;

	} else if (flp & 0x2) {	/* first structure is a QH */
		cpssp->NAME.qhp = flp & ~0xf;
		NAME_(read_qh)(cpssp);

	} else { /* first structure is a TD */
		cpssp->NAME.td[0] = flp;
		cpssp->NAME.q_context = 0;
	}

	/* we haven't done any communications up to now, so we're not waiting
	 * for packets already */
	cpssp->NAME.tds_sent = 0;
	cpssp->NAME.cont = 0;

	NAME_(work_frame)(cpssp);
}

static void
NAME_(work_frame)(struct cpssp *cpssp)
{
	assert(cpssp->NAME.tds_sent == 0);

	if (cpssp->NAME.cont) {
		goto cont;
	}
 
	do {
		/*
		 * cf. UHCI Spec., Rev. 1.1, chapter 3.4.2 "Transfer Queueing",
		 * "USB Schedule List Traversal Decision Table" (incomplete!)
		 * and "USB Linked List Traversal State Diagram"
		 */

		/* states #1, #2, #3 */
		if (!cpssp->NAME.q_context) {
			/*
			 * Not in a queue context.
			 */
			/* execute TD pointed at by cpssp->NAME.tdlp */
			cpssp->NAME.tdp = cpssp->NAME.td[0] & ~0xf;
			NAME_(read_td)(cpssp);
			cpssp->NAME.cont = 1;
			NAME_(execute_td)(cpssp);

		} else {
			/*
			 * Queue context, states #4-#12
			 */
			/* states #4, #5, #6, #9, #11 */
			if (! (cpssp->NAME.td[0] & 0x1)
			 && ! (cpssp->NAME.td[0] & 0x2)) {
				/* TD below */
				/* execute TD pointed at by cpssp->NAME.tdlp */
				cpssp->NAME.tdp = cpssp->NAME.td[0] & ~0xf;
				NAME_(read_td)(cpssp);
				cpssp->NAME.cont = 1;
				NAME_(execute_td)(cpssp);
				/* we are probably waiting for answers now */

			/* state #8 */
			} else if (! (cpssp->NAME.td[0] & 0x1)
				&& (cpssp->NAME.td[0] & 0x2)) {
				/* QH below */
				cpssp->NAME.qhp = cpssp->NAME.qh[1] & ~0xf;
				NAME_(read_qh)(cpssp);
				/* we are not yet waiting for answers, loop! */

			/* states #7, #10, #12 */
			} else { /* td_t is set */
				/* state #10 */
				if (cpssp->NAME.qh[0] & 0x1) {
					/* QH with nothing to the right */
					/* end of frame */
					NAME_(finish_frame)(cpssp);
					/* we are done for this frame, leaving the loop */

				/* state #7 */
				} else if (! (cpssp->NAME.qh[0] & 0x2)) {
					/* QH with TD to the right */
					cpssp->NAME.td[0] = cpssp->NAME.qh[0];
					/* leaving queue context! specs are buggy here! */
					cpssp->NAME.q_context = 0;
					/* we are not yet waiting for answers, loop! */

				/* state #12 */
				} else if (cpssp->NAME.qh[0] & 0x2) {
					/* QH with QH to the right */
					cpssp->NAME.qhp = cpssp->NAME.qh[0] & ~0xf;
					NAME_(read_qh)(cpssp);
					/* we are not yet waiting for answers, loop! */
				}
			}
		}
	cont:	;
		cpssp->NAME.cont = 0;

		if (cpssp->NAME.tds_sent == 0
		 && USB_MAX_QH_PER_FRAME <= cpssp->NAME.qh_count) {
			/*
			 * If the simulation hits this point, two constants
			 * might need to be adjusted:
			 * - USB_MAX_QH_PER_FRAME might need to be increased in
			 *   case the in-memory QH chain gets too long (too
			 *   many devices attached, too many endpoints per
			 *   device)
			 */
			NAME_(finish_frame)(cpssp);
		}
	} while (cpssp->NAME.tds_sent == 0
	      && cpssp->NAME.working);
}

static void
NAME_(finish_frame)(struct cpssp *cpssp)
{
	int do_interrupt = 0;

	cpssp->NAME.working = 0;
	assert(cpssp->NAME.tds_sent == 0);

	/* generate interrupt? */
	/* cf. UHCI Spec. Rev. 1.1, chapter 4, "Interrupts" */
	if (cpssp->NAME.timeout_interrupt) {
		if (cpssp->NAME.usbintr & USBINTR_TIE) {
			do_interrupt = 1;
		}
		cpssp->NAME.usbsts |= USBSTS_ERROR_INTERRUPT;
	}
	if (cpssp->NAME.ioc) {
		if (cpssp->NAME.usbintr & USBINTR_IOC) {
			do_interrupt = 1;
		}
		cpssp->NAME.usbsts |= USBSTS_INTERRUPT;
	}
	if (cpssp->NAME.spd) {
		if (cpssp->NAME.usbintr & USBINTR_SPIE) {
			do_interrupt = 1;
		}
		cpssp->NAME.usbsts |= USBSTS_INTERRUPT;
	}
	/* TODO: resume received missing */
	if (cpssp->NAME.babble_interrupt
	 || cpssp->NAME.stalled_interrupt
	 || cpssp->NAME.data_buffer_error_interrupt
	 || cpssp->NAME.bit_stuff_error_interrupt
	 || cpssp->NAME.hc_process_error_interrupt
	 || cpssp->NAME.host_system_error_interrupt) {
		do_interrupt = 1; /* non-maskable */
		if (cpssp->NAME.babble_interrupt
	 	 || cpssp->NAME.stalled_interrupt
		 || cpssp->NAME.data_buffer_error_interrupt
		 || cpssp->NAME.bit_stuff_error_interrupt) {
			cpssp->NAME.usbsts |= USBSTS_ERROR_INTERRUPT;
		}
		if (cpssp->NAME.hc_process_error_interrupt) {
			cpssp->NAME.usbsts |= USBSTS_HC_PROCESS_ERROR;
			cpssp->NAME.usbsts |= USBSTS_HC_HALTED;
			cpssp->NAME.usbcmd &= ~USBCMD_RUN_STOP;
		}
		if (cpssp->NAME.host_system_error_interrupt) {
			cpssp->NAME.usbsts |= USBSTS_HC_HALTED;
			cpssp->NAME.usbcmd &= ~USBCMD_RUN_STOP;
		}
	}

	if (do_interrupt) {
		NAME_(set_irq)(cpssp, 1);
	}

	cpssp->NAME.frnum++;
	cpssp->NAME.frnum &= FRNUM_FRNUM;
}

static void
NAME_(step)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

again:	;
	if (! cpssp->state_power) {
		/* No Power */
		cpssp->NAME.process.inst_cnt = cpssp->NAME.process.inst_limit;

	} else if (! cpssp->NAME.master_enable
		|| ! (cpssp->NAME.usbcmd & USBCMD_RUN_STOP)
		|| cpssp->NAME.usbsts & USBSTS_HC_HALTED) {
		/* Halted */
		cpssp->NAME.process.inst_cnt = cpssp->NAME.process.inst_limit;
		cpssp->NAME.working = 0;

	} else {
		/* Running */
		NAME_(start_frame)(cpssp);
		cpssp->NAME.process.inst_cnt = cpssp->NAME.process.inst_limit; /* FIXME */
	}

	sched_to_scheduler();
	goto again;
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	NAME_(reset)(cpssp);

	cpssp->NAME.process.inst_hz = 32*1024*1024; /* PCI Bus */
	sched_process_init(&cpssp->NAME.process, NAME_(step), cpssp);
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
