/*
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "compiler.h"

#include "assert.h"
#include "stdio.h"
#include "serial.h"

#define SER_RWBUFFER	(serport + 0)
#define SER_DIVLO	(serport + 0)
#define SER_DIVHI	(serport + 1)
#define SER_INTENABLE	(serport + 1)
#define SER_LINECTRL	(serport + 3)
#define SER_MODEMCTRL	(serport + 4)
#define SER_LINESTATUS	(serport + 5)
#define SER_MODEMSTATUS	(serport + 6)

/* ==================== RUNTIME_RM ==================== */
#ifdef RUNTIME_RM

#include "var.h"
#include "const.h"
#include "io.h"
#include "ptrace.h"

CODE16;

unsigned char
serial_write(unsigned char c, unsigned short port)
{
	unsigned short serport;

	serport = var_get(addr_ser[port]);

	/* DTR + RTS*/
	outb(0x03, SER_MODEMCTRL);
	/* wait for: DSR + CTS */
	while ((inb(SER_MODEMSTATUS) & 0x30) != 0x30);
	/* wait for transmitter */
	while ((inb(SER_LINESTATUS) & 0x20) != 0x20);
	/* write byte */
	outb(c, SER_RWBUFFER);

	return(inb(SER_LINESTATUS));
}

unsigned short
serial_getstatus(unsigned short port)
{
	unsigned short serport, ret;

	serport = var_get(addr_ser[port]);

	ret = inb(SER_LINESTATUS) << 8;
	ret += inb(SER_MODEMSTATUS);

	return(ret);
}

unsigned short
serial_read(unsigned short port)
{
	unsigned short serport, ret;

	serport = var_get(addr_ser[port]);

	/* DTR */
	outb(0x01, SER_MODEMCTRL);
	/* wait for: DSR */
	while ((inb(SER_MODEMSTATUS) & 0x20) != 0x20);
	/* wait for: RECV */
	while ((ret = (inb(SER_LINESTATUS) & 0x01)) != 0x01);
	/* mask error codes */
	ret &= 0x1e;
	ret <<= 8;
	/* get char */
	ret |= inb(SER_RWBUFFER);
	return(ret);
}

/*
 * Serial Function 00: Initialize serial adapter
 *
 * In:  AH      = 00h
 *	AL	= Init parameters
 *	DX	= Serial port
 *
 * Out: AL	= Modem status
 *	AH	= Line status
 */
static void
bios_14_00xx(struct regs *regs)
{
#if 0
	AX = serial_init(AL, DX);
#else
	F |= (1 << 0);	/* Set carry. FIXME VOSSI */
#endif
}

/*
 * Serial Function 01: Send character
 *
 * In:  AH      = 01h
 *	AL	= character to transmit
 *	DX	= Serial port
 *
 * Out:	AH	= Line status
 */
static void
bios_14_01xx(struct regs *regs)
{
	AH = serial_write(AL, DX);
}

/*
 * Serial Function 02: Receive character
 *
 * In:  AH      = 02h
 *	DX	= Serial port
 *
 * Out:	AL	= Character received
 *	AH	= Line status
 */
static void
bios_14_02xx(struct regs *regs)
{
	AX = serial_read(DX);
}

/*
 * Serial Function 03: Return serial port status
 *
 * In:  AH      = 03h
 *	DX	= Serial port
 *
 * Out:	AL	= Modem status
 *	AH	= Line status
 */
static void
bios_14_03xx(struct regs *regs)
{
	AX = serial_getstatus(DX);
}

C_ENTRY void
bios_14_xxxx(struct regs *regs)
{
	if (AH == 0x00) {
		bios_14_00xx(regs);
	} else if (AH == 0x01) {
		bios_14_01xx(regs);
	} else if (AH == 0x02) {
		bios_14_02xx(regs);
	} else if (AH == 0x03) {
		bios_14_03xx(regs);
	} else {
		/* Unknown sub-function. */
		dprintf("int $0x%02x: unknown sub-function 0x%02x\n", 0x14, AH);
		F |= 0x00000001;
	}
}

#endif /* RUNTIME_RM */
/* ===================== REAL MODE INIT ============================ */
#ifdef INIT_RM

#include "var.h"
#include "const.h"
#include "io.h"
#include "ptrace.h"


static CONST char col_vga2term[] = {
	'0',	/* 0	= Black		*/
	'4',	/* 1	= Blue		*/
	'2',	/* 2	= Green 	*/
	'6',	/* 3	= Cyan		*/
	'1',	/* 4	= Red		*/
	'5',	/* 5	= Magenta	*/
	'7',	/* 6	= Brown		*/
	'7',	/* 7	= Light grey	*/
	'7',	/* 8	= Dark grey	*/
	'4',	/* 9	= Light blue	*/
	'2',	/* A	= Light green	*/
	'6',	/* B	= Light cyan	*/
	'1',	/* C	= Light red	*/
	'5',	/* D	= Light magenta	*/
	'3',	/* E	= Yellow	*/
	'7'	/* F	= White		*/
};

CODE16;

void
serial_putcstr(CONST char *str, unsigned char port)
{
       for (;;) {
		char c;

		c = const_get(*str);
		if (c == '\0') {
			break;
		}
		serial_write(c, port);
                str++;
        }
}

void
serial_colwrite(unsigned char c, unsigned char col, unsigned char port)
{
	/* select fg/bg color */
	serial_putcstr("\e[3", port);
	serial_write(const_get(col_vga2term[col & 0xf]), port);
	serial_putcstr(";4", port);
	serial_write(const_get(col_vga2term[col >> 4]), port);
	serial_write('m', port);
	/* print char */
	serial_write(c, port);
	/* back to normal */
	serial_putcstr("\e[0m", port);
}

void
serial_writenum(unsigned char num)
{
	if (num < 10) {
		serial_write(num + '0', 0);
	} else {
		serial_writenum(num/10);
		serial_write((num % 10) + '0', 0);
	}
}

void
serial_gotoxy(unsigned char x, unsigned y)
{
	/* serial gotoxy */
	serial_putcstr("\e[", 0);
	/* counting starts with 1 here */
	y++; x++;
	serial_writenum(y);
	serial_write(';', 0);
	serial_writenum(x);
	serial_write('H', 0);
}

void
serial_reset(void)
{
	/* 8,1,no parity */
	serial_init(0xe3, 0);

	/* clear screen and home cursor */
	serial_putcstr("\e[2J", 0);

	/* goto top left */
	serial_gotoxy(0, 0);
}

unsigned short
serial_init(unsigned char param, unsigned short port)
{
	unsigned short serport, ret;

	static CONST unsigned short divisor[] = {
		0x0417,	/*    110 baud */
		0x0300, /*    150 baud */
		0x0180, /*    300 baud */
		0x00C0, /*    600 baud */
		0x0060, /*   1200 baud */
		0x0030, /*   2400 baud */
		0x0018, /*   4800 baud */
		0x000C, /*   9600 baud */
		0x0006, /*  19200 baud */
		0x0003, /*  38400 baud */
		0x0002, /*  56700 baud */
		0x0001  /* 115200 baud */
	};

	/* AL:	bits	value	meaning
	 *	1,0
	 *		10	7 data bits
	 *		11	8 data bits
	 *	2	0	1 stop bit
	 *		1	2 stop bits
	 *	4,3	00	no parity
	 *		10	no parity
	 *		01	odd parity
	 *		11	even parity
	 *	7-5	000	110 baud - divisor 0x417
	 *		001	150 baud - divisor 0x300
	 *		010	300 baud - divisor 0x180
	 *		011	600 baud - divisor 0x0C0
	 *		100	1200 baud - divisor 0x060
	 *		101	2400 baud - divisor 0x030
	 *		110	4800 baud - divisor 0x018
	 *		111	9600 baud - divisor 0x00C
	 */

	serport = var_get(addr_ser[port]);

	/* set DLAB (divisor latch access bit) on */
	outb(0x80, SER_LINECTRL);
	/* divisor latch */
	/* low byte */
	outb(const_get(divisor[param>>5]) & 0xff, SER_DIVLO);
	/* high byte */
	outb((const_get(divisor[param>>5]) >> 8) & 0xff, SER_DIVHI);
	/* disable interrupts - needs DLAB set*/
	outb(0x00, SER_INTENABLE);

	/* set parity, data/stop bits */
	outb(param & 0x1f, SER_LINECTRL);
	/* set RTS and DTR */
	outb(0x03, SER_MODEMCTRL);

	ret = inb(SER_LINESTATUS) << 8;
	ret += inb(SER_MODEMSTATUS);

	return(ret);
}

#endif /* INIT_RM */
