/*
 * Copyright (C) 2022 HVML Community. All rights reserved.
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include <wtf/HashMap.h>
#include <wtf/HashSet.h>
#include <wtf/Vector.h>

namespace WebCore {

class Element;
class HTMLImageElement;
class HTMLLabelElement;
class HTMLMapElement;
class TreeScope;

class TreeScopeHandleMap {
    WTF_MAKE_FAST_ALLOCATED;
public:
    void add(uint64_t, Element&, const TreeScope&);
    void remove(uint64_t, Element&);
    void clear();

    bool contains(uint64_t) const;
    bool containsSingle(uint64_t) const;
    bool containsMultiple(uint64_t) const;

#if ENABLE(HVML_ATTRS)
    Element* getElementByHVMLHandle(uint64_t, const TreeScope&) const;
    const Vector<Element*>* getAllElementsByHVMLHandle(uint64_t, const TreeScope&) const;
#endif

    const Vector<uint64_t> keys() const;

    static uint64_t convertFrom(const AtomString&);
    static uint64_t convertFrom(const String&);
    static uint64_t convertFrom(const StringView&);

private:
    template <typename KeyMatchingFunction>
    Element* get(uint64_t, const TreeScope&, const KeyMatchingFunction&) const;

    struct MapEntry {
        MapEntry() { }
        explicit MapEntry(Element* firstElement)
            : element(firstElement)
            , count(1)
        { }

        Element* element { nullptr };
        unsigned count { 0 };
        Vector<Element*> orderedList;
#if ASSERT_ENABLED || ENABLE(SECURITY_ASSERTIONS)
        HashSet<Element*> registeredElements;
#endif
    };

    typedef HashMap<uint64_t, MapEntry, WTF::IntHash<uint64_t>, WTF::UnsignedWithZeroKeyHashTraits<uint64_t>> Map;

    mutable Map m_map;
};

inline bool TreeScopeHandleMap::containsSingle(uint64_t handle) const
{
    auto it = m_map.find(handle);
    return it != m_map.end() && it->value.count == 1;
}

inline bool TreeScopeHandleMap::contains(uint64_t handle) const
{
    return m_map.contains(handle);
}

inline bool TreeScopeHandleMap::containsMultiple(uint64_t handle) const
{
    auto it = m_map.find(handle);
    return it != m_map.end() && it->value.count > 1;
}

} // namespace WebCore
