/*
 * Copyright (C) 2015 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package libcore.icu;

import android.icu.text.DateFormat;
import android.icu.text.DateTimePatternGenerator;
import android.icu.text.DisplayContext;
import android.icu.text.SimpleDateFormat;
import android.icu.util.Calendar;
import android.icu.util.ULocale;

import libcore.util.BasicLruCache;

/**
 * A formatter that outputs a single date/time.
 */
public class DateTimeFormat {
  private static final FormatterCache CACHED_FORMATTERS = new FormatterCache();

  static class FormatterCache extends BasicLruCache<String, DateFormat> {
    FormatterCache() {
      super(8);
    }
  }

  private DateTimeFormat() {
  }

  public static String format(ULocale icuLocale, Calendar time, int flags,
      DisplayContext displayContext) {
    String skeleton = DateUtilsBridge.toSkeleton(time, flags);
    String key = skeleton + "\t" + icuLocale + "\t" + time.getTimeZone();
    synchronized(CACHED_FORMATTERS) {
      DateFormat formatter = CACHED_FORMATTERS.get(key);
      if (formatter == null) {
        DateTimePatternGenerator generator = DateTimePatternGenerator.getInstance(icuLocale);
        formatter = new SimpleDateFormat(generator.getBestPattern(skeleton), icuLocale);
        CACHED_FORMATTERS.put(key, formatter);
      }
      formatter.setContext(displayContext);
      return formatter.format(time);
    }
  }
}
