#!/usr/bin/python3
# -*- coding: UTF-8 -*-

'''Hardware and driver package detection functionality for UOS systems.'''

from cmath import inf
import os
import fnmatch
import subprocess
import re
import json
import gzip

from click import command

JSON_DIR="/usr/share/doc/deepin-nvidia-installer"

class NVDetectError(Exception):
    def __init__(self, message):
        self.message = message

class NVDetect(object):
    def __init__(self):
        self._drv_versions = self._get_driver_version()
        self._has_intel_gpu =False
        self._nvidia_gpu_sysfs = None
        self._nvidia_gpu_modalias = None
        self._nvidia_gpu_vendor = None
        self._current_driver = None

        gpus_info = self._get_gpus_info()
        for _gpu in  gpus_info:
            if _gpu['vendor'].find("pci 0x10de") >= 0:
                self._nvidia_gpu_sysfs = _gpu["sysfs"]
                self._nvidia_gpu_modalias = _gpu["alias"]
                self._nvidia_gpu_vendor = _gpu["vendor"]
            elif _gpu['vendor'].find("pci 0x8086") >= 0:
                self._has_intel_gpu = True
        
        command = 'dpkg -l |grep -E "^[a-z]{2}[\t ]*nvidia-[0-9]{3}|^[a-z]{2}[\t ]*nvidia-driver-[0-9]{3}"'
        ret_code, pkg_info = self._run_shell_command(command)

        if ret_code != 0:
            return

        for item in pkg_info.split("\n"):
            if item == '':
                continue
            self._current_driver = item.split()[1] if item.split()[0] == 'ii' else None

        # if not self._nvidia_gpu_sysfs:
        #     raise NVDetectError("NVIDIA GPU not found.")

        return

    def _nvidia_drm_enable(self):
        modprobe_conf = "/etc/nvidia/current/nvidia-modprobe.conf"
        if self._has_intel_gpu:
            return
        
        if not os.path.exists(modprobe_conf):
            return
        
        str="install nvidia-drm modprobe nvidia-modeset ; modprobe -i nvidia-current-drm \$CMDLINE_OPTS"
        command = 'sed -i "s/%s/%s modeset=1/" %s' % (str, str, modprobe_conf)
        self._run_shell_command(command)

    def _nvidia_drm_clean(self):
        modprobe_conf = "/etc/nvidia/current/nvidia-modprobe.conf"

        if not os.path.exists(modprobe_conf):
            return

        command = 'sed -i "/^install nvidia-drm/s/modeset=1//g" %s' % modprobe_conf
        self._run_shell_command(command)


    def get_nvidia_gpu_sysfs(self):
        return self._nvidia_gpu_sysfs
    
    def get_nvidia_gpu_modalias(self):
        return self._nvidia_gpu_modalias
    
    def get_nvidia_gpu_vendor(self):
        return self._nvidia_gpu_vendor
    
    def get_drv_versions(self):
        return self._drv_versions
    
    def get_current_driver(self):
        return self._current_driver

    def _get_kernel_version(self):
        command = 'uname -r'
        major = 0
        minor = 0
        ret_code, _out = self._run_shell_command(command)
        if ret_code == 0:
            major = _out.split('.')[0]
            minor = _out.split('.')[1]
        return int(major), int(minor)

    def system_nvidia_driver_packages(self):
        packages={}
        ver_list = self._get_driver_version()
        for ver in ver_list:
            if self._check_driver_support_gpu(ver):
                package_name = "nvidia-%s" % ver \
                              if ver == "340" \
                              else "nvidia-driver-%s" % ver
                packages[ver] = package_name
        return packages

    def _gpu_match(self, json_alias):
        for alias in json_alias:
            if fnmatch.fnmatch(self._nvidia_gpu_modalias.lower(), alias.lower()):
                return True
        return False


    def _get_driver_version(self):
        ver = []
        kernel_ver_major, kernel_ver_minor = self._get_kernel_version()
        for item in os.listdir(JSON_DIR):
            if re.findall("supported-gpus.json.gz", item):
                if item.split('-')[0] == "340" and \
                    kernel_ver_major >= 5 and kernel_ver_minor > 10:
                    continue
                ver.append(item.split('-')[0])
        return ver


    def _prase_json_to_alias(self, json_file):
        template = 'pci:v0000%sd0000%ssv%ssd%sbc03sc*i*'
        vendor = "10de"
        alias_list = []
        with gzip.GzipFile(json_file) as f:
            data = json.load(f)
            for chip in data["chips"]:
                dev_id = chip.get('devid')[2:]

                # subvendor_id = '0000%s' % (chip.get('subvendorid')[2:]
                #                        if "subvendorid" in chip.keys()
                #                        else '*')

                # subdev_id = '0000%s' % (chip.get('subdev_id')[2:]
                #                        if "subdev_id" in chip.keys()
                #                        else '*')

                subvendor_id = '*'
                subdev_id = '*'

                line = template % (vendor, dev_id, subvendor_id, subdev_id)
                alias_list.append(line)
        return alias_list

    def _check_driver_support_gpu(self, version=None):
        if version:
            json_path = '%s/%s-%s' % (JSON_DIR, version, "supported-gpus.json.gz")
            if not os.path.exists(json_path):
                print("The version %s is invalid." % version)
                return False

            alias = self._prase_json_to_alias(json_path)
            if self._gpu_match(alias):
                return True
        return False



    def _run_shell_command(self, command):
        ret = subprocess.run(command,shell=True,stdout=subprocess.PIPE,stderr=subprocess.PIPE,encoding="utf-8")
        if ret.returncode == 0:
            return(ret.returncode, ret.stdout)
        else:
            return(ret.returncode, ret.stderr)

    def _prase_gpu_str(self, str_list):
        module_alias = []
        sysfs_id = []
        vendor = []
        for item in str_list:
            if len(module_alias) == 0:
                module_alias = re.findall("Module Alias:.*", item)

            if len(sysfs_id) == 0:
                sysfs_id = re.findall("SysFS ID:.*", item)

            if len(vendor) == 0:
                vendor = re.findall("Vendor:.*", item)

        if len(module_alias) and len(sysfs_id) and len(vendor):
            return {"sysfs":sysfs_id[0].split(": ")[1].strip('"'),
                    "alias":module_alias[0].split(": ")[1].strip('"'),
                    "vendor":vendor[0].split(": ")[1].strip('"')}


    def _get_gpus_info(self):
        gpus_list = []
        result = []
        ret_code, gpuinfo=self._run_shell_command("hwinfo --display")
        if ret_code == 0:
            lines = gpuinfo.splitlines()
            start = 0
            end = 0
            for line in lines:
                end = end + 1
                if len(line):
                    continue
                if end > start:
                    gpus_list.append(lines[start:end])
                    start = end

            if end > start:
                gpus_list.append(lines[start:end])

            for _gpu in gpus_list:
                if len(_gpu) <= 1:
                    continue
                info = self._prase_gpu_str(_gpu)
                if info:
                    result.append(info)

        return result

    def get_support_json(self, version=None):
        if version:
            json_path = '%s/%s-%s' % (JSON_DIR, version, "supported-gpus.json.gz")
            if not os.path.exists(json_path):
                print("The version %s is invalid." % version)
                return None
            with gzip.GzipFile(json_path) as f:
                data = json.load(f)
        return str(data).replace("'", "\"")

    def apt_install_package(self, pkg_name=None, mini_install=False):
        command = 'apt -y -o Dpkg::Options::="--force-confdef" -o Dpkg::Options::="--force-confold" --allow-unauthenticated'
        if mini_install:
            command = command + ' --no-install-recommends --no-install-suggests'
        command = command + ' %s %s' %('install', pkg_name)
        ret = subprocess.run(command, shell=True, env={"DEBIAN_FRONTEND":"noninteractive"})
        if ret.returncode != 0:
            print("Install %s failed!" % pkg_name)
            return -1
        return 0

    def install_nvidia_driver(self, version=None, mini_install=False):
        euid = os.geteuid()
        if euid != 0:
            print("Script not started as root.")
            return -1
        drivers = self.system_nvidia_driver_packages()
        if version not in drivers.keys():
            print("The version %s is not supported." % version)
            return
        self._nvidia_drm_clean()
        #name = '%s-%s' % ("nvidia-driver", version)
        name = '%s-%s' % ("nvidia" if version == "340" else "nvidia-driver", version)
        if not self.apt_install_package(name, mini_install):
            self._nvidia_drm_enable()
            self._current_driver = name