package utils

import (
	"fmt"
	"reflect"

	"pkg.deepin.io/daemon/sync/infrastructure/log"

	"github.com/godbus/dbus"
	"github.com/godbus/dbus/introspect"
	"github.com/godbus/dbus/prop"
)

type DBusCore interface {
	MakeSignals() []introspect.Signal
	MakePropSpec() map[string]map[string]*prop.Prop

	GetDBusName() string
	GetDBusPath() dbus.ObjectPath
	GetDBusInterface() string
}

type DBusProxy struct {
	core DBusCore
	conn *dbus.Conn
}

func (d *DBusProxy) SetupDBus(conn *dbus.Conn, core DBusCore) error {
	err := conn.Export(core, core.GetDBusPath(), core.GetDBusInterface())
	if err != nil {
		return err
	}
	props := prop.New(conn, core.GetDBusPath(), core.MakePropSpec())
	propsNode := &introspect.Node{
		Name: core.GetDBusName(),
		Interfaces: []introspect.Interface{
			introspect.IntrospectData,
			prop.IntrospectData,
			{
				Name:       core.GetDBusInterface(),
				Methods:    introspect.Methods(core),
				Properties: props.Introspection(core.GetDBusInterface()),
				Signals:    core.MakeSignals(),
			},
		},
	}
	err = conn.Export(introspect.NewIntrospectable(propsNode), core.GetDBusPath(), "org.freedesktop.DBus.Introspectable")
	if err != nil {
		return err
	}
	// must after export, otherwise will occur error(Not found methods on object)
	// when by dbus-daemon called
	reply, err := conn.RequestName(core.GetDBusName(), dbus.NameFlagDoNotQueue)
	if err != nil {
		return err
	}
	if reply != dbus.RequestNameReplyPrimaryOwner {
		return fmt.Errorf("name '%s' has exists", core.GetDBusName())
	}

	d.core = core
	d.conn = conn

	return nil
}

func (d *DBusProxy) SetMapProp(refer *map[string]interface{}, name string, value map[string]interface{}) {
	if reflect.DeepEqual(*refer, value) {
		log.Warning("skip send change message!!!")
		return
	}
	*refer = value
	d.Emit(name, value)
}

func (d *DBusProxy) SetBoolProp(refer *bool, name string, value bool) {
	if *refer == value {
		return
	}
	*refer = value
	d.Emit(name, value)
}

func (d *DBusProxy) SetInt64Prop(refer *int64, name string, value int64) {
	if *refer == value {
		return
	}
	*refer = value
	d.Emit(name, value)
}

func (d *DBusProxy) SetStrProp(refer *string, name, value string) {
	if *refer == value {
		return
	}
	*refer = value
	d.Emit(name, value)
}

func (d *DBusProxy) Emit(name string, value interface{}) {
	// err := d.conn.Emit(d.core.GetDBusPath(), d.core.GetDBusInterface()+"."+name, value)
	// if err != nil {
	// 	log.Error("Failed to emit prop changed:", name, value)
	// }
	err := d.conn.Emit(d.core.GetDBusPath(),
		"org.freedesktop.DBus.Properties.PropertiesChanged",
		d.core.GetDBusInterface(),
		map[string]interface{}{
			name: value,
		},
		[]string{})
	if err != nil {
		log.Error("Failed to emit prop changed:", name, value)
	}
}
