package modules

import (
	"encoding/json"

	Logger "pkg.deepin.io/daemon/sync/infrastructure/log"
	"pkg.deepin.io/daemon/sync/infrastructure/storage"
)

type ModuleIndex struct {
	Appearance  string `json:"appearance"`
	Audio       string `json:"audio"`
	Background  string `json:"background"`
	Dock        string `json:"dock"`
	Launcher    string `json:"launcher"`
	Network     string `json:"network"`
	Peripherals string `json:"peripherals"`
	Power       string `json:"power"`
	ScreenEdge  string `json:"screen_edge"`
	Screensaver string `json:"screensaver"`
	Updater     string `json:"updater"`
}

type SyncError struct {
	Code    StateType `json:"code"`
	Module  string    `json:"module"`
	Message string    `json:"message"`
}

type StateType int32

const (
	cloudKeyIndex = "index"
)

const (
	CodeSynchronizing StateType = 100
	CodeSuccess       StateType = 200

	CodeLoginErr StateType = iota + 1000
	CodeNetwrokErr
	CodeTokenErr
	CodeIndexUploadErr
	CodeIndexDownloadErr
	CodeIndexFormatErr
	CodeModuleUploadErr
	CodeModuleDownloadErr
	CodeModuleSetErr
	CodeModuleGetErr
	CodeModuleFormatErr
)

func (code StateType) Message() string {
	switch code {
	case CodeSynchronizing:
		return "Synchronizing"
	case CodeSuccess:
		return "Synchronized"
	}
	return ""
}

func (e SyncError) Error() string {
	data, _ := json.Marshal(&e)
	return string(data)
}

func NewSyncError(code StateType, module, msg string) *SyncError {
	return &SyncError{
		Code:    code,
		Module:  module,
		Message: msg,
	}
}

func (m *Manager) Sync() (bool, *SyncError) {
	Logger.Debug("[Modules] start sync...")
	if !m.SwitcherGet(switcherEnabled) {
		Logger.Debug("[Modules] sync has been disabled")
		return false, nil
	}
	cloud, err := m.getCloudStorage()
	if cloud == nil {
		return false, NewSyncError(CodeTokenErr, "", err.Error())
	}
	idxInfo, serr := m.getIndexCloudInfo(cloud)
	if err != nil {
		return false, serr
	}
	if idxInfo == nil {
		// first sync
		idxInfo = &ModuleIndex{}
	}
	// Logger.Debug("[Sync] index info:", *idxInfo)

	var idxSet = make(map[string]string)
	for _, item := range m.items {
		if !m.SwitcherGet(item.Name()) {
			Logger.Debugf("The module(%s) has been disabled",
				item.Name())
			continue
		}
		itemVersion, e := m.SyncModule(cloud, item,
			idxInfo.Get(item.Name()))
		if e != nil {
			Logger.Warning("[Modules] [Sync] Failed:", item.Name(), e)
			continue
		}
		if len(itemVersion) != 0 {
			Logger.Debug("[Sync] module changed:", item.Name(), itemVersion)
			idxSet[item.Name()] = itemVersion
		}
	}
	if len(idxSet) == 0 {
		Logger.Debug("[Modules] no changes")
		return false, nil
	}

	// update index
	for k, v := range idxSet {
		idxInfo.Update(k, v)
	}
	// Logger.Debug("[Modules] will update index to:", string(idxInfo.Bytes()))
	_, err = cloud.Set(cloudKeyIndex, idxInfo.Bytes(),
		&storage.SetOptions{NewVersion: true})
	if err != nil {
		return false, NewSyncError(CodeIndexUploadErr,
			cloudKeyIndex, err.Error())
	}
	return true, nil
}

func (index *ModuleIndex) Update(name, version string) bool {
	var changed = false
	switch name {
	case "appearance":
		changed = updateStrValue(&index.Appearance, version)
	case "audio":
		changed = updateStrValue(&index.Audio, version)
	case "background":
		changed = updateStrValue(&index.Background, version)
	case "dock":
		changed = updateStrValue(&index.Dock, version)
	case "launcher":
		changed = updateStrValue(&index.Launcher, version)
	case "network":
		changed = updateStrValue(&index.Network, version)
	case "peripherals":
		changed = updateStrValue(&index.Peripherals, version)
	case "power":
		changed = updateStrValue(&index.Power, version)
	case "screen_edge":
		changed = updateStrValue(&index.ScreenEdge, version)
	case "screensaver":
		changed = updateStrValue(&index.Screensaver, version)
	case "updater":
		changed = updateStrValue(&index.Updater, version)
	}
	return changed
}

func (index *ModuleIndex) Get(name string) string {
	switch name {
	case "appearance":
		return index.Appearance
	case "audio":
		return index.Audio
	case "background":
		return index.Background
	case "dock":
		return index.Dock
	case "launcher":
		return index.Launcher
	case "network":
		return index.Network
	case "peripherals":
		return index.Peripherals
	case "power":
		return index.Power
	case "screen_edge":
		return index.ScreenEdge
	case "screensaver":
		return index.Screensaver
	case "updater":
		return index.Updater
	}
	return ""
}

func (index *ModuleIndex) Bytes() []byte {
	data, _ := json.Marshal(index)
	return data
}

func updateStrValue(src *string, value string) bool {
	if *src == value {
		return false
	}
	*src = value
	return true
}

func jsonMarshal(v interface{}) []byte {
	data, _ := json.Marshal(v)
	return data
}
