/*
 * Copyright (C) 2002-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of 
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * Derived from QEMU sources.
 * Modified for FAUmachine by Volkmar Sieh.
 */

#ifndef __CPU_CORE_H_INCLUDED 
#define __CPU_CORE_H_INCLUDED 

extern void __attribute__((__noreturn__))
CHIP_(raise_interrupt)(struct cpssp *cpssp, int intno, int is_int, int error_code, int next_eip_addend);
extern void __attribute__((__noreturn__))
CHIP_(raise_exception_err)(struct cpssp *cpssp, int exception_index, int error_code);
extern void __attribute__((__noreturn__))
CHIP_(raise_exception)(struct cpssp *cpssp, int exception_index);

/*
 * Types
 */
typedef struct CCTable {
	int (*compute_all)(struct cpssp *cpssp); /* return all the flags */
	int (*compute_c)(struct cpssp *cpssp);  /* return the C flag */
} CCTable;

/*
 * Variables
 */
extern const CCTable CHIP_(cc_table)[];

extern const uint8_t CHIP_(parity_table)[];

/*
 * Inline functions
 */
static inline uint32_t
compute_all(struct cpssp *cpssp)
{
	return CHIP_(cc_table)[CC_OP(cpssp)].compute_all(cpssp);
}

static inline uint32_t compute_eflags(struct cpssp *cpssp)
{
	return cpssp->eflags | compute_all(cpssp) | (DF(cpssp) & CPU_DF_MASK);
}

/* NOTE: CC_OP must be modified manually to CC_OP_EFLAGS */
static inline void
load_eflags(struct cpssp *cpssp, int eflags, int update_mask)
{
	cpssp->cc_src = eflags & (CC_O | CC_S | CC_Z | CC_A | CC_P | CC_C);
	cpssp->df = 1 - (2 * ((eflags >> 10) & 1));
	cpssp->eflags = (cpssp->eflags & ~update_mask) | (eflags & update_mask);
}

/* n must be a constant to be efficient */
static inline target_long
lshift(target_long x, int n)
{
	if (n >= 0)
		return x << n;
	else
		return x >> (-n);
}

/*
 * Global functions
 */
extern void
CHIP_(load_seg)(struct cpssp *cpssp, int seg_reg, int selector);
extern void
CHIP_(helper_ljmp_protected_T0_T1)(struct cpssp *cpssp, int next_eip_addend, uint16_t t0, target_ulong t1);
extern void
CHIP_(helper_lcall_real_T0_T1)(struct cpssp *cpssp, int shift, int next_eip, uint16_t t0, target_ulong t1);
extern void
CHIP_(helper_lcall_protected_T0_T1)(struct cpssp *cpssp, int shift, int next_eip_addend, uint16_t t0, target_ulong t1);
extern void
CHIP_(helper_iret_real)(struct cpssp *cpssp, int shift);
extern void
CHIP_(helper_iret_protected)(struct cpssp *cpssp, int shift, int next_eip);
extern void
CHIP_(helper_lret_protected)(struct cpssp *cpssp, int shift, int addend);
extern void
CHIP_(helper_lldt_T0)(struct cpssp *cpssp, uint16_t t0);
extern void
CHIP_(helper_ltr_T0)(struct cpssp *cpssp, uint16_t t0);
extern void
CHIP_(helper_movl_crN_T0)(struct cpssp *cpssp, int reg, target_ulong t0);
extern void
CHIP_(helper_movl_drN_T0)(struct cpssp *cpssp, int reg, target_ulong t0);
#if 80486 <= CONFIG_CPU
extern void
CHIP_(helper_invlpg)(struct cpssp *cpssp, target_ulong addr);
#endif
extern void
CHIP_(lock)(struct cpssp *cpssp);
extern void
CHIP_(unlock)(struct cpssp *cpssp);
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
extern void
CHIP_(helper_divq_EAX_T0)(struct cpssp *cpssp, uint64_t t0);
extern void
CHIP_(helper_idivq_EAX_T0)(struct cpssp *cpssp, uint64_t t0);
extern void
CHIP_(helper_mulq_EAX_T0)(struct cpssp *cpssp, uint64_t t0);
extern void
CHIP_(helper_imulq_EAX_T0)(struct cpssp *cpssp, uint64_t t0);
extern uint64_t
CHIP_(helper_imulq_T0_T1)(struct cpssp *cpssp, uint64_t t0, uint64_t t1);
extern uint64_t
CHIP_(helper_bswapq_T0)(struct cpssp *cpssp, uint64_t t0);
#endif /* CONFIG_CPU_LM_SUPPORT */
extern void
CHIP_(helper_cmpxchg8b)(struct cpssp *cpssp, target_ulong a0);
extern void
CHIP_(helper_cpuid)(struct cpssp *cpssp);
extern void
CHIP_(helper_enter_level)(struct cpssp *cpssp, int level, int data32, target_ulong t1);
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
extern void
CHIP_(helper_enter64_level)(struct cpssp *cpssp, int level, int data64, target_ulong t1);
#endif /* CONFIG_CPU_LM_SUPPORT */
extern void
CHIP_(helper_syscall)(struct cpssp *cpssp, int next_eip_addend);
extern void
CHIP_(helper_sysret)(struct cpssp *cpssp, int dflag);
extern void
CHIP_(helper_sysenter)(struct cpssp *cpssp);
extern void
CHIP_(helper_sysexit)(struct cpssp *cpssp);
extern void
CHIP_(helper_rdtsc)(struct cpssp *cpssp);
extern void
CHIP_(helper_rdmsr)(struct cpssp *cpssp);
extern void
CHIP_(helper_wrmsr)(struct cpssp *cpssp);
extern uint32_t
CHIP_(helper_lsl)(struct cpssp *cpssp, uint16_t t0);
extern uint32_t
CHIP_(helper_lar)(struct cpssp *cpssp, uint16_t t0);
extern void
CHIP_(helper_verr)(struct cpssp *cpssp, uint16_t t0);
extern void
CHIP_(helper_verw)(struct cpssp *cpssp, uint16_t t0);
extern void
CHIP_(helper_rsm)(struct cpssp *cpssp);
extern void
CHIP_(check_iob_T0)(struct cpssp *cpssp, uint16_t t0);
extern void
CHIP_(check_iow_T0)(struct cpssp *cpssp, uint16_t t0);
extern void
CHIP_(check_iol_T0)(struct cpssp *cpssp, uint16_t t0);
extern void
CHIP_(check_iob_DX)(struct cpssp *cpssp);
extern void
CHIP_(check_iow_DX)(struct cpssp *cpssp);
extern void
CHIP_(check_iol_DX)(struct cpssp *cpssp);

#endif /* __CPU_CORE_H_INCLUDED */
