# 43 "clexer.mll"
 
open Cparser
open Pretty
exception Eof
exception InternalError of string
module E = Errormsg
module H = Hashtbl

let matchingParsOpen = ref 0

let currentLoc () = Cabshelper.currentLoc ()

(* string -> unit *)
let addComment c =
  let l = currentLoc() in
  let i = GrowArray.max_init_index Cabshelper.commentsGA in
  GrowArray.setg Cabshelper.commentsGA (i+1) (l,c,false)

(* track whitespace for the current token *)
let white = ref ""  
let addWhite lexbuf =
    let w = Lexing.lexeme lexbuf in 
    white := !white ^ w
let clear_white () = white := ""
let get_white () = !white

let lexeme = ref ""
let addLexeme lexbuf =
    let l = Lexing.lexeme lexbuf in
    lexeme := !lexeme ^ l
let clear_lexeme () = lexeme := ""
let get_extra_lexeme () = !lexeme 

let int64_to_char value =
  if (compare value (Int64.of_int 255) > 0) || (compare value Int64.zero < 0) then
    begin
      let msg = Printf.sprintf "clexer:intlist_to_string: character 0x%Lx too big" value in
      E.parse_error msg;
    end
  else
    Char.chr (Int64.to_int value)

(* takes a not-nul-terminated list, and converts it to a string. *)
let rec intlist_to_string (str: int64 list):string =
  match str with
    [] -> ""  (* add nul-termination *)
  | value::rest ->
      let this_char = int64_to_char value in
      (String.make 1 this_char) ^ (intlist_to_string rest)

(* Some debugging support for line numbers *)
let dbgToken (t: token) = 
  if false then begin
    ignore (E.log "%a" insert
              (match t with 
                IDENT (n, l) -> dprintf "IDENT(%s,%d)\n" n l.Cabs.lineno
              | LBRACE l -> dprintf "LBRACE(%d)\n" l.Cabs.lineno
              | RBRACE l -> dprintf "RBRACE(%d)\n" l.Cabs.lineno
              | IF l -> dprintf "IF(%d)\n" l.Cabs.lineno
              | SWITCH l -> dprintf "SWITCH(%d)\n" l.Cabs.lineno
              | RETURN l -> dprintf "RETURN(%d)\n" l.Cabs.lineno
              | _ -> nil));
    t
  end else
    t


(*
** Keyword hashtable
*)
let lexicon = H.create 211
let init_lexicon _ =
  H.clear lexicon;
  List.iter 
    (fun (key, builder) -> H.add lexicon key builder)
    [ ("auto", fun loc -> AUTO loc);
      ("const", fun loc -> CONST loc);
      ("__const", fun loc -> CONST loc);
      ("__const__", fun loc -> CONST loc);
      ("static", fun loc -> STATIC loc);
      ("extern", fun loc -> EXTERN loc);
      ("long", fun loc -> LONG loc);
      ("short", fun loc -> SHORT loc);
      ("register", fun loc -> REGISTER loc);
      ("signed", fun loc -> SIGNED loc);
      ("__signed", fun loc -> SIGNED loc);
      ("unsigned", fun loc -> UNSIGNED loc);
      ("volatile", fun loc -> VOLATILE loc);
      ("__volatile", fun loc -> VOLATILE loc);
      (* WW: see /usr/include/sys/cdefs.h for why __signed and __volatile
       * are accepted GCC-isms *)
      ("char", fun loc -> CHAR loc);
      ("_Bool", fun loc -> BOOL loc);
      ("int", fun loc -> INT loc);
      ("float", fun loc -> FLOAT loc);
      ("double", fun loc -> DOUBLE loc);
      ("void", fun loc -> VOID loc);
      ("enum", fun loc -> ENUM loc);
      ("struct", fun loc -> STRUCT loc);
      ("typedef", fun loc -> TYPEDEF loc);
      ("union", fun loc -> UNION loc);
      ("break", fun loc -> BREAK loc);
      ("continue", fun loc -> CONTINUE loc);
      ("goto", fun loc -> GOTO loc); 
      ("return", fun loc -> dbgToken (RETURN loc));
      ("switch", fun loc -> dbgToken (SWITCH loc));
      ("case", fun loc -> CASE loc); 
      ("default", fun loc -> DEFAULT loc);
      ("while", fun loc -> WHILE loc);  
      ("do", fun loc -> DO loc);  
      ("for", fun loc -> FOR loc);
      ("if", fun loc -> dbgToken (IF loc));
      ("else", fun _ -> ELSE);
      (*** Implementation specific keywords ***)
      ("__signed__", fun loc -> SIGNED loc);
      ("__inline__", fun loc -> INLINE loc);
      ("inline", fun loc -> INLINE loc); 
      ("__inline", fun loc -> INLINE loc);
      ("_inline", fun loc ->
                      if !Cprint.msvcMode then 
                        INLINE loc
                      else 
                        IDENT ("_inline", loc));
      ("__attribute__", fun loc -> ATTRIBUTE loc);
      ("__attribute", fun loc -> ATTRIBUTE loc);
(*
      ("__attribute_used__", fun loc -> ATTRIBUTE_USED loc);
*)
      ("__blockattribute__", fun _ -> BLOCKATTRIBUTE);
      ("__blockattribute", fun _ -> BLOCKATTRIBUTE);
      ("__asm__", fun loc -> ASM loc);
      ("asm", fun loc -> ASM loc);
      ("__typeof__", fun loc -> TYPEOF loc);
      ("__typeof", fun loc -> TYPEOF loc);
      ("typeof", fun loc -> TYPEOF loc); 
      ("__alignof", fun loc -> ALIGNOF loc);
      ("__alignof__", fun loc -> ALIGNOF loc);
      ("__volatile__", fun loc -> VOLATILE loc);
      ("__volatile", fun loc -> VOLATILE loc);

      ("__FUNCTION__", fun loc -> FUNCTION__ loc);
      ("__func__", fun loc -> FUNCTION__ loc); (* ISO 6.4.2.2 *)
      ("__PRETTY_FUNCTION__", fun loc -> PRETTY_FUNCTION__ loc);
      ("__label__", fun _ -> LABEL__);
      (*** weimer: GCC arcana ***)
      ("__restrict", fun loc -> RESTRICT loc);
      ("restrict", fun loc -> RESTRICT loc);
(*      ("__extension__", EXTENSION); *)
      (**** MS VC ***)
      ("__int64", fun _ -> INT64 (currentLoc ()));
      ("__int32", fun loc -> INT loc);
      ("_cdecl",  fun _ -> MSATTR ("_cdecl", currentLoc ())); 
      ("__cdecl", fun _ -> MSATTR ("__cdecl", currentLoc ()));
      ("_stdcall", fun _ -> MSATTR ("_stdcall", currentLoc ())); 
      ("__stdcall", fun _ -> MSATTR ("__stdcall", currentLoc ()));
      ("_fastcall", fun _ -> MSATTR ("_fastcall", currentLoc ())); 
      ("__fastcall", fun _ -> MSATTR ("__fastcall", currentLoc ()));
      ("__w64", fun _ -> MSATTR("__w64", currentLoc ()));
      ("__declspec", fun loc -> DECLSPEC loc);
      ("__forceinline", fun loc -> INLINE loc); (* !! we turn forceinline 
                                                 * into inline *)
      ("__try", fun loc -> TRY loc);
      ("__except", fun loc -> EXCEPT loc);
      ("__finally", fun loc -> FINALLY loc);
      (* weimer: some files produced by 'GCC -E' expect this type to be
       * defined *)
      ("__builtin_va_list", 
       fun _ -> NAMED_TYPE ("__builtin_va_list", currentLoc ()));
      ("__builtin_va_arg", fun loc -> BUILTIN_VA_ARG loc);
      ("__builtin_types_compatible_p", fun loc -> BUILTIN_TYPES_COMPAT loc);
      ("__builtin_offsetof", fun loc -> BUILTIN_OFFSETOF loc);
      (* On some versions of GCC __thread is a regular identifier *)
      ("__thread", fun loc -> 
                      if Machdep.__thread_is_keyword then 
                         THREAD loc
                       else 
                         IDENT ("__thread", loc));
    ]

(* Mark an identifier as a type name. The old mapping is preserved and will 
 * be reinstated when we exit this context *)
let add_type name =
   (* ignore (print_string ("adding type name " ^ name ^ "\n"));  *)
   H.add lexicon name (fun loc -> NAMED_TYPE (name, loc))

let context : string list list ref = ref []

let push_context _ = context := []::!context

let pop_context _ = 
  match !context with
    [] -> raise (InternalError "Empty context stack")
  | con::sub ->
		(context := sub;
		List.iter (fun name -> 
                           (* ignore (print_string ("removing lexicon for " ^ name ^ "\n")); *)
                            H.remove lexicon name) con)

(* Mark an identifier as a variable name. The old mapping is preserved and 
 * will be reinstated when we exit this context  *)
let add_identifier name =
  match !context with
    [] -> () (* Just ignore raise (InternalError "Empty context stack") *)
  | con::sub ->
      (context := (name::con)::sub;
       (*                print_string ("adding IDENT for " ^ name ^ "\n"); *)
       H.add lexicon name (fun loc -> 
         dbgToken (IDENT (name, loc))))


(*
** Useful primitives
*)
let scan_ident id =
  let here = currentLoc () in
  try (H.find lexicon id) here
  (* default to variable name, as opposed to type *)
  with Not_found -> dbgToken (IDENT (id, here))


(*
** Buffer processor
*)
 

let init ~(filename: string) : Lexing.lexbuf =
  init_lexicon ();
  (* Inititialize the pointer in Errormsg *)
  Lexerhack.add_type := add_type;
  Lexerhack.push_context := push_context;
  Lexerhack.pop_context := pop_context;
  Lexerhack.add_identifier := add_identifier;
  E.startParsing filename


let finish () = 
  E.finishParsing ()

(*** Error handling ***)
let error msg =
  E.parse_error msg


(*** escape character management ***)
let scan_escape (char: char) : int64 =
  let result = match char with
    'n' -> '\n'
  | 'r' -> '\r'
  | 't' -> '\t'
  | 'b' -> '\b'
  | 'f' -> '\012'  (* ASCII code 12 *)
  | 'v' -> '\011'  (* ASCII code 11 *)
  | 'a' -> '\007'  (* ASCII code 7 *)
  | 'e' | 'E' -> '\027'  (* ASCII code 27. This is a GCC extension *)
  | '\'' -> '\''    
  | '"'-> '"'     (* '"' *)
  | '?' -> '?'
  | '(' when not !Cprint.msvcMode -> '('
  | '{' when not !Cprint.msvcMode -> '{'
  | '[' when not !Cprint.msvcMode -> '['
  | '%' when not !Cprint.msvcMode -> '%'
  | '\\' -> '\\' 
  | other -> error ("Unrecognized escape sequence: \\" ^ (String.make 1 other))
  in
  Int64.of_int (Char.code result)

let scan_hex_escape str =
  let radix = Int64.of_int 16 in
  let the_value = ref Int64.zero in
  (* start at character 2 to skip the \x *)
  for i = 2 to (String.length str) - 1 do
    let thisDigit = Cabshelper.valueOfDigit (String.get str i) in
    (* the_value := !the_value * 16 + thisDigit *)
    the_value := Int64.add (Int64.mul !the_value radix) thisDigit
  done;
  !the_value

let scan_oct_escape str =
  let radix = Int64.of_int 8 in
  let the_value = ref Int64.zero in
  (* start at character 1 to skip the \x *)
  for i = 1 to (String.length str) - 1 do
    let thisDigit = Cabshelper.valueOfDigit (String.get str i) in
    (* the_value := !the_value * 8 + thisDigit *)
    the_value := Int64.add (Int64.mul !the_value radix) thisDigit
  done;
  !the_value

let lex_hex_escape remainder lexbuf =
  let prefix = scan_hex_escape (Lexing.lexeme lexbuf) in
  prefix :: remainder lexbuf

let lex_oct_escape remainder lexbuf =
  let prefix = scan_oct_escape (Lexing.lexeme lexbuf) in
  prefix :: remainder lexbuf

let lex_simple_escape remainder lexbuf =
  let lexchar = Lexing.lexeme_char lexbuf 1 in
  let prefix = scan_escape lexchar in
  prefix :: remainder lexbuf

let lex_unescaped remainder lexbuf =
  let prefix = Int64.of_int (Char.code (Lexing.lexeme_char lexbuf 0)) in
  prefix :: remainder lexbuf

let lex_comment remainder lexbuf =
  let ch = Lexing.lexeme_char lexbuf 0 in
  let prefix = Int64.of_int (Char.code ch) in
  if ch = '\n' then E.newline();
  prefix :: remainder lexbuf

let make_char (i:int64):char =
  let min_val = Int64.zero in
  let max_val = Int64.of_int 255 in
  (* if i < 0 || i > 255 then error*)
  if compare i min_val < 0 || compare i max_val > 0 then begin
    let msg = Printf.sprintf "clexer:make_char: character 0x%Lx too big" i in
    error msg
  end;
  Char.chr (Int64.to_int i)


(* ISO standard locale-specific function to convert a wide character
 * into a sequence of normal characters. Here we work on strings. 
 * We convert L"Hi" to "H\000i\000" 
  matth: this seems unused.
let wbtowc wstr =
  let len = String.length wstr in 
  let dest = String.make (len * 2) '\000' in 
  for i = 0 to len-1 do 
    dest.[i*2] <- wstr.[i] ;
  done ;
  dest
*)

(* This function converst the "Hi" in L"Hi" to { L'H', L'i', L'\0' }
  matth: this seems unused.
let wstr_to_warray wstr =
  let len = String.length wstr in
  let res = ref "{ " in
  for i = 0 to len-1 do
    res := !res ^ (Printf.sprintf "L'%c', " wstr.[i])
  done ;
  res := !res ^ "}" ;
  !res
*)

(* Pragmas get explicit end-of-line tokens.
 * Elsewhere they are silently discarded as whitespace. *)
let pragmaLine = ref false


# 355 "clexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base = 
   "\000\000\182\255\183\255\091\000\113\000\194\000\195\255\196\255\
    \197\255\198\255\199\255\200\255\201\255\202\255\203\255\204\255\
    \205\255\035\000\067\000\070\000\071\000\177\000\092\000\072\000\
    \073\000\090\000\093\000\121\000\015\001\027\001\065\001\246\255\
    \138\001\248\255\225\001\250\255\001\000\252\255\208\000\180\000\
    \235\255\254\255\255\255\251\255\005\000\056\002\143\002\230\002\
    \061\003\148\003\235\003\066\004\153\004\240\004\071\005\158\005\
    \245\005\076\006\163\006\250\006\081\007\168\007\255\007\086\008\
    \173\008\004\009\091\009\178\009\009\010\096\010\183\010\014\011\
    \245\255\247\255\091\011\106\001\125\011\083\000\028\001\137\011\
    \117\011\191\011\084\000\154\011\189\011\183\011\185\000\079\001\
    \099\001\039\001\097\000\186\000\203\011\100\001\103\011\098\000\
    \244\255\006\012\017\012\046\012\056\012\038\001\085\012\214\001\
    \095\012\045\002\156\012\194\012\101\000\022\001\025\001\079\011\
    \124\012\233\012\132\002\243\255\102\000\107\000\076\000\133\000\
    \008\013\041\013\073\013\219\002\172\000\239\255\225\255\107\000\
    \120\000\110\000\200\000\240\255\219\255\238\255\217\255\218\255\
    \237\255\236\255\234\255\209\255\233\255\210\255\232\255\231\255\
    \224\255\174\000\230\255\223\255\175\000\229\255\226\255\117\013\
    \204\013\035\014\122\014\209\014\140\000\119\000\140\000\139\000\
    \160\000\148\000\170\000\186\000\179\000\204\000\246\000\031\001\
    \238\000\249\000\248\000\189\255\006\001\010\001\018\001\022\001\
    \021\001\030\001\023\001\188\255\026\001\025\001\187\255\034\001\
    \051\001\186\255\190\001\118\001\021\002\067\015\249\255\125\001\
    \253\255\089\003\072\015\057\001\073\001\082\015\091\003\125\001\
    \126\001\086\011\099\011\112\001\105\001\075\001\132\001\088\001\
    \098\001\133\001\169\001\174\001\187\001\176\003\077\015\218\001\
    \185\001\173\001\174\001\204\001\216\001\218\001\252\001\017\002\
    \224\001\245\001\005\002\011\002\007\002\016\002\251\255\013\002\
    \080\002\047\002\039\002\034\002\053\002\085\002\097\002\092\002\
    \094\002\089\002\101\002\096\002\124\002\126\002\135\002\142\002\
    \124\002\187\002\190\002\174\002\186\002\181\002\183\002\194\002\
    \220\002\220\002\218\002\214\002\225\002\004\003\004\003\242\002\
    \247\002\254\002\022\003\011\003\021\003\041\003\030\003\055\003\
    \045\003\049\003\052\003\047\003\212\010\042\011\178\003\219\011\
    \196\003\197\003\153\010\180\012\224\003\152\015\175\015\055\004\
    \150\010\207\015\229\004\232\015\031\016\060\005\046\011\181\004\
    \070\003\013\011\091\003\095\003\078\003\085\003";
  Lexing.lex_backtrk = 
   "\255\255\255\255\255\255\071\000\073\000\071\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\033\000\035\000\034\000\049\000\047\000\048\000\043\000\
    \041\000\040\000\039\000\044\000\061\000\014\000\014\000\255\255\
    \071\000\255\255\071\000\255\255\073\000\255\255\002\000\042\000\
    \255\255\255\255\255\255\255\255\255\255\071\000\071\000\071\000\
    \071\000\071\000\071\000\006\000\071\000\071\000\071\000\071\000\
    \063\000\071\000\071\000\071\000\071\000\064\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\070\000\
    \255\255\255\255\255\255\255\255\011\000\255\255\014\000\014\000\
    \014\000\013\000\255\255\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\255\255\014\000\014\000\014\000\014\000\255\255\
    \255\255\255\255\011\000\255\255\011\000\255\255\011\000\255\255\
    \011\000\255\255\255\255\012\000\255\255\012\000\012\000\012\000\
    \255\255\011\000\255\255\255\255\012\000\012\000\012\000\255\255\
    \255\255\255\255\011\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\027\000\255\255\255\255\028\000\255\255\255\255\071\000\
    \071\000\071\000\071\000\062\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\065\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\001\000\255\255\255\255\255\255\006\000\
    \255\255\001\000\255\255\006\000\006\000\002\000\001\000\255\255\
    \002\000\002\000\002\000\002\000\002\000\002\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\005\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\003\000\001\000\255\255\
    \255\255\255\255\255\255\004\000\002\000\003\000\001\000\002\000\
    \255\255\004\000\002\000\003\000\001\000\002\000\255\255\000\000\
    \255\255\255\255\002\000\255\255\255\255\255\255";
  Lexing.lex_default = 
   "\001\000\000\000\000\000\255\255\255\255\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\000\000\255\255\000\000\255\255\000\000\255\255\255\255\
    \000\000\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\000\000\255\255\
    \255\255\255\255\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\000\000\000\000\255\255\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\000\000\255\255\
    \255\255\000\000\041\000\255\255\041\000\190\000\000\000\255\255\
    \000\000\255\255\190\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\037\000\023\001\255\255\023\001\
    \192\000\041\000\043\000\037\000\255\255\255\255\255\255\255\255\
    \043\000\037\000\255\255\255\255\255\255\255\255\192\000\255\255\
    \041\000\192\000\255\255\255\255\255\255\255\255";
  Lexing.lex_trans = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\038\000\037\000\043\000\038\000\038\000\044\000\043\000\
    \000\000\000\000\044\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \038\000\027\000\031\000\035\000\003\000\023\000\021\000\033\000\
    \009\000\008\000\024\000\026\000\006\000\025\000\028\000\039\000\
    \030\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\015\000\007\000\019\000\017\000\018\000\016\000\
    \004\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\032\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\011\000\036\000\010\000\020\000\034\000\
    \150\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\005\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\013\000\022\000\012\000\014\000\003\000\
    \147\000\148\000\145\000\144\000\143\000\138\000\137\000\135\000\
    \132\000\095\000\090\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\086\000\091\000\136\000\
    \134\000\140\000\133\000\119\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\126\000\115\000\
    \115\000\115\000\003\000\115\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\157\000\141\000\
    \139\000\038\000\125\000\115\000\038\000\038\000\042\000\156\000\
    \128\000\129\000\130\000\041\000\158\000\159\000\003\000\076\000\
    \076\000\131\000\127\000\146\000\149\000\183\000\142\000\180\000\
    \038\000\040\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\172\000\160\000\075\000\075\000\
    \002\000\161\000\162\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\163\000\075\000\075\000\
    \164\000\003\000\165\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\151\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\124\000\166\000\098\000\
    \098\000\098\000\098\000\098\000\098\000\098\000\098\000\098\000\
    \098\000\076\000\076\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\076\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \075\000\075\000\117\000\167\000\168\000\116\000\169\000\079\000\
    \093\000\170\000\171\000\173\000\075\000\174\000\115\000\076\000\
    \078\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\029\000\029\000\175\000\176\000\076\000\177\000\108\000\
    \075\000\075\000\118\000\178\000\077\000\077\000\075\000\080\000\
    \094\000\179\000\181\000\182\000\075\000\079\000\115\000\184\000\
    \078\000\076\000\076\000\086\000\075\000\105\000\078\000\105\000\
    \185\000\074\000\104\000\104\000\104\000\104\000\104\000\104\000\
    \104\000\104\000\104\000\104\000\086\000\042\000\075\000\043\000\
    \075\000\075\000\077\000\209\000\072\000\080\000\003\000\086\000\
    \091\000\073\000\207\000\206\000\075\000\192\000\078\000\192\000\
    \192\000\074\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\086\000\192\000\208\000\037\000\
    \075\000\075\000\204\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\192\000\210\000\199\000\
    \187\000\003\000\205\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \211\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\212\000\213\000\013\001\255\000\042\000\
    \246\000\245\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\046\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\241\000\236\000\233\000\232\000\
    \045\000\231\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\104\000\104\000\104\000\
    \104\000\104\000\104\000\104\000\104\000\104\000\104\000\226\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\225\000\227\000\228\000\229\000\224\000\230\000\
    \230\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\230\000\234\000\235\000\230\000\003\000\
    \237\000\054\000\003\000\003\000\003\000\052\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \053\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\113\000\113\000\113\000\113\000\
    \113\000\113\000\113\000\113\000\113\000\113\000\255\255\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\238\000\239\000\240\000\230\000\242\000\243\000\244\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\230\000\230\000\247\000\248\000\003\000\249\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\047\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\042\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \250\000\251\000\252\000\253\000\254\000\230\000\000\001\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\001\001\002\001\003\001\004\001\003\000\005\001\048\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\193\000\006\001\198\000\193\000\193\000\198\000\
    \198\000\007\001\008\001\009\001\010\001\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\011\001\
    \012\001\193\000\230\000\198\000\014\001\015\001\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \016\001\017\001\018\001\019\001\003\000\230\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\049\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\214\000\043\001\022\001\214\000\214\000\022\001\022\001\
    \044\001\045\001\041\000\042\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\042\000\042\000\
    \214\000\000\000\022\001\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\003\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\050\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \031\001\031\001\031\001\031\001\031\001\031\001\031\001\031\001\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\255\255\000\000\
    \000\000\000\000\003\000\000\000\051\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\003\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\039\001\000\000\
    \000\000\039\001\039\001\000\000\041\000\255\255\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\039\001\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \003\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\062\000\003\000\003\000\003\000\037\001\037\001\037\001\
    \037\001\037\001\037\001\037\001\037\001\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\003\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\057\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\000\000\000\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\003\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\055\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\003\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\056\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\003\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\003\000\000\000\058\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\003\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\059\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\003\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\060\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \003\000\000\000\061\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\003\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\003\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\063\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\003\000\000\000\003\000\
    \003\000\003\000\003\000\064\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\003\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\065\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\003\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\066\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\003\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\067\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\003\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\068\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \003\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\069\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\042\000\000\000\042\000\000\000\070\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\022\001\042\000\000\000\
    \022\001\022\001\000\000\000\000\000\000\000\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\033\001\000\000\022\001\027\001\021\001\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\071\000\042\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\255\255\000\000\000\000\255\255\039\001\
    \000\000\000\000\039\001\039\001\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \042\000\000\000\000\000\000\000\192\000\000\000\039\001\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\042\001\003\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\106\000\042\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\076\000\255\255\000\000\
    \000\000\255\255\000\000\000\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\203\000\076\000\115\000\000\000\000\000\000\000\
    \000\000\041\000\000\000\192\000\075\000\098\000\098\000\098\000\
    \098\000\098\000\098\000\098\000\098\000\098\000\098\000\076\000\
    \192\000\000\000\075\000\116\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\097\000\096\000\115\000\000\000\000\000\000\000\
    \076\000\096\000\091\000\192\000\075\000\000\000\075\000\203\000\
    \000\000\000\000\000\000\091\000\255\255\092\000\000\000\000\000\
    \192\000\000\000\075\000\000\000\000\000\000\000\091\000\075\000\
    \000\000\092\000\097\000\096\000\255\255\076\000\088\000\255\255\
    \000\000\096\000\091\000\076\000\000\000\076\000\075\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\029\000\
    \029\000\076\000\000\000\000\000\075\000\192\000\091\000\075\000\
    \000\000\000\000\075\000\082\000\075\000\000\000\089\000\000\000\
    \000\000\087\000\000\000\084\000\086\000\255\255\000\000\000\000\
    \075\000\000\000\086\000\000\000\083\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\075\000\000\000\000\000\000\000\
    \091\000\000\000\075\000\087\000\075\000\000\000\000\000\000\000\
    \082\000\000\000\255\255\085\000\086\000\000\000\255\255\000\000\
    \075\000\103\000\086\000\103\000\083\000\000\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \091\000\098\000\098\000\098\000\098\000\098\000\098\000\098\000\
    \098\000\098\000\098\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\099\000\096\000\
    \000\000\101\000\000\000\101\000\000\000\096\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\000\000\000\000\000\000\000\000\099\000\096\000\
    \000\000\000\000\000\000\000\000\000\000\096\000\096\000\000\000\
    \000\000\000\000\000\000\000\000\096\000\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\104\000\
    \104\000\104\000\104\000\104\000\104\000\104\000\104\000\104\000\
    \104\000\000\000\000\000\096\000\000\000\000\000\096\000\000\000\
    \000\000\096\000\000\000\000\000\096\000\096\000\000\000\114\000\
    \000\000\114\000\000\000\096\000\113\000\113\000\113\000\113\000\
    \113\000\113\000\113\000\113\000\113\000\113\000\000\000\000\000\
    \000\000\000\000\000\000\096\000\000\000\000\000\000\000\000\000\
    \000\000\096\000\000\000\000\000\000\000\096\000\000\000\000\000\
    \000\000\000\000\000\000\096\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\000\000\028\001\028\001\028\001\028\001\
    \028\001\028\001\028\001\028\001\000\000\000\000\000\000\000\000\
    \106\000\000\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\000\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\000\000\000\000\000\000\029\001\000\000\110\000\000\000\
    \000\000\000\000\112\000\000\000\000\000\000\000\000\000\109\000\
    \000\000\113\000\113\000\113\000\113\000\113\000\113\000\113\000\
    \113\000\113\000\113\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\000\000\000\000\108\000\029\001\000\000\111\000\096\000\
    \000\000\000\000\112\000\000\000\000\000\096\000\000\000\109\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\120\000\120\000\120\000\120\000\120\000\120\000\096\000\
    \000\000\000\000\000\000\000\000\123\000\096\000\123\000\000\000\
    \121\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\120\000\120\000\120\000\120\000\120\000\120\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \121\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\096\000\
    \000\000\000\000\000\000\000\000\000\000\096\000\000\000\000\000\
    \000\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\096\000\
    \000\000\000\000\000\000\000\000\255\255\096\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\003\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\152\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\003\000\000\000\003\000\003\000\003\000\
    \003\000\153\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\003\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\154\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\003\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \155\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \003\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\193\000\042\000\000\000\193\000\
    \193\000\198\000\042\000\000\000\198\000\198\000\214\000\000\000\
    \000\000\214\000\214\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\193\000\000\000\035\000\000\000\000\000\
    \198\000\000\000\000\000\192\000\037\000\214\000\000\000\000\000\
    \000\000\000\000\191\000\000\000\000\000\000\000\000\000\000\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\222\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\201\000\000\000\
    \000\000\215\000\000\000\000\000\000\000\000\000\000\000\200\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\218\000\000\000\
    \000\000\000\000\000\000\000\000\196\000\000\000\221\000\000\000\
    \195\000\000\000\000\000\199\000\219\000\000\000\202\000\216\000\
    \220\000\000\000\217\000\000\000\223\000\000\000\000\000\200\000\
    \030\001\030\001\030\001\030\001\030\001\030\001\030\001\030\001\
    \030\001\030\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\030\001\030\001\030\001\030\001\030\001\030\001\030\001\
    \030\001\030\001\030\001\030\001\030\001\030\001\030\001\030\001\
    \030\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \030\001\030\001\030\001\030\001\030\001\030\001\000\000\000\000\
    \000\000\030\001\030\001\030\001\030\001\030\001\030\001\034\001\
    \034\001\034\001\034\001\034\001\034\001\034\001\034\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \030\001\030\001\030\001\030\001\030\001\030\001\000\000\000\000\
    \036\001\036\001\036\001\036\001\036\001\036\001\036\001\036\001\
    \036\001\036\001\000\000\000\000\000\000\000\000\000\000\035\001\
    \000\000\036\001\036\001\036\001\036\001\036\001\036\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\000\000\000\000\035\001\
    \255\255\036\001\036\001\036\001\036\001\036\001\036\001\036\001\
    \036\001\036\001\036\001\036\001\036\001\036\001\036\001\036\001\
    \036\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \036\001\036\001\036\001\036\001\036\001\036\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \036\001\036\001\036\001\036\001\036\001\036\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    ";
  Lexing.lex_check = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\036\000\000\000\000\000\036\000\044\000\
    \255\255\255\255\044\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \017\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \018\000\018\000\019\000\019\000\020\000\023\000\024\000\025\000\
    \026\000\077\000\082\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\090\000\095\000\025\000\
    \025\000\022\000\026\000\108\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\027\000\117\000\
    \118\000\119\000\003\000\116\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\004\000\021\000\
    \022\000\038\000\124\000\116\000\038\000\038\000\039\000\004\000\
    \127\000\128\000\129\000\039\000\004\000\004\000\005\000\086\000\
    \091\000\130\000\027\000\145\000\148\000\156\000\021\000\157\000\
    \038\000\039\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\158\000\159\000\086\000\091\000\
    \000\000\160\000\161\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\162\000\086\000\091\000\
    \163\000\005\000\164\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\028\000\165\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\029\000\078\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\089\000\101\000\101\000\
    \101\000\101\000\101\000\101\000\101\000\101\000\101\000\101\000\
    \029\000\078\000\109\000\166\000\167\000\110\000\168\000\029\000\
    \078\000\169\000\170\000\172\000\089\000\173\000\110\000\030\000\
    \029\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\174\000\175\000\087\000\176\000\109\000\
    \029\000\078\000\109\000\177\000\029\000\078\000\030\000\029\000\
    \078\000\178\000\180\000\181\000\089\000\030\000\110\000\183\000\
    \029\000\088\000\093\000\089\000\087\000\075\000\030\000\075\000\
    \184\000\030\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\087\000\187\000\030\000\191\000\
    \088\000\093\000\030\000\195\000\032\000\030\000\032\000\088\000\
    \093\000\032\000\196\000\199\000\087\000\204\000\030\000\205\000\
    \206\000\030\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\087\000\203\000\207\000\208\000\
    \088\000\093\000\200\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\203\000\209\000\200\000\
    \186\000\032\000\200\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\034\000\103\000\103\000\
    \103\000\103\000\103\000\103\000\103\000\103\000\103\000\103\000\
    \210\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\211\000\212\000\215\000\216\000\188\000\
    \217\000\218\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\219\000\220\000\221\000\222\000\
    \034\000\224\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\045\000\105\000\105\000\105\000\
    \105\000\105\000\105\000\105\000\105\000\105\000\105\000\225\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\223\000\226\000\227\000\228\000\223\000\229\000\
    \231\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\232\000\233\000\234\000\235\000\045\000\
    \236\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\046\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\186\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\237\000\238\000\239\000\240\000\241\000\242\000\243\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\244\000\245\000\246\000\247\000\046\000\248\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\047\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\188\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \249\000\250\000\251\000\252\000\253\000\254\000\255\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\000\001\001\001\002\001\003\001\047\000\004\001\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\048\000\193\000\005\001\198\000\193\000\193\000\198\000\
    \198\000\006\001\007\001\008\001\009\001\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\010\001\
    \011\001\193\000\012\001\198\000\013\001\014\001\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \015\001\016\001\017\001\018\001\048\000\019\001\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \049\000\213\000\042\001\022\001\213\000\213\000\022\001\022\001\
    \043\001\044\001\045\001\040\001\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\024\001\025\001\
    \213\000\255\255\022\001\255\255\255\255\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\255\255\
    \255\255\255\255\255\255\049\000\255\255\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\050\000\
    \028\001\028\001\028\001\028\001\028\001\028\001\028\001\028\001\
    \255\255\255\255\255\255\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\040\001\255\255\
    \255\255\255\255\050\000\255\255\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\051\000\031\001\
    \031\001\031\001\031\001\031\001\031\001\031\001\031\001\255\255\
    \255\255\255\255\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\255\255\255\255\255\255\
    \255\255\051\000\255\255\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\052\000\039\001\255\255\
    \255\255\039\001\039\001\255\255\024\001\025\001\255\255\255\255\
    \255\255\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\255\255\255\255\039\001\255\255\255\255\
    \255\255\255\255\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\255\255\255\255\255\255\255\255\
    \052\000\255\255\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\053\000\034\001\034\001\034\001\
    \034\001\034\001\034\001\034\001\034\001\255\255\255\255\255\255\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\255\255\255\255\255\255\255\255\053\000\
    \255\255\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\054\000\037\001\037\001\037\001\037\001\
    \037\001\037\001\037\001\037\001\255\255\255\255\255\255\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\255\255\255\255\255\255\255\255\054\000\255\255\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\055\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\255\255\255\255\255\255\255\255\055\000\255\255\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\056\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \255\255\255\255\255\255\255\255\056\000\255\255\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \057\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\255\255\
    \255\255\255\255\255\255\057\000\255\255\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\058\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\255\255\255\255\
    \255\255\255\255\058\000\255\255\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\059\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\255\255\255\255\255\255\
    \255\255\059\000\255\255\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\060\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\255\255\255\255\255\255\255\255\
    \060\000\255\255\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\061\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\255\255\255\255\255\255\255\255\061\000\
    \255\255\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\062\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\255\255\255\255\255\255\255\255\062\000\255\255\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\063\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\255\255\255\255\255\255\255\255\063\000\255\255\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\064\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \255\255\255\255\255\255\255\255\064\000\255\255\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \065\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\255\255\
    \255\255\255\255\255\255\065\000\255\255\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\066\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\255\255\255\255\
    \255\255\255\255\066\000\255\255\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\067\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\255\255\255\255\255\255\
    \255\255\067\000\255\255\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\068\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\255\255\255\255\255\255\255\255\
    \068\000\255\255\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\069\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\026\001\255\255\032\001\255\255\069\000\
    \255\255\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\070\000\255\255\020\001\020\001\255\255\
    \020\001\020\001\255\255\255\255\255\255\255\255\255\255\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\255\255\032\001\255\255\020\001\026\001\020\001\255\255\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\255\255\255\255\255\255\255\255\070\000\041\001\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\071\000\021\001\255\255\255\255\021\001\038\001\
    \255\255\255\255\038\001\038\001\255\255\255\255\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \041\001\255\255\255\255\255\255\021\001\255\255\038\001\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\255\255\255\255\255\255\041\001\071\000\255\255\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\074\000\041\001\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\094\000\032\001\255\255\
    \255\255\026\001\255\255\255\255\074\000\074\000\074\000\074\000\
    \074\000\074\000\201\000\080\000\111\000\255\255\255\255\255\255\
    \255\255\038\001\255\255\201\000\094\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\079\000\
    \202\000\255\255\080\000\111\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\076\000\076\000\111\000\255\255\255\255\255\255\
    \083\000\076\000\080\000\201\000\094\000\255\255\079\000\202\000\
    \255\255\255\255\255\255\094\000\020\001\079\000\255\255\255\255\
    \202\000\255\255\080\000\255\255\255\255\255\255\079\000\083\000\
    \255\255\080\000\076\000\076\000\023\001\085\000\083\000\023\001\
    \255\255\076\000\080\000\084\000\255\255\081\000\079\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\092\000\255\255\255\255\085\000\023\001\079\000\083\000\
    \255\255\255\255\084\000\083\000\081\000\255\255\083\000\255\255\
    \255\255\084\000\255\255\081\000\085\000\041\001\255\255\255\255\
    \092\000\255\255\084\000\255\255\081\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\085\000\255\255\255\255\255\255\
    \092\000\255\255\084\000\085\000\081\000\255\255\255\255\255\255\
    \081\000\255\255\021\001\081\000\085\000\255\255\038\001\255\255\
    \092\000\097\000\084\000\097\000\081\000\255\255\097\000\097\000\
    \097\000\097\000\097\000\097\000\097\000\097\000\097\000\097\000\
    \092\000\098\000\098\000\098\000\098\000\098\000\098\000\098\000\
    \098\000\098\000\098\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\098\000\098\000\
    \255\255\099\000\255\255\099\000\255\255\098\000\099\000\099\000\
    \099\000\099\000\099\000\099\000\099\000\099\000\099\000\099\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\255\255\255\255\255\255\255\255\098\000\098\000\
    \255\255\255\255\255\255\255\255\255\255\098\000\100\000\255\255\
    \255\255\255\255\255\255\255\255\100\000\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\104\000\
    \104\000\104\000\104\000\104\000\104\000\104\000\104\000\104\000\
    \104\000\255\255\255\255\102\000\255\255\255\255\100\000\255\255\
    \255\255\102\000\255\255\255\255\100\000\104\000\255\255\112\000\
    \255\255\112\000\255\255\104\000\112\000\112\000\112\000\112\000\
    \112\000\112\000\112\000\112\000\112\000\112\000\255\255\255\255\
    \255\255\255\255\255\255\102\000\255\255\255\255\255\255\255\255\
    \255\255\102\000\255\255\255\255\255\255\104\000\255\255\255\255\
    \255\255\255\255\255\255\104\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\255\255\255\255\
    \255\255\255\255\255\255\023\001\255\255\106\000\106\000\106\000\
    \106\000\106\000\106\000\255\255\027\001\027\001\027\001\027\001\
    \027\001\027\001\027\001\027\001\255\255\255\255\255\255\255\255\
    \107\000\255\255\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\255\255\106\000\106\000\106\000\
    \106\000\106\000\106\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\255\255\255\255\255\255\027\001\255\255\107\000\255\255\
    \255\255\255\255\107\000\255\255\255\255\255\255\255\255\107\000\
    \255\255\113\000\113\000\113\000\113\000\113\000\113\000\113\000\
    \113\000\113\000\113\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\255\255\255\255\107\000\027\001\255\255\107\000\113\000\
    \255\255\255\255\107\000\255\255\255\255\113\000\255\255\107\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\120\000\120\000\120\000\120\000\120\000\120\000\113\000\
    \255\255\255\255\255\255\255\255\121\000\113\000\121\000\255\255\
    \120\000\121\000\121\000\121\000\121\000\121\000\121\000\121\000\
    \121\000\121\000\121\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\120\000\120\000\120\000\120\000\120\000\120\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \120\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\122\000\
    \255\255\255\255\255\255\255\255\255\255\122\000\255\255\255\255\
    \255\255\151\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\122\000\
    \255\255\255\255\255\255\255\255\027\001\122\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \255\255\255\255\255\255\255\255\151\000\255\255\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \152\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\255\255\
    \255\255\255\255\255\255\152\000\255\255\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\153\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\255\255\255\255\
    \255\255\255\255\153\000\255\255\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\154\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\255\255\255\255\255\255\
    \255\255\154\000\255\255\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\155\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\255\255\255\255\255\255\255\255\
    \155\000\255\255\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\189\000\189\000\255\255\189\000\
    \189\000\194\000\194\000\255\255\194\000\194\000\214\000\255\255\
    \255\255\214\000\214\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\189\000\255\255\189\000\255\255\255\255\
    \194\000\255\255\255\255\189\000\189\000\214\000\255\255\255\255\
    \255\255\255\255\189\000\255\255\255\255\255\255\255\255\255\255\
    \194\000\194\000\194\000\194\000\194\000\194\000\194\000\194\000\
    \194\000\194\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\214\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\197\000\255\255\
    \255\255\214\000\255\255\255\255\255\255\255\255\255\255\197\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\214\000\255\255\
    \255\255\255\255\255\255\255\255\194\000\255\255\214\000\255\255\
    \194\000\255\255\255\255\197\000\214\000\255\255\197\000\214\000\
    \214\000\255\255\214\000\255\255\214\000\255\255\255\255\197\000\
    \029\001\029\001\029\001\029\001\029\001\029\001\029\001\029\001\
    \029\001\029\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\029\001\029\001\029\001\029\001\029\001\029\001\030\001\
    \030\001\030\001\030\001\030\001\030\001\030\001\030\001\030\001\
    \030\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \030\001\030\001\030\001\030\001\030\001\030\001\255\255\255\255\
    \255\255\029\001\029\001\029\001\029\001\029\001\029\001\033\001\
    \033\001\033\001\033\001\033\001\033\001\033\001\033\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \030\001\030\001\030\001\030\001\030\001\030\001\255\255\255\255\
    \035\001\035\001\035\001\035\001\035\001\035\001\035\001\035\001\
    \035\001\035\001\255\255\255\255\255\255\255\255\255\255\033\001\
    \255\255\035\001\035\001\035\001\035\001\035\001\035\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\189\000\255\255\255\255\255\255\033\001\
    \194\000\035\001\035\001\035\001\035\001\035\001\035\001\036\001\
    \036\001\036\001\036\001\036\001\036\001\036\001\036\001\036\001\
    \036\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \036\001\036\001\036\001\036\001\036\001\036\001\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \036\001\036\001\036\001\036\001\036\001\036\001\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\033\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    ";
  Lexing.lex_base_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\002\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\004\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_default_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\000\000\001\000\001\000\001\000\001\000\001\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\000\000\001\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_check_code = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\213\000\255\255\214\000\213\000\213\000\214\000\214\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \213\000\255\255\214\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255";
  Lexing.lex_code = 
   "\255\001\255\255\000\001\255";
}

let rec initial lexbuf =
    __ocaml_lex_initial_rec lexbuf 0
and __ocaml_lex_initial_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 446 "clexer.mll"
               ( let il = comment lexbuf in
	                                  let sl = intlist_to_string il in
					  addComment sl;
                                          addWhite lexbuf;
                                          initial lexbuf)
# 1774 "clexer.ml"

  | 1 ->
# 451 "clexer.mll"
                                        ( let il = onelinecomment lexbuf in
                                          let sl = intlist_to_string il in
                                          addComment sl;
                                          E.newline();
                                          addWhite lexbuf;
                                          initial lexbuf
                                           )
# 1785 "clexer.ml"

  | 2 ->
# 458 "clexer.mll"
           ( addWhite lexbuf; initial lexbuf)
# 1790 "clexer.ml"

  | 3 ->
# 459 "clexer.mll"
                                        ( E.newline ();
                                          if !pragmaLine then
                                            begin
                                              pragmaLine := false;
                                              PRAGMA_EOL
                                            end
                                          else begin
                                            addWhite lexbuf;
                                            initial lexbuf
                                          end)
# 1804 "clexer.ml"

  | 4 ->
# 469 "clexer.mll"
                                        ( addWhite lexbuf;
                                          E.newline ();
                                          initial lexbuf
                                        )
# 1812 "clexer.ml"

  | 5 ->
# 473 "clexer.mll"
         ( addWhite lexbuf; hash lexbuf)
# 1817 "clexer.ml"

  | 6 ->
# 474 "clexer.mll"
                                   ( PRAGMA (currentLoc ()) )
# 1822 "clexer.ml"

  | 7 ->
# 475 "clexer.mll"
          ( CST_CHAR (chr lexbuf, currentLoc ()))
# 1827 "clexer.ml"

  | 8 ->
# 476 "clexer.mll"
          ( CST_WCHAR (chr lexbuf, currentLoc ()) )
# 1832 "clexer.ml"

  | 9 ->
# 477 "clexer.mll"
         ( addLexeme lexbuf; (* '"' *)
(* matth: BUG:  this could be either a regular string or a wide string.
 *  e.g. if it's the "world" in 
 *     L"Hello, " "world"
 *  then it should be treated as wide even though there's no L immediately
 *  preceding it.  See test/small1/wchar5.c for a failure case. *)
                                          try CST_STRING (str lexbuf, currentLoc ())
                                          with e -> 
                                             raise (InternalError 
                                                     ("str: " ^ 
                                                      Printexc.to_string e)))
# 1847 "clexer.ml"

  | 10 ->
# 488 "clexer.mll"
           ( (* weimer: wchar_t string literal *)
                                          try CST_WSTRING(str lexbuf, currentLoc ())
                                          with e -> 
                                             raise (InternalError 
                                                     ("wide string: " ^ 
                                                      Printexc.to_string e)))
# 1857 "clexer.ml"

  | 11 ->
# 494 "clexer.mll"
             (CST_FLOAT (Lexing.lexeme lexbuf, currentLoc ()))
# 1862 "clexer.ml"

  | 12 ->
# 495 "clexer.mll"
            (CST_INT (Lexing.lexeme lexbuf, currentLoc ()))
# 1867 "clexer.ml"

  | 13 ->
# 496 "clexer.mll"
            (CST_INT (Lexing.lexeme lexbuf, currentLoc ()))
# 1872 "clexer.ml"

  | 14 ->
# 497 "clexer.mll"
            (CST_INT (Lexing.lexeme lexbuf, currentLoc ()))
# 1877 "clexer.ml"

  | 15 ->
# 498 "clexer.mll"
             (EOF)
# 1882 "clexer.ml"

  | 16 ->
# 499 "clexer.mll"
           (ELLIPSIS)
# 1887 "clexer.ml"

  | 17 ->
# 500 "clexer.mll"
          (PLUS_EQ)
# 1892 "clexer.ml"

  | 18 ->
# 501 "clexer.mll"
          (MINUS_EQ)
# 1897 "clexer.ml"

  | 19 ->
# 502 "clexer.mll"
          (STAR_EQ)
# 1902 "clexer.ml"

  | 20 ->
# 503 "clexer.mll"
          (SLASH_EQ)
# 1907 "clexer.ml"

  | 21 ->
# 504 "clexer.mll"
          (PERCENT_EQ)
# 1912 "clexer.ml"

  | 22 ->
# 505 "clexer.mll"
          (PIPE_EQ)
# 1917 "clexer.ml"

  | 23 ->
# 506 "clexer.mll"
          (AND_EQ)
# 1922 "clexer.ml"

  | 24 ->
# 507 "clexer.mll"
          (CIRC_EQ)
# 1927 "clexer.ml"

  | 25 ->
# 508 "clexer.mll"
           (INF_INF_EQ)
# 1932 "clexer.ml"

  | 26 ->
# 509 "clexer.mll"
           (SUP_SUP_EQ)
# 1937 "clexer.ml"

  | 27 ->
# 510 "clexer.mll"
          (INF_INF)
# 1942 "clexer.ml"

  | 28 ->
# 511 "clexer.mll"
          (SUP_SUP)
# 1947 "clexer.ml"

  | 29 ->
# 512 "clexer.mll"
           (EQ_EQ)
# 1952 "clexer.ml"

  | 30 ->
# 513 "clexer.mll"
           (EXCLAM_EQ)
# 1957 "clexer.ml"

  | 31 ->
# 514 "clexer.mll"
          (INF_EQ)
# 1962 "clexer.ml"

  | 32 ->
# 515 "clexer.mll"
          (SUP_EQ)
# 1967 "clexer.ml"

  | 33 ->
# 516 "clexer.mll"
          (EQ)
# 1972 "clexer.ml"

  | 34 ->
# 517 "clexer.mll"
          (INF)
# 1977 "clexer.ml"

  | 35 ->
# 518 "clexer.mll"
          (SUP)
# 1982 "clexer.ml"

  | 36 ->
# 519 "clexer.mll"
          (PLUS_PLUS (currentLoc ()))
# 1987 "clexer.ml"

  | 37 ->
# 520 "clexer.mll"
          (MINUS_MINUS (currentLoc ()))
# 1992 "clexer.ml"

  | 38 ->
# 521 "clexer.mll"
          (ARROW)
# 1997 "clexer.ml"

  | 39 ->
# 522 "clexer.mll"
          (PLUS (currentLoc ()))
# 2002 "clexer.ml"

  | 40 ->
# 523 "clexer.mll"
          (MINUS (currentLoc ()))
# 2007 "clexer.ml"

  | 41 ->
# 524 "clexer.mll"
          (STAR (currentLoc ()))
# 2012 "clexer.ml"

  | 42 ->
# 525 "clexer.mll"
          (SLASH)
# 2017 "clexer.ml"

  | 43 ->
# 526 "clexer.mll"
          (PERCENT)
# 2022 "clexer.ml"

  | 44 ->
# 527 "clexer.mll"
         (EXCLAM (currentLoc ()))
# 2027 "clexer.ml"

  | 45 ->
# 528 "clexer.mll"
          (AND_AND (currentLoc ()))
# 2032 "clexer.ml"

  | 46 ->
# 529 "clexer.mll"
          (PIPE_PIPE)
# 2037 "clexer.ml"

  | 47 ->
# 530 "clexer.mll"
          (AND (currentLoc ()))
# 2042 "clexer.ml"

  | 48 ->
# 531 "clexer.mll"
          (PIPE)
# 2047 "clexer.ml"

  | 49 ->
# 532 "clexer.mll"
          (CIRC)
# 2052 "clexer.ml"

  | 50 ->
# 533 "clexer.mll"
          (QUEST)
# 2057 "clexer.ml"

  | 51 ->
# 534 "clexer.mll"
          (COLON)
# 2062 "clexer.ml"

  | 52 ->
# 535 "clexer.mll"
               (TILDE (currentLoc ()))
# 2067 "clexer.ml"

  | 53 ->
# 537 "clexer.mll"
               (dbgToken (LBRACE (currentLoc ())))
# 2072 "clexer.ml"

  | 54 ->
# 538 "clexer.mll"
               (dbgToken (RBRACE (currentLoc ())))
# 2077 "clexer.ml"

  | 55 ->
# 539 "clexer.mll"
          (LBRACKET)
# 2082 "clexer.ml"

  | 56 ->
# 540 "clexer.mll"
          (RBRACKET)
# 2087 "clexer.ml"

  | 57 ->
# 541 "clexer.mll"
               (dbgToken (LPAREN (currentLoc ())) )
# 2092 "clexer.ml"

  | 58 ->
# 542 "clexer.mll"
          (RPAREN)
# 2097 "clexer.ml"

  | 59 ->
# 543 "clexer.mll"
               (dbgToken (SEMICOLON (currentLoc ())) )
# 2102 "clexer.ml"

  | 60 ->
# 544 "clexer.mll"
          (COMMA)
# 2107 "clexer.ml"

  | 61 ->
# 545 "clexer.mll"
          (DOT)
# 2112 "clexer.ml"

  | 62 ->
# 546 "clexer.mll"
             (SIZEOF (currentLoc ()))
# 2117 "clexer.ml"

  | 63 ->
# 547 "clexer.mll"
                                        ( if !Cprint.msvcMode then 
                                             MSASM (msasm lexbuf, currentLoc ()) 
                                          else (ASM (currentLoc ())) )
# 2124 "clexer.ml"

  | 64 ->
# 552 "clexer.mll"
                                        ( matchingParsOpen := 0;
                                          let _ = matchingpars lexbuf in 
                                          addWhite lexbuf;
                                          initial lexbuf 
                                        )
# 2133 "clexer.ml"

  | 65 ->
# 559 "clexer.mll"
                                        (AT_TRANSFORM (currentLoc ()))
# 2138 "clexer.ml"

  | 66 ->
# 560 "clexer.mll"
                                        (AT_TRANSFORMEXPR (currentLoc ()))
# 2143 "clexer.ml"

  | 67 ->
# 561 "clexer.mll"
                                        (AT_SPECIFIER (currentLoc ()))
# 2148 "clexer.ml"

  | 68 ->
# 562 "clexer.mll"
                                        (AT_EXPR (currentLoc ()))
# 2153 "clexer.ml"

  | 69 ->
# 563 "clexer.mll"
                                        (AT_NAME)
# 2158 "clexer.ml"

  | 70 ->
# 567 "clexer.mll"
                                        (addWhite lexbuf; initial lexbuf )
# 2163 "clexer.ml"

  | 71 ->
# 568 "clexer.mll"
           (scan_ident (Lexing.lexeme lexbuf))
# 2168 "clexer.ml"

  | 72 ->
# 569 "clexer.mll"
         (EOF)
# 2173 "clexer.ml"

  | 73 ->
# 570 "clexer.mll"
       (E.parse_error "Invalid symbol")
# 2178 "clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_initial_rec lexbuf __ocaml_lex_state

and comment lexbuf =
    __ocaml_lex_comment_rec lexbuf 186
and __ocaml_lex_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 573 "clexer.mll"
                     ( addWhite lexbuf; [] )
# 2189 "clexer.ml"

  | 1 ->
# 575 "clexer.mll"
         ( addWhite lexbuf; lex_comment comment lexbuf )
# 2194 "clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_comment_rec lexbuf __ocaml_lex_state

and onelinecomment lexbuf =
    __ocaml_lex_onelinecomment_rec lexbuf 188
and __ocaml_lex_onelinecomment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 579 "clexer.mll"
                (addWhite lexbuf; [])
# 2205 "clexer.ml"

  | 1 ->
# 580 "clexer.mll"
                (addWhite lexbuf; lex_comment onelinecomment lexbuf )
# 2210 "clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_onelinecomment_rec lexbuf __ocaml_lex_state

and matchingpars lexbuf =
    __ocaml_lex_matchingpars_rec lexbuf 189
and __ocaml_lex_matchingpars_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 583 "clexer.mll"
                ( addWhite lexbuf; E.newline (); matchingpars lexbuf )
# 2221 "clexer.ml"

  | 1 ->
# 584 "clexer.mll"
                ( addWhite lexbuf; matchingpars lexbuf )
# 2226 "clexer.ml"

  | 2 ->
# 585 "clexer.mll"
                ( addWhite lexbuf; incr matchingParsOpen; matchingpars lexbuf )
# 2231 "clexer.ml"

  | 3 ->
# 586 "clexer.mll"
                ( addWhite lexbuf; decr matchingParsOpen;
                  if !matchingParsOpen = 0 then 
                     ()
                  else 
                     matchingpars lexbuf
                )
# 2241 "clexer.ml"

  | 4 ->
# 592 "clexer.mll"
         ( addWhite lexbuf; let il = comment lexbuf in
                  let sl = intlist_to_string il in
		  addComment sl;
                  matchingpars lexbuf)
# 2249 "clexer.ml"

  | 5 ->
# 596 "clexer.mll"
        ( addWhite lexbuf; (* '"' *)
                  let _ = str lexbuf in 
                  matchingpars lexbuf
                 )
# 2257 "clexer.ml"

  | 6 ->
# 600 "clexer.mll"
                 ( addWhite lexbuf; matchingpars lexbuf )
# 2262 "clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_matchingpars_rec lexbuf __ocaml_lex_state

and hash lexbuf =
  lexbuf.Lexing.lex_mem <- Array.create 2 (-1) ;   __ocaml_lex_hash_rec lexbuf 194
and __ocaml_lex_hash_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 604 "clexer.mll"
        ( addWhite lexbuf; E.newline (); initial lexbuf)
# 2273 "clexer.ml"

  | 1 ->
# 605 "clexer.mll"
         ( addWhite lexbuf; hash lexbuf)
# 2278 "clexer.ml"

  | 2 ->
# 606 "clexer.mll"
         ( addWhite lexbuf; (* We are seeing a line number. This is the number for the 
                   * next line *)
                 let s = Lexing.lexeme lexbuf in
                 let lineno = try
                   int_of_string s
                 with Failure ("int_of_string") ->
                   (* the int is too big. *)
                   E.warn "Bad line number in preprocessed file: %s" s;
                   (-1)
                 in
                  E.setCurrentLine (lineno - 1);
                  (* A file name may follow *)
		  file lexbuf )
# 2295 "clexer.ml"

  | 3 ->
# 619 "clexer.mll"
                ( addWhite lexbuf; hash lexbuf )
# 2300 "clexer.ml"

  | 4 ->
let
# 622 "clexer.mll"
                                     pragmaName
# 2306 "clexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 623 "clexer.mll"
                ( let here = currentLoc () in
                  PRAGMA_LINE (pragmaName ^ pragma lexbuf, here)
                )
# 2312 "clexer.ml"

  | 5 ->
# 626 "clexer.mll"
                ( pragmaLine := true; PRAGMA (currentLoc ()) )
# 2317 "clexer.ml"

  | 6 ->
# 627 "clexer.mll"
            ( addWhite lexbuf; endline lexbuf)
# 2322 "clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_hash_rec lexbuf __ocaml_lex_state

and file lexbuf =
    __ocaml_lex_file_rec lexbuf 276
and __ocaml_lex_file_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 630 "clexer.mll"
                      (addWhite lexbuf; E.newline (); initial lexbuf)
# 2333 "clexer.ml"

  | 1 ->
# 631 "clexer.mll"
          (addWhite lexbuf; file lexbuf)
# 2338 "clexer.ml"

  | 2 ->
# 632 "clexer.mll"
                                ( addWhite lexbuf;  (* '"' *)
                                   let n = Lexing.lexeme lexbuf in
                                   let n1 = String.sub n 1 
                                       ((String.length n) - 2) in
                                   E.setCurrentFile n1;
				 endline lexbuf)
# 2348 "clexer.ml"

  | 3 ->
# 639 "clexer.mll"
      (addWhite lexbuf; endline lexbuf)
# 2353 "clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_file_rec lexbuf __ocaml_lex_state

and endline lexbuf =
    __ocaml_lex_endline_rec lexbuf 280
and __ocaml_lex_endline_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 642 "clexer.mll"
                ( addWhite lexbuf; E.newline (); initial lexbuf)
# 2364 "clexer.ml"

  | 1 ->
# 643 "clexer.mll"
                                ( EOF )
# 2369 "clexer.ml"

  | 2 ->
# 644 "clexer.mll"
      ( addWhite lexbuf; endline lexbuf)
# 2374 "clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_endline_rec lexbuf __ocaml_lex_state

and pragma lexbuf =
    __ocaml_lex_pragma_rec lexbuf 281
and __ocaml_lex_pragma_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 647 "clexer.mll"
                        ( E.newline (); "" )
# 2385 "clexer.ml"

  | 1 ->
# 648 "clexer.mll"
                        ( let cur = Lexing.lexeme lexbuf in 
                          cur ^ (pragma lexbuf) )
# 2391 "clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_pragma_rec lexbuf __ocaml_lex_state

and str lexbuf =
    __ocaml_lex_str_rec lexbuf 282
and __ocaml_lex_str_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 652 "clexer.mll"
                        ([])
# 2402 "clexer.ml"

  | 1 ->
# 653 "clexer.mll"
             (addLexeme lexbuf; lex_hex_escape str lexbuf)
# 2407 "clexer.ml"

  | 2 ->
# 654 "clexer.mll"
             (addLexeme lexbuf; lex_oct_escape str lexbuf)
# 2412 "clexer.ml"

  | 3 ->
# 655 "clexer.mll"
          (addLexeme lexbuf; lex_simple_escape str lexbuf)
# 2417 "clexer.ml"

  | 4 ->
# 656 "clexer.mll"
     (addLexeme lexbuf; lex_unescaped str lexbuf)
# 2422 "clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_str_rec lexbuf __ocaml_lex_state

and chr lexbuf =
    __ocaml_lex_chr_rec lexbuf 288
and __ocaml_lex_chr_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 659 "clexer.mll"
              ([])
# 2433 "clexer.ml"

  | 1 ->
# 660 "clexer.mll"
             (lex_hex_escape chr lexbuf)
# 2438 "clexer.ml"

  | 2 ->
# 661 "clexer.mll"
             (lex_oct_escape chr lexbuf)
# 2443 "clexer.ml"

  | 3 ->
# 662 "clexer.mll"
          (lex_simple_escape chr lexbuf)
# 2448 "clexer.ml"

  | 4 ->
# 663 "clexer.mll"
     (lex_unescaped chr lexbuf)
# 2453 "clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_chr_rec lexbuf __ocaml_lex_state

and msasm lexbuf =
    __ocaml_lex_msasm_rec lexbuf 294
and __ocaml_lex_msasm_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 666 "clexer.mll"
                        ( msasm lexbuf )
# 2464 "clexer.ml"

  | 1 ->
# 667 "clexer.mll"
                        ( msasminbrace lexbuf )
# 2469 "clexer.ml"

  | 2 ->
# 668 "clexer.mll"
                        ( let cur = Lexing.lexeme lexbuf in 
                          cur ^ (msasmnobrace lexbuf) )
# 2475 "clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_msasm_rec lexbuf __ocaml_lex_state

and msasminbrace lexbuf =
    __ocaml_lex_msasminbrace_rec lexbuf 296
and __ocaml_lex_msasminbrace_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 672 "clexer.mll"
                        ( "" )
# 2486 "clexer.ml"

  | 1 ->
# 673 "clexer.mll"
                        ( let cur = Lexing.lexeme lexbuf in 
                          cur ^ (msasminbrace lexbuf) )
# 2492 "clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_msasminbrace_rec lexbuf __ocaml_lex_state

and msasmnobrace lexbuf =
    __ocaml_lex_msasmnobrace_rec lexbuf 297
and __ocaml_lex_msasmnobrace_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 676 "clexer.mll"
                        ( lexbuf.Lexing.lex_curr_pos <- 
                               lexbuf.Lexing.lex_curr_pos - 1;
                          "" )
# 2505 "clexer.ml"

  | 1 ->
# 679 "clexer.mll"
                        ( lexbuf.Lexing.lex_curr_pos <- 
                               lexbuf.Lexing.lex_curr_pos - 5;
                          "" )
# 2512 "clexer.ml"

  | 2 ->
# 682 "clexer.mll"
                        ( let cur = Lexing.lexeme lexbuf in 

                          cur ^ (msasmnobrace lexbuf) )
# 2519 "clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_msasmnobrace_rec lexbuf __ocaml_lex_state

;;

# 686 "clexer.mll"
 


# 2529 "clexer.ml"
