/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/pareto-type1/cdf' );
import Pareto1 = require( './../../../../../base/dists/pareto-type1/ctor' );
import entropy = require( './../../../../../base/dists/pareto-type1/entropy' );
import kurtosis = require( './../../../../../base/dists/pareto-type1/kurtosis' );
import logcdf = require( './../../../../../base/dists/pareto-type1/logcdf' );
import logpdf = require( './../../../../../base/dists/pareto-type1/logpdf' );
import mean = require( './../../../../../base/dists/pareto-type1/mean' );
import median = require( './../../../../../base/dists/pareto-type1/median' );
import mode = require( './../../../../../base/dists/pareto-type1/mode' );
import pdf = require( './../../../../../base/dists/pareto-type1/pdf' );
import quantile = require( './../../../../../base/dists/pareto-type1/quantile' );
import skewness = require( './../../../../../base/dists/pareto-type1/skewness' );
import stdev = require( './../../../../../base/dists/pareto-type1/stdev' );
import variance = require( './../../../../../base/dists/pareto-type1/variance' );

/**
* Interface describing the `pareto-type1` namespace.
*/
interface Namespace {
	/**
	* Pareto (Type I) distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param alpha - shape parameter
	* @param beta - scale parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 2.0, 1.0, 1.0 );
	* // returns 0.5
	*
	* y = ns.cdf( 5.0, 2.0, 4.0 );
	* // returns ~0.36
	*
	* y = ns.cdf( 4.0, 2.0, 2.0 );
	* // returns 0.75
	*
	* y = ns.cdf( 1.9, 2.0, 2.0 );
	* // returns 0.0
	*
	* y = ns.cdf( +Infinity, 4.0, 2.0 );
	* // returns 1.0
	*
	* var mycdf = ns.cdf.factory( 10.0, 2.0 );
	* y = mycdf( 3.0 );
	* // returns ~0.983
	*
	* y = mycdf( 2.5 );
	* // returns ~0.893
	*/
	cdf: typeof cdf;

	/**
	* Pareto (Type I) distribution.
	*/
	Pareto1: typeof Pareto1;

	/**
	* Returns the differential entropy of a Pareto (Type I) distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - scale parameter
	* @returns differential entropy
	*
	* @example
	* var v = ns.entropy( 1.0, 1.0 );
	* // returns 2.0
	*
	* @example
	* var v = ns.entropy( 4.0, 12.0 );
	* // returns ~2.349
	*
	* @example
	* var v = ns.entropy( 8.0, 2.0 );
	* // returns ~-0.261
	*
	* @example
	* var v = ns.entropy( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN, 2.0 );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of a Pareto (Type I) distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 4` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - scale parameter
	* @returns excess kurtosis
	*
	* @example
	* var v = ns.kurtosis( 5.0, 1.0 );
	* // returns ~70.8
	*
	* @example
	* var v = ns.kurtosis( 7.0, 12.0 );
	* // returns ~24.857
	*
	* @example
	* var v = ns.kurtosis( 8.0, 2.0 );
	* // returns ~19.725
	*
	* @example
	* var v = ns.kurtosis( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN, 2.0 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Pareto (Type I) distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param alpha - shape parameter
	* @param beta - scale parameter
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 2.0, 1.0, 1.0 );
	* // returns ~-0.693
	*
	* y = ns.logcdf( 5.0, 2.0, 4.0 );
	* // returns ~-1.022
	*
	* y = ns.logcdf( 4.0, 2.0, 2.0 );
	* // returns ~-0.288
	*
	* y = ns.logcdf( 1.9, 2.0, 2.0 );
	* // returns -Infinity
	*
	* y = ns.logcdf( +Infinity, 4.0, 2.0 );
	* // returns 0.0
	*
	* var mylogcdf = ns.logcdf.factory( 10.0, 2.0 );
	* y = mylogcdf( 3.0 );
	* // returns ~-0.017
	*
	* y = mylogcdf( 2.5 );
	* // returns ~-0.113
	*/
	logcdf: typeof logcdf;

	/**
	* Pareto (Type I) distribution natural logarithm of the probability density function (logPDF).
	*
	* @param x - input value
	* @param alpha - shape parameter
	* @param beta - scale parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 4.0, 1.0, 1.0 );
	* // returns ~-2.773
	*
	* y = ns.logpdf( 20.0, 1.0, 10.0 );
	* // returns ~-3.689
	*
	* y = ns.logpdf( 7.0, 2.0, 6.0 );
	* // returns ~-1.561
	*
	* var mylogpdf = ns.logpdf.factory( 0.5, 0.5 );
	*
	* y = mylogpdf( 0.8 );
	* // returns ~-0.705
	*
	* y = mylogpdf( 2.0 );
	* // returns ~-2.079
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value of a Pareto (Type I) distribution.
	*
	* ## Notes
	*
	* -   If `0 < alpha <= 1`, the function returns `Infinity`.
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - scale parameter
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 1.0, 1.0 );
	* // returns Infinity
	*
	* @example
	* var v = ns.mean( 4.0, 12.0 );
	* // returns 16.0
	*
	* @example
	* var v = ns.mean( 8.0, 2.0 );
	* // returns ~2.286
	*
	* @example
	* var v = ns.mean( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN, 2.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median of a Pareto (Type I) distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - scale parameter
	* @returns median
	*
	* @example
	* var v = ns.median( 1.0, 1.0 );
	* // returns 2.0
	*
	* @example
	* var v = ns.median( 4.0, 12.0 );
	* // returns ~14.27
	*
	* @example
	* var v = ns.median( 8.0, 2.0 );
	* // returns ~2.181
	*
	* @example
	* var v = ns.median( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.median( NaN, 2.0 );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Returns the mode of a Pareto (Type I) distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - scale parameter
	* @returns mode
	*
	* @example
	* var v = ns.mode( 1.0, 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.mode( 4.0, 12.0 );
	* // returns 12.0
	*
	* @example
	* var v = ns.mode( 8.0, 2.0 );
	* // returns 2.0
	*
	* @example
	* var v = ns.mode( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN, 2.0 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Pareto (Type I) distribution probability density function (PDF).
	*
	* @param x - input value
	* @param alpha - shape parameter
	* @param beta - scale parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 4.0, 1.0, 1.0 );
	* // returns ~0.044
	*
	* y = ns.pdf( 20.0, 1.0, 10.0 );
	* // returns 0.025
	*
	* y = ns.pdf( 7.0, 2.0, 6.0 );
	* // returns ~0.21
	*
	* var mypdf = ns.pdf.factory( 0.5, 0.5 );
	*
	* y = mypdf( 0.8 );
	* // returns ~0.494
	*
	* y = mypdf( 2.0 );
	* // returns ~0.125
	*/
	pdf: typeof pdf;

	/**
	* Pareto (Type I) distribution quantile function.
	*
	* @param p - input value
	* @param alpha - shape parameter
	* @param beta - scale parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 2.0, 1.0 );
	* // returns ~2.236
	*
	* y = ns.quantile( 0.8, 1.0, 10.0 );
	* // returns ~50.0
	*
	* y = ns.quantile( 0.1, 1.0, 10.0 );
	* // returns ~10.541
	*
	* var myquantile = ns.quantile.factory( 2.5, 0.5 );
	* y = myquantile( 0.5 );
	* // returns ~0.66
	*
	* y = myquantile( 0.8 );
	* // returns ~0.952
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a Pareto (Type I) distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 3` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - scale parameter
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 3.5, 1.0 );
	* // returns ~11.784
	*
	* @example
	* var v = ns.skewness( 4.0, 12.0 );
	* // returns ~7.071
	*
	* @example
	* var v = ns.skewness( 8.0, 2.0 );
	* // returns ~3.118
	*
	* @example
	* var v = ns.skewness( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN, 2.0 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a Pareto (Type I) distribution.
	*
	* ## Notes
	*
	* -   If `0 < alpha <= 2` and `beta > 0`, the function returns positive infinity.
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - scale parameter
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 4.0, 12.0 );
	* // returns ~5.657
	*
	* @example
	* var v = ns.stdev( 8.0, 2.0 );
	* // returns ~0.33
	*
	* @example
	* var v = ns.stdev( 1.0, 1.0 );
	* // returns Infinity
	*
	* @example
	* var v = ns.stdev( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN, 2.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a Pareto (Type I) distribution.
	*
	* ## Notes
	*
	* -   If `0 < alpha <= 2` and `beta > 0`, the function returns positive infinity.
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - scale parameter
	* @returns variance
	*
	* @example
	* var v = ns.variance( 4.0, 12.0 );
	* // returns 32.0
	*
	* @example
	* var v = ns.variance( 8.0, 2.0 );
	* // returns ~0.109
	*
	* @example
	* var v = ns.variance( 1.0, 1.0 );
	* // returns Infinity
	*
	* @example
	* var v = ns.variance( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN, 2.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Pareto (Type I) distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
