// SPDX-FileCopyrightText: 2023 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: LGPL-3.0-or-later

/* IPwatchD - IP conflict detection tool for Linux
 * Copyright (C) 2007-2010 Jaroslav Imrich <jariq(at)jariq(dot)sk>
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *  
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */

/** \file message.c
 * \brief Contains logic used for manipulating program messages
 */


#include "ipwatchd.h"


extern int syslog_flag;
extern IPWD_S_CONFIG config;

//! Handles output of messages generated by IPwatchD
/*!
 * \param type Type of message.
 * \param format Message to output in format similar to printf.
 */
void ipwd_message (IPWD_MSG_TYPE type, const char *format, ...)
{
        if (type > config.log_level){
            return;
        }

	va_list arguments;
	char msg[IPWD_MSG_BUFSIZ];

	/* Put formatted message to msg buffer */
	va_start(arguments, format);
	vsnprintf(msg, IPWD_MSG_BUFSIZ, format, arguments);
	va_end(arguments);

	/* Every message is recorded by syslog no matter if process is daemonized or not */
	switch (type)
	{

		case IPWD_MSG_TYPE_INFO:
			syslog (LOG_INFO, "ipwatchd: %s", msg);
			break;

		case IPWD_MSG_TYPE_ERROR:
			syslog (LOG_ERR, "ipwatchd: %s", msg);
			break;

		case IPWD_MSG_TYPE_ALERT:
			syslog (LOG_ALERT, "ipwatchd: %s", msg);
			break;

		case IPWD_MSG_TYPE_DEBUG:
			syslog (LOG_DEBUG, "ipwatchd: %s", msg);
			break;

		default:
			syslog (LOG_ERR, "ipwatchd: %s", msg);
			break;

	}

	/* Output message also to terminal if process is not daemonized */
	if (!syslog_flag)
	{
		switch (type)
		{

			case IPWD_MSG_TYPE_INFO:
				fprintf (stdout, "ipwatchd: %s\n", msg);
				break;

			case IPWD_MSG_TYPE_ERROR:
				fprintf (stderr, "ipwatchd: %s\n", msg);
				break;

			case IPWD_MSG_TYPE_ALERT:
				fprintf (stderr, "ipwatchd: %s\n", msg);
				break;

			case IPWD_MSG_TYPE_DEBUG:
				fprintf (stdout, "ipwatchd: %s\n", msg);
				break;

			default:
				fprintf (stderr, "ipwatchd: %s\n", msg);
				break;

		}
	}

}

