// SPDX-FileCopyrightText: 2022 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include <DLog>
#include <QStandardPaths>
#include <QCoreApplication>
#include <QDir>
#include <QJsonArray>
#include <QJsonDocument>

#include "base/command.h"
#include "controller/search_db.h"
#include "dpinyin.h"

int main(int argc, char **argv)
{
    qputenv("LANG", "en_US.UTF8");
    qputenv("LC_ALL", "en_US.UTF8");
    qputenv("LANGUAGE", "en_US");
    qDebug() << "Set locale to en_US.UTF8";

    QCoreApplication app(argc, argv);
    qDebug() << "QCoreApplication initialized";

    //设置当前工作目录为 manual/src/web
    qDebug() << "Setting working directory to:" << DMAN_SEARCH_WORK_DIR;
    QDir::setCurrent(DMAN_SEARCH_WORK_DIR);

//    //获取manual/manual-assets目录下的文件夹列表
//    QStringList list = QDir(DMAN_ORIG_MANUAL_DIR).entryList(QDir::NoDotAndDotDot | QDir::Dirs);

    //遍历manual/manual-assets目录下的文件夹列表
    for (QString &dbType :
            QDir(DMAN_ORIG_MANUAL_DIR).entryList(QDir::NoDotAndDotDot | QDir::Dirs)) {
        qDebug() << "Processing dbType:" << dbType;
        QString strCreateDbPath = DMAN_SEARCH_CREATE_DB_PATH;
        strCreateDbPath += "/" + dbType;

        QDir dir(strCreateDbPath);
        if (!dir.exists()) {
            qDebug() << "Creating directory:" << strCreateDbPath;
            dir.mkpath(strCreateDbPath);
        } else {
            qDebug() << "Directory exists:" << strCreateDbPath;
        }
        strCreateDbPath += "/search.db";

//        dman::SearchDb db;
        SearchDb db;
        db.initDb();
        db.initSearchTable();

        QStringList list = {"zh_CN", "en_US"};
//        list << "zh_CN"
//             << "en_US";
        for (QString &locale : list) {
            qDebug() << "Processing locale:" << locale;
            QString strManualDir = DMAN_ORIG_MANUAL_DIR;
            strManualDir += "/" + dbType;
            for (QString &app_name :
                    QDir(strManualDir).entryList(QDir::NoDotAndDotDot | QDir::Dirs)) {
                const QString md_file = QStringList {strManualDir, app_name, locale, "index.md"}.join(QDir::separator());
                if (!QFileInfo(md_file).isFile()) {
                    qWarning() << md_file << "not found";
                    continue;
                }

                QDir manualDir = QDir(DMAN_ORIG_MANUAL_DIR);
                manualDir.cdUp();
                qDebug() << manualDir.path();
                QString searchIndexFilePath = QString("%1/%2/%3/%4")
                                              .arg(manualDir.path(), "src", "web", "toSearchIndex.js");
                qDebug() << searchIndexFilePath;
                QString out, err;
                //            QStringList cmdList = {"node"};
                //            const bool queryNode = dman::SpawnCmd("which", cmdList);
                //            if (!queryNode) {
                //                qWarning() << err << queryNode << md_file;
                //            }

                const QStringList cmd = {searchIndexFilePath, "-f", md_file};
                const bool ok = dman::SpawnCmd("/usr/bin/node", cmd, out, err);
                if (!ok) {
                    qWarning() << err << ok << md_file;
                    continue;
                }

                qDebug() << "====>" << out;
                // Parse json
                QJsonDocument document = QJsonDocument::fromJson(out.toLocal8Bit());
                if (!document.isArray()) {
                    qWarning() << md_file << "Invalid json object:" << out;
                    continue;
                }

                const QJsonArray array = document.array();
                QStringList anchors;
                QStringList anchorIdList;
                QStringList anchorInitialList;
                QStringList anchorSpellList;
                QStringList contents;
                bool invalid_entry = false;
                for (const QJsonValue &value : array) {
                    if (!value.isArray()) {
                        invalid_entry = true;
                        qWarning() << md_file << "is not an array:" << value;
                        break;
                    }

                    const QJsonArray anchor = value.toArray();
                    const QString id = anchor.at(0).toString();
                    anchorIdList.append(id);
                    const QString title_ch = anchor.at(1).toString();
                    QString title_us = anchor.at(1).toString();
                    anchors.append(title_ch);
                    if (locale == "zh_CN") {
#if (QT_VERSION < QT_VERSION_CHECK(6, 0, 0))
                        QRegExp regExp("[1-9]");
#else
                        QRegularExpression regExp("[1-9]");
#endif
                        QString str = Dtk::Core::Chinese2Pinyin(title_ch).remove(regExp);
                        anchorSpellList.append(str);
                        if (id == "h0") {
                            QString anchorInitial;
                            for (int i = 0; i < title_ch.length(); i++) {
                                anchorInitial.append(Dtk::Core::Chinese2Pinyin(title_ch.at(i)).left(1));
                            }
                            anchorInitialList.append(anchorInitial);
                        } else {
                            anchorInitialList.append("");
                        }
                    } else if (locale == "en_US") {
                        if (id == "h0") {
                            QStringList listTitle = title_us.split(" ");
                            QString anchorInitial;
                            for (QString str : listTitle) {
                                anchorInitial.append(str.left(1));
                            }
                            anchorInitialList.append(anchorInitial);
                        } else {
                            anchorInitialList.append("");
                        }
                        anchorSpellList.append(title_us.remove(" "));
                    }
                    const QString content = anchor.at(2).toString();
                    contents.append(content);
                }

                if (!invalid_entry) {
                    qDebug() << "add search entry" << app_name << locale << anchors;
                    db.addSearchEntry(app_name, locale, anchors, anchorInitialList, anchorSpellList, anchorIdList, contents);
                }
            }
        }
    }
    qInfo() << "generate_search_db completed successfully";
    return 0;
}
