/*
 *   Copyright 2008 Marco Martin <notmart@gmail.com>
 *   Copyright 2008 Aaron Seigo <aseigo@kde.org>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as
 *   published by the Free Software Foundation; either version 2, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "toolbutton.h"

#include <QAction>
#include <QPainter>
#include <QPaintEvent>
#include <QStyle>
#include <QStyleOptionToolButton>

#include <plasma/paintutils.h>
#include <plasma/theme.h>

ToolButton::ToolButton(QWidget *parent)
    : QToolButton(parent),
      m_action(0)
{
}

void ToolButton::setAction(QAction *action)
{
    if (!action) {
        return;
    }

    if (m_action) {
        disconnect(m_action, SIGNAL(changed()), this, SLOT(syncToAction()));
        disconnect(this, SIGNAL(clicked()), m_action, SLOT(trigger()));
    }

    m_action = action;
    connect(m_action, SIGNAL(changed()), this, SLOT(syncToAction()));
    connect(this, SIGNAL(clicked()), m_action, SLOT(trigger()));
    connect(m_action, SIGNAL(destroyed(QObject*)), this, SLOT(actionDestroyed(QObject*)));
    syncToAction();
}

void ToolButton::syncToAction()
{
    if (!m_action) {
        return;
    }

    setIcon(m_action->icon());
    setText(m_action->text());

    if (toolButtonStyle() == Qt::ToolButtonIconOnly) {
        setToolTip(m_action->text());
    }

    setCheckable(m_action->isCheckable());
    if (m_action->actionGroup()) {
        setAutoExclusive(m_action->actionGroup()->isExclusive());
    }

    setEnabled(m_action->isEnabled());
}

void ToolButton::actionDestroyed(QObject *)
{
    m_action = 0;
}

void ToolButton::paintEvent(QPaintEvent *event)
{
    QPainter painter(this);
    painter.setRenderHint(QPainter::Antialiasing, true);

    painter.translate(0.5, 0.5);

    QStyleOptionToolButton buttonOpt;
    initStyleOption(&buttonOpt);

    QColor backgroundColor;
    if ((buttonOpt.state & QStyle::State_MouseOver) || isChecked()) {
        backgroundColor = Plasma::Theme::defaultTheme()->color(Plasma::Theme::TextColor);
    } else {
        backgroundColor = Plasma::Theme::defaultTheme()->color(Plasma::Theme::BackgroundColor);
    }

    backgroundColor.setAlphaF(0.4);
    QColor textColor = Plasma::Theme::defaultTheme()->color(Plasma::Theme::TextColor);

    buttonOpt.palette.setColor(QPalette::Foreground, textColor);
    buttonOpt.palette.setColor(QPalette::ButtonText, textColor);

    textColor.setAlphaF(0.4);
    painter.setPen(textColor);
    painter.setBrush(backgroundColor);
    painter.drawPath(Plasma::PaintUtils::roundedRectangle(event->rect().adjusted(1,1,-1,-1), 4));

    style()->drawControl(QStyle::CE_ToolButtonLabel, &buttonOpt, &painter, this);
}

#include "toolbutton.moc"

