/*
    Copyright (C) 2011  Martin Klapetek <mklapetek@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/



#include "persondetailsview.h"

#include <Nepomuk2/Resource>
#include <Nepomuk2/Vocabulary/NCO>
#include <Nepomuk2/Variant>
#include <KDebug>
#include <KIconLoader>
#include <KStandardDirs>
#include <KIcon>

#include <QVBoxLayout>
#include <QLabel>
#include <QAction>
#include <QToolButton>
#include <QModelIndex>

#include "persondata.h"
#include "personsmodel.h"
#include "abstractpersondetailswidget.h"

#include "plugins/emaildetailswidget.h"
#include "plugins/phonedetailswidget.h"
#include "plugins/mergecontactswidget.h"

#include <KService>
#include <KServiceTypeTrader>
#include <KPluginInfo>
#include <KPluginLoader>
#include <KPluginFactory>


#include <QList>
#include "abstractpersonplugin.h"

using namespace Nepomuk2::Vocabulary;
using namespace KPeople;

namespace KPeople {

class PersonDetailsViewPrivate{
public:
    PersonDataPtr m_person;
    QVBoxLayout *m_mainLayout;

    QLabel *m_contactPixmap;
    QLabel *m_contactNameLabel;
    QLabel *m_contactBirthdayLabel;
    QLabel *m_contactStatusLabel;
    QWidget *m_contactsListWidget;

    QList<AbstractPersonDetailsWidget*> m_detailWidgets;

    QPixmap iconForPresence(const QString &presenceString);
};
}

class DetailsGroupWidget : public QWidget
{
public:
    DetailsGroupWidget(AbstractPersonDetailsWidget *detailsWidget, QWidget *parent);
};



DetailsGroupWidget::DetailsGroupWidget(AbstractPersonDetailsWidget *detailsWidget, QWidget *parent)
    :QWidget(parent)
{
    QGridLayout *layout = new QGridLayout(this);
    layout->setColumnStretch(1, 1);

    QLabel *iconLabel = new QLabel(this);
    QLabel *titleLabel = new QLabel(this);

    QFont f;
    f.setPixelSize(16); //NOTE: HACK
    titleLabel->setFont(f);

    layout->addWidget(iconLabel, 0, 0);
    layout->addWidget(titleLabel, 0, 1);

    layout->addWidget(detailsWidget, 1, 1);

    titleLabel->setText(detailsWidget->title());
    iconLabel->setPixmap(detailsWidget->icon().pixmap(KIconLoader::SizeSmall, KIconLoader::SizeSmall));

    setVisible(detailsWidget->active());
    connect(detailsWidget, SIGNAL(activeChanged(bool)), SLOT(setVisible(bool)));

    setLayout(layout);
}


bool pluginWeightLess(const KPluginInfo &p1, const KPluginInfo &p2) {
    return p1.property("X-KPeople-Weight").toInt() < p2.property("X-KPeople-Weight").toInt();
}

PersonDetailsView::PersonDetailsView(QWidget *parent)
    : QWidget(parent),
      d_ptr(new PersonDetailsViewPrivate())
{
    Q_D(PersonDetailsView);
    d->m_mainLayout = new QVBoxLayout(this);
    d->m_contactPixmap = new QLabel(this);
    d->m_contactNameLabel = new QLabel(this);
    QFont f;
    f.setPixelSize(18);
    d->m_contactNameLabel->setFont(f);
    d->m_contactStatusLabel = new QLabel(this);
    d->m_contactBirthdayLabel = new QLabel(this);

    QGridLayout *namePresenceLayout = new QGridLayout(this);
    namePresenceLayout->addWidget(d->m_contactPixmap, 0, 0, 2, 1);
    namePresenceLayout->addWidget(d->m_contactNameLabel, 0, 1);
    namePresenceLayout->addWidget(d->m_contactStatusLabel, 0, 2);
    namePresenceLayout->addWidget(d->m_contactBirthdayLabel, 1, 1);
    namePresenceLayout->addItem(new QSpacerItem(1, 1, QSizePolicy::Expanding), 0, 3);

    QFrame *line = new QFrame(this);
    line->setFrameShape(QFrame::HLine);

    namePresenceLayout->addWidget(line, 2, 0, 1, 4);

    d->m_contactsListWidget = new QWidget(this);

    d->m_mainLayout->addLayout(namePresenceLayout);
    d->m_mainLayout->addWidget(d->m_contactsListWidget);

    // load every KPeopleWidgets Plugin
    KService::List pluginList = KServiceTypeTrader::self()->query( QLatin1String("KPeopleWidgets/Plugin"));

    QList<KPluginInfo> plugins = KPluginInfo::fromServices(pluginList);
    qSort(plugins.begin(), plugins.end(), pluginWeightLess );

    Q_FOREACH(const KPluginInfo &p, plugins) {
        QString error;
        AbstractPersonDetailsWidget *plugin = p.service()->createInstance<AbstractPersonDetailsWidget>(this, QVariantList(), &error);

        if (plugin) {
            d->m_detailWidgets.append(plugin);
        }
    }
    kDebug() << "List of loaded plugins :"  << d->m_detailWidgets;

     Q_FOREACH(AbstractPersonDetailsWidget *detailsWidget, d->m_detailWidgets) {
         d->m_mainLayout->addWidget(new DetailsGroupWidget(detailsWidget, this));
     }

    d->m_mainLayout->addSpacerItem(new QSpacerItem(32, 32, QSizePolicy::Minimum, QSizePolicy::Expanding));
    setLayout(d->m_mainLayout);
}

PersonDetailsView::~PersonDetailsView()
{
    delete d_ptr;
}

void PersonDetailsView::setPerson(const PersonDataPtr &person)
{
    Q_D(PersonDetailsView);
    if (!d->m_person.isNull()) {
        disconnect(d->m_person.data(), SIGNAL(dataChanged()), this, SLOT(reload()));
        //d->m_person is QSharedPointer and will get deleted automagically when out of scope
    }

    d->m_person = person;
    connect(d->m_person.data(), SIGNAL(dataChanged()), this, SLOT(reload()));
    reload();
}

void PersonDetailsView::setPersonsModel(PersonsModel *model)
{
    Q_D(PersonDetailsView);
    Q_FOREACH (AbstractPersonDetailsWidget *detailsWidget, d->m_detailWidgets) {
        detailsWidget->setPersonsModel(model);
    }
}

void PersonDetailsView::reload()
{
    Q_D(PersonDetailsView);
    kDebug();
    QPixmap avatar;
    QString avatarPath = d->m_person->avatar().toLocalFile();
    if (!avatarPath.isEmpty()) {
        avatar.load(avatarPath);
    }

    if (avatar.isNull()) {
        //FIXME (also move inside PersonData?)
        avatar.load(KStandardDirs::locate("data", "person-viewer/dummy_avatar.png"));
    }
    d->m_contactPixmap->setPixmap(avatar.scaled(96, 96, Qt::KeepAspectRatio, Qt::SmoothTransformation));
    d->m_contactNameLabel->setText(d->m_person->name());
    d->m_contactStatusLabel->setPixmap(d->iconForPresence(d->m_person->status()));
    d->m_contactBirthdayLabel->setText(d->m_person->groups().join(", "));

    Q_FOREACH (AbstractPersonDetailsWidget* detailsWidget, d->m_detailWidgets) {
        detailsWidget->setPerson(d->m_person.data());
    }
}

QPixmap PersonDetailsViewPrivate::iconForPresence(const QString &presenceString)
{
    KIconLoader il;
    if (presenceString == "available") {
        return il.loadIcon("user-online", KIconLoader::Toolbar);
    }
    if (presenceString == "away") {
        return il.loadIcon("user-away", KIconLoader::Toolbar);
    }
    if (presenceString == "busy" || presenceString == "dnd") {
        return il.loadIcon("user-busy", KIconLoader::Toolbar);
    }
    if (presenceString == "xa") {
        return il.loadIcon("user-online", KIconLoader::Toolbar);
    }
    if (presenceString == "offline") {
        return il.loadIcon("user-offline", KIconLoader::Toolbar);
    }

    return QPixmap();
}
