#!/usr/bin/python
# vim: encoding=utf-8
# expand Windows-specific *.in files
# for Visual Studio Builds

import os
import sys
import re
import string
import argparse


def open_compat(src, mode):
    if (sys.version_info.major < 3):
        return open(src, mode)
    else:
        return open(src, mode, encoding='utf-8', errors='surrogateescape')

def get_version(srcroot):
    ver = {}
    RE_VERSION = re.compile(r'^m4_define\(\[(gtk_\w+)\],\s*\[(\d+)\]\)')
    RE_FLOAT_VERSION = re.compile(r'^m4_define\(\[(gtk_\w+)\],\s*\[(\d+\.\d*)\]\)')
    with open(os.path.join(srcroot, 'configure.ac'), 'r') as ac:
        for i in ac:
            mo = RE_VERSION.search(i)
            if mo:
                ver[mo.group(1).upper()] = int(mo.group(2))
            mo = RE_FLOAT_VERSION.search(i)
            if mo:
                ver[mo.group(1).upper()] = float(mo.group(2))

    ver['GTK_BINARY_AGE'] = 100 * ver['GTK_MINOR_VERSION'] + ver['GTK_MICRO_VERSION']
    ver['GTK_VERSION'] = '%d.%d.%d' % (ver['GTK_MAJOR_VERSION'],
                                        ver['GTK_MINOR_VERSION'],
                                        ver['GTK_MICRO_VERSION'])
    ver['LT_RELEASE'] = '%d.%d' % (ver['GTK_MAJOR_VERSION'], ver['GTK_MINOR_VERSION'])
    ver['LT_CURRENT'] = 100 * \
                        ver['GTK_MINOR_VERSION'] + \
                        ver['GTK_MICRO_VERSION'] - \
                        ver['GTK_INTERFACE_AGE']
    ver['LT_REVISION'] = ver['GTK_INTERFACE_AGE']
    ver['LT_AGE'] = ver['GTK_BINARY_AGE'] - ver['GTK_INTERFACE_AGE']
    ver['LT_CURRENT_MINUS_AGE'] = ver['LT_CURRENT'] - ver['LT_AGE']
    return ver

def process_in(src, dest, vars):
    RE_VARS = re.compile(r'@(\w+?)@')

    with open_compat(src, 'r') as s:
        with open_compat(dest, 'w') as d:
            for i in s:
                i = RE_VARS.sub(lambda x: str(vars[x.group(1)]), i)
                d.write(i)

def get_srcroot():
    if not os.path.isabs(__file__):
        path = os.path.abspath(__file__)
    else:
        path = __file__
    dirname = os.path.dirname(path)
    return os.path.abspath(os.path.join(dirname, '..', '..'))

def main(argv):
    prog_desc = 'Create Various autogenerated Win32-specific Source Files'
    parser = argparse.ArgumentParser(description=prog_desc)
    parser.add_argument('--gtkwin32rc', dest='gtkwin32rc', action='store_const',
                        const=1,
                        help='Generate gtk-win32.rc')

    parser.add_argument('--gtk3manifest', dest='gtk3manifest', action='store_const',
                   const=1,
                   help='Generate libgtk3.manifest')

    args = parser.parse_args()
    no_args = True

    if args.gtkwin32rc is not None:
        srcroot = get_srcroot()

        ver = get_version(srcroot)

        target = os.path.join(srcroot, 'gtk', 'gtk-win32.rc')
        process_in(os.path.join(srcroot, 'gtk', 'gtk-win32.rc.in'),
                   target + '.intermediate',
                   ver)
        fp_r = open_compat(target + '.intermediate', 'r')
        lines = fp_r.readlines()
        fp_r.close()
        fp_w = open_compat(target, 'w')
        fp_w.writelines([item for item in lines[:-1]])

        fp_w.close()
        os.unlink(target + '.intermediate')

        no_args = False

    if args.gtk3manifest is not None:
        manifest = {}
        manifest['EXE_MANIFEST_ARCHITECTURE'] = '*'
        process_in(os.path.join(srcroot, 'gtk', 'libgtk3.manifest.in'),
                   os.path.join(srcroot, 'gtk', 'libgtk3.manifest'),
                   manifest)
        no_args = False

    if no_args is True:
        raise SystemExit('Action argument required.  Please see %s --help for details.' % __file__)

if __name__ == '__main__':
    sys.exit(main(sys.argv))
