// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_UI_PASSWORDS_WELL_KNOWN_CHANGE_PASSWORD_NAVIGATION_THROTTLE_H_
#define CHROME_BROWSER_UI_PASSWORDS_WELL_KNOWN_CHANGE_PASSWORD_NAVIGATION_THROTTLE_H_

#include <memory>

#include "content/public/browser/navigation_throttle.h"

#include "components/password_manager/core/browser/well_known_change_password_state.h"
#include "components/password_manager/core/browser/well_known_change_password_util.h"
#include "services/metrics/public/cpp/ukm_source_id.h"

class GURL;
namespace content {
class NavigationHandle;
}  // namespace content

namespace password_manager {
class AffiliationService;
class ChangePasswordUrlService;
}  // namespace password_manager

// This NavigationThrottle checks whether a site supports the
// .well-known/change-password url. To check whether a site supports the
// change-password url, we also request a .well-known path that is defined to
// return a 404. When that one returns a 404 and the change password path a 200
// we assume the site supports the change-password url. If the site does not
// support the change password url, the user gets redirected to the base path
// '/'.
class WellKnownChangePasswordNavigationThrottle
    : public content::NavigationThrottle,
      public password_manager::WellKnownChangePasswordStateDelegate {
 public:
  static std::unique_ptr<WellKnownChangePasswordNavigationThrottle>
  MaybeCreateThrottleFor(content::NavigationHandle* handle);

  explicit WellKnownChangePasswordNavigationThrottle(
      content::NavigationHandle* handle);

  ~WellKnownChangePasswordNavigationThrottle() override;

  // We don't need to override WillRedirectRequest since a redirect is the
  // expected behaviour and does not need manual intervention.
  // content::NavigationThrottle:
  ThrottleCheckResult WillStartRequest() override;
  ThrottleCheckResult WillFailRequest() override;
  ThrottleCheckResult WillProcessResponse() override;
  const char* GetNameForLogging() override;

 private:
  // password_manager::WellKnownChangePasswordStateDelegate:
  void OnProcessingFinished(bool is_supported) override;
  // Redirects to a given URL in the same tab.
  void Redirect(const GURL& url);
  // Records the given UKM metric.
  void RecordMetric(password_manager::WellKnownChangePasswordResult result);

  // Stores `navigation_handle()->GetURL()` if the first navigation was to
  // .well-known/change-password. It is later used to derive the URL for the
  // non-existing resource, and to provide fallback logic.
  const GURL request_url_;
  password_manager::WellKnownChangePasswordState
      well_known_change_password_state_{this};
  ukm::SourceId source_id_ = ukm::kInvalidSourceId;
  password_manager::ChangePasswordUrlService* change_password_url_service_ =
      nullptr;
  password_manager::AffiliationService* affiliation_service_ = nullptr;
};

#endif  // CHROME_BROWSER_UI_PASSWORDS_WELL_KNOWN_CHANGE_PASSWORD_NAVIGATION_THROTTLE_H_
