#!/bin/sh
# SPDX-License-Identifier: GPL-2.0+
# Copyright (C) 2020 Denis 'GNUtoo' Carikli

progname="install-uboot4extlinux.sh"
u_boot_with_spl="/usr/lib/u-boot/uboot4extlinux-a13-olinuxinom/u-boot-sunxi-with-spl.bin"
u_boot_with_spl_offset=8192

usage()
{
	echo "Usage: ${progname} [INSTALL_DEVICE]"
}

install_image()
{
	install_device="$1"
	offset="$2"
	image="$3"

	# We need root permissions for now:
	# - blkid is used to bypass caching and it needs root access
	if [ "$(id -u)" != 0 ] ; then
		echo "Error: This script needs to be run as root"
		exit 1
	fi

	# We only support MBR for now
	# It will also catch errors like /dev/sdb1 and mmcbk0p1 as they
	# don't have the PTTYPE tag
	PTTYPE="$(blkid --probe --match-tag PTTYPE ${install_device} | \
		  awk '{print $2}' | sed 's#^PTTYPE="##' | sed 's#"$##')"
	if [ -z "${PTTYPE}" ] ; then
		echo "Error: Could not find an (MBR) formating inside " \
		     "${install_device}"
		echo "       Possible causes:"
		echo "       - Maybe {instal_device} is a partition"
		echo "       - or Maybe it's completelyor unformated"
		exit 1
	elif [ "${PTTYPE}" != "dos" ] ; then
		echo "Error: ${install_device} is not in MBR format"
		echo "       Currently, only the MBR format is supported"
		exit 1
	fi

	# Check if offset > MBR partition table part end
	if [ ${offset} -lt 512 ] ; then
	    echo "Error: The offset (${offset}) is < 512"
	    echo "       offsets < 512 aren't supported (yet)"
	    exit 1
	fi

	# Get the image size
	image_size="$(du --bytes --apparent-size ${image} | awk '{print $1}')"

	# With MBR, most partitioning tools leave makes the first partition
	# start at 1MiB. So we need to check if the bootloader doesn't end
	# up overwriting the first partition.
	if [ $(expr ${image_size} + ${offset}) -gt $(expr 1024 \* 1024) ] ; then
		echo "Error: ${image} is too big:"
		echo "       offset: ${offset}"
		echo "       size:   ${image_size}"
		echo "       By default, "\
		     "partitioing tools makes the first partition start at 1MiB"
		echo "       Instaling ${images} "\
		     "would overwrite that first partition (if it's present)."
		echo "       Please contact the Parabola developers " \
		     "so they could fix that issue"
		exit 1
	fi

	# Copies with with a block size of 1 can be quite slow in practice
	if [ "$(expr ${offset} % 512)" = 0 ] ; then
		dd conv=notrunc "if=${image}" "of=${install_device}" \
		   bs=512 "seek=$(expr ${offset} / 512)"
	else
		echo "Warning: slow copy"
		dd conv=notrunc "if=${image}" "of=${install_device}" \
		   bs=1 "seek=${offset}"
	fi

	sync "${install_device}"
}

if [ $# -ne 1 ] ; then
    usage
else
    install_image "$1" "${u_boot_with_spl_offset}" "${u_boot_with_spl}"
fi
