/**********************************************************************
 * $Id: mitab_bounds.cpp,v 1.8 2008-01-29 20:53:10 dmorissette Exp $
 *
 * Name:     mitab_bounds.cpp
 * Project:  MapInfo TAB Read/Write library
 * Language: C++
 * Purpose:  Implementation of bound lookup tables for known projections.
 * Author:   Daniel Morissette, dmorissette@dmsolutions.ca
 *
 **********************************************************************
 * Copyright (c) 2001, Daniel Morissette
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 **********************************************************************
 *
 * $Log: mitab_bounds.cpp,v $
 * Revision 1.8  2008-01-29 20:53:10  dmorissette
 * Added bounds for PSAD56 (Patch from AJD sent for bug #1754)
 *
 * Revision 1.7  2005/09/29 18:31:28  dmorissette
 * New bounds entry for Finnish KKJ and Swedish projections (AJD, bug 1155)
 *
 * Revision 1.6  2005/03/31 22:00:38  dmorissette
 * Added bounds entry to match datum 1011 based on MapInfo's "Svenska
 * rikssystemet, 2,5 gon väst (RT 90 7 parametrar)" (bug 997)
 *
 * Revision 1.5  2005/03/22 23:24:54  dmorissette
 * Added support for datum id in .MAP header (bug 910)
 *
 * Revision 1.4  2004/06/30 20:29:03  dmorissette
 * Fixed refs to old address danmo@videotron.ca
 *
 * Revision 1.3  2001/02/14 21:17:33  daniel
 * Check only if first char is "#" for comments in MITABLoadCoordSysTable()
 *
 * Revision 1.2  2001/01/23 22:06:50  daniel
 * Added MITABCoordSysTableLoaded()
 *
 * Revision 1.1  2001/01/23 21:23:41  daniel
 * Added projection bounds lookup table, called from TABFile::SetProjInfo()
 *
 **********************************************************************/

#include "mitab.h"

typedef struct
{
    TABProjInfo sProj;          /* Projection/datum definition */
    double      dXMin;          /* Default bounds for that coordsys */
    double      dYMin;
    double      dXMax;
    double      dYMax;
} MapInfoBoundsInfo;

typedef struct
{
    TABProjInfo       sProjIn;
    MapInfoBoundsInfo sBoundsInfo;
} MapInfoRemapProjInfo;

/*-----------------------------------------------------------------
 * List of known coordsys bounds.
 * 0xff in nEllipsoidId or nUnitsId fields means any value can match.
 *
 * __TODO__: nDatumId is always set to zero in this list, we'd have to
 * reprocess the whole list to properly set all datum ids and accelerate
 * bounds lookups
 *----------------------------------------------------------------*/
static MapInfoRemapProjInfo *gpasExtBoundsList = NULL;
static int nExtBoundsListCount = -1;
static const MapInfoBoundsInfo gasBoundsList[] = {
{{1, 0xff, 0xff, {0,0,0,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0}, -1000, -1000, 1000, 1000},  /* Lat/Lon */

{{2, 29, 0, {-85.5,13,0,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -18500.7190263237, -4067.43878447928, 30025.7571082958, 4067.43878447928},
{{2, 29, 0, {20,0,0,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26284.8753911183, -3963.19059194305, 23518.0464025796, 3963.19059194305},
{{2, 7, 7, {0,30,0,0,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -34706360.1398239, -7364918.36397399, 34706360.1398239, 7364918.36397399},
{{3, 0, 3, {-109.5,44.25,45,49,1968503.937,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -376322393.49652, -357755728.255206, 380259401.37052, 398826066.611833},
{{3, 0, 3, {-111.5,36.6666666667,37.2166666667,38.35,1640419.948,9842519.685}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -281416969.95067, -245782664.730374, 284697809.84667, 320332115.066966},
{{3, 0, 3, {-111.5,38.3333333333,39.0166666667,40.65,1640419.948,6561679.79}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -299820220.76226, -269235441.863424, 303101060.65826, 333685839.557096},
{{3, 0, 3, {-111.5,40.3333333333,40.7166666667,41.7833333333,1640419.948,3280839.895}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -313479418.366583, -287610480.882755, 316760258.262583, 342629195.746411},
{{3, 0, 3, {-120.5,41.6666666667,42.3333333333,44,4921259.843,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -329872743.533369, -311905915.324464, 339715263.219369, 357682091.428273},
{{3, 0, 3, {-120.5,43.6666666667,44.3333333333,46,8202099.738,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -348623368.682272, -335442185.295993, 365027568.158272, 378208751.544552},
{{3, 0, 3, {-81,31.8333333333,32.5,34.8333333333,2000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -248789436.724623, -218682485.302253, 252789436.724623, 282896388.146993},
{{3, 0, 3, {-84.3666666667,41.5,42.1,43.6666666667,13123359.58,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -318674512.717618, -308729738.1419, 344921231.877618, 354866006.453336},
{{3, 0, 3, {-84.3666666667,43.3166666667,44.1833333333,45.7,19685039.37,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -334588341.234808, -332680007.139814, 373958419.974808, 375866754.069803},
{{3, 0, 3, {-87,44.7833333333,45.4833333333,47.0833333333,26246719.16,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -343541891.912548, -349200540.720143, 396035330.232548, 390376681.424953},
{{3, 0, 7, {-100,39.8333333333,40,43,500000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -96293653.747449, -89392122.913416, 97293653.747449, 104195184.581482},
{{3, 0, 7, {-100,43.8333333333,44.4166666667,45.6833333333,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -107757768.605122, -101845373.546917, 108957768.605122, 114870163.663327},
{{3, 0, 7, {-100.3333333333,29.6666666667,30.1166666667,31.8833333333,700000,3000000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -70389655.9882633, -57319094.8848422, 71789655.9882633, 84860217.0916844},
{{3, 0, 7, {-100.3333333333,42.3333333333,42.8333333333,44.4,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -102916158.881298, -96669936.9429582, 104116158.881298, 110362380.819638},
{{3, 0, 7, {-100.5,45.6666666667,46.1833333333,47.4833333333,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -114111552.091083, -108589212.496103, 115311552.091083, 120833891.686064},
{{3, 0, 7, {-100.5,47,47.4333333333,48.7333333333,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -118782512.590452, -113525792.131232, 119982512.590452, 125239233.049672},
{{3, 0, 7, {-101.5,34,34.65,36.1833333333,200000,1000000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -80190916.4774175, -70255345.5878226, 80590916.4774175, 90526487.3670124},
{{3, 0, 7, {-105.5,36.6666666667,37.2333333333,38.4333333333,914401.8289,304800.6096}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -85492362.7230086, -77749948.5363837, 87321166.3808086, 95063580.5674335},
{{3, 0, 7, {-105.5,37.8333333333,38.45,39.75,914401.8289,304800.6096}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -88909656.3330413, -81520557.8132071, 90738459.9908412, 98127558.5106754},
{{3, 0, 7, {-105.5,39.3333333333,39.7166666667,40.7833333333,914401.8289,304800.6096}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -92173099.6583073, -85136649.2531605, 94001903.3161073, 101038353.721254},
{{3, 0, 7, {-109.5,44.25,45,49,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -114703065.537737, -109043945.972187, 115903065.537737, 121562185.103287},
{{3, 0, 7, {-111.5,36.6666666667,37.2166666667,38.35,500000,3000000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -85775892.4411146, -74914556.209806, 86775892.4411146, 97637228.6724232},
{{3, 0, 7, {-111.5,38.3333333333,39.0166666667,40.65,500000,2000000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -91385203.2884872, -82062962.6799637, 92385203.2884872, 101707443.897011},
{{3, 0, 7, {-111.5,40.3333333333,40.7166666667,41.7833333333,500000,1000000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -95548526.7182849, -87663674.5730598, 96548526.7182849, 104433378.86351},
{{3, 0, 7, {-116.25,32.1666666667,32.7833333333,33.8833333333,2000000,500000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -73735809.4129763, -65399717.6233228, 77735809.4129763, 86071901.2026297},
{{3, 0, 7, {-118,33.5,34.0333333333,35.4666666667,2000000,500000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -76848317.8487893, -69006561.7099004, 80848317.8487893, 88690073.9876782},
{{3, 0, 7, {-119,35.3333333333,36,37.25,2000000,500000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -81316774.1198701, -74083546.7405704, 85316774.1198701, 92550001.4991699},
{{3, 0, 7, {-120.5,36.5,37.0666666667,38.4333333333,2000000,500000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -84187586.7378795, -77301811.5655565, 88187586.7378795, 95073361.9102026},
{{3, 0, 7, {-120.5,41.6666666667,42.3333333333,44,1500000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -100545212.229117, -95068922.9908967, 103545212.229117, 109021501.467338},
{{3, 0, 7, {-120.5,43.6666666667,44.3333333333,46,2500000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -106260402.774499, -102242778.078219, 111260402.774499, 115278027.47078},
{{3, 0, 7, {-120.5,45.3333333333,45.8333333333,47.3333333333,500000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -113297926.255298, -107613973.979824, 114297926.255298, 119981878.530772},
{{3, 0, 7, {-120.8333333333,47,47.5,48.7333333333,500000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -119009737.681158, -113655959.077325, 120009737.681158, 125363516.284991},
{{3, 0, 7, {-122,37.6666666667,38.3333333333,39.8333333333,2000000,500000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -87776918.3325266, -81257129.4018421, 91776918.3325266, 98296707.2632112},
{{3, 0, 7, {-122,39.3333333333,40,41.6666666667,2000000,500000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -92797918.1664438, -86741363.5256259, 96797918.1664438, 102854472.807262},
{{3, 0, 7, {-176,51,51.8333333333,53.8333333333,1000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -137707686.600156, -133658946.217207, 139707686.600156, 143756426.983104},
{{3, 0, 7, {-66.4333333333,17.8333333333,18.0333333333,18.4333333333,200000,200000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -56733778.1428648, -37322071.9454256, 57133778.1428648, 58748927.6361153},
{{3, 0, 7, {-70.5,41,41.2833333333,41.4833333333,500000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -95953926.4298888, -89161935.9801186, 96953926.4298888, 103745916.879659},
{{3, 0, 7, {-71.5,41,41.7166666667,42.6833333333,200000,750000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -98769146.9690858, -91041445.2286386, 99169146.9690858, 106896848.709533},
{{3, 0, 7, {-72.75,40.8333333333,41.2,41.8666666667,304800.6096,152400.3048}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -96604898.0590896, -89468373.8450348, 97214499.2782896, 104351023.492344},
{{3, 0, 7, {-74,40.1666666667,40.6666666667,41.0333333333,300000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -94551938.838961, -87389402.5378217, 95151938.838961, 102314475.1401},
{{3, 0, 7, {-77,37.6666666667,38.3,39.45,400000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -88804178.7516629, -81148556.0144016, 89604178.7516629, 97259801.4889241},
{{3, 0, 7, {-77.75,39.3333333333,39.9333333333,40.9666666667,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -93070575.1551149, -86055381.3970464, 94270575.1551149, 101285768.913183},
{{3, 0, 7, {-77.75,40.1666666667,40.8833333333,41.95,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -95953285.96374, -89173937.5969296, 97153285.96374, 103932634.33055},
{{3, 0, 7, {-78.5,36.3333333333,36.7666666667,37.9666666667,3500000,1000000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -81693800.1596124, -75717098.3537021, 88693800.1596124, 94670501.9655228},
{{3, 0, 7, {-78.5,37.6666666667,38.0333333333,39.2,3500000,2000000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -84998823.9067757, -78398508.2242156, 91998823.9067757, 98599139.5893358},
{{3, 0, 7, {-79,33.75,34.3333333333,36.1666666667,609601.22,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -79389023.1316542, -70798838.7584427, 80608225.5716542, 89198409.9448658},
{{3, 0, 7, {-79.5,38.5,39,40.25,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -90694785.2338336, -83456997.4383959, 91894785.2338336, 99132573.0292712},
{{3, 0, 7, {-81,31.8333333333,32.5,34.8333333333,609600,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -75831020.313665, -66654421.5201266, 77050220.313665, 86226819.1072034},
{{3, 0, 7, {-81,37,37.4833333333,38.8833333333,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -86731868.6538258, -79080928.1042007, 87931868.6538258, 95582809.203451},
{{3, 0, 7, {-82.5,38,38.7333333333,40.0333333333,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -90013727.4321062, -82680858.3947178, 91213727.4321062, 98546596.4694945},
{{3, 0, 7, {-82.5,39.6666666667,40.4333333333,41.7,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -94896218.4967998, -88010766.8984708, 96096218.4967998, 102981670.095129},
{{3, 0, 7, {-84.25,37.5,37.9666666667,38.9666666667,500000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -87593445.377663, -79948138.9064763, 88593445.377663, 96238751.8488498},
{{3, 0, 7, {-84.3666666667,41.5,42.1,43.6666666667,4000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -97131991.4763139, -94100824.185651, 105131991.476314, 108163158.766977},
{{3, 0, 7, {-84.3666666667,43.3166666667,44.1833333333,45.7,6000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -101982526.408346, -101400866.176215, 113982526.408346, 114564186.640476},
{{3, 0, 7, {-84.5,29,29.5833333333,30.75,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -68981040.7901211, -58469775.5831551, 70181040.7901211, 80692305.9970871},
{{3, 0, 7, {-85.75,36.3333333333,36.7333333333,37.9333333333,500000,500000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -84608108.5314714, -76125029.0762607, 85608108.5314714, 94091187.9866821},
{{3, 0, 7, {-86,34.3333333333,35.25,36.4166666667,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -80782527.8217375, -72373896.5103859, 81982527.8217375, 90391159.1330892},
{{3, 0, 7, {-87,44.7833333333,45.4833333333,47.0833333333,8000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -104711568.654913, -106436324.8115, 120711568.654913, 118986812.498326},
{{3, 0, 7, {-90,42,42.7333333333,44.0666666667,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -102206338.477554, -95896048.3229835, 103406338.477554, 109716628.632125},
{{3, 0, 7, {-90,43.8333333333,44.25,45.5,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -107154105.286573, -101222137.29396, 108354105.286573, 114286073.279185},
{{3, 0, 7, {-90,45.1666666667,45.5666666667,46.7666666667,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -111693759.01384, -106048829.503351, 112893759.01384, 118538688.524328},
{{3, 0, 7, {-91.3333333333,25.5,26.1666666667,27.8333333333,1000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -63537867.0156485, -51847003.2875935, 65537867.0156485, 77228730.7437035},
{{3, 0, 7, {-91.3333333333,28.5,29.3,30.7,1000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -68286769.6701213, -58065091.5621744, 70286769.6701213, 80508447.7780682},
{{3, 0, 7, {-92,32.6666666667,33.3,34.7666666667,400000,400000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -76844795.0772024, -67240210.8482154, 77644795.0772024, 87249379.3061895},
{{3, 0, 7, {-92,34.3333333333,34.9333333333,36.2333333333,400000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -80385696.4806497, -71723301.2825226, 81185696.4806497, 89848091.6787767},
{{3, 0, 7, {-92.5,30.5,31.1666666667,32.6666666667,1000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -71843187.1479918, -62437428.102125, 73843187.1479918, 83248946.1938587},
{{3, 0, 7, {-93.1,46.5,47.0333333333,48.6333333333,800000,100000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -117632875.188048, -112398161.714158, 119232875.188048, 124467588.661938},
{{3, 0, 7, {-93.5,40,40.6166666667,41.7833333333,500000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -95397349.164015, -88468304.6501118, 96397349.164015, 103326393.677918},
{{3, 0, 7, {-93.5,41.5,42.0666666667,43.2666666667,1500000,1000000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -98941207.3645198, -92381467.3382571, 101941207.36452, 108500947.390783},
{{3, 0, 7, {-94,43,43.7833333333,45.2166666667,800000,100000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -105671574.194712, -99704447.3453072, 107271574.194712, 113238701.044116},
{{3, 0, 7, {-94.25,45,45.6166666667,47.05,800000,100000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -112092591.799175, -106546357.855489, 113692591.799175, 119238825.742861},
{{3, 0, 7, {-98,33.3333333333,33.9333333333,35.2333333333,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -77870984.1222039, -69071740.6080561, 79070984.1222039, 87870227.6363517},
{{3, 0, 7, {-98,35,35.5666666667,36.7666666667,600000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -81588328.3763743, -73324068.4505433, 82788328.3763743, 91052588.3022054},
{{3, 0, 7, {-98,38.3333333333,38.7166666667,39.7833333333,400000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -89841929.3777815, -82323689.9068513, 90641929.3777815, 98160168.8487116},
{{3, 0, 7, {-98.5,25.6666666667,26.1666666667,27.8333333333,300000,5000000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -64237867.0156485, -46865470.5583462, 64837867.0156485, 82210263.4729508},
{{3, 0, 7, {-98.5,31.6666666667,32.1333333333,33.9666666667,600000,2000000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -74535881.4099808, -63171655.6562241, 75735881.4099808, 87100107.1637374},
{{3, 0, 7, {-98.5,36.6666666667,38.5666666667,37.2666666667,400000,400000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -86225535.4567815, -77888976.4101326, 87025535.4567815, 95362094.5034304},
{{3, 0, 7, {-99,27.8333333333,28.3833333333,30.2833333333,600000,4000000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -67541882.489505, -52618369.9310933, 68741882.489505, 83665395.0479167},
{{3, 0, 8, {-100,39.8333333333,40,43,1640416.667,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -315923429.002755, -293280656.591766, 319204262.336755, 341847034.747745},
{{3, 0, 8, {-100,43.8333333333,44.4166666667,45.6833333333,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -353535279.165304, -334137696.37851, 357472279.165304, 376869861.952098},
{{3, 0, 8, {-100.3333333333,29.6666666667,30.1166666667,31.8833333333,2296583.333,9842500}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -230936729.688494, -188054397.134687, 235529896.354494, 278412228.908301},
{{3, 0, 8, {-100.3333333333,42.3333333333,42.8333333333,44.4,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -337650764.596392, -317157951.453689, 341587764.596392, 362080577.739096},
{{3, 0, 8, {-100.5,45.6666666667,46.1833333333,47.4833333333,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -374380983.81883, -356263107.997632, 378317983.81883, 396435859.640027},
{{3, 0, 8, {-100.5,47,47.4333333333,48.7333333333,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -389705626.72384, -372459203.017216, 393642626.72384, 410889050.430464},
{{3, 0, 8, {-101.5,34,34.65,36.1833333333,656166.6667,3280833.333}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -263093031.809627, -230496079.649714, 264405365.143027, 297002317.30294},
{{3, 0, 8, {-105.5,36.6666666667,37.2333333333,38.4333333333,3000000,1000000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -280486193.367387, -255084622.823115, 286486193.367387, 311887763.911659},
{{3, 0, 8, {-105.5,37.8333333333,38.45,39.75,3000000,1000000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -291697764.152969, -267455363.425493, 297697764.152969, 321940164.880445},
{{3, 0, 8, {-105.5,39.3333333333,39.7166666667,40.7833333333,3000000,1000000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -302404577.795946, -279319156.758073, 308404577.795946, 331489998.833818},
{{3, 0, 8, {-109.5,44.25,45,49,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -376321640.851725, -357755012.743749, 380258640.851725, 398825268.9597},
{{3, 0, 8, {-111.5,36.6666666667,37.2166666667,38.35,1640416.667,9842500}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -281416407.11689, -245782173.165005, 284697240.45089, 320331474.402775},
{{3, 0, 8, {-111.5,38.3333333333,39.0166666667,40.65,1640416.667,6561666.667}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -299819621.121978, -269234903.392181, 303100454.455978, 333685172.185776},
{{3, 0, 8, {-111.5,40.3333333333,40.7166666667,41.7833333333,1640416.667,3280833.333}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -313478791.407906, -287609905.662114, 316759624.741906, 342628510.487699},
{{3, 0, 8, {-116.25,32.1666666667,32.7833333333,33.8833333333,6561666.667,1640416.667}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -241914901.382073, -214565573.568852, 255038234.716073, 282387562.529294},
{{3, 0, 8, {-118,33.5,34.0333333333,35.4666666667,6561666.667,1640416.667}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -252126522.80857, -226399027.876232, 265249856.14257, 290977351.074907},
{{3, 0, 8, {-119,35.3333333333,36,37.25,6561666.667,1640416.667}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -266786783.091274, -243055769.597688, 279910116.425274, 303641129.91886},
{{3, 0, 8, {-120.5,36.5,37.0666666667,38.4333333333,6561666.667,1640416.667}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -276205440.822193, -253614360.110997, 289328774.156193, 311919854.86739},
{{3, 0, 8, {-120.5,41.6666666667,42.3333333333,44,4921250,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -329872083.788362, -311905291.512634, 339714583.788362, 357681376.064091},
{{3, 0, 8, {-120.5,43.6666666667,44.3333333333,46,8202083.333,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -348622671.436336, -335441514.411622, 365026838.102336, 378207995.127049},
{{3, 0, 8, {-120.5,45.3333333333,45.8333333333,47.3333333333,1640416.667,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -371711613.055591, -353063512.965473, 374992446.389591, 393640546.479709},
{{3, 0, 8, {-120.8333333333,47,47.5,48.7333333333,1640416.667,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -390451114.375266, -372886259.072857, 393731947.709266, 411296803.011675},
{{3, 0, 8, {-122,37.6666666667,38.3333333333,39.8333333333,6561666.667,1640416.667}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -287981439.562298, -266591098.71221, 301104772.896298, 322495113.746385},
{{3, 0, 8, {-122,39.3333333333,40,41.6666666667,6561666.667,1640416.667}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -304454503.184074, -284583956.833324, 317577836.518074, 337448382.868824},
{{3, 0, 8, {-176,51,51.8333333333,53.8333333333,3280833.333,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -451795968.454344, -438512726.047621, 458357635.120344, 471640877.527066},
{{3, 0, 8, {-66.4333333333,17.8333333333,18.0333333333,18.4333333333,656166.6667,656166.6667}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -186134070.457016, -122447497.707584, 187446403.790416, 192745440.086188},
{{3, 0, 8, {-70.5,41,41.2833333333,41.4833333333,1640416.667,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -314808840.29506, -292525451.628106, 318089673.62906, 340373062.296015},
{{3, 0, 8, {-71.5,41,41.7166666667,42.6833333333,656166.6667,2460625}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -324045109.681042, -298691808.220958, 325357443.014442, 350710744.474526},
{{3, 0, 8, {-72.75,40.8333333333,41.2,41.8666666667,1000000,500000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -316944569.715526, -293530823.189916, 318944569.715526, 342358316.241135},
{{3, 0, 8, {-74,40.1666666667,40.6666666667,41.0333333333,984250,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -310209152.674158, -286710064.82617, 312177652.674158, 335676740.522146},
{{3, 0, 8, {-77,37.6666666667,38.3,39.45,1312333.333,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -291351709.788081, -266234887.523916, 293976376.454081, 319093198.718245},
{{3, 0, 8, {-77.75,39.3333333333,39.9333333333,40.9666666667,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -305349045.321406, -282333363.800143, 309286045.321406, 332301726.842669},
{{3, 0, 8, {-77.75,40.1666666667,40.8833333333,41.95,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -314806739.032704, -292564826.932593, 318743739.032704, 340985651.132814},
{{3, 0, 8, {-78.5,36.3333333333,36.7666666667,37.9666666667,11482916.67,3280833.333}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -268023742.686995, -248415180.182438, 290989576.026995, 310598138.531553},
{{3, 0, 8, {-78.5,37.6666666667,38.0333333333,39.2,11482916.67,6561666.667}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -278866974.764147, -257212439.065281, 301832808.104147, 323487343.803013},
{{3, 0, 8, {-79,33.75,34.3333333333,36.1666666667,2000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -260462153.393719, -232279190.159991, 264462153.393719, 292645116.627447},
{{3, 0, 8, {-79.5,38.5,39,40.25,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -297554474.554669, -273808499.095804, 301491474.554669, 325237450.013534},
{{3, 0, 8, {-81,31.8333333333,32.5,34.8333333333,1999996,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -248788939.145749, -218682047.937282, 252788931.145749, 282895822.354217},
{{3, 0, 8, {-81,37,37.4833333333,38.8833333333,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -284552805.74176, -259451344.955198, 288489805.74176, 313591266.528322},
{{3, 0, 8, {-82.5,38,38.7333333333,40.0333333333,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -295320037.416835, -271262116.250003, 299257037.416835, 323314958.583666},
{{3, 0, 8, {-82.5,39.6666666667,40.4333333333,41.7,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -311338676.851584, -288748657.732733, 315275676.851584, 337865695.970435},
{{3, 0, 8, {-84.25,37.5,37.9666666667,38.9666666667,1640416.667,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -287379495.376216, -262296519.062331, 290660328.710216, 315743305.024101},
{{3, 0, 8, {-84.3666666667,41.5,42.1,43.6666666667,13123333.33,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -318673875.371873, -308729120.682423, 344920542.031873, 354865296.721323},
{{3, 0, 8, {-84.3666666667,43.3166666667,44.1833333333,45.7,19685000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -334587672.058047, -332679341.779799, 373957672.058047, 375866002.336295},
{{3, 0, 8, {-84.5,29,29.5833333333,30.75,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -226315297.992256, -191829588.725735, 230252297.992256, 264738007.258777},
{{3, 0, 8, {-85.75,36.3333333333,36.7333333333,37.9333333333,1640416.667,1640416.667}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -277585102.740002, -249753532.894032, 280865936.074002, 308697505.919973},
{{3, 0, 8, {-86,34.3333333333,35.25,36.4166666667,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -265034010.028484, -237446692.134491, 268971010.028484, 296558327.922477},
{{3, 0, 8, {-87,44.7833333333,45.4833333333,47.0833333333,26246666.67,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -343541204.825326, -349199842.319061, 396034538.165326, 390375900.67159},
{{3, 0, 8, {-90,42,42.7333333333,44.0666666667,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -335321962.15511, -314618951.872988, 339258962.15511, 359961972.437231},
{{3, 0, 8, {-90,43.8333333333,44.25,45.5,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -351554760.427697, -332092962.105268, 355491760.427697, 374953558.750127},
{{3, 0, 8, {-90,45.1666666667,45.5666666667,46.7666666667,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -366448607.697906, -347928534.795579, 370385607.697906, 388905680.600232},
{{3, 0, 8, {-91.3333333333,25.5,26.1666666667,27.8333333333,3280833.333,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -208457152.034173, -170101376.61938, 215018818.700173, 253374594.114967},
{{3, 0, 8, {-91.3333333333,28.5,29.3,30.7,3280833.333,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -224037510.159723, -190501887.900234, 230599176.825723, 264134799.085212},
{{3, 0, 8, {-92,32.6666666667,33.3,34.7666666667,1312333.333,1312333.333}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -252114965.182788, -220603925.09152, 254739631.848788, 286250671.940057},
{{3, 0, 8, {-92,34.3333333333,34.9333333333,36.2333333333,1312333.333,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -263732072.537265, -235312197.62441, 266356739.203265, 294776614.11612},
{{3, 0, 8, {-92.5,30.5,31.1666666667,32.6666666667,3280833.333,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -235705523.16837, -204846795.365055, 242267189.83437, 273125917.637685},
{{3, 0, 8, {-93.1,46.5,47.0333333333,48.6333333333,2624666.667,328083.3333}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -385933858.012454, -368759635.557233, 391183191.346454, 408357413.801674},
{{3, 0, 8, {-93.5,40,40.6166666667,41.7833333333,1640416.667,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -312982803.048606, -290249762.839575, 316263636.382606, 338996676.591636},
{{3, 0, 8, {-93.5,41.5,42.0666666667,43.2666666667,4921250,3280833.333}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -324609611.161762, -303088197.425932, 334452111.161762, 355973524.897592},
{{3, 0, 8, {-94,43,43.7833333333,45.2166666667,2624666.667,328083.3333}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -346690823.003483, -327113674.332095, 351940156.337483, 371517305.008871},
{{3, 0, 8, {-94.25,45,45.6166666667,47.05,2624666.667,328083.3333}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -367757111.594128, -349560842.397585, 373006444.928128, 391202714.124671},
{{3, 0, 8, {-98,33.3333333333,33.9333333333,35.2333333333,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -255481720.407597, -226612868.978264, 259418720.407597, 288287571.836931},
{{3, 0, 8, {-98,35,35.5666666667,36.7666666667,1968500,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -267677707.348155, -240564047.908157, 271614707.348155, 298728366.788152},
{{3, 0, 8, {-98,38.3333333333,38.7166666667,39.7833333333,1312333.333,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -294756396.633938, -270090305.969395, 297381063.299938, 322047153.964481},
{{3, 0, 8, {-98.5,25.6666666667,26.1666666667,27.8333333333,984250,16404166.67}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -210753735.367173, -153757797.986841, 212722235.367173, 269718172.747506},
{{3, 0, 8, {-98.5,31.6666666667,32.1333333333,33.9666666667,1968500,6561666.667}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -244539804.259245, -207255673.598462, 248476804.259245, 285760934.920029},
{{3, 0, 8, {-98.5,36.6666666667,38.5666666667,37.2666666667,1312333.333,1312333.333}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -282891610.911457, -255540750.10591, 285516277.577457, 312867138.383004},
{{3, 0, 8, {-99,27.8333333333,28.3833333333,30.2833333333,1968500,13123333.33}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -221593659.467651, -172632102.018929, 225530659.467651, 274492216.916373},
{{3, 2, 7, {135,-24,-18,-36,0,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -63926410.6698201, -76575533.9276959, 63926410.6698201, 51277287.4119443},
{{3, 2, 7, {145,-37,-36,-38,2500000,4500000}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -81753864.458242, -88226525.7784545, 86753864.458242, 80281203.1380295},
{{3, 2, 7, {147,0,-32.666,-35.333,1000000,10000000}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -76161714.4889037, -80605291.2154594, 78161714.4889037, 73718137.7623481},
{{3, 28, 7, {23,-23,-18,-32,0,0}, 0,-134.73,-110.92,-292.66, {0,0,0,1,0}, 0,0,0,0,0,0,0,0},  -61566719.246568, -72414618.8198515, 61566719.246568, 47796808.4991836},
{{3, 29, 0, {110,10,25,40,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -45730.409093877, -38000.6689484435, 45730.409093877, 53460.1492393104},
{{3, 29, 0, {132.5,-10,-21.5,-33.5,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -40383.2188994685, -49161.7062224971, 40383.2188994685, 31604.73157644},
{{3, 29, 0, {25,35,40,65,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -84908.0213013157, -80759.383210806, 84908.0213013157, 89056.6593918254},
{{3, 29, 0, {47.5,25,15,35,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -38011.5438463059, -29679.737315404, 38011.5438463059, 46069.8166618704},
{{3, 29, 0, {95,40,20,60,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -55750.1705370063, -51403.1783693689, 55750.1705370063, 60097.1627046438},
{{3, 30, 7, {0,42.165,41.560387840948,42.76766346965,234.358,185861.369}, 0,-168,-60,320, {0,0,0,0,2.337229166667}, 0,0,0,0,0,0,0,0},  -98847613.927946, -91608686.7437833, 98848082.643946, 106087009.828109},
{{3, 30, 7, {0,42.165,41.560387840948,42.76766346965,234.358,4185861.369}, 0,-168,-60,320, {0,0,0,0,2.337229166667}, 0,0,0,0,0,0,0,0},  -98847613.927946, -87608686.7437833, 98848082.643946, 110087009.828109},
{{3, 30, 7, {0,44.1,43.199291275544,44.996093814511,600000,200000}, 0,-168,-60,320, {0,0,0,0,2.337229166667}, 0,0,0,0,0,0,0,0},  -104503824.398662, -98311919.3139696, 105703824.398662, 111895729.483354},
{{3, 30, 7, {0,44.1,43.199291275544,44.996093814511,600000,3200000}, 0,-168,-60,320, {0,0,0,0,2.337229166667}, 0,0,0,0,0,0,0,0},  -104503824.398662, -95311919.3139696, 105703824.398662, 114895729.483354},
{{3, 30, 7, {0,46.8,45.898918964419,47.696014502038,600000,200000}, 0,-168,-60,320, {0,0,0,0,2.337229166667}, 0,0,0,0,0,0,0,0},  -113967455.416715, -108367759.648713, 115167455.416715, 120767151.184716},
{{3, 30, 7, {0,46.8,45.898918964419,47.696014502038,600000,2200000}, 0,-168,-60,320, {0,0,0,0,2.337229166667}, 0,0,0,0,0,0,0,0},  -113967455.416715, -106367759.648713, 115167455.416715, 122767151.184716},
{{3, 30, 7, {0,49.5,48.598522847174,50.395911631678,600000,1200000}, 0,-168,-60,320, {0,0,0,0,2.337229166667}, 0,0,0,0,0,0,0,0},  -124264257.877732, -118206641.203686, 125464257.877732, 131521874.551778},
{{3, 30, 7, {0,49.5,48.598522847174,50.395911631678,600000,200000}, 0,-168,-60,320, {0,0,0,0,2.337229166667}, 0,0,0,0,0,0,0,0},  -124264257.877732, -119206641.203686, 125464257.877732, 130521874.551778},
{{3, 4, 7, {17,29.77930555,42,56,2679984.29,-484330}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -119981631.268354, -115489555.67494, 125341599.848354, 129833675.441769},
{{3, 4, 7, {4.3569397222,90,49.8333333333,51.1666666667,150000.01256,5400088.4378}, 0,81,120,129, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -128761663.907607, -123511575.482367, 129061663.932727, 134311752.357967},
{{3, 4, 7, {4.367975,90,49.8333333333,51.1666666667,150000,5400000}, 0,81,120,129, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -128761663.920167, -123511663.920167, 129061663.920167, 134311663.920167},
{{3, 6, 7, {23,-23,-18,-32,0,0}, 0,-136,-108,-292, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -61564419.064164, -72412828.2264313, 61564419.064164, 47794154.9885407},
{{3, 7, 7, {-68.5,44,46,60,0.99999912,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -139220351.696306, -133454155.398424, 139220353.696304, 144986549.994186},
{{3, 7, 7, {-96,23,20,60,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -89717066.7318629, -80920710.9207624, 89717066.7318629, 98513422.5429635},
{{3, 7, 7, {-96,23,33,45,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -89295530.2887796, -79679575.056002, 89295530.2887796, 98911485.5215573},
{{3, 7, 7, {-96,39,33,45,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -89295530.2887796, -81466209.2421514, 89295530.2887796, 97124851.3354078},
{{3, 7, 8, {-100,41.3333333333,41.85,42.8166666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -324055199.858072, -302686222.906259, 328055199.858072, 349424176.809884},
{{3, 7, 8, {-100,43.8333333333,44.4166666667,45.6833333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -353482329.11673, -334115632.201571, 357482329.11673, 376849026.031889},
{{3, 7, 8, {-100.3333333333,29.6666666667,30.1166666667,31.8833333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -231224626.213763, -197887504.402976, 235224626.213763, 268561748.024549},
{{3, 7, 8, {-100.3333333333,42.3333333333,42.8333333333,44.4,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -337599541.873847, -317137605.090357, 341599541.873847, 362061478.657338},
{{3, 7, 8, {-100.5,45.6666666667,46.1833333333,47.4833333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -374325734.927512, -356238757.711242, 378325734.927512, 396412712.143782},
{{3, 7, 8, {-100.5,47,47.4333333333,48.7333333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -389648664.421659, -372433148.680405, 393648664.421659, 410864180.162913},
{{3, 7, 8, {-101.5,34,34.65,36.1833333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -261737418.19268, -233764458.26155, 265737418.19268, 293710378.12381},
{{3, 7, 8, {-105.5,36.6666666667,37.2333333333,38.4333333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -281472391.565436, -256070159.37991, 285472391.565436, 310874623.750962},
{{3, 7, 8, {-105.5,37.8333333333,38.45,39.75,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -292682801.597057, -268439748.039304, 296682801.597057, 320925855.154811},
{{3, 7, 8, {-105.5,39.3333333333,39.7166666667,40.7833333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -303388496.172414, -280302427.035458, 307388496.172414, 330474565.30937},
{{3, 7, 8, {-109.5,44,44.8666666667,46.4,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -360164256.91475, -341067435.395783, 364164256.91475, 383261078.433716},
{{3, 7, 8, {-109.5,45.8333333333,46.45,47.8833333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -378353013.025643, -360433206.501952, 382353013.025643, 400272819.549334},
{{3, 7, 8, {-109.5,47,47.85,48.7166666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -392158606.473501, -375000733.125562, 396158606.473501, 413316479.821439},
{{3, 7, 8, {-111.5,36.6666666667,37.2166666667,38.35,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -281043066.114894, -255610253.091634, 285043066.114894, 310475879.138154},
{{3, 7, 8, {-111.5,38.3333333333,39.0166666667,40.65,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -299444370.803389, -275780256.64851, 303444370.803389, 327108484.958269},
{{3, 7, 8, {-111.5,40.3333333333,40.7166666667,41.7833333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -313102102.853289, -290872992.274509, 317102102.853289, 339331213.432069},
{{3, 7, 8, {-116.25,32.1666666667,32.7833333333,33.8833333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -246466336.059689, -216195067.142672, 250466336.059689, 280737604.976705},
{{3, 7, 8, {-118,33.5,34.0333333333,35.4666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -256676923.297906, -228027497.817793, 260676923.297906, 289326348.778019},
{{3, 7, 8, {-118.3333333333,34.1333333333,33.8666666667,34.4166666667,4186692.58,4160926.74}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -250023955.862257, -219155253.035527, 258397341.022257, 289266043.848988},
{{3, 7, 8, {-119,35.3333333333,36,37.25,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -271335689.379641, -244682757.78249, 275335689.379641, 301988620.976793},
{{3, 7, 8, {-120.5,36.5,37.0666666667,38.4333333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -280753380.155715, -255240388.034962, 284753380.155715, 310266372.276468},
{{3, 7, 8, {-120.5,41.6666666667,42.3333333333,44,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -332774131.475311, -311885463.700041, 336774131.475311, 357662799.250581},
{{3, 7, 8, {-120.5,43.6666666667,44.3333333333,46,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -354803161.418059, -335419308.868092, 358803161.418059, 378187013.968026},
{{3, 7, 8, {-120.5,45.3333333333,45.8333333333,47.3333333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -371328614.306844, -353039494.544633, 375328614.306844, 393617734.069055},
{{3, 7, 8, {-120.8333333333,47,47.5,48.7333333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -390066021.607763, -372860157.907014, 394066021.607763, 411271885.308512},
{{3, 7, 8, {-122,37.6666666667,38.3333333333,39.8333333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -292528160.248962, -268215917.695861, 296528160.248962, 320840402.802063},
{{3, 7, 8, {-122,39.3333333333,40,41.6666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -308999499.517601, -286207062.905244, 312999499.517601, 335791936.129957},
{{3, 7, 8, {-176,51,51.8333333333,53.8333333333,3000000,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -452044058.020189, -438479429.247576, 458044058.020189, 471608686.792802},
{{3, 7, 8, {-66.4333333333,18.4333333333,18.0333333333,18.4333333333,500000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -186287246.382945, -123317624.596919, 187287246.382945, 191870266.743549},
{{3, 7, 8, {-66.4333333333,18.4333333333,18.0333333333,18.4333333333,500000,100000}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -186287246.382945, -123217624.596919, 187287246.382945, 191970266.743549},
{{3, 7, 8, {-70.5,41,41.2833333333,41.4833333333,800000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -315632009.889084, -292507560.721895, 317232009.889084, 340356459.056272},
{{3, 7, 8, {-71.5,41,41.7166666667,42.6833333333,600000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -324083153.067583, -301133676.4023, 325283153.067583, 348232629.732865},
{{3, 7, 8, {-72.75,40.8333333333,41.2,41.8666666667,600000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -317327164.550983, -294012777.09286, 318527164.550983, 341841552.009105},
{{3, 7, 8, {-74,40.5,40.6666666667,41.0333333333,2000000,100000}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -309176711.160841, -286714165.719965, 313176711.160841, 335639256.601717},
{{3, 7, 8, {-77,37.8333333333,38.3,39.45,800000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -291849291.606495, -266280179.434122, 293449291.606495, 319018403.778868},
{{3, 7, 8, {-77.75,39.3333333333,39.9333333333,40.9666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -305301262.728919, -282316435.790654, 309301262.728919, 332286089.667183},
{{3, 7, 8, {-77.75,40.1666666667,40.8833333333,41.95,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -314757958.485081, -292546908.072089, 318757958.485081, 340969008.898072},
{{3, 7, 8, {-78.5,36.3333333333,36.7666666667,37.9666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -277493267.234049, -251681955.591517, 281493267.234049, 307304578.87658},
{{3, 7, 8, {-78.5,37.6666666667,38.0333333333,39.2,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -288335380.041045, -263758935.876341, 292335380.041045, 316911824.205748},
{{3, 7, 8, {-79,33.75,34.3333333333,36.1666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -260450504.74125, -232266866.952514, 264450504.74125, 292634142.529987},
{{3, 7, 8, {-79.5,38.5,39,40.25,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -297507509.009345, -273792382.933741, 301507509.009345, 325222635.08495},
{{3, 7, 8, {-81,31.8333333333,32.3333333333,33.6666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -244165860.608621, -213488972.345334, 248165860.608621, 278842748.871908},
{{3, 7, 8, {-81,33,33.7666666667,34.9666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -253843797.715595, -224716073.21255, 257843797.715595, 286971522.21864},
{{3, 7, 8, {-81,37,37.4833333333,38.8833333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -284507191.09272, -259436571.044567, 288507191.09272, 313577811.140872},
{{3, 7, 8, {-82.5,38,38.7333333333,40.0333333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -295273305.101978, -271246233.791215, 299273305.101978, 323300376.41274},
{{3, 7, 8, {-82.5,39.6666666667,40.4333333333,41.7,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -311290263.635938, -288731104.977908, 315290263.635938, 337849422.293968},
{{3, 7, 8, {-84.25,37.5,37.9666666667,38.9666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -287005538.125454, -262281485.830837, 291005538.125454, 315729590.42007},
{{3, 7, 8, {-84.5,29,29.5833333333,30.75,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -226275614.557084, -191820690.371176, 230275614.557084, 264730538.742992},
{{3, 7, 8, {-85.75,36.3333333333,36.7333333333,37.9333333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -277212156.180922, -251379920.019457, 281212156.180922, 307044392.342388},
{{3, 7, 8, {-86,34.6666666667,35.25,36.4166666667,2000000,100000}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -264990397.68906, -237455247.728469, 268990397.68906, 296525547.64965},
{{3, 7, 8, {-90,42,42.7333333333,44.0666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -335270989.964058, -314598855.147961, 339270989.964058, 359943124.780154},
{{3, 7, 8, {-90,43.8333333333,44.25,45.5,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -351502026.802048, -332071112.456575, 355502026.802048, 374932941.147521},
{{3, 7, 8, {-90,45.1666666667,45.5666666667,46.7666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -366394237.721428, -347905058.027747, 370394237.721428, 388883417.41511},
{{3, 7, 8, {-91.3333333333,25.6666666667,26.1666666667,27.8333333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -209731525.091816, -170154762.702282, 213731525.091816, 253308287.48135},
{{3, 7, 8, {-91.3333333333,28.6666666667,29.3,30.67,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -225224715.171064, -190451550.801895, 229224715.171064, 263997879.540232},
{{3, 7, 8, {-92,32.6666666667,33.3,34.7666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -251416565.855924, -221904841.787545, 255416565.855924, 284928289.924303},
{{3, 7, 8, {-92,34.3333333333,34.9333333333,36.2333333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -263032493.27188, -235299609.879708, 267032493.27188, 294765376.664052},
{{3, 7, 8, {-92.5,30.6666666667,31.1666666667,32.6666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -236977085.666136, -204897455.65052, 240977085.666136, 273056715.681752},
{{3, 7, 8, {-93.1,46.5,47.0333333333,48.6333333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -386533413.283953, -369062013.990686, 390533413.283953, 408004812.577219},
{{3, 7, 8, {-93.5,40,40.6166666667,41.7833333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -312606167.17083, -290232070.277002, 316606167.17083, 338980264.064658},
{{3, 7, 8, {-93.5,41.5,42.0666666667,43.2666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -327512222.250473, -306349761.225561, 331512222.250473, 352674683.275385},
{{3, 7, 8, {-94,43,43.7833333333,45.2166666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -347294715.331617, -327420366.588323, 351294715.331617, 371169064.074911},
{{3, 7, 8, {-94.25,45,45.6166666667,47.05,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -368358691.706647, -349865234.580916, 372358691.706647, 390852148.832379},
{{3, 7, 8, {-97.5,31.6666666667,32.1333333333,33.9666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -244498272.760526, -213806618.399239, 248498272.760526, 279189927.121813},
{{3, 7, 8, {-98,33.3333333333,33.9333333333,35.2333333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -255439079.607865, -226601046.512961, 259439079.607865, 288277112.70277},
{{3, 7, 8, {-98,35,35.5666666667,36.7666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -267633825.34702, -240550993.657801, 271633825.34702, 298716657.03624},
{{3, 7, 8, {-98,38.3333333333,38.7166666667,39.7833333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -294053624.246013, -270074545.620282, 298053624.246013, 322032702.871744},
{{3, 7, 8, {-98.5,25.6666666667,26.1666666667,27.8333333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -209731525.091816, -170154762.702282, 213731525.091816, 253308287.48135},
{{3, 7, 8, {-98.5,36.6666666667,38.5666666667,37.2666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -282190068.629093, -256838547.449771, 286190068.629093, 311541589.808415},
{{3, 7, 8, {-99,27.8333333333,28.3833333333,30.2833333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -221554461.124832, -185747019.927933, 225554461.124832, 261361902.321731},
{{3, 7, 8, {-99.5,39.6666666667,40.2833333333,41.7166666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -310633016.402964, -288042234.606454, 314633016.402964, 337223798.199475},
{{3, 8, 8, {-84.3333333333,41.5,42.1,43.6666666667,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -329791001.829521, -308721404.605132, 333791001.829521, 354860599.053909},
{{3, 8, 8, {-84.3333333333,43.3166666667,44.1833333333,45.7,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -352264889.757925, -332670121.355321, 356264889.757925, 375859658.160529},
{{3, 8, 8, {-87,44.7833333333,45.4833333333,47.0833333333,2000000,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -367778976.228709, -349189556.133667, 371778976.228709, 390368396.32375},
{{4, 7, 7, {0,90,90,0,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -9020145.99449487, -9020145.99449487, 9020145.99449487, 9020145.99449487},
{{6, 2, 1, {134,-90,-18,-36,0,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25412.1329074842, -30820.2630332478, 25412.1329074842, 20004.0027817205},
{{6, 2, 1, {147,-32.5,-29.5,-35.5,0,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -23611.4811992266, -33623.6505009443, 23611.4811992266, 13599.3118975089},
{{6, 2, 7, {134,-90,-18,-36,0,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25412132.9074842, -30820263.0332478, 25412132.9074842, 20004002.7817205},
{{6, 7, 7, {-96,23,20,60,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -21729866.6858831, -12546277.7889483, 21729866.6858831, 30913455.5828178},
{{6, 7, 7, {-96,23,29.5,45.5,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -22421877.8189369, -12546277.7889483, 22421877.8189369, 32297477.8489255},
{{7, 0, 7, {-133.6666666667,57,-36.8698976458,0.9999,5000000,-5000000}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  715546.220413176, -14325907.9924165, 9284453.77958682, 4325907.99241646},
{{7, 0, 8, {-133.6666666667,57,-36.8698976458,0.9999,16404166.67,-16404166.67}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  2347587.89480556, -47000916.4751197, 30460745.4451944, 14192583.1351197},
{{7, 7, 8, {-133.6666666667,57,-36.8698976458,0.9999,16404166.6667,-16404166.6667}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  2347270.72925546, -47001747.5537593, 30461062.6041445, 14193414.2203593},
{{8, 0, 3, {-110.1666666667,31,0.9999,700000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26361414.1707081, -44066918.5002725, 27761414.1707081, 21556214.9175316},
{{8, 0, 3, {-111.9166666667,31,0.9999,700000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26361414.1707081, -44066918.5002725, 27761414.1707081, 21556214.9175316},
{{8, 0, 3, {-113.75,31,0.9999333333,700000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26362316.3071587, -44068387.5429912, 27762316.3071587, 21556933.5291715},
{{8, 0, 7, {-104.3333333333,31,0.9999090909,165000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8083394.03137459, -13431718.8763978, 8413394.03137459, 6570394.04308942},
{{8, 0, 7, {-105,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604573, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-105.1666666667,40.5,0.9999375,200000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8048628.38213008, -14486109.042394, 8448628.38213009, 5516572.17078242},
{{8, 0, 7, {-106.25,31,0.9999,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748319.03923183, -13431596.758883, 8748319.03923182, 6570334.30686364},
{{8, 0, 7, {-107.3333333333,40.5,0.9999375,400000,100000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7848628.38213008, -14386109.042394, 8648628.38213008, 5616572.17078242},
{{8, 0, 7, {-107.8333333333,31,0.9999166667,830000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7418456.52523935, -13431820.641665, 9078456.52523936, 6570443.82360611},
{{8, 0, 7, {-108.75,40.5,0.9999375,600000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7648628.38213009, -14486109.042394, 8848628.38213009, 5516572.17078242},
{{8, 0, 7, {-110.0833333333,40.5,0.9999375,800000,100000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7448628.38213009, -14386109.042394, 9048628.38213008, 5616572.17078242},
{{8, 0, 7, {-110.1666666667,31,0.9999,213360,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8034959.03923183, -13431596.758883, 8461679.03923182, 6570334.30686364},
{{8, 0, 7, {-111,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-111.9166666667,31,0.9999,213360,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8034959.03923183, -13431596.758883, 8461679.03923182, 6570334.30686364},
{{8, 0, 7, {-112.1666666667,41.6666666667,0.9999473684,200000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8048709.78798227, -14615809.9655587, 8448709.78798228, 5387068.65441496},
{{8, 0, 7, {-113.75,31,0.9999333333,213360,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8035234.01042198, -13432044.5231037, 8461954.01042198, 6570553.33969148},
{{8, 0, 7, {-114,41.6666666667,0.9999473684,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748709.78798228, -14615809.9655587, 8748709.78798228, 5387068.65441496},
{{8, 0, 7, {-115.5833333333,34.75,0.9999,200000,8000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8048319.03923182, -5847439.17666561, 8448319.03923183, 14154491.8890811},
{{8, 0, 7, {-115.75,41.6666666667,0.9999333333,800000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7448594.01042197, -14615604.8204071, 9048594.01042198, 5386993.0423881},
{{8, 0, 7, {-116.6666666667,34.75,0.9999,500000,6000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748319.03923183, -7847439.17666561, 8748319.03923183, 12154491.8890811},
{{8, 0, 7, {-117,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-118.5833333333,34.75,0.9999,800000,4000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7448319.03923183, -9847439.17666561, 9048319.03923182, 10154491.8890811},
{{8, 0, 7, {-123,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604573, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-129,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-135,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604573, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-141,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604573, 9997964.94315451},
{{8, 0, 7, {-142,54,0.9999,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748319.03923183, -15986282.9700963, 8748319.03923183, 4015648.0956504},
{{8, 0, 7, {-146,54,0.9999,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748319.03923182, -15986282.9700963, 8748319.03923183, 4015648.0956504},
{{8, 0, 7, {-147,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604573, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-15,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-150,54,0.9999,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748319.03923183, -15986282.9700963, 8748319.03923183, 4015648.0956504},
{{8, 0, 7, {-153,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-154,54,0.9999,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748319.03923182, -15986282.9700963, 8748319.03923183, 4015648.0956504},
{{8, 0, 7, {-155.5,18.8333333333,0.9999666667,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748868.98243704, -12084782.4965247, 8748868.98243704, 7918482.16531935},
{{8, 0, 7, {-156.6666666667,20.3333333333,0.9999666667,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748868.98243704, -12250825.7354572, 8748868.98243703, 7752438.92638694},
{{8, 0, 7, {-158,21.1666666667,0.99999,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7749061.46218765, -12343372.1823083, 8749061.46218765, 7660359.23726966},
{{8, 0, 7, {-158,54,0.9999,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748319.03923182, -15986282.9700963, 8748319.03923182, 4015648.0956504},
{{8, 0, 7, {-159,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-159.5,21.8333333333,0.99999,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7749061.46218765, -12417187.1755718, 8749061.46218765, 7586544.24400615},
{{8, 0, 7, {-160.1666666667,21.6666666667,1,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7749143.95362718, -12398856.862762, 8749143.95362719, 7605074.59613055},
{{8, 0, 7, {-162,54,0.9999,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748319.03923183, -15986282.9700963, 8748319.03923183, 4015648.0956504},
{{8, 0, 7, {-165,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604573, 9997964.94315451},
{{8, 0, 7, {-166,54,0.9999,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748319.03923182, -15986282.9700963, 8748319.03923183, 4015648.0956504},
{{8, 0, 7, {-170,54,0.9999,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748319.03923182, -15986282.9700963, 8748319.03923183, 4015648.0956504},
{{8, 0, 7, {-171,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-177,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-2,49,0.9996012717,400000,-100000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7845854.7864821, -15525440.3489618, 8645854.7864821, 4470514.97634689},
{{8, 0, 7, {-21,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-27,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-33,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604573, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-39,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-45,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-51,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-55.5,0,0.9999,304800,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943519.03923183, -10000965.5328733, 8553119.03923183, 10000965.5328733},
{{8, 0, 7, {-57,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-58.5,0,0.9999,304800,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943519.03923183, -10000965.5328733, 8553119.03923182, 10000965.5328733},
{{8, 0, 7, {-61.5,0,0.9999,304800,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943519.03923182, -10000965.5328733, 8553119.03923183, 10000965.5328733},
{{8, 0, 7, {-63,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-64.5,0,0.9999,304800,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943519.03923182, -10000965.5328733, 8553119.03923182, 10000965.5328733},
{{8, 0, 7, {-67.5,0,0.9999,304800,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943519.03923183, -10000965.5328733, 8553119.03923182, 10000965.5328733},
{{8, 0, 7, {-68.5,43.6666666667,0.9999,300000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7948319.03923182, -14837267.8945768, 8548319.03923182, 5164663.17116993},
{{8, 0, 7, {-69,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-70.1666666667,42.8333333333,0.9999666667,900000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7348868.98243704, -14745678.889568, 9148868.98243704, 5257585.77227607},
{{8, 0, 7, {-70.5,0,0.9999,304800,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943519.03923183, -10000965.5328733, 8553119.03923182, 10000965.5328733},
{{8, 0, 7, {-71.5,41.0833333333,0.99999375,100000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8149092.39647748, -14551702.6314571, 8349092.39647748, 5452103.80286384},
{{8, 0, 7, {-71.6666666667,42.5,0.9999666667,300000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7948868.98243704, -14708651.3754574, 8548868.98243704, 5294613.28638668},
{{8, 0, 7, {-72.5,42.5,0.9999642857,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748849.34122529, -14708616.3529911, 8748849.34122528, 5294600.67949221},
{{8, 0, 7, {-73.5,0,0.9999,304800,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943519.03923182, -10000965.5328733, 8553119.03923183, 10000965.5328733},
{{8, 0, 7, {-74.5,38.8333333333,0.9999,150000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8098319.03923183, -14300537.2022438, 8398319.03923182, 5701393.86350293},
{{8, 0, 7, {-74.5,38.8333333334,0.9999,150000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8098319.03923183, -14300537.2022549, 8398319.03923182, 5701393.86349183},
{{8, 0, 7, {-75,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-75.4166666667,38,0.999995,200000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8049102.70790742, -14209392.7012779, 8449102.70790742, 5794438.73795742},
{{8, 0, 7, {-76.5,0,0.9999,304800,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943519.03923183, -10000965.5328733, 8553119.03923182, 10000965.5328733},
{{8, 0, 7, {-76.5833333333,40,0.9999375,250000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7998628.38213008, -14430592.791401, 8498628.38213008, 5572088.42177536},
{{8, 0, 7, {-78.5833333333,40,0.9999375,350000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7898628.38213008, -14430592.791401, 8598628.38213008, 5572088.42177536},
{{8, 0, 7, {-79.5,0,0.9999,304800,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943519.03923182, -10000965.5328733, 8553119.03923183, 10000965.5328733},
{{8, 0, 7, {-81,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604573, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-81,24.3333333333,0.9999411765,200000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8048658.71010783, -12693427.8789688, 8448658.71010783, 7309326.87866157},
{{8, 0, 7, {-82,24.3333333333,0.9999411765,200000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8048658.71010783, -12693427.8789688, 8448658.71010783, 7309326.87866157},
{{8, 0, 7, {-82.1666666667,30,0.9999,200000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8048319.03923182, -13320746.9194058, 8448319.03923183, 6681184.14634087},
{{8, 0, 7, {-84.1666666667,30,0.9999,700000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7548319.03923183, -13320746.9194058, 8948319.03923182, 6681184.14634087},
{{8, 0, 7, {-85.6666666667,37.5,0.9999666667,100000,250000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8148868.98243704, -13903496.0736175, 8348868.98243704, 6099768.58822661},
{{8, 0, 7, {-85.8333333333,30.5,0.99996,200000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8048813.98786904, -13376972.3620314, 8448813.98786904, 6626158.9396028},
{{8, 0, 7, {-87,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-87.0833333333,37.5,0.9999666667,900000,250000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7348868.98243704, -13903496.0736175, 9148868.98243703, 6099768.58822661},
{{8, 0, 7, {-87.5,30,0.9999333333,600000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7648594.01042198, -13321190.988266, 8848594.01042197, 6681406.87452919},
{{8, 0, 7, {-88.3333333333,36.6666666667,0.999975,300000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7948937.72502834, -14061133.6597386, 8548937.72502835, 5942297.70086748},
{{8, 0, 7, {-88.8333333333,29.5,0.99995,300000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7948731.4964295, -13265991.6727511, 8548731.4964295, 6736939.58956854},
{{8, 0, 7, {-9,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {-90.1666666667,36.6666666667,0.9999411765,700000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7548658.71010783, -14060658.0510941, 8948658.71010783, 5942096.70653633},
{{8, 0, 7, {-90.3333333333,29.5,0.99995,700000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7548731.49642951, -13265991.6727511, 8948731.4964295, 6736939.58956854},
{{8, 0, 7, {-90.5,35.8333333333,0.9999333333,250000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7998594.01042198, -13968084.2221719, 8498594.01042197, 6034513.64062328},
{{8, 0, 7, {-92.5,35.8333333333,0.9999333333,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748594.01042198, -13968084.2221719, 8748594.01042197, 6034513.64062328},
{{8, 0, 7, {-93,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604573, 9997964.94315451},
{{8, 0, 7, {-94.5,36.1666666667,0.9999411765,850000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7398658.71010783, -14005177.9421902, 9098658.71010783, 5997576.81544022},
{{8, 0, 7, {-99,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {105,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, 2035.0568454858, 8745844.29604573, 19997964.9431545},
{{8, 0, 7, {111,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, 2035.0568454858, 8745844.29604574, 19997964.9431545},
{{8, 0, 7, {117,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, 2035.0568454858, 8745844.29604574, 19997964.9431545},
{{8, 0, 7, {12,0,0.99995,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7748731.4964295, -10001465.6311598, 8748731.4964295, 10001465.6311598},
{{8, 0, 7, {123,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, 2035.0568454858, 8745844.29604573, 19997964.9431545},
{{8, 0, 7, {129,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, 2035.0568454858, 8745844.29604574, 19997964.9431545},
{{8, 0, 7, {135,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604573, 2035.0568454858, 8745844.29604573, 19997964.9431545},
{{8, 0, 7, {141,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604573, 2035.0568454858, 8745844.29604574, 19997964.9431545},
{{8, 0, 7, {147,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, 2035.0568454858, 8745844.29604573, 19997964.9431545},
{{8, 0, 7, {15,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {15,0,1,900000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7349143.95362719, -10001965.7294463, 9149143.95362719, 10001965.7294463},
{{8, 0, 7, {153,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, 2035.0568454858, 8745844.29604574, 19997964.9431545},
{{8, 0, 7, {159,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, 2035.0568454858, 8745844.29604574, 19997964.9431545},
{{8, 0, 7, {165,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604573, 2035.0568454858, 8745844.29604574, 19997964.9431545},
{{8, 0, 7, {9,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, -9997964.94315451, 8745844.29604574, 9997964.94315451},
{{8, 0, 7, {9.5,0,0.99995,200000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8048731.4964295, -10001465.6311598, 8448731.49642951, 10001465.6311598},
{{8, 0, 7, {99,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29604574, 2035.0568454858, 8745844.29604574, 19997964.9431545},
{{8, 0, 8, {-104.3333333333,31,0.9999090909,541337.5,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26520268.5846015, -44067231.0136484, 27602943.5846015, 21556367.7897025},
{{8, 0, 8, {-105.1666666667,40.5,0.9999375,656166.6667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26406208.2836718, -47526509.4165876, 27718541.6170718, 18098953.863642},
{{8, 0, 8, {-106.25,31,0.9999,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25420943.3808798, -44066830.3664355, 28701776.7148797, 21556171.8051018},
{{8, 0, 8, {-107.3333333333,40.5,0.9999375,1312333.333,328083.3333,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25750041.6173718, -47198426.0832876, 28374708.2833718, 18427037.196942},
{{8, 0, 8, {-107.8333333333,31,0.9999166667,2723091.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -24338719.4495561, -44067564.8885294, 29784902.7835561, 21556531.111281},
{{8, 0, 8, {-108.75,40.5,0.9999375,1968500,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25093874.9503718, -47526509.4165876, 29030874.9503718, 18098953.863642},
{{8, 0, 8, {-110.0833333333,40.5,0.9999375,2624666.667,328083.3333,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -24437708.2833718, -47198426.0832876, 29687041.6173718, 18427037.196942},
{{8, 0, 8, {-110.1666666667,31,0.9999,699998.6,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26361361.4478798, -44066830.3664355, 27761358.6478797, 21556171.8051018},
{{8, 0, 8, {-111.9166666667,31,0.9999,699998.6,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26361361.4478797, -44066830.3664355, 27761358.6478797, 21556171.8051018},
{{8, 0, 8, {-112.1666666667,41.6666666667,0.9999473684,656166.6667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26406475.3627052, -47952036.5286703, 27718808.6961052, 17674074.4103597},
{{8, 0, 8, {-113.75,31,0.9999333333,699998.6,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26362263.5825261, -44068299.4062161, 27762260.7825261, 21556890.4153045},
{{8, 0, 8, {-114,41.6666666667,0.9999473684,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25422225.3624052, -47952036.5286703, 28703058.6964052, 17674074.4103597},
{{8, 0, 8, {-115.5833333333,34.75,0.9999,656166.6667,26246666.67,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26405193.3811797, -19184473.3621104, 27717526.7145797, 46438528.8094268},
{{8, 0, 8, {-115.75,41.6666666667,0.9999333333,2624666.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -24437595.5155261, -47951363.4816189, 29686928.8495261, 17673826.3399016},
{{8, 0, 8, {-116.6666666667,34.75,0.9999,1640416.667,19685000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25420943.3808797, -25746140.0321104, 28701776.7148797, 39876862.1394268},
{{8, 0, 8, {-118.5833333333,34.75,0.9999,2624666.667,13123333.33,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -24436693.3808797, -32307806.7021104, 29686026.7148797, 33315195.4694268},
{{8, 0, 8, {-142,54,0.9999,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25420943.3808797, -52448330.0443909, 28701776.7148797, 13174672.1271464},
{{8, 0, 8, {-146,54,0.9999,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25420943.3808797, -52448330.0443909, 28701776.7148797, 13174672.1271464},
{{8, 0, 8, {-150,54,0.9999,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25420943.3808797, -52448330.0443909, 28701776.7148797, 13174672.1271464},
{{8, 0, 8, {-154,54,0.9999,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25420943.3808797, -52448330.0443909, 28701776.7148797, 13174672.1271464},
{{8, 0, 8, {-155.5,18.8333333333,0.9999666667,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25422747.6528788, -39648157.2406816, 28703580.9868788, 25979220.2373852},
{{8, 0, 8, {-156.6666666667,20.3333333333,0.9999666667,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25422747.6528789, -40192917.4337457, 28703580.9868788, 25434460.0443211},
{{8, 0, 8, {-158,21.1666666667,0.99999,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25423379.1468606, -40496546.9014566, 28704212.4808606, 25132361.9309422},
{{8, 0, 8, {-158,54,0.9999,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25420943.3808797, -52448330.0443909, 28701776.7148797, 13174672.1271464},
{{8, 0, 8, {-159.5,21.8333333333,0.99999,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25423379.1468606, -40738721.5918553, 28704212.4808606, 24890187.2405435},
{{8, 0, 8, {-160.1666666667,21.6666666667,1,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25423649.7875252, -40678582.8905784, 28704483.1215252, 24950982.2374716},
{{8, 0, 8, {-162,54,0.9999,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25420943.3808797, -52448330.0443909, 28701776.7148797, 13174672.1271464},
{{8, 0, 8, {-166,54,0.9999,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25420943.3808797, -52448330.0443909, 28701776.7148797, 13174672.1271464},
{{8, 0, 8, {-170,54,0.9999,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25420943.3808797, -52448330.0443909, 28701776.7148797, 13174672.1271464},
{{8, 0, 8, {-68.5,43.6666666667,0.9999,984250,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26077110.0478797, -48678603.0841239, 28045610.0478797, 16944399.0874133},
{{8, 0, 8, {-70.1666666667,42.8333333333,0.9999666667,2952750,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -24110414.3198788, -48378114.8235244, 30015914.3198788, 17249262.6545424},
{{8, 0, 8, {-71.5,41.0833333333,0.99999375,328083.3333,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26735813.9708099, -47741711.0500389, 27391980.6374099, 17887443.8932291},
{{8, 0, 8, {-71.6666666667,42.5,0.9999666667,984250,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26078914.3198788, -48256633.7209798, 28047414.3198788, 17370743.757087},
{{8, 0, 8, {-72.5,42.5,0.9999642857,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25422683.2133366, -48256518.8181048, 28703516.5473366, 17370702.3959674},
{{8, 0, 8, {-74.5,38.8333333333,0.9999,492125,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26569235.0478797, -46917679.1376948, 27553485.0478797, 18705323.0338425},
{{8, 0, 8, {-74.5,38.8333333334,0.9999,492125,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26569235.0478797, -46917679.1377312, 27553485.0478797, 18705323.0338061},
{{8, 0, 8, {-75.4166666667,38,0.999995,656166.6667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26407764.4674929, -46618649.2207758, 27720097.8008929, 19010587.7594486},
{{8, 0, 8, {-76.5833333333,40,0.9999375,820208.3333,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26242166.6170718, -47344369.8497883, 27882583.2836718, 18281093.4304413},
{{8, 0, 8, {-78.5833333333,40,0.9999375,1148291.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25914083.2833718, -47344369.8497883, 28210666.6173718, 18281093.4304413},
{{8, 0, 8, {-81,24.3333333333,0.9999411765,656166.6667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26406307.7847121, -41645021.2995836, 27718641.1181121, 23980683.2677422},
{{8, 0, 8, {-82,24.3333333333,0.9999411765,656166.6667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26406307.7847121, -41645021.2995836, 27718641.1181121, 23980683.2677422},
{{8, 0, 8, {-82.1666666667,30,0.9999,656166.6667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26405193.3811797, -43703150.5180839, 27717526.7145797, 21919851.6534533},
{{8, 0, 8, {-84.1666666667,30,0.9999,2296583.333,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -24764776.7148797, -43703150.5180839, 29357943.3808797, 21919851.6534533},
{{8, 0, 8, {-85.6666666667,37.5,0.9999666667,328083.3333,820208.3333,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26735080.9865789, -45615053.3682267, 27391247.6531788, 20012324.1098401},
{{8, 0, 8, {-85.8333333333,30.5,0.99996,656166.6667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26406817.225167, -43887616.8244314, 27719150.558567, 21739323.1210135},
{{8, 0, 8, {-87.0833333333,37.5,0.9999666667,2952750,820208.3333,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -24110414.3198788, -45615053.3682267, 30015914.3198788, 20012324.1098401},
{{8, 0, 8, {-87.5,30,0.9999333333,1968500,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25093762.1825261, -43704607.4340027, 29030762.1825261, 21920582.3875179},
{{8, 0, 8, {-88.3333333333,36.6666666667,0.999975,984250,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26079139.8528638, -46132236.0153258, 28047639.8528638, 19495688.373596},
{{8, 0, 8, {-88.8333333333,29.5,0.99995,984250,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26078463.2512025, -43523507.6796842, 28046963.2512025, 22102775.9701094},
{{8, 0, 8, {-90.1666666667,36.6666666667,0.9999411765,2296583.333,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -24765891.1184121, -46130675.6226312, 29359057.7844121, 19495028.9446946},
{{8, 0, 8, {-90.3333333333,29.5,0.99995,2296583.333,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -24766129.9182025, -43523507.6796842, 29359296.5842025, 22102775.9701094},
{{8, 0, 8, {-90.5,35.8333333333,0.9999333333,820208.3333,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26242053.8492261, -45826956.318909, 27882470.5158261, 19798233.5026115},
{{8, 0, 8, {-92.5,35.8333333333,0.9999333333,1640416.667,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25421845.5155261, -45826956.318909, 28702678.8495261, 19798233.5026115},
{{8, 0, 8, {-94.5,36.1666666667,0.9999411765,2788708.333,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -24273766.1184121, -45948654.6320023, 29851182.7844121, 19677049.9353235},
{{8, 10, 7, {12,0,1,4500000,0,0}, 0,582,105,414, {-1.04,-0.35,3.08,8.3,0}, 0,0,0,0,0,0,0,0},  -3748143.32560618, -10000855.7646457, 12748143.3256062, 10000855.7646457},
{{8, 10, 7, {123,0,0.9996,500000,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7744844.06827594, -9996855.42233989, 8744844.06827594, 9996855.42233989},
{{8, 10, 7, {124,26,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127362, -12876121.1385208, 8247318.51127362, 7123590.21961776},
{{8, 10, 7, {127.5,26,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127362, -12876121.1385208, 8247318.51127362, 7123590.21961776},
{{8, 10, 7, {129,0,0.9996,500000,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7744844.06827594, -9996855.42233989, 8744844.06827594, 9996855.42233989},
{{8, 10, 7, {129.5,33,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127362, -13651876.9787944, 8247318.51127362, 6347834.37934419},
{{8, 10, 7, {131,26,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127363, -12876121.1385208, 8247318.51127362, 7123590.21961776},
{{8, 10, 7, {131,33,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127363, -13651876.9787944, 8247318.51127362, 6347834.37934419},
{{8, 10, 7, {132.166666,36,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127362, -13984603.2178459, 8247318.51127362, 6015108.14029263},
{{8, 10, 7, {133.5,33,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127362, -13651876.9787944, 8247318.51127363, 6347834.37934419},
{{8, 10, 7, {134.333333,36,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127362, -13984603.2178459, 8247318.51127362, 6015108.14029263},
{{8, 10, 7, {135,0,0.9996,500000,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7744844.06827594, -9996855.42233989, 8744844.06827594, 9996855.42233989},
{{8, 10, 7, {136,20,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127363, -12211786.0141993, 8247318.51127362, 7787925.34393926},
{{8, 10, 7, {136,36,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127363, -13984603.2178459, 8247318.51127362, 6015108.14029263},
{{8, 10, 7, {137.166666,36,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127362, -13984603.2178459, 8247318.51127362, 6015108.14029263},
{{8, 10, 7, {138.5,36,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127363, -13984603.2178459, 8247318.51127362, 6015108.14029263},
{{8, 10, 7, {139.833333,36,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127362, -13984603.2178459, 8247318.51127362, 6015108.14029263},
{{8, 10, 7, {140.25,44,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127362, -14872697.8989466, 8247318.51127362, 5127013.45919198},
{{8, 10, 7, {140.833333,40,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127362, -14428497.5605602, 8247318.51127362, 5571213.7975784},
{{8, 10, 7, {141,0,0.9996,500000,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7744844.06827594, -9996855.42233989, 8744844.06827595, 9996855.42233989},
{{8, 10, 7, {142,26,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127362, -12876121.1385208, 8247318.51127362, 7123590.21961776},
{{8, 10, 7, {142.25,44,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127362, -14872697.8989466, 8247318.51127362, 5127013.45919198},
{{8, 10, 7, {144.25,44,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127363, -14872697.8989466, 8247318.51127362, 5127013.45919198},
{{8, 10, 7, {147,0,0.9996,500000,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7744844.06827594, -9996855.42233989, 8744844.06827593, 9996855.42233989},
{{8, 10, 7, {15,0,1,5500000,0,0}, 0,582,105,414, {-1.04,-0.35,3.08,8.3,0}, 0,0,0,0,0,0,0,0},  -2748143.32560618, -10000855.7646457, 13748143.3256062, 10000855.7646457},
{{8, 10, 7, {15.8082777778,0,1,1500000,0,0}, 0,498,-36,568, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -6748143.32560618, -10000855.7646457, 9748143.32560618, 10000855.7646457},
{{8, 10, 7, {15.8082777778,0,1,1500000,0,0}, 0,419.3836,99.3335,591.3451, {-0.850389, -1.817277, 7.862238, -0.99496, 0}, 0,0,0,0,0,0,0,0},  -6748143.32560618, -10000855.7646457, 9748143.32560618, 10000855.7646457},
{{8, 10, 7, {153,0,0.9996,500000,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7744844.06827594, -9996855.42233989, 8744844.06827594, 9996855.42233989},
{{8, 10, 7, {154,26,0.9999,0,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8247318.51127362, -12876121.1385208, 8247318.51127362, 7123590.21961776},
{{8, 10, 7, {159,0,0.9996,500000,0,0}, 0,-128,481,664, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7744844.06827594, -9996855.42233989, 8744844.06827594, 9996855.42233989},
{{8, 10, 7, {3,0,1,1500000,0,0}, 0,582,105,414, {-1.04,-0.35,3.08,8.3,0}, 0,0,0,0,0,0,0,0},  -6748143.32560618, -10000855.7646457, 9748143.32560618, 10000855.7646457},
{{8, 10, 7, {6,0,1,2500000,0,0}, 0,582,105,414, {-1.04,-0.35,3.08,8.3,0}, 0,0,0,0,0,0,0,0},  -5748143.32560618, -10000855.7646457, 10748143.3256062, 10000855.7646457},
{{8, 10, 7, {9,0,1,3500000,0,0}, 0,582,105,414, {-1.04,-0.35,3.08,8.3,0}, 0,0,0,0,0,0,0,0},  -4748143.32560618, -10000855.7646457, 11748143.3256062, 10000855.7646457},
{{8, 13, 7, {-8,53.5,1.000035,200000,250000,0}, 0,506,-122,611, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8048349.95153666, -15680948.9714353, 8448349.95153666, 4321303.19013035},
{{8, 2, 7, {105,0,0.9996,500000,10000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {105,0,0.9996,500000,10000000,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {111,0,0.9996,500000,10000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {111,0,0.9996,500000,10000000,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {117,0,0.9996,500000,10000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {117,0,0.9996,500000,10000000,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {123,0,0.9996,500000,10000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {123,0,0.9996,500000,10000000,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {129,0,0.9996,500000,10000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {129,0,0.9996,500000,10000000,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {135,0,0.9996,500000,10000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {135,0,0.9996,500000,10000000,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {141,0,0.9996,500000,10000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491762, 19998000.5903039},
{{8, 2, 7, {141,0,0.9996,500000,10000000,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491762, 19998000.5903039},
{{8, 2, 7, {141,0,0.99994,300000,5000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7948679.10409615, -5001401.27077682, 8548679.10409616, 15001401.2707768},
{{8, 2, 7, {143,0,0.99994,300000,5000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7948679.10409616, -5001401.27077682, 8548679.10409615, 15001401.2707768},
{{8, 2, 7, {145,0,0.99994,300000,5000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7948679.10409615, -5001401.27077682, 8548679.10409616, 15001401.2707768},
{{8, 2, 7, {147,0,0.9996,500000,10000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491762, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {147,0,0.9996,500000,10000000,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491762, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {147,0,0.99994,300000,5000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7948679.10409616, -5001401.27077682, 8548679.10409615, 15001401.2707768},
{{8, 2, 7, {149,0,0.99994,300000,5000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7948679.10409615, -5001401.27077682, 8548679.10409616, 15001401.2707768},
{{8, 2, 7, {149.0092948333,0,1.000086,200000,4510193.4939,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8049883.48350812, -5492668.06907988, 8449883.48350812, 14513055.0568799},
{{8, 2, 7, {151,0,0.99994,300000,5000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7948679.10409615, -5001401.27077682, 8548679.10409616, 15001401.2707768},
{{8, 2, 7, {153,0,0.9996,500000,10000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {153,0,0.9996,500000,10000000,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {153,0,0.99994,300000,5000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7948679.10409616, -5001401.27077682, 8548679.10409616, 15001401.2707768},
{{8, 2, 7, {155,0,0.99994,300000,5000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7948679.10409615, -5001401.27077682, 8548679.10409615, 15001401.2707768},
{{8, 2, 7, {159,0,0.9996,500000,10000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {159,0,0.9996,500000,10000000,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {165,0,0.9996,500000,10000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {165,0,0.9996,500000,10000000,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {99,0,0.9996,500000,10000000,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 2, 7, {99,0,0.9996,500000,10000000,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 24, 7, {-33,0,0.9996,500000,10000000,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 24, 7, {-39,0,0.9996,500000,10000000,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 24, 7, {-45,0,0.9996,500000,10000000,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 24, 7, {-51,0,0.9996,500000,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393},
{{8, 24, 7, {-51,0,0.9996,500000,10000000,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 24, 7, {-57,0,0.9996,500000,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393},
{{8, 24, 7, {-57,0,0.9996,500000,10000000,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 24, 7, {-63,0,0.9996,500000,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393},
{{8, 24, 7, {-63,0,0.9996,500000,10000000,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 24, 7, {-69,0,0.9996,500000,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393},
{{8, 24, 7, {-69,0,0.9996,500000,10000000,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 24, 7, {-75,0,0.9996,500000,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393},
{{8, 24, 7, {-75,0,0.9996,500000,10000000,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 24, 7, {-81,0,0.9996,500000,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393},
{{8, 24, 7, {-81,0,0.9996,500000,10000000,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, 1999.40969607119, 8745874.38491761, 19998000.5903039},
{{8, 28, 7, {-105,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-105,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-111,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-111,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-117,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-117,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-123,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-123,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-129,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-129,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-135,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597411, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-135,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597411, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-141,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597413, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-141,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597413, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-147,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597411, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-147,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597411, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-15,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-15,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-153,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-153,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-159,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-159,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-165,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-165,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-171,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597413, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-171,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597413, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-177,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-177,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-21,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-21,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-27,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-27,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-3,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-3,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-33,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-33,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-39,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-39,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-45,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-45,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-51,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-51,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-57,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-57,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-63,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-63,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-69,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-69,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-75,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-75,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-8.1319061111,39.6666666667,1,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8249143.95355554, -14394484.2826752, 8249143.95355554, 5609447.17638195},
{{8, 28, 7, {-81,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597411, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-81,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597411, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-87,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-87,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-9,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-9,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-93,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-93,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {-99,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {-99,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {105,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {105,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {111,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {111,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {117,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {117,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {123,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {123,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {129,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {129,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {135,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {135,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {141,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597411, -9997964.94323674, 8745844.29597413, 9997964.94323674},
{{8, 28, 7, {141,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597411, 2035.05676326129, 8745844.29597413, 19997964.9432367},
{{8, 28, 7, {147,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597411, 9997964.94323674},
{{8, 28, 7, {147,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597411, 19997964.9432367},
{{8, 28, 7, {15,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {15,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {153,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {153,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {159,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {159,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {165,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {165,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {17,0,1,0,0,0}, 0,-134.73,-110.92,-292.66, {0,0,0,1,0}, 0,0,0,0,0,0,0,0},  -8249143.95355554, -10001965.7295286, 8249143.95355554, 10001965.7295286},
{{8, 28, 7, {171,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597413, 9997964.94323674},
{{8, 28, 7, {171,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597413, 19997964.9432367},
{{8, 28, 7, {177,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {177,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {19,0,1,0,0,0}, 0,-134.73,-110.92,-292.66, {0,0,0,1,0}, 0,0,0,0,0,0,0,0},  -8249143.95355554, -10001965.7295286, 8249143.95355554, 10001965.7295286},
{{8, 28, 7, {19,0,1,0,3700000,0}, 0,-134.73,-110.92,-292.66, {0,0,0,1,0}, 0,0,0,0,0,0,0,0},  -8249143.95355554, -6301965.72952855, 8249143.95355554, 13701965.7295286},
{{8, 28, 7, {21,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {21,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {21,0,1,0,0,0}, 0,-134.73,-110.92,-292.66, {0,0,0,1,0}, 0,0,0,0,0,0,0,0},  -8249143.95355554, -10001965.7295286, 8249143.95355554, 10001965.7295286},
{{8, 28, 7, {23,0,1,0,0,0}, 0,-134.73,-110.92,-292.66, {0,0,0,1,0}, 0,0,0,0,0,0,0,0},  -8249143.95355554, -10001965.7295286, 8249143.95355554, 10001965.7295286},
{{8, 28, 7, {25,0,1,0,0,0}, 0,-134.73,-110.92,-292.66, {0,0,0,1,0}, 0,0,0,0,0,0,0,0},  -8249143.95355554, -10001965.7295286, 8249143.95355554, 10001965.7295286},
{{8, 28, 7, {27,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {27,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {27,0,1,0,0,0}, 0,-134.73,-110.92,-292.66, {0,0,0,1,0}, 0,0,0,0,0,0,0,0},  -8249143.95355554, -10001965.7295286, 8249143.95355554, 10001965.7295286},
{{8, 28, 7, {29,0,1,0,0,0}, 0,-134.73,-110.92,-292.66, {0,0,0,1,0}, 0,0,0,0,0,0,0,0},  -8249143.95355554, -10001965.7295286, 8249143.95355554, 10001965.7295286},
{{8, 28, 7, {3,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {3,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {31,0,1,0,0,0}, 0,-134.73,-110.92,-292.66, {0,0,0,1,0}, 0,0,0,0,0,0,0,0},  -8249143.95355554, -10001965.7295286, 8249143.95355554, 10001965.7295286},
{{8, 28, 7, {33,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {33,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {39,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {39,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {45,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {45,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {51,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {51,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {57,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {57,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {6.166666667,49.83333333,1,80000,100000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8169143.95355554, -15424274.8616842, 8329143.95355554, 4579656.59737291},
{{8, 28, 7, {63,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {63,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {69,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {69,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {75,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {75,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {81,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {81,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {87,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {87,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {9,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {9,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {93,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {93,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 28, 7, {99,0,0.9996,500000,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, -9997964.94323674, 8745844.29597412, 9997964.94323674},
{{8, 28, 7, {99,0,0.9996,500000,10000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745844.29597412, 2035.05676326129, 8745844.29597412, 19997964.9432367},
{{8, 29, 0, {78,0,1,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -5125.7804133582, -6214.93337007411, 5125.7804133582, 6214.93337007411},
{{8, 3, 7, {-171,0,1,32500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  24250718.46099, -10002137.4977586, 40749281.53901, 10002137.4977586},
{{8, 3, 7, {-177,0,1,31500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  23250718.46099, -10002137.4977586, 39749281.53901, 10002137.4977586},
{{8, 3, 7, {105,0,1,18500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  10250718.46099, -10002137.4977586, 26749281.53901, 10002137.4977586},
{{8, 3, 7, {111,0,1,19500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  11250718.46099, -10002137.4977586, 27749281.53901, 10002137.4977586},
{{8, 3, 7, {117,0,1,20500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  12250718.46099, -10002137.4977586, 28749281.53901, 10002137.4977586},
{{8, 3, 7, {123,0,1,21500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  13250718.46099, -10002137.4977586, 29749281.53901, 10002137.4977586},
{{8, 3, 7, {129,0,1,22500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  14250718.46099, -10002137.4977586, 30749281.53901, 10002137.4977586},
{{8, 3, 7, {135,0,1,23500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  15250718.46099, -10002137.4977586, 31749281.53901, 10002137.4977586},
{{8, 3, 7, {141,0,1,24500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  16250718.46099, -10002137.4977586, 32749281.53901, 10002137.4977586},
{{8, 3, 7, {147,0,1,25500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  17250718.46099, -10002137.4977586, 33749281.53901, 10002137.4977586},
{{8, 3, 7, {15,0,1,3500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  -4749281.53900998, -10002137.4977586, 11749281.53901, 10002137.4977586},
{{8, 3, 7, {153,0,1,26500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  18250718.46099, -10002137.4977586, 34749281.53901, 10002137.4977586},
{{8, 3, 7, {159,0,1,27500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  19250718.46099, -10002137.4977586, 35749281.53901, 10002137.4977586},
{{8, 3, 7, {165,0,1,28500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  20250718.46099, -10002137.4977586, 36749281.53901, 10002137.4977586},
{{8, 3, 7, {171,0,1,29500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  21250718.46099, -10002137.4977586, 37749281.53901, 10002137.4977586},
{{8, 3, 7, {177,0,1,30500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  22250718.46099, -10002137.4977586, 38749281.53901, 10002137.4977586},
{{8, 3, 7, {21,0,1,4500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  -3749281.53900997, -10002137.4977586, 12749281.53901, 10002137.4977586},
{{8, 3, 7, {27,0,1,5500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  -2749281.53900998, -10002137.4977586, 13749281.53901, 10002137.4977586},
{{8, 3, 7, {3,0,1,1500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  -6749281.53900998, -10002137.4977586, 9749281.53900998, 10002137.4977586},
{{8, 3, 7, {33,0,1,6500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  -1749281.53900998, -10002137.4977586, 14749281.53901, 10002137.4977586},
{{8, 3, 7, {39,0,1,7500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  -749281.539009975, -10002137.4977586, 15749281.53901, 10002137.4977586},
{{8, 3, 7, {45,0,1,8500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  250718.460990024, -10002137.4977586, 16749281.53901, 10002137.4977586},
{{8, 3, 7, {51,0,1,9500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  1250718.46099002, -10002137.4977586, 17749281.53901, 10002137.4977586},
{{8, 3, 7, {57,0,1,10500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  2250718.46099002, -10002137.4977586, 18749281.53901, 10002137.4977586},
{{8, 3, 7, {63,0,1,11500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  3250718.46099002, -10002137.4977586, 19749281.53901, 10002137.4977586},
{{8, 3, 7, {69,0,1,12500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  4250718.46099003, -10002137.4977586, 20749281.53901, 10002137.4977586},
{{8, 3, 7, {75,0,1,13500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  5250718.46099002, -10002137.4977586, 21749281.53901, 10002137.4977586},
{{8, 3, 7, {81,0,1,14500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  6250718.46099003, -10002137.4977586, 22749281.53901, 10002137.4977586},
{{8, 3, 7, {87,0,1,15500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  7250718.46099003, -10002137.4977586, 23749281.53901, 10002137.4977586},
{{8, 3, 7, {9,0,1,2500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  -5749281.53900998, -10002137.4977586, 10749281.53901, 10002137.4977586},
{{8, 3, 7, {93,0,1,16500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  8250718.46099003, -10002137.4977586, 24749281.53901, 10002137.4977586},
{{8, 3, 7, {99,0,1,17500000,0,0}, 0,24,-123,-94, {-0.02,0.25,0.13,1.1,0}, 0,0,0,0,0,0,0,0},  9250718.46099002, -10002137.4977586, 25749281.53901, 10002137.4977586},
{{8, 4, 4, {171.5,-44,1,500000,500000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8521796.21510011, -5108460.96074551, 9521796.2151001, 16768810.0348996},
{{8, 4, 4, {175.5,-39,1,300000,400000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8721796.2151001, -5815782.48470089, 9321796.21510011, 16061488.5109442},
{{8, 4, 7, {-15,0,0.9996,500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, -9998287.38388927, 8746230.6469039, 9998287.38388927},
{{8, 4, 7, {-3,0,0.9996,500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, -9998287.38388927, 8746230.6469039, 9998287.38388927},
{{8, 4, 7, {-33,0,0.9996,500000,10000000,0}, 0,-206,172,-6, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, 1712.61611072717, 8746230.6469039, 19998287.3838893},
{{8, 4, 7, {-39,0,0.9996,500000,10000000,0}, 0,-206,172,-6, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, 1712.61611072717, 8746230.6469039, 19998287.3838893},
{{8, 4, 7, {-45,0,0.9996,500000,10000000,0}, 0,-206,172,-6, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, 1712.61611072717, 8746230.6469039, 19998287.3838893},
{{8, 4, 7, {-51,0,0.9996,500000,0,0}, 0,-206,172,-6, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, -9998287.38388927, 8746230.6469039, 9998287.38388927},
{{8, 4, 7, {-51,0,0.9996,500000,10000000,0}, 0,-206,172,-6, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, 1712.61611072717, 8746230.6469039, 19998287.3838893},
{{8, 4, 7, {-57,0,0.9996,500000,0,0}, 0,-206,172,-6, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, -9998287.38388927, 8746230.6469039, 9998287.38388927},
{{8, 4, 7, {-57,0,0.9996,500000,10000000,0}, 0,-206,172,-6, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, 1712.61611072717, 8746230.6469039, 19998287.3838893},
{{8, 4, 7, {-63,0,0.9996,500000,0,0}, 0,-206,172,-6, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, -9998287.38388927, 8746230.6469039, 9998287.38388927},
{{8, 4, 7, {-63,0,0.9996,500000,10000000,0}, 0,-206,172,-6, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, 1712.61611072717, 8746230.6469039, 19998287.3838893},
{{8, 4, 7, {-69,0,0.9996,500000,0,0}, 0,-206,172,-6, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, -9998287.38388927, 8746230.6469039, 9998287.38388927},
{{8, 4, 7, {-69,0,0.9996,500000,10000000,0}, 0,-206,172,-6, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, 1712.61611072717, 8746230.6469039, 19998287.3838893},
{{8, 4, 7, {-75,0,0.9996,500000,10000000,0}, 0,-206,172,-6, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, 1712.61611072717, 8746230.6469039, 19998287.3838893},
{{8, 4, 7, {-8.1319061111,39.6666666667,1,0,0,0}, 0,-303,-62,105, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8249530.45908754, -14394881.7325041, 8249530.45908754, 5609694.86591381},
{{8, 4, 7, {-8.1319061111,39.6666666667,1,180.598,-86.99,0}, 0,-223,110,37, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8249349.86108754, -14394968.7225041, 8249711.05708754, 5609607.87591381},
{{8, 4, 7, {-8.1319061111,39.6666666667,1,200000,300000,0}, 0,-303,-62,105, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8049530.45908754, -14094881.7325041, 8449530.45908754, 5909694.86591381},
{{8, 4, 7, {-9,0,0.9996,500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, -9998287.38388927, 8746230.6469039, 9998287.38388927},
{{8, 4, 7, {114.178555,22.3121333,0.99995,836694.05,819069.8,0}, 0,-156,-271,-189, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7412423.93256458, -11650990.6841779, 9085812.03256459, 8352585.68541009},
{{8, 4, 7, {114.178555,22.3121333,1,836694.05,819069.8,0}, 0,-156,-271,-189, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7412836.40908754, -11651614.2183788, 9086224.50908754, 8352962.38003909},
{{8, 4, 7, {15,0,0.9996,2520000,0,0}, 0,-225,-65,9, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -5726230.6469039, -9998287.38388927, 10766230.6469039, 9998287.38388927},
{{8, 4, 7, {15,0,0.9996,500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, -9998287.38388927, 8746230.6469039, 9998287.38388927},
{{8, 4, 7, {167.738861778,-45.563726167,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908753, -4254598.13766705, 8549530.45908754, 15749978.4607509},
{{8, 4, 7, {168.342872,-46.600009611,1,300002.66,699999.58,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949527.79908754, -4139408.83270982, 8549533.11908754, 15865167.7657081},
{{8, 4, 7, {168.398641194,-45.132902583,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908753, -4302480.81326099, 8549530.45908754, 15702095.7851569},
{{8, 4, 7, {168.606267,-43.977802889,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908753, -4430843.14592791, 8549530.45908754, 15573733.45249},
{{8, 4, 7, {169.467755083,-44.735267972,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -4346671.56116845, 8549530.45908754, 15657905.0372495},
{{8, 4, 7, {170.260925833,-43.110128139,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -4527247.70696346, 8549530.45908754, 15477328.8914545},
{{8, 4, 7, {170.282589111,-45.861513361,0.99996,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949200.47786918, -4221302.42846714, 8549200.47786917, 15782473.9868868},
{{8, 4, 7, {170.628595167,-45.816196611,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -4226536.32741985, 8549530.45908754, 15778040.2709981},
{{8, 4, 7, {170.9799935,-42.886322361,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -4552111.65805664, 8549530.45908754, 15452464.9403613},
{{8, 4, 7, {171.057250833,-44.402220361,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -4383682.10598832, 8549530.45908753, 15620894.4924296},
{{8, 4, 7, {171.360748472,-43.748711556,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -4456298.18308237, 8549530.45908754, 15548278.4153355},
{{8, 4, 7, {171.549771306,-42.333694278,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -4613502.29276327, 8549530.45908753, 15391074.3056546},
{{8, 4, 7, {171.581260056,-41.810802861,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908753, -4671584.03492437, 8549530.45908754, 15332992.5634935},
{{8, 4, 7, {172.109028194,-41.289911528,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -4729438.31462099, 8549530.45908753, 15275138.2837969},
{{8, 4, 7, {172.6720465,-40.714759056,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -4793313.13554601, 8549530.45908754, 15211263.4628719},
{{8, 4, 7, {172.727193583,-43.590637583,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -4473861.66485194, 8549530.45908753, 15530714.933566},
{{8, 4, 7, {173.010133389,-42.689116583,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -4574019.640636, 8549530.45908754, 15430556.9577819},
{{8, 4, 7, {173.299316806,-41.274544722,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908753, -4731144.99254062, 8549530.45908754, 15273431.6058773},
{{8, 4, 7, {173.802074111,-41.544486667,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -4701163.86142202, 8549530.45908754, 15303412.7369959},
{{8, 4, 7, {174.22801175,-39.135758306,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -4968639.63588687, 8549530.45908754, 15035936.962531},
{{8, 4, 7, {174.764339361,-36.879865278,0.9999,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7948705.50604163, -5218451.08592923, 8548705.50604163, 14784125.0548288},
{{8, 4, 7, {174.776623111,-41.301319639,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908753, -4728171.29681831, 8549530.45908754, 15276405.3015996},
{{8, 4, 7, {175.488099611,-40.241947139,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908753, -4845817.49876657, 8549530.45908754, 15158759.0996513},
{{8, 4, 7, {175.640036806,-39.512470389,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -4926815.26039816, 8549530.45908754, 15077761.3380198},
{{8, 4, 7, {175.647349667,-40.925532639,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -4769905.95772054, 8549530.45908754, 15234670.6406974},
{{8, 4, 7, {176.46619725,-37.761249806,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908753, -5121221.11127107, 8549530.45908754, 14883355.4871468},
{{8, 4, 7, {176.673680528,-39.650929306,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908753, -4911442.20418344, 8549530.45908754, 15093134.3942345},
{{8, 4, 7, {177.885636278,-38.624702778,1,300000,700000,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7949530.45908754, -5025375.11246985, 8549530.45908753, 14979201.4859481},
{{8, 4, 7, {21,0,0.9996,500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, -9998287.38388927, 8746230.6469039, 9998287.38388927},
{{8, 4, 7, {21,0,1,1500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -6749530.45908754, -10002288.299209, 9749530.45908754, 10002288.299209},
{{8, 4, 7, {24,0,1,2500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -5749530.45908754, -10002288.299209, 10749530.4590875, 10002288.299209},
{{8, 4, 7, {27,0,0.9996,500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, -9998287.38388927, 8746230.6469039, 9998287.38388927},
{{8, 4, 7, {27,0,1,3500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -4749530.45908754, -10002288.299209, 11749530.4590875, 10002288.299209},
{{8, 4, 7, {3,0,0.9996,500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, -9998287.38388927, 8746230.6469039, 9998287.38388927},
{{8, 4, 7, {30,0,1,4500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -3749530.45908754, -10002288.299209, 12749530.4590875, 10002288.299209},
{{8, 4, 7, {33,0,0.9996,500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, -9998287.38388927, 8746230.6469039, 9998287.38388927},
{{8, 4, 7, {39,0,0.9996,500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, -9998287.38388927, 8746230.6469039, 9998287.38388927},
{{8, 4, 7, {45,0,0.9996,500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, -9998287.38388927, 8746230.6469039, 9998287.38388927},
{{8, 4, 7, {9,0,0.9996,1500000,0,0}, 0,-225,-65,9, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -6746230.6469039, -9998287.38388927, 9746230.6469039, 9998287.38388927},
{{8, 4, 7, {9,0,0.9996,500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746230.6469039, -9998287.38388927, 8746230.6469039, 9998287.38388927},

// Added bounds for PSAD56 (AJD, Encom 2005)
{{8, 4, 7, {-75,0,0.9996,500000,0,0}, 0,-288, 175, -376, {0,0,0,0,0}, 0,0,0,0,0,0,0,0}, -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393}, // PSAD56 18N (Encom 2005)
{{8, 4, 7, {-69,0,0.9996,500000,0,0}, 0,-288, 175, -376, {0,0,0,0,0}, 0,0,0,0,0,0,0,0}, -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393}, // PASD56 19N (Encom 2005)
{{8, 4, 7, {-63,0,0.9996,500000,0,0}, 0,-288, 175, -376, {0,0,0,0,0}, 0,0,0,0,0,0,0,0}, -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393}, // PSAD56 20N (Encom 2005)
{{8, 4, 7, {-57,0,0.9996,500000,0,0}, 0,-288, 175, -376, {0,0,0,0,0}, 0,0,0,0,0,0,0,0}, -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393}, // PSAD56 21N (Encom 2005)
{{8, 4, 7, {-81,0,0.9996,500000,10000000,0}, 0,-288, 175, -376, {0,0,0,0,0}, 0,0,0,0,0,0,0,0}, -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393}, // PSAD56 17S (Encom 2005)
{{8, 4, 7, {-75,0,0.9996,500000,10000000,0}, 0,-288, 175, -376, {0,0,0,0,0}, 0,0,0,0,0,0,0,0}, -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393}, // PASD56 18S (Encom 2005)
{{8, 4, 7, {-69,0,0.9996,500000,10000000,0}, 0,-288, 175, -376, {0,0,0,0,0}, 0,0,0,0,0,0,0,0}, -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393}, // PSAD56 19S (Encom 2005)
{{8, 4, 7, {-63,0,0.9996,500000,10000000,0}, 0,-288, 175, -376, {0,0,0,0,0}, 0,0,0,0,0,0,0,0}, -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393}, // PSAD56 20S (Encom 2005)
{{8, 4, 7, {-80.5,0,0.99983008,222000,1426834.743,0}, 0,-288, 175, -376, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393}, // PERU WEST ZONE (Encom 2005)
{{8, 4, 7, {-76,0,0.99932994,720000,1039979.159,0}, 0,-288, 175, -376, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},    -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393}, // PERU CENTRAL ZONE (Encom 2005)
{{8, 4, 7, {-70.5,0,0.99952992,1324000,1040084.558,0}, 0,-288, 175, -376, {0,0,0,0,0}, 0,0,0,0,0,0,0,0}, -7745874.38491761, -9998000.59030393, 8745874.38491761, 9998000.59030393}, // PERU EAST ZONE (Encom 2005)

{{8, 6, 7, {17,0,1,0,0,0}, 0,-136,-108,-292, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8249527.70018454, -10001867.5518777, 8249527.70018454, 10001867.5518777},
{{8, 6, 7, {19,0,1,0,0,0}, 0,-136,-108,-292, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8249527.70018454, -10001867.5518777, 8249527.70018454, 10001867.5518777},
{{8, 6, 7, {19,0,1,0,3700000,0}, 0,-136,-108,-292, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8249527.70018454, -6301867.55187774, 8249527.70018454, 13701867.5518777},
{{8, 6, 7, {21,0,1,0,0,0}, 0,-136,-108,-292, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8249527.70018454, -10001867.5518777, 8249527.70018454, 10001867.5518777},
{{8, 6, 7, {23,0,1,0,0,0}, 0,-136,-108,-292, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8249527.70018454, -10001867.5518777, 8249527.70018454, 10001867.5518777},
{{8, 6, 7, {25,0,1,0,0,0}, 0,-136,-108,-292, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8249527.70018454, -10001867.5518777, 8249527.70018454, 10001867.5518777},
{{8, 6, 7, {27,0,1,0,0,0}, 0,-136,-108,-292, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8249527.70018454, -10001867.5518777, 8249527.70018454, 10001867.5518777},
{{8, 6, 7, {29,0,1,0,0,0}, 0,-136,-108,-292, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8249527.70018454, -10001867.5518777, 8249527.70018454, 10001867.5518777},
{{8, 6, 7, {31,0,1,0,0,0}, 0,-136,-108,-292, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -8249527.70018454, -10001867.5518777, 8249527.70018454, 10001867.5518777},
{{8, 7, 7, {-102,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-105,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578484, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-105,0,0.9996,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578484, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-105,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-108,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-111,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-111,0,0.9996,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-111,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-114,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-117,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-117,0,0.9996,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-117,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-120,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-123,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578484, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-123,0,0.9996,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578484, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-123,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-126,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-129,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-129,0,0.9996,500000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-129,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-132,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-135,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578484, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-135,0,0.9996,500000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578484, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-135,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-138,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463713, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-141,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578484, 9997887.28799036},
{{8, 7, 7, {-141,0,0.9996,500000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578484, 9997887.28799036},
{{8, 7, 7, {-141,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463713, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-147,0,0.9996,500000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578484, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-15,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-153,0,0.9996,500000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-159,0,0.9996,500000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-165,0,0.9996,500000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578484, 9997887.28799036},
{{8, 7, 7, {-171,0,0.9996,500000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-177,0,0.9996,500000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-21,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-27,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-33,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578484, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-39,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-45,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-51,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-53,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-55.5,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-56,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-57,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-58.5,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-61.5,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-63,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-64.5,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-67.5,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-69,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-69,0,0.9996,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-70.5,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-73.5,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-75,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-75,0,0.9996,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-76.5,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-79.5,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-81,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578484, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-81,0,0.9996,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578484, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-81,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-82.5,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463713, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-84,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-87,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-87,0,0.9996,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-87,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463713, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-9,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-90,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-93,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578484, 9997887.28799036},
{{8, 7, 7, {-93,0,0.9996,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578484, 9997887.28799036},
{{8, 7, 7, {-93,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-96,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 7, {-99,0,0.9996,500000,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-99,0,0.9996,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7746096.41578485, -9997887.28799036, 8746096.41578485, 9997887.28799036},
{{8, 7, 7, {-99,0,0.9999,304800,0,0}, 0,-10,158,187, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7943771.23463712, -10000887.8544033, 8553371.23463712, 10000887.8544033},
{{8, 7, 8, {-104.3333333333,31,0.9999090909,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562433.5032163, -44066372.5582056, 27562433.5032163, 21556716.5402837},
{{8, 7, 8, {-105.1666666667,40.6666666667,0.9999411765,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563301.8965776, -47586452.7340335, 27563301.8965776, 18038742.112075},
{{8, 7, 8, {-106.25,31,0.9999,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562187.458972, -44065971.9187976, 27562187.458972, 21556520.5525122},
{{8, 7, 8, {-107.3333333333,40.6666666667,0.9999411765,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563301.8965776, -47586452.7340335, 27563301.8965776, 18038742.112075},
{{8, 7, 8, {-107.8333333333,31,0.9999166667,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562638.5414399, -44066706.4265826, 27562638.5414399, 21556879.8645045},
{{8, 7, 8, {-108.75,40.6666666667,0.9999411765,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563301.8965776, -47586452.7340335, 27563301.8965776, 18038742.112075},
{{8, 7, 8, {-110.0833333333,40.6666666667,0.9999411765,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563301.8965776, -47586452.7340335, 27563301.8965776, 18038742.112075},
{{8, 7, 8, {-110.1666666667,31,0.9999,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562187.458972, -44065971.9187976, 27562187.458972, 21556520.5525122},
{{8, 7, 8, {-111.9166666667,31,0.9999,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562187.458972, -44065971.9187976, 27562187.458972, 21556520.5525122},
{{8, 7, 8, {-112.1666666667,41.6666666667,0.9999473684,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563469.4796944, -47951082.5341968, 27563469.4796945, 17674518.6804598},
{{8, 7, 8, {-113.75,31,0.9999333333,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563089.6212014, -44067440.9299605, 27563089.6212014, 21557239.1743409},
{{8, 7, 8, {-114,41.6666666667,0.9999473684,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563469.4796945, -47951082.5341968, 27563469.4796945, 17674518.6804598},
{{8, 7, 8, {-115.5833333333,34.75,0.9999,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562187.458972, -45430239.3046952, 27562187.458972, 20192253.1666146},
{{8, 7, 8, {-115.75,41.6666666667,0.9999333333,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563089.6212014, -47950409.5005354, 27563089.6212014, 17674270.6037659},
{{8, 7, 8, {-116.6666666667,34.75,0.9999,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562187.458972, -45430239.3046952, 27562187.458972, 20192253.1666146},
{{8, 7, 8, {-118.5833333333,34.75,0.9999,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562187.458972, -45430239.3046952, 27562187.458972, 20192253.1666146},
{{8, 7, 8, {-142,54,0.9999,500000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562187.458972, -52447364.8503891, 27562187.458972, 13175127.6209207},
{{8, 7, 8, {-146,54,0.9999,500000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562187.458972, -52447364.8503891, 27562187.458972, 13175127.6209207},
{{8, 7, 8, {-150,54,0.9999,500000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562187.458972, -52447364.8503891, 27562187.458972, 13175127.6209207},
{{8, 7, 8, {-154,54,0.9999,500000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562187.458972, -52447364.8503891, 27562187.458972, 13175127.6209207},
{{8, 7, 8, {-155.5,18.8333333333,0.9999666667,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563991.7861373, -39647492.5170134, 27563991.7861373, 25979375.2268425},
{{8, 7, 8, {-156.6666666667,20.3333333333,0.9999666667,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563991.7861374, -40192224.7269965, 27563991.7861373, 25434643.0168594},
{{8, 7, 8, {-158,21.1666666667,0.99999,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26564623.2994273, -40495839.074036, 27564623.2994273, 25132560.0122577},
{{8, 7, 8, {-158,54,0.9999,500000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562187.458972, -52447364.8503891, 27562187.458972, 13175127.6209207},
{{8, 7, 8, {-159.5,21.8333333333,0.99999,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26564623.2994273, -40738001.9175015, 27564623.2994273, 24890397.1687922},
{{8, 7, 8, {-160.1666666667,21.6666666667,1,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26564893.9483668, -40677866.1507286, 27564893.9483668, 24951189.2261189},
{{8, 7, 8, {-162,54,0.9999,700000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26362187.458972, -52447364.8503891, 27762187.458972, 13175127.6209207},
{{8, 7, 8, {-166,54,0.9999,500000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562187.458972, -52447364.8503891, 27562187.458972, 13175127.6209207},
{{8, 7, 8, {-170,54,0.9999,600000,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26462187.458972, -52447364.8503891, 27662187.458972, 13175127.6209207},
{{8, 7, 8, {-68.5,43.8333333333,0.9999,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562187.458972, -48738387.3899167, 27562187.458972, 16884105.0813931},
{{8, 7, 8, {-70.1666666667,42.8333333333,0.9999666667,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563991.7861373, -48377155.1233007, 27563991.7861373, 17249712.6205552},
{{8, 7, 8, {-71.5,41.0833333333,0.99999375,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26564724.7927796, -47740760.2162526, 27564724.7927796, 17887884.9789988},
{{8, 7, 8, {-71.6666666667,42.5,0.9999666667,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563991.7861373, -48255675.547233, 27563991.7861373, 17371192.1966229},
{{8, 7, 8, {-72.5,42.5,0.9999642857,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563927.3446249, -48255560.6466395, 27563927.3446248, 17371150.8344356},
{{8, 7, 8, {-74.3333333333,40,0.9999666667,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563991.7861373, -47344806.5947758, 27563991.7861373, 18282061.1490801},
{{8, 7, 8, {-74.6666666667,38.8333333333,0.999975,2000000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -25064217.3260181, -46920262.1007603, 29064217.3260181, 18707152.5497028},
{{8, 7, 8, {-75.4166666667,38,0.999995,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26564758.6238971, -46617719.2903186, 27564758.6238971, 19011007.941252},
{{8, 7, 8, {-76.5833333333,40,0.9999375,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563202.392495, -47343425.6569741, 27563202.392495, 18281527.9039123},
{{8, 7, 8, {-78.5833333333,40,0.9999375,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563202.392495, -47343425.6569741, 27563202.392495, 18281527.9039123},
{{8, 7, 8, {-81,24.3333333333,0.9999411765,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563301.8965776, -41644259.2047719, 27563301.8965776, 23980935.6413366},
{{8, 7, 8, {-82,24.3333333333,0.9999411765,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563301.8965776, -41644259.2047719, 27563301.8965776, 23980935.6413366},
{{8, 7, 8, {-82.1666666667,30,0.9999,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562187.458972, -43702304.8531186, 27562187.458972, 21920187.6181912},
{{8, 7, 8, {-84.1666666667,30,0.9999,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26562187.458972, -43702304.8531186, 27562187.458972, 21920187.6181912},
{{8, 7, 8, {-85.6666666667,37.5,0.9999666667,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563991.7861373, -46434335.8091192, 27563991.7861373, 19192531.9347367},
{{8, 7, 8, {-85.8333333333,30.5,0.99996,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563811.3526089, -43886764.6396605, 27563811.3526089, 21739665.5749719},
{{8, 7, 8, {-87.0833333333,37.5,0.9999666667,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563991.7861373, -46434335.8091192, 27563991.7861373, 19192531.9347367},
{{8, 7, 8, {-87.5,30,0.9999333333,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563089.6212014, -43703761.7408458, 27563089.6212014, 21920918.3634556},
{{8, 7, 8, {-88.3333333333,36.6666666667,0.999975,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26564217.3260181, -46131317.1820757, 27564217.3260181, 19496097.4683874},
{{8, 7, 8, {-88.8333333333,29.6666666667,0.999996,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26564785.688791, -43585279.1789159, 27564785.688791, 22043513.6817101},
{{8, 7, 8, {-90.1666666667,36.6666666667,0.9999411765,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563301.8965776, -46129756.82046, 27563301.8965776, 19495438.0256486},
{{8, 7, 8, {-90.3333333333,30.5,0.9999411765,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563301.8965776, -43885938.5041009, 27563301.8965776, 21739256.3420077},
{{8, 7, 8, {-90.5,35.8333333333,0.9999333333,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563089.6212014, -45826045.0626944, 27563089.6212014, 19798635.041607},
{{8, 7, 8, {-92.5,35.8333333333,0.9999333333,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563089.6212014, -45826045.0626944, 27563089.6212014, 19798635.041607},
{{8, 7, 8, {-94.5,36.1666666667,0.9999411765,500000,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26563301.8965776, -45947740.2967367, 27563301.8965776, 19677454.5493719},
{{8, 9, 7, {-2,49,0.9996012717,400000,-100000,0}, 0,375,-111,431, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -7845061.1011034, -15524202.1641258, 8645061.1011034, 4470074.53373206},
{{8,  10,7, {15.808277777800001, 0.0, 1.0, 1500000, 0.0, 0.0}, 0, 419.3836, 99.3335, 591.3451, {-0.850389, -1.817277, 7.862238, -0.99496, 0}, 0,0,0,0,0,0,0,0}, -1e9, -1e9, 1e9, 1e9}, // Encom 2005 (AJD) - to Support Swedish
{{9, 2, 7, {132,0,-36,-18,0,0}, 0,-133,-48,148, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20488603.5475955, -35940818.6722945, 20488603.5475955, 5036388.42289653},
{{9, 2, 7, {132,0,-36,-18,0,0}, 0,-134,-48,149, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20488603.5475955, -35940818.6722945, 20488603.5475955, 5036388.42289653},
{{9, 28, 7, {23,-23,-18,-32,0,0}, 0,-134.73,-110.92,-292.66, {0,0,0,1,0}, 0,0,0,0,0,0,0,0},  -21500589.3169017, -34197433.4604502, 21500589.3169017, 7592182.07528184},
{{9, 29, 0, {110,10,25,40,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -11362.8133775181, -3609.79239961301, 11362.8133775181, 19115.8343554232},
{{9, 29, 0, {132.5,-10,-21.5,-33.5,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -12557.486987589, -21368.9438624525, 12557.486987589, 3746.0301127255},
{{9, 29, 0, {25,35,40,65,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -9044.42142037436, -4777.87654505631, 9044.42142037436, 13310.9662956924},
{{9, 29, 0, {47.5,25,15,35,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -13390.3137726925, -4887.09300250369, 13390.3137726925, 21675.1568723853},
{{9, 29, 0, {78,23,22,33,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -12554.0796051134, -4624.86888238625, 12554.0796051134, 20483.2903278405},
{{9, 29, 0, {95,40,20,60,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -10369.4020979755, -5634.63503836129, 10369.4020979755, 15104.1691575898},
{{9, 6, 7, {23,-23,-18,-32,0,0}, 0,-136,-108,-292, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -21500683.2503467, -34197901.8785539, 21500683.2503467, 7591931.73862156},
{{9, 7, 7, {-154,50,55,65,0,0}, 0,-5,135,172, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -13752073.5330064, -8947886.69972899, 13752073.5330064, 18556260.3662838},
{{9, 7, 7, {-157,3,8,18,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -24775369.1253827, -6004013.76934506, 34739284.1330387, 43890186.5919339},
{{9, 7, 7, {-96,23,20,60,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -16687870.3794973, -7101873.97827787, 16687870.3794973, 26273866.7807168},
{{9, 7, 7, {-96,23,29.5,45.5,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -16900972.6938504, -6971893.13585582, 16900972.6938504, 26830052.251845},
{{11, 7, 7, {0,0,0,0,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -40075452.7386388, -14691640.6260036, 40075452.7386388, 14691640.6260036},
{{12, 7, 7, {0,0,0,0,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -34012036.7392828, -8625248.51472, 34012036.7392828, 8625248.51472},
{{13, 7, 7, {0,0,0,0,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -36080583.9779795, -9020145.99431898, 36080583.9779795, 9020145.99431898},
{{14, 29, 0, {-60,0,0,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -17523.9133905318, -5257.17401715949, 24533.4787467445, 5257.17401715949},
{{14, 7, 7, {0,0,0,0,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -33842774.0824052, -8460693.52060123, 33842774.0824052, 8460693.52060123},
{{15, 7, 7, {0,0,0,0,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -35347572.9807667, -8836893.24519168, 35347572.9807667, 8836893.24519168},
{{16, 7, 7, {0,0,0,0,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -40075452.7386388, -10018863.1846597, 40075452.7386388, 10018863.1846597},
{{18, 4, 7, {173,-41,2510000,6023150,0,0}, 0,84,-22,209, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  1001587.21174105, 4254077.4935345, 4082370.4786608, 7690559.24125294},
{{20, 0, 7, {0,0,1,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -100000000, -100000000, 100000000, 100000000},
{{20, 10, 7, {5.387638889,52.156160556,0.9999079,155000,463000,0}, 0,593,26,478, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -99845000, -99537000, 100155000, 100463000},
{{20, 28, 7, {0,-90,0.994,2000000,2000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -98000000, -98000000, 102000000, 102000000},
{{20, 28, 7, {0,90,0.994,2000000,2000000,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -98000000, -98000000, 102000000, 102000000},
{{21, 4, 7, {9,0,0.9996,500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -688662.0096834, -1193504.42233962, 42993.625661512, 466993.370035908},
{{22, 4, 7, {9,0,0.9996,500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -2981314.87489887, -52667593.719744, 93265053.7016014, 912774.431510712},
{{23, 4, 7, {15,0,0.9996,500000,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -142680.459281219, -57483.8396628927, 56778.2298793441, 142611.243106357},
{{24, 4, 7, {21,0,1,1500000,0,0}, 0,-96.062, -82.428, -121.754, {-4.801, -0.345, 1.376, 1.496, 0}, 0,0,0,0,0,0,0,0},   1250000, 6500000, 1750000, 8000000}, // Encom 2005 (AJD) - to support Finnish KKJ Zone 1
{{24, 4, 7, {24,0,1,2500000,0,0}, 0,-96.062, -82.428, -121.754, {-4.801, -0.345, 1.376, 1.496, 0}, 0,0,0,0,0,0,0,0},   2250000, 6500000, 2750000, 8000000}, // Encom 2005 (AJD) - to support Finnish KKJ Zone 2
{{24, 4, 7, {27,0,1,3500000,0,0}, 0,-96.062, -82.428, -121.754, {-4.801, -0.345, 1.376, 1.496, 0}, 0,0,0,0,0,0,0,0},   2850000, 6500000, 3850000, 8000000}, // Encom 2005 (AJD) - to support Finnish KKJ Zone 3
{{24, 4, 7, {30,0,1,4500000,0,0}, 0,-96.062, -82.428, -121.754, {-4.801, -0.345, 1.376, 1.496, 0}, 0,0,0,0,0,0,0,0},   4250000, 6500000, 4750000, 8000000}, // Encom 2005 (AJD) - to support Finnish KKJ Zone 4
{{25, 10, 7, {7.4395833333,46.9524055555,0,0,0,0}, 0,660.077,13.551,369.344, {0.804816,0.577692,0.952236,5.66,0}, 0,0,0,0,0,0,0,0},  -100000000, -100000000, 100000000, 100000000},
{{25, 10, 7, {7.4395833333,46.9524055555,600000,200000,0,0}, 0,660.077,13.551,369.344, {0.804816,0.577692,0.952236,5.66,0}, 0,0,0,0,0,0,0,0},  -99400000, -99800000, 100600000, 100200000},
{{26, 29, 0, {-60,0,0,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20751.2174140408, -12383.4550682633, 29051.7043796571, 12383.4550682633},
{{26, 29, 0, {-85.5,0,0,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -18987.3639338473, -12383.4550682633, 30815.5578598506, 12383.4550682633},
{{26, 29, 0, {20,0,0,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -26284.8753911183, -12383.4550682633, 23518.0464025796, 12383.4550682633},
{{26, 4, 7, {0,60,0,0,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20089005.2889087, -9990138.39344294, 20089005.2889087, 9990138.39344294},
{{26, 4, 7, {0,70,0,0,0,0}, 0,-87,-98,-121, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -13747868.5914983, -6836730.23471024, 13747868.5914983, 6836730.23471024},
{{26, 7, 7, {0,0,0,0,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -40075452.7386388, -19928981.8895549, 40075452.7386388, 19928981.8895549},
{{27, 11, 7, {78,0,1200000,0,0,0}, 0,289,734,257, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -18834804.515364, -15340324.8240999, 21234804.515364, 15340324.8240999},
{{27, 24, 7, {-37,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-38,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-39,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-41,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-42,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-43,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-45,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-48,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-49,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-51,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-52,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-54,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-56,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-59,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-62,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-63,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-65,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 24, 7, {-70,0,0,0,0,0}, 0,-57,1,-41, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037580.5994203, -15342381.4311037, 20037580.5994203, 15342381.4311037},
{{27, 28, 7, {-100,40,0,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037508.3427892, -19771855.3330942, 20037508.3427892, 10912797.2721118},
{{27, 28, 7, {0,0,0,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -20037508.3427892, -15342326.302603, 20037508.3427892, 15342326.302603},
{{27, 28, 7, {78,0,1200000,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -18837508.3427892, -15342326.302603, 21237508.3427892, 15342326.302603},
{{28, 0, 7, {-100,-80,180,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -19889431.1404394, -20026376.3929389, 19889431.1404395, 20026376.3929385},
{{28, 0, 7, {-100,-80,90,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -10014588.6119777, -10018754.1713946, 10014588.6119777, 10018754.1713946},
{{28, 0, 7, {-100,40,180,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -19610690.9824974, -20026376.3936683, 19610690.9824974, 20026376.3936695},
{{28, 0, 7, {-100,40,90,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -10018734.8959557, -10018754.1713946, 10018734.8959557, 10018754.1713946},
{{28, 0, 7, {0,-60,180,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -19758589.084291, -20026376.393667, 19758589.084291, 20026376.3936708},
{{28, 0, 7, {0,-60,90,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -10018731.6857622, -10018754.1713946, 10018731.6857622, 10018754.1713946},
{{28, 0, 7, {0,80,180,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -19889431.1404395, -20026376.3929385, 19889431.1404395, 20026376.3929389},
{{28, 0, 7, {0,80,90,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -10014588.6119777, -10018754.1713946, 10014588.6119777, 10018754.1713946},
{{28, 7, 7, {0,-90,90,0,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -10018863.1846597, -10018863.1846597, 10018863.1846597, 10018863.1846597},
{{28, 7, 7, {0,90,90,0,0,0}, 0,-8,160,176, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  -10018863.1846597, -10018863.1846597, 10018863.1846597, 10018863.1846597},

{{0xff, 0, 0, {0,0,0,0,0,0}, 0,0,0,0, {0,0,0,0,0}, 0,0,0,0,0,0,0,0},  0, 0, 0, 0}
};


#define TAB_EQUAL(a, b, eps) (fabs((a)-(b)) < eps)

static char szPreviousMitabBoundsFile[2048] = { 0 };
static VSIStatBufL sStatBoundsFile;

/**********************************************************************
 *                     MITABLookupCoordSysBounds()
 *
 * Lookup bounds for specified TABProjInfo struct.
 *
 * This can modify that passed TABProjInfo struct if a match is found
 * in an external bound file with proj remapping.
 *
 * Returns TRUE if valid bounds were found, FALSE otherwise.
 **********************************************************************/
GBool MITABLookupCoordSysBounds(TABProjInfo *psCS,
                                double &dXMin, double &dYMin,
                                double &dXMax, double &dYMax,
                                int bOnlyUserTable)
{
    GBool bFound = FALSE;
    const MapInfoBoundsInfo *psList;

    /*-----------------------------------------------------------------
    * Try to load the user defined table if not loaded yet .
    *----------------------------------------------------------------*/
    const char * pszMitabBoundsFile = CPLGetConfigOption("MITAB_BOUNDS_FILE", NULL);
    if (pszMitabBoundsFile != NULL && pszMitabBoundsFile[0] != '\0' )
    {
        if( strcmp(pszMitabBoundsFile, szPreviousMitabBoundsFile) != 0)
        {
            CPLStrlcpy(szPreviousMitabBoundsFile, pszMitabBoundsFile,
                       sizeof(szPreviousMitabBoundsFile));
            MITABLoadCoordSysTable(pszMitabBoundsFile);
            if( VSIStatL(pszMitabBoundsFile, &sStatBoundsFile) != 0 )
            {
                sStatBoundsFile.st_mtime = 0;
            }
        }
        else
        {
            /* Reload file if its modification file has changed */
            VSIStatBufL sStat;
            if( VSIStatL(pszMitabBoundsFile, &sStat) == 0 )
            {
                if( sStat.st_mtime != sStatBoundsFile.st_mtime )
                {
                    MITABLoadCoordSysTable(pszMitabBoundsFile);
                    memcpy(&sStatBoundsFile, &sStat, sizeof(sStat));
                }
            }
        }
    }
    else if ( szPreviousMitabBoundsFile[0] != '\0' )
    {
        MITABFreeCoordSysTable();
        strcpy(szPreviousMitabBoundsFile, "");
    }

    for(int iLoop=0; !bFound && iLoop < 2; iLoop++)
    {
        /* MapInfo uses a hack to differentiate some SRS that have the same */
        /* definition, but different bounds, e.g. Lambet 93 France with French */
        /* Bounds or with European bounds. It alters slightly one of the projection */
        /* parameters, e.g. std_parallel_1 = 49.00000000001 or 49.00000000002 */
        double eps = (iLoop == 0) ? 1e-12 : 1e-6;

        /*-----------------------------------------------------------------
        * Lookup table...
        * Lookup external file if one was loaded, then lookup internal table.
        *
        * Note that entries in lookup table with 0xff for projId, UnitsId,
        * means ignore that param, and 0xff in ellipsoidId means ignore the
        * whole datum.
        *----------------------------------------------------------------*/
        for( int i = 0 ; !bFound && i < nExtBoundsListCount; i++)
        {
            TABProjInfo *p = &(gpasExtBoundsList[i].sProjIn);

            if (p->nProjId == psCS->nProjId &&
                (p->nUnitsId == 0xff || p->nUnitsId == psCS->nUnitsId) &&
                (p->nEllipsoidId == 0xff ||
                (p->nEllipsoidId == psCS->nEllipsoidId &&
                ( (p->nDatumId > 0 && p->nDatumId == psCS->nDatumId) ||
                    ((p->nDatumId <= 0 || psCS->nDatumId <= 0) &&
                    TAB_EQUAL(p->dDatumShiftX, psCS->dDatumShiftX, eps) &&
                    TAB_EQUAL(p->dDatumShiftY, psCS->dDatumShiftY, eps) &&
                    TAB_EQUAL(p->dDatumShiftZ, psCS->dDatumShiftZ, eps) &&
                    TAB_EQUAL(p->adDatumParams[0], psCS->adDatumParams[0], eps) &&
                    TAB_EQUAL(p->adDatumParams[1], psCS->adDatumParams[1], eps) &&
                    TAB_EQUAL(p->adDatumParams[2], psCS->adDatumParams[2], eps) &&
                    TAB_EQUAL(p->adDatumParams[3], psCS->adDatumParams[3], eps) &&
                    TAB_EQUAL(p->adDatumParams[4], psCS->adDatumParams[4], eps) )))) &&
                (TAB_EQUAL(p->adProjParams[0], psCS->adProjParams[0], eps) &&
                TAB_EQUAL(p->adProjParams[1], psCS->adProjParams[1], eps) &&
                TAB_EQUAL(p->adProjParams[2], psCS->adProjParams[2], eps) &&
                TAB_EQUAL(p->adProjParams[3], psCS->adProjParams[3], eps) &&
                TAB_EQUAL(p->adProjParams[4], psCS->adProjParams[4], eps) &&
                TAB_EQUAL(p->adProjParams[5], psCS->adProjParams[5], eps) )  )
            {
                memcpy(psCS, &gpasExtBoundsList[i].sBoundsInfo.sProj,
                       sizeof(TABProjInfo));
                dXMin = gpasExtBoundsList[i].sBoundsInfo.dXMin;
                dYMin = gpasExtBoundsList[i].sBoundsInfo.dYMin;
                dXMax = gpasExtBoundsList[i].sBoundsInfo.dXMax;
                dYMax = gpasExtBoundsList[i].sBoundsInfo.dYMax;
                bFound = TRUE;
            }
        }

        psList = gasBoundsList;
        for( ; !bOnlyUserTable && !bFound && psList->sProj.nProjId!=0xff; psList++)
        {
            const TABProjInfo *p = &(psList->sProj);

            if (p->nProjId == psCS->nProjId &&
                (p->nUnitsId == 0xff || p->nUnitsId == psCS->nUnitsId) &&
                (p->nEllipsoidId == 0xff ||
                (p->nEllipsoidId == psCS->nEllipsoidId &&
                ( (p->nDatumId > 0 && p->nDatumId == psCS->nDatumId) ||
                    ((p->nDatumId <= 0 || psCS->nDatumId <= 0) &&
                    TAB_EQUAL(p->dDatumShiftX, psCS->dDatumShiftX, eps) &&
                    TAB_EQUAL(p->dDatumShiftY, psCS->dDatumShiftY, eps) &&
                    TAB_EQUAL(p->dDatumShiftZ, psCS->dDatumShiftZ, eps) &&
                    TAB_EQUAL(p->adDatumParams[0], psCS->adDatumParams[0], eps) &&
                    TAB_EQUAL(p->adDatumParams[1], psCS->adDatumParams[1], eps) &&
                    TAB_EQUAL(p->adDatumParams[2], psCS->adDatumParams[2], eps) &&
                    TAB_EQUAL(p->adDatumParams[3], psCS->adDatumParams[3], eps) &&
                    TAB_EQUAL(p->adDatumParams[4], psCS->adDatumParams[4], eps) )))) &&
                (TAB_EQUAL(p->adProjParams[0], psCS->adProjParams[0], eps) &&
                TAB_EQUAL(p->adProjParams[1], psCS->adProjParams[1], eps) &&
                TAB_EQUAL(p->adProjParams[2], psCS->adProjParams[2], eps) &&
                TAB_EQUAL(p->adProjParams[3], psCS->adProjParams[3], eps) &&
                TAB_EQUAL(p->adProjParams[4], psCS->adProjParams[4], eps) &&
                TAB_EQUAL(p->adProjParams[5], psCS->adProjParams[5], eps) )  )
            {
                dXMin = psList->dXMin;
                dYMin = psList->dYMin;
                dXMax = psList->dXMax;
                dYMax = psList->dYMax;
                bFound = TRUE;
            }
        }
    }

    return bFound;
}


/**********************************************************************
 *                     MITABLoadCoordSysTable()
 *
 * Load a Coordsys bounds lookup table from an external file.
 * The entries from that table will be looked up in priority BEFORE the
 * entries from gasBoundsList[] defined above.
 *
 * This allows users to override the default bounds for existing
 * projections, and to define bounds for new projections not listed in
 * the table above.
 *
 * The format of the file is a simple text file with one CoordSys string
 * per line.  The CoordSys lines should follow the MIF specs, and MUST
 * include the optional Bounds definition at the end of the line.
 *
 * e.g.
 *  CoordSys Earth Projection 8, 24, "m", -63, 0, 0.9996, 500000, 0 Bounds \
 *  (-7746230.6469039, -9998287.383889269) (8746230.6469039, 9998287.383889269)
 *
 * Returns 0 on success, -1 on error.
 **********************************************************************/
int MITABLoadCoordSysTable(const char *pszFname)
{
    VSILFILE *fp;
    int nStatus = 0, iLine = 0;

    MITABFreeCoordSysTable();

    if ((fp = VSIFOpenL(pszFname, "rt")) != NULL)
    {
        const char *pszLine;
        int         iEntry=0, numEntries=100;

        gpasExtBoundsList = (MapInfoRemapProjInfo *)CPLMalloc(numEntries*
                                                  sizeof(MapInfoRemapProjInfo));

        while( (pszLine = CPLReadLineL(fp)) != NULL)
        {
            double dXMin, dYMin, dXMax, dYMax;
            int bHasProjIn = FALSE;
            TABProjInfo sProjIn;
            TABProjInfo sProj;

            iLine++;

            if (strlen(pszLine) < 10 || EQUALN(pszLine, "#", 1))
                continue;  // Skip empty lines/comments

            if( EQUALN(pszLine, "Source", strlen("Source")) )
            {
                const char* pszEqual = strchr(pszLine, '=');
                if( !pszEqual )
                {
                    CPLError(CE_Warning, CPLE_IllegalArg, "Invalid format at line %d", iLine);
                    break;
                }
                pszLine = pszEqual + 1;
                if ((nStatus = MITABCoordSys2TABProjInfo(pszLine, &sProjIn)) != 0)
                {
                    break;  // Abort and return
                }
                if( strstr(pszLine, "Bounds") != NULL )
                {
                    CPLError(CE_Warning, CPLE_IllegalArg, "Unexpected Bounds parameter at line %d",
                             iLine);
                }
                bHasProjIn = TRUE;

                iLine++;
                pszLine = CPLReadLineL(fp);
                if( pszLine == NULL ||
                    !EQUALN(pszLine, "Destination", strlen("Destination")) ||
                    (pszEqual = strchr(pszLine, '=')) == NULL )
                {
                    CPLError(CE_Warning, CPLE_IllegalArg, "Invalid format at line %d", iLine);
                        break;
                }
                pszLine = pszEqual + 1;
            }

            if ((nStatus = MITABCoordSys2TABProjInfo(pszLine, &sProj)) != 0)
            {
                break;  // Abort and return
            }

            if (!MITABExtractCoordSysBounds(pszLine, dXMin,dYMin,dXMax,dYMax))
            {
                CPLError(CE_Warning, CPLE_IllegalArg,
                         "Missing Bounds parameters in line %d of %s",
                         iLine, pszFname);
                continue;  // Just skip this line.
            }

            if (iEntry >= numEntries-1)
            {
                numEntries+= 100;
                gpasExtBoundsList =
                    (MapInfoRemapProjInfo *)CPLRealloc(gpasExtBoundsList,
                                        numEntries* sizeof(MapInfoRemapProjInfo));
            }

            gpasExtBoundsList[iEntry].sProjIn = (bHasProjIn) ? sProjIn : sProj;
            gpasExtBoundsList[iEntry].sBoundsInfo.sProj = sProj;
            gpasExtBoundsList[iEntry].sBoundsInfo.dXMin = dXMin;
            gpasExtBoundsList[iEntry].sBoundsInfo.dYMin = dYMin;
            gpasExtBoundsList[iEntry].sBoundsInfo.dXMax = dXMax;
            gpasExtBoundsList[iEntry].sBoundsInfo.dYMax = dYMax;
            iEntry ++;
        }
        nExtBoundsListCount = iEntry;

        VSIFCloseL(fp);
    }
    else
    {
        CPLError(CE_Failure, CPLE_FileIO, "Cannot open %s", pszFname);
    }

    return nStatus;
}


/**********************************************************************
 *                     MITABFreeCoordSysTable()
 *
 * Flush memory used by the coordsys table loaded by MITABLoadCoordSysTable()
 **********************************************************************/
void MITABFreeCoordSysTable()
{
    CPLFree(gpasExtBoundsList);
    gpasExtBoundsList = NULL;
    nExtBoundsListCount = -1;
}

/**********************************************************************
 *                     MITABCoordSysTableLoaded()
 *
 * Returns TRUE if a coordsys table was loaded, FALSE otherwise.
 **********************************************************************/
GBool MITABCoordSysTableLoaded()
{
    return (nExtBoundsListCount >= 0);
}
