static char rcsid[] = "$Id$";
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifndef HAVE_MEMCPY
#define memcpy(d,s,n) bcopy((s),(d),(n))
#endif

#include "kmer-search.h"

#include <string.h>		/* For strlen */
#include "assert.h"
#include "mem.h"
#include "types.h"
#include "chrnum.h"
#include "reader.h"
#include "oligo.h"

#ifdef LARGE_GENOMES
#include "intersect-large.h"
#include "intersect-approx-uint8.h"
#include "intersect-indices-large.h"
#else
#include "intersect-small.h"
#include "intersect-approx-uint4.h"
#include "intersect-indices-small.h"
#endif

#include "transcript.h"
#include "popcount.h"
#include "junction.h"

#include "univdiag.h"
#include "univdiagdef.h"
#include "genomebits_count.h"
#include "genomebits_kmer.h"
#include "splice.h"
#include "indel.h"
#include "intron.h"
#include "maxent_hr.h"
#include "sedgesort.h"
#include "path-solve.h"


#ifndef LARGE_GENOMES
#include "merge-diagonals-simd-uint4.h"
#elif !defined(HAVE_AVX512) && !defined(HAVE_AVX2)
#include "merge-diagonals-heap.h" /* For Merge_diagonals_large */
#include "merge-diagonals-simd-uint4.h"
#else
#include "merge-diagonals-simd-uint8.h" /* For Merge_diagonals_large */
#include "merge-diagonals-simd-uint4.h"
#endif

#include "simd.h"

#if 0
#if defined(HAVE_SSE2)
#include <emmintrin.h>
#endif
#ifdef HAVE_SSE4_1
#include <smmintrin.h>
#endif
#ifdef HAVE_AVX2
#include <immintrin.h>
#endif
#ifdef HAVE_AVX512
#include <immintrin.h>
#endif
#endif


#ifdef LARGE_GENOMES
#define GETPOS(high,low) (((UINT8) high << 32) + low)
#endif

#define TRIM_AT_GENOME_BOUNDS 1

/* Causes problems with counting nmismatches */
/* #define TRIM_AT_CHROMOSOME_BOUNDS 1 */

#define MIN_SUPPORT_INDEL 6
#define NEED_CHRNUM 1


/* General flow */
#ifdef DEBUG
#define debug(x) x
#else
#define debug(x)
#endif

/* most_prevalent_univcoord */
#ifdef DEBUG0
#define debug0(x) x
#else
#define debug0(x)
#endif

/* Kmer_anypair_univdiagonals */
#ifdef DEBUG2
#define debug2(x) x
#else
#define debug2(x)
#endif

/* Kmer_search_complete */
#ifdef DEBUG9
#define debug9(x) x
#else
#define debug9(x)
#endif



/* Merging faster than count table */
#define USE_MERGE 1

#define MAX_NEIGHBORS 3		/* Cannot be 0 */
#define SUBOPT 3

#define LONG_END 6
#define ALLOWED_END_MISMATCHES 2 /* For long ends */
#define ALLOWED_TRANSCRIPTOME_TRIM 3


static int index1part;
static int index1interval;
static Indexdb_T indexdb;

static EF64_T chromosome_ef64;
static Genomebits_T genomebits;
static Genomebits_T genomebits_alt;
static Univcoord_T genomelength;

static bool splicingp;


/* Indexdb: oligo + diagterm -> streams of diagonals */
/* Merge: an array of diagonals with duplicates */
/* Path: genomic endpoints + gaps + trnums */

/* All calls to Substring_new are for transcriptome.  May need to make call to Univ_IIT_update_chrnum */

/* Genome */
/* Ultrafast: check ends only */
/* find_local_sets: merged->diagonals -> middle_diagonal, left_diagonals, right_diagonals */
/* Algorithm 1a: find_best_path_genome -> complete_path (list of Univdiag_T) */
/* Algorithm 1b: Kmer_search_genome (solve_via_segments_genome): complete_path -> hits */
/* Algorithm 2: Stage3end_complete_path_run_gmap: complete_path -> hits */


int
Kmer_exact1 (Univcoord_T **univdiagonals_gplus, int *nunivdiagonals_gplus,
	     Univcoord_T **univdiagonals_gminus, int *nunivdiagonals_gminus,
	     Stage1_T stage1, int querylength) {

  Univcoord_T **mod_univdiagonals_gplus, **mod_univdiagonals_gminus;
  int *mod_nunivdiagonals_gplus, *mod_nunivdiagonals_gminus;
  int query_lastpos = querylength - index1part;
  int mod5, mod3, adj;
  int querypos5, querypos3;

  debug(printf("Entered Kmer_exact1\n"));

  mod_univdiagonals_gplus = (Univcoord_T **) MALLOC(index1interval*sizeof(Univcoord_T *));
  mod_nunivdiagonals_gplus = (int *) MALLOC((index1interval + 1)*sizeof(int)); /* Add 1 needed for Merge_diagonals */
  mod_univdiagonals_gminus = (Univcoord_T **) MALLOC(index1interval*sizeof(Univcoord_T *));
  mod_nunivdiagonals_gminus = (int *) MALLOC((index1interval + 1)*sizeof(int)); /* Add 1 needed for Merge_diagonals */

  for (mod5 = 0; mod5 < index1interval; mod5++) {
    querypos5 = mod5;
    adj = (querylength - index1part) % index1interval;
    mod3 = (index1interval + adj - mod5) % index1interval;
    querypos3 = query_lastpos - mod3;
    /* printf("mod5 %d => querypos5 %d => mod3 %d => querypos3 %d\n",mod5,querypos5,mod3,querypos3); */

    /* gplus */
#ifdef LARGE_GENOMES
    mod_univdiagonals_gplus[mod5] =
      Intersect_large(&(mod_nunivdiagonals_gplus[mod5]),
		      stage1->plus_positions_high[querypos5],stage1->plus_positions[querypos5],
		      stage1->plus_npositions[querypos5],/*diagterm1*/querylength - querypos5,
		      stage1->plus_positions_high[querypos3],stage1->plus_positions[querypos3],
		      stage1->plus_npositions[querypos3],/*diagterm2*/querylength - querypos3);
#else
    mod_univdiagonals_gplus[mod5] =
      Intersect_small(&(mod_nunivdiagonals_gplus[mod5]),
		      stage1->plus_positions[querypos5],stage1->plus_npositions[querypos5],
		      /*diagterm1*/querylength - querypos5,
		      stage1->plus_positions[querypos3],stage1->plus_npositions[querypos3],
		      /*diagterm2*/querylength - querypos3);
#endif

    debug(printf("gplus mod5 %d, mod3 %d: diagterm5 %d, diagterm3 %d, %d univdiagonals\n",
		 mod5,mod3,querylength - querypos5,querylength - querypos3,
		 mod_nunivdiagonals_gplus[mod5]));

    /* gminus */
#ifdef LARGE_GENOMES
    mod_univdiagonals_gminus[mod3] =
      Intersect_large(&(mod_nunivdiagonals_gminus[mod3]),
		      stage1->minus_positions_high[querypos5],stage1->minus_positions[querypos5],
		      stage1->minus_npositions[querypos5],/*diagterm1*/querypos5 + index1part,
		      stage1->minus_positions_high[querypos3],stage1->minus_positions[querypos3],
		      stage1->minus_npositions[querypos3],/*diagterm2*/querypos3 + index1part);
#else
    mod_univdiagonals_gminus[mod3] =
      Intersect_small(&(mod_nunivdiagonals_gminus[mod3]),
		      stage1->minus_positions[querypos5],stage1->minus_npositions[querypos5],
		      /*diagterm1*/querypos5 + index1part,
		      stage1->minus_positions[querypos3],stage1->minus_npositions[querypos3],
		      /*diagterm2*/querypos3 + index1part);
#endif
    
    debug(printf("gminus mod5 %d, mod3 %d: diagterm5 %d, diagterm3 %d, %d univdiagonals\n",
		 mod5,mod3,querypos5 + index1part,querypos3 + index1part,
		 mod_nunivdiagonals_gminus[mod3]));
  }
    
#ifdef LARGE_GENOMES  
  *univdiagonals_gplus = Merge_diagonals_uint8(&(*nunivdiagonals_gplus),
					       mod_univdiagonals_gplus,mod_nunivdiagonals_gplus,
					       /*nstreams*/index1interval,stage1->mergeinfo);
  *univdiagonals_gminus = Merge_diagonals_uint8(&(*nunivdiagonals_gminus),
						mod_univdiagonals_gminus,mod_nunivdiagonals_gminus,
						/*nstreams*/index1interval,stage1->mergeinfo);
#else
  *univdiagonals_gplus = Merge_diagonals_uint4(&(*nunivdiagonals_gplus),
					       mod_univdiagonals_gplus,mod_nunivdiagonals_gplus,
					       /*nstreams*/index1interval,stage1->mergeinfo);
  *univdiagonals_gminus = Merge_diagonals_uint4(&(*nunivdiagonals_gminus),
						mod_univdiagonals_gminus,mod_nunivdiagonals_gminus,
						/*nstreams*/index1interval,stage1->mergeinfo);
#endif


  for (mod3 = 0; mod3 < index1interval; mod3++) {
    FREE(mod_univdiagonals_gminus[mod3]);
  }
  FREE(mod_univdiagonals_gminus);
  FREE(mod_nunivdiagonals_gminus);

  for (mod5 = 0; mod5 < index1interval; mod5++) {
    FREE(mod_univdiagonals_gplus[mod5]);
  }
  FREE(mod_univdiagonals_gplus);
  FREE(mod_nunivdiagonals_gplus);

#ifdef DEBUG
  printf("Kmer_exact1 returning %d plus and %d minus hits\n",
	 (*nunivdiagonals_gplus),(*nunivdiagonals_gminus));

  for (int i = 0; i < *nunivdiagonals_gplus; i++) {
    printf("plus %u\n",(*univdiagonals_gplus)[i]);
  }
  for (int i = 0; i < *nunivdiagonals_gminus; i++) {
    printf("minus %u\n",(*univdiagonals_gminus)[i]);
  }
#endif

  return (*nunivdiagonals_gplus) + (*nunivdiagonals_gminus);
}


static int
keep_duplicates (Univcoord_T *diagonals, int ndiagonals) {
  int k = 0, i, j;

  i = 0;
  while (i < ndiagonals) {
    j = i + 1;
    while (j < ndiagonals && diagonals[j] == diagonals[i]) {
      j++;
    }
    if (j - i > 1) {
      diagonals[k++] = diagonals[i];
    }
    
    i = j;
  }

  return k;
}


int
Kmer_exact2 (Univcoord_T **univdiagonals_gplus, int *nunivdiagonals_gplus,
	     Univcoord_T **univdiagonals_gminus, int *nunivdiagonals_gminus,
	     Stage1_T stage1, int querylength) {

  int query_lastpos = querylength - index1part;
  int ndiagonals;
  int nstreams;
  int mod5, mod3;
  int querypos;


  debug(printf("Entered Kmer_exact2\n"));
  /* Stage1_dump(stage1,querylength); */

  nstreams = 0;
  for (mod5 = 0; mod5 < index1interval; mod5++) {
    querypos = mod5;
#ifdef LARGE_GENOMES
    stage1->gplus_stream_high_array_5[nstreams] = stage1->plus_positions_high[querypos];
    stage1->gplus_stream_low_array_5[nstreams] = stage1->plus_positions[querypos];
#else
    stage1->gplus_stream_array_5[nstreams] = stage1->plus_positions[querypos];
#endif
    stage1->gplus_streamsize_array_5[nstreams] = stage1->plus_npositions[querypos];
    stage1->gplus_diagterm_array_5[nstreams++] = stage1->plus_diagterms[querypos];
    if (querypos + index1part <= query_lastpos) {
#ifdef LARGE_GENOMES
      stage1->gplus_stream_high_array_5[nstreams] = stage1->plus_positions_high[querypos + index1part];
      stage1->gplus_stream_low_array_5[nstreams] = stage1->plus_positions[querypos + index1part];
#else
      stage1->gplus_stream_array_5[nstreams] = stage1->plus_positions[querypos + index1part];
#endif
      stage1->gplus_streamsize_array_5[nstreams] = stage1->plus_npositions[querypos + index1part];
      stage1->gplus_diagterm_array_5[nstreams++] = stage1->plus_diagterms[querypos + index1part];
    }
  }

  for (mod3 = 0; mod3 < index1interval; mod3++) {
    querypos = query_lastpos - mod3;
#ifdef LARGE_GENOMES
    stage1->gplus_stream_high_array_5[nstreams] = stage1->plus_positions_high[querypos];
    stage1->gplus_stream_low_array_5[nstreams] = stage1->plus_positions[querypos];
#else
    stage1->gplus_stream_array_5[nstreams] = stage1->plus_positions[querypos];
#endif
    stage1->gplus_streamsize_array_5[nstreams] = stage1->plus_npositions[querypos];
    stage1->gplus_diagterm_array_5[nstreams++] = stage1->plus_diagterms[querypos];
    if (querypos - index1part >= 0) {
#ifdef LARGE_GENOMES
      stage1->gplus_stream_high_array_5[nstreams] = stage1->plus_positions_high[querypos - index1part];
      stage1->gplus_stream_low_array_5[nstreams] = stage1->plus_positions[querypos - index1part];
#else
      stage1->gplus_stream_array_5[nstreams] = stage1->plus_positions[querypos - index1part];
#endif
      stage1->gplus_streamsize_array_5[nstreams] = stage1->plus_npositions[querypos - index1part];
      stage1->gplus_diagterm_array_5[nstreams++] = stage1->plus_diagterms[querypos - index1part];
    }
  }
    

#ifdef LARGE_GENOMES
  *univdiagonals_gplus = Merge_diagonals_large(&ndiagonals,
					       stage1->gplus_stream_high_array_5,
					       stage1->gplus_stream_low_array_5,
					       stage1->gplus_streamsize_array_5,
					       stage1->gplus_diagterm_array_5,
					       nstreams,stage1->mergeinfo);
#else
  *univdiagonals_gplus = Merge_diagonals(&ndiagonals,
					 stage1->gplus_stream_array_5,
					 stage1->gplus_streamsize_array_5,
					 stage1->gplus_diagterm_array_5,
					 nstreams,stage1->mergeinfo);
#endif

  *nunivdiagonals_gplus = keep_duplicates(*univdiagonals_gplus,ndiagonals);


  nstreams = 0;
  for (mod5 = 0; mod5 < index1interval; mod5++) {
    querypos = mod5;
#ifdef LARGE_GENOMES
    stage1->gminus_stream_high_array_5[nstreams] = stage1->minus_positions_high[querypos];
    stage1->gminus_stream_low_array_5[nstreams] = stage1->minus_positions[querypos];
#else
    stage1->gminus_stream_array_5[nstreams] = stage1->minus_positions[querypos];
#endif
    stage1->gminus_streamsize_array_5[nstreams] = stage1->minus_npositions[querypos];
    stage1->gminus_diagterm_array_5[nstreams++] = stage1->minus_diagterms[querypos];
    if (querypos + index1part <= query_lastpos) {
#ifdef LARGE_GENOMES
      stage1->gminus_stream_high_array_5[nstreams] = stage1->minus_positions_high[querypos + index1part];
      stage1->gminus_stream_low_array_5[nstreams] = stage1->minus_positions[querypos + index1part];
#else
      stage1->gminus_stream_array_5[nstreams] = stage1->minus_positions[querypos + index1part];
#endif
      stage1->gminus_streamsize_array_5[nstreams] = stage1->minus_npositions[querypos + index1part];
      stage1->gminus_diagterm_array_5[nstreams++] = stage1->minus_diagterms[querypos + index1part];
    }
  }

  for (mod3 = 0; mod3 < index1interval; mod3++) {
    querypos = query_lastpos - mod3;
#ifdef LARGE_GENOMES
    stage1->gminus_stream_high_array_5[nstreams] = stage1->minus_positions_high[querypos];
    stage1->gminus_stream_low_array_5[nstreams] = stage1->minus_positions[querypos];
#else
    stage1->gminus_stream_array_5[nstreams] = stage1->minus_positions[querypos];
#endif
    stage1->gminus_streamsize_array_5[nstreams] = stage1->minus_npositions[querypos];
    stage1->gminus_diagterm_array_5[nstreams++] = stage1->minus_diagterms[querypos];
    if (querypos - index1part >= 0) {
#ifdef LARGE_GENOMES
      stage1->gminus_stream_high_array_5[nstreams] = stage1->minus_positions_high[querypos - index1part];
      stage1->gminus_stream_low_array_5[nstreams] = stage1->minus_positions[querypos - index1part];
#else
      stage1->gminus_stream_array_5[nstreams] = stage1->minus_positions[querypos - index1part];
#endif
      stage1->gminus_streamsize_array_5[nstreams] = stage1->minus_npositions[querypos - index1part];
      stage1->gminus_diagterm_array_5[nstreams++] = stage1->minus_diagterms[querypos - index1part];
    }
  }

#ifdef LARGE_GENOMES
  *univdiagonals_gminus = Merge_diagonals_large(&ndiagonals,
						stage1->gminus_stream_high_array_5,
						stage1->gminus_stream_low_array_5,
						stage1->gminus_streamsize_array_5,
						stage1->gminus_diagterm_array_5,
						nstreams,stage1->mergeinfo);
#else
  *univdiagonals_gminus = Merge_diagonals(&ndiagonals,
					  stage1->gminus_stream_array_5,
					  stage1->gminus_streamsize_array_5,
					  stage1->gminus_diagterm_array_5,
					  nstreams,stage1->mergeinfo);
#endif

  *nunivdiagonals_gminus = keep_duplicates(*univdiagonals_gminus,ndiagonals);


#ifdef DEBUG
  printf("Kmer_exact2 returning %d plus and %d minus hits\n",
	 (*nunivdiagonals_gplus),(*nunivdiagonals_gminus));

  for (int i = 0; i < *nunivdiagonals_gplus; i++) {
    printf("plus %u\n",(*univdiagonals_gplus)[i]);
  }
  for (int i = 0; i < *nunivdiagonals_gminus; i++) {
    printf("minus %u\n",(*univdiagonals_gminus)[i]);
  }
#endif

  return (*nunivdiagonals_gplus) + (*nunivdiagonals_gminus);
}


#if 0
/* Not as effective as merging all 12 positions, and not as fast */
/* Also, needs to add diagterms when adding or subtracting index1part is not possible */
int
Kmer_exact2 (Univcoord_T **univdiagonals_gplus, int *nunivdiagonals_gplus,
	     Univcoord_T **univdiagonals_gminus, int *nunivdiagonals_gminus,
	     Stage1_T stage1, int querylength) {

  Univcoord_T **mod_univdiagonals;
  int *mod_nunivdiagonals;

  int query_lastpos = querylength - index1part;
  int mod5, mod3;
  int querypos0, querypos1;
  int nstreams;


  debug(printf("Entered Kmer_exact2\n"));
  /* Stage1_dump(stage1,querylength); */

  mod_univdiagonals = (Univcoord_T **) MALLOC(2*index1interval*sizeof(Univcoord_T *));
  mod_nunivdiagonals = (int *) MALLOC((2*index1interval + 1)*sizeof(int)); /* Add 1 needed for Merge_diagonals */


  /* plus */
  nstreams = 0;
  for (mod5 = 0; mod5 < index1interval; mod5++) {
    querypos0 = mod5;
    if ((querypos1 = querypos0 + index1part) > query_lastpos) {
      /* Need to add diagterm */
      mod_univdiagonals[nstreams] = stage1->plus_positions[querypos0];
      mod_nunivdiagonals[nstreams] = stage1->plus_npositions[querypos0];
    } else {
#ifdef LARGE_GENOMES
      mod_univdiagonals[nstreams] =
	Intersect_large(&(mod_nunivdiagonals[nstreams]),
			stage1->plus_positions_high[querypos0],
			stage1->plus_positions[querypos0],stage1->plus_npositions[querypos0],
			/*diagterm1*/querylength - querypos0,
			stage1->plus_positions[querypos1],stage1->plus_npositions[querypos1],
			/*diagterm2*/querylength - querypos1);
#else
      mod_univdiagonals[nstreams] =
	Intersect_small(&(mod_nunivdiagonals[nstreams]),
			stage1->plus_positions[querypos0],stage1->plus_npositions[querypos0],
			/*diagterm1*/querylength - querypos0,
			stage1->plus_positions[querypos1],stage1->plus_npositions[querypos1],
			/*diagterm2*/querylength - querypos1);

#endif
    }
    nstreams++;
  }

  for (mod3 = 0; mod3 < index1interval; mod3++) {
    querypos0 = query_lastpos - mod3;
    if ((querypos1 = querypos0 - index1part) < 0) {
      /* Need to add diagterm */
      mod_univdiagonals[nstreams] = stage1->plus_positions[querypos0];
      mod_nunivdiagonals[nstreams] = stage1->plus_npositions[querypos0];
    } else {
#ifdef LARGE_GENOMES
      mod_univdiagonals[nstreams] =
	Intersect_large(&(mod_nunivdiagonals[nstreams]),
			stage1->plus_positions_high[querypos0],
			stage1->plus_positions[querypos0],stage1->plus_npositions[querypos0],
			/*diagterm1*/querylength - querypos0,
			stage1->plus_positions[querypos1],stage1->plus_npositions[querypos1],
			/*diagterm2*/querylength - querypos1);
#else
      mod_univdiagonals[nstreams] =
	Intersect_small(&(mod_nunivdiagonals[nstreams]),
			stage1->plus_positions[querypos0],stage1->plus_npositions[querypos0],
			/*diagterm1*/querylength - querypos0,
			stage1->plus_positions[querypos1],stage1->plus_npositions[querypos1],
			/*diagterm2*/querylength - querypos1);
#endif
    }
    nstreams++;
  }
    
#ifdef LARGE_GENOMES  
  *univdiagonals_gplus = Merge_diagonals_uint8(&(*nunivdiagonals_gplus),
					       mod_univdiagonals,mod_nunivdiagonals,
					       nstreams,stage1->mergeinfo);
#else
  *univdiagonals_gplus = Merge_diagonals_uint4(&(*nunivdiagonals_gplus),
					       mod_univdiagonals,mod_nunivdiagonals,
					       nstreams,stage1->mergeinfo);
#endif

  nstreams = 0;
  for (mod5 = 0; mod5 < index1interval; mod5++) {
    querypos0 = mod5;
    if ((querypos1 = querypos0 + index1part) > query_lastpos) {
      /* Skip */
    } else {
      FREE(mod_univdiagonals[nstreams]);
    }
    nstreams++;
  }
  for (mod3 = 0; mod3 < index1interval; mod3++) {
    querypos0 = query_lastpos - mod3;
    if ((querypos1 = querypos0 - index1part) < 0) {
      /* Skip */
    } else {
      FREE(mod_univdiagonals[nstreams]);
    }
    nstreams++;
  }



  /* minus */
  nstreams = 0;
  for (mod5 = 0; mod5 < index1interval; mod5++) {
    querypos0 = mod5;
    if ((querypos1 = querypos0 + index1part) > query_lastpos) {
      /* Need to add diagterm */
      mod_univdiagonals[nstreams] = stage1->minus_positions[querypos0];
      mod_nunivdiagonals[nstreams] = stage1->minus_npositions[querypos0];
    } else {
#ifdef LARGE_GENOMES
      mod_univdiagonals[nstreams] =
	Intersect_large(&(mod_nunivdiagonals[nstreams]),
			stage1->minus_positions_high[querypos0],
			stage1->minus_positions[querypos0],stage1->minus_npositions[querypos0],
			/*diagterm1*/querypos0 + index1part,
			stage1->minus_positions[querypos1],stage1->minus_npositions[querypos1],
			/*diagterm2*/querypos1 + index1part);
#else
      mod_univdiagonals[nstreams] =
	Intersect_small(&(mod_nunivdiagonals[nstreams]),
			stage1->minus_positions[querypos0],stage1->minus_npositions[querypos0],
			/*diagterm1*/querypos0 + index1part,
			stage1->minus_positions[querypos1],stage1->minus_npositions[querypos1],
			/*diagterm2*/querypos1 + index1part);
#endif
    }
    nstreams++;
  }

  for (mod3 = 0; mod3 < index1interval; mod3++) {
    querypos0 = query_lastpos - mod3;
    if ((querypos1 = querypos0 - index1part) < 0) {
      /* Need to add diagterm */
      mod_univdiagonals[nstreams] = stage1->minus_positions[querypos0];
      mod_nunivdiagonals[nstreams] = stage1->minus_npositions[querypos0];
    } else {
#ifdef LARGE_GENOMES
      mod_univdiagonals[nstreams] =
	Intersect_large(&(mod_nunivdiagonals[nstreams]),
			stage1->minus_positions_high[querypos0],
			stage1->minus_positions[querypos0],stage1->minus_npositions[querypos0],
			/*diagterm1*/querypos0 + index1part,
			stage1->minus_positions[querypos1],stage1->minus_npositions[querypos1],
			/*diagterm2*/querypos1 + index1part);
#else
      mod_univdiagonals[nstreams] =
	Intersect_small(&(mod_nunivdiagonals[nstreams]),
			stage1->minus_positions[querypos0],stage1->minus_npositions[querypos0],
			/*diagterm1*/querypos0 + index1part,
			stage1->minus_positions[querypos1],stage1->minus_npositions[querypos1],
			/*diagterm2*/querypos1 + index1part);
#endif
    }
    nstreams++;
  }
    
#ifdef LARGE_GENOMES  
  *univdiagonals_gminus = Merge_diagonals_uint8(&(*nunivdiagonals_gminus),
					       mod_univdiagonals,mod_nunivdiagonals,
					       nstreams,stage1->mergeinfo);
#else
  *univdiagonals_gminus = Merge_diagonals_uint4(&(*nunivdiagonals_gminus),
					       mod_univdiagonals,mod_nunivdiagonals,
					       nstreams,stage1->mergeinfo);
#endif

  nstreams = 0;
  for (mod5 = 0; mod5 < index1interval; mod5++) {
    querypos0 = mod5;
    if ((querypos1 = querypos0 + index1part) > query_lastpos) {
      /* Skip */
    } else {
      FREE(mod_univdiagonals[nstreams]);
    }
    nstreams++;
  }
  for (mod3 = 0; mod3 < index1interval; mod3++) {
    querypos0 = query_lastpos - mod3;
    if ((querypos1 = querypos0 - index1part) < 0) {
      /* Skip */
    } else {
      FREE(mod_univdiagonals[nstreams]);
    }
    nstreams++;
  }



#ifdef DEBUG
  printf("Kmer_exact2 returning %d plus and %d minus hits\n",
	 (*nunivdiagonals_gplus),(*nunivdiagonals_gminus));

  for (int i = 0; i < *nunivdiagonals_gplus; i++) {
    printf("plus %u\n",(*univdiagonals_gplus)[i]);
  }
  for (int i = 0; i < *nunivdiagonals_gminus; i++) {
    printf("minus %u\n",(*univdiagonals_gminus)[i]);
  }
#endif

  return (*nunivdiagonals_gplus) + (*nunivdiagonals_gminus);
}
#endif




/* Want a pair with sufficient separation */
static int
check_indexpairs (bool **separatedp, int *indexpairs, int nindexpairs,
		  int querypos_j, int *accumulated_qmin, int *accumulated_qmax,
		  int min_separation) {
  int nseparated = 0;
  int i, k;
  int index2;

  *separatedp = (bool *) CALLOC(nindexpairs,sizeof(bool));
  for (i = 0, k = 0; i < nindexpairs; i++, k += 2) {
    index2 = indexpairs[k+1];
    debug2(printf("Comparing querypos_j %d with querypos_i %d..%d\n",
		  querypos_j,accumulated_qmin[index2],accumulated_qmax[index2]));
    if (querypos_j > /*querypos_i*/accumulated_qmin[index2] + min_separation) {
      (*separatedp)[i] = true;
      nseparated++;
    } else if (/*querypos_i*/accumulated_qmax[index2] > querypos_j + min_separation) {
      (*separatedp)[i] = true;
      nseparated++;
    }
  }

  if (nseparated == 0) {
    FREE(*separatedp);
  }

  return nseparated;
}


static void
revise_accumulated (Univcoord_T **accumulated_diagonals, int **accumulated_qmin,
		    int **accumulated_qmax, int *accumulated_ndiagonals,
#ifdef LARGE_GENOMES
		    unsigned char *positions_high_j, UINT4 *positions_j,
#else
		    Univcoord_T *positions_j,
#endif
		    int npositions_j, int querypos_j, int diagterm_j) {

  Univcoord_T *new_diagonals, univdiagonal_i, univdiagonal_j;
  int *new_qmin, *new_qmax;
  int new_ndiagonals, i, j, k = 0;
  

  new_ndiagonals = (*accumulated_ndiagonals) + npositions_j; /* Includes duplicates */
  new_diagonals = (Univcoord_T *) MALLOC(new_ndiagonals * sizeof(Univcoord_T));
  new_qmin = (int *) MALLOC(new_ndiagonals * sizeof(int));
  new_qmax = (int *) MALLOC(new_ndiagonals * sizeof(int));

  i = j = 0;
  while (i < (*accumulated_ndiagonals) && j < npositions_j) {
    univdiagonal_i = (*accumulated_diagonals)[i];
#ifdef LARGE_GENOMES
    univdiagonal_j = GETPOS(positions_high_j[j],positions_j[j]) + diagterm_j;
#else
    univdiagonal_j = positions_j[j] + diagterm_j;
#endif

    if (univdiagonal_i < univdiagonal_j) {
      new_diagonals[k] = univdiagonal_i;
      new_qmin[k] = (*accumulated_qmin)[i];
      new_qmax[k++] = (*accumulated_qmax)[i];
      i++;
    } else if (univdiagonal_j < univdiagonal_i) {
      new_diagonals[k] = univdiagonal_j;
      new_qmin[k] = querypos_j;
      new_qmax[k++] = querypos_j;
      j++;
    } else {
      new_diagonals[k] = univdiagonal_i;
      if (querypos_j < (*accumulated_qmin)[i]) {
	new_qmin[k] = querypos_j;
      } else {
	new_qmin[k] =  (*accumulated_qmin)[i];
      }
      if (querypos_j > (*accumulated_qmax)[i]) {
	new_qmax[k++] = querypos_j;
      } else {
	new_qmax[k++] =  (*accumulated_qmax)[i];
      }
      i++; j++;
    }
  }
	
  while (i < (*accumulated_ndiagonals)) {
    new_diagonals[k] = /*univdiagonal_i =*/ (*accumulated_diagonals)[i];
    new_qmin[k] = (*accumulated_qmin)[i];
    new_qmax[k++] = (*accumulated_qmax)[i];
    i++;
  }

  while (j < npositions_j) {
#ifdef LARGE_GENOMES
    new_diagonals[k] = /*univdiagonal_j =*/ GETPOS(positions_high_j[j],positions_j[j]) + diagterm_j;
#else
    new_diagonals[k] = /*univdiagonal_j =*/ positions_j[j] + diagterm_j;
#endif
    new_qmin[k] = querypos_j;
    new_qmax[k++] = querypos_j;
    j++;
  }
	
  FREE(*accumulated_qmax);
  FREE(*accumulated_qmin);
  FREE(*accumulated_diagonals);
  *accumulated_diagonals = new_diagonals;
  *accumulated_qmin = new_qmin;
  *accumulated_qmax = new_qmax;
  *accumulated_ndiagonals = k;	/* Excludes duplicates */

  return;
}


/* Previously returned qmin and qmax, but no longer returning because
   Kmer_exact and Kmer_merge do not, and we can use genomebits first
   and last kmer to find qmin/qmax */
/* Iterate until we find our first match */
/* Results are aligned to be consistent with Kmer_exact and Kmer_anypair */
int
Kmer_anypair (Univcoord_T **univdiagonals_gplus, int *nunivdiagonals_gplus,
	      Univcoord_T **univdiagonals_gminus, int *nunivdiagonals_gminus,
	      Stage1_T stage1, int querylength) {

  int *indexpairs_gplus = NULL, *indexpairs_gminus = NULL;
  int nindexpairs_gplus = 0, nindexpairs_gminus = 0;

  Univcoord_T *accumulated_diagonals_gplus, *accumulated_diagonals_gminus;
#ifdef LARGE_GENOMES
  unsigned char *positions_high_j_gplus, *positions_high_j_gminus;
  UINT4 *positions_j_gplus, *positions_j_gminus;
#else
  Univcoord_T *positions_j_gplus, *positions_j_gminus;
#endif
  int accumulated_ndiagonals_gplus, accumulated_ndiagonals_gminus,
    npositions_j_gplus, npositions_j_gminus, i, j, k_gplus, k_gminus;
  int *accumulated_qmin_gplus, *accumulated_qmax_gplus, *accumulated_qmin_gminus, *accumulated_qmax_gminus,
    qpos_j_gplus, diagterm_j_gplus, qpos_j_gminus, diagterm_j_gminus;

  int *order_gplus, *order_gminus;

  /* npositions should be filled from 0 through query_lastpos, inclusive */
  int len = querylength - index1part + 1, index2, k;
  bool *separatedp_gplus = NULL, *separatedp_gminus = NULL, donep;
  int nseparated_gplus = 0, nseparated_gminus = 0;


  debug(printf("Entered Kmer_anypair\n"));
  debug2(Stage1_dump(stage1,querylength));

  /* Initialize accumulated_ndiagonals, plus */
  /* Note: This will overwrite stage1->plus_npositions[query_lastpos+1] */
  order_gplus = Sedgesort_order_int(stage1->plus_npositions,len);

  accumulated_diagonals_gplus = (Univcoord_T *) NULL;
  accumulated_qmin_gplus = (int *) NULL;
  accumulated_qmax_gplus = (int *) NULL;
  accumulated_ndiagonals_gplus = 0;

  k_gplus = 0;
  while (k_gplus < len && accumulated_ndiagonals_gplus == 0) {
    qpos_j_gplus = order_gplus[k_gplus];
    if ((accumulated_ndiagonals_gplus = stage1->plus_npositions[qpos_j_gplus]) == 0) {
      /* Skip */
      debug2(printf("Skipping initial plus qpos %d with no positions\n",qpos_j_gplus));

    } else {
      accumulated_diagonals_gplus = (Univcoord_T *) MALLOC(accumulated_ndiagonals_gplus * sizeof(Univcoord_T));
      accumulated_qmin_gplus = (int *) MALLOC(accumulated_ndiagonals_gplus * sizeof(int));
      accumulated_qmax_gplus = (int *) MALLOC(accumulated_ndiagonals_gplus * sizeof(int));
#ifdef LARGE_GENOMES
      positions_high_j_gplus = stage1->plus_positions_high[qpos_j_gminus];
#endif
      positions_j_gplus = stage1->plus_positions[qpos_j_gplus];
      diagterm_j_gplus = querylength - qpos_j_gplus; /* plus */
      for (j = 0; j < accumulated_ndiagonals_gplus; j++) {
#ifdef LARGE_GENOMES
	accumulated_diagonals_gplus[j] = GETPOS(positions_high_j_gplus[j],positions_j_gplus[j]) + diagterm_j_gplus;
#else
	accumulated_diagonals_gplus[j] = positions_j_gplus[j] + diagterm_j_gplus;
#endif
	accumulated_qmin_gplus[j] = qpos_j_gplus;
	accumulated_qmax_gplus[j] = qpos_j_gplus;
      }
    }
    k_gplus++;
  }


  /* Initialize accumulated_ndiagonals, minus */
  /* Note: This will overwrite stage1->minus_npositions[query_lastpos+1] */
  order_gminus = Sedgesort_order_int(stage1->minus_npositions,len);

  accumulated_diagonals_gminus = (Univcoord_T *) NULL;
  accumulated_qmin_gminus = (int *) NULL;
  accumulated_qmax_gminus = (int *) NULL;
  accumulated_ndiagonals_gminus = 0;

  k_gminus = 0;
  while (k_gminus < len && accumulated_ndiagonals_gminus == 0) {
    qpos_j_gminus = order_gminus[k_gminus];
    if ((accumulated_ndiagonals_gminus = stage1->minus_npositions[qpos_j_gminus]) == 0) {
      /* Skip */
      debug2(printf("Skipping initial minus qpos %d with no positions\n",qpos_j_gminus));

    } else {
      accumulated_diagonals_gminus = (Univcoord_T *) MALLOC(accumulated_ndiagonals_gminus * sizeof(Univcoord_T));
      accumulated_qmin_gminus = (int *) MALLOC(accumulated_ndiagonals_gminus * sizeof(int));
      accumulated_qmax_gminus = (int *) MALLOC(accumulated_ndiagonals_gminus * sizeof(int));
#ifdef LARGE_GENOMES
      positions_high_j_gminus = stage1->minus_positions_high[qpos_j_gminus];
#endif
      positions_j_gminus = stage1->minus_positions[qpos_j_gminus];
      diagterm_j_gminus = qpos_j_gminus + index1part; /* minus */
      for (j = 0; j < accumulated_ndiagonals_gminus; j++) {
#ifdef LARGE_GENOMES
	accumulated_diagonals_gminus[j] = GETPOS(positions_high_j_gminus[j],positions_j_gminus[j]) + diagterm_j_gminus;
#else
	accumulated_diagonals_gminus[j] = positions_j_gminus[j] + diagterm_j_gminus;
#endif
	accumulated_qmin_gminus[j] = qpos_j_gminus;
	accumulated_qmax_gminus[j] = qpos_j_gminus;
      }
    }
    k_gminus++;
  }

  /* Perform intersections with accumulated diagonals.  Search plus and minus in parallel. */
  donep = false;
  indexpairs_gplus = indexpairs_gminus = (int *) NULL;
  nindexpairs_gplus = nindexpairs_gminus = 0;
  while (donep == false && k_gplus < len && k_gminus < len) {
    /* plus */
    qpos_j_gplus = order_gplus[k_gplus];
    npositions_j_gplus = stage1->plus_npositions[qpos_j_gplus]; /* Must be > 0 after initial loop */
#ifdef LARGE_GENOMES
    positions_high_j_gplus = stage1->plus_positions_high[qpos_j_gplus];
#endif
    positions_j_gplus = stage1->plus_positions[qpos_j_gplus];
    diagterm_j_gplus = querylength - qpos_j_gplus; /* plus */

    FREE(indexpairs_gplus);
    indexpairs_gplus = (int *) MALLOC(2 * npositions_j_gplus * sizeof(int));
    debug2(printf("Performing plus intersection of qpos %d (%d positions) with %d accumulated diagonals\n",
		  qpos_j_gplus,npositions_j_gplus,accumulated_ndiagonals_gplus));

#ifdef LARGE_GENOMES
    nindexpairs_gplus = Intersect_indices_large(indexpairs_gplus,positions_high_j_gplus,positions_j_gplus,npositions_j_gplus,
						/*diagterm1*/diagterm_j_gplus,
						accumulated_diagonals_gplus,accumulated_ndiagonals_gplus,
						/*diagterm2*/0);
#else
    nindexpairs_gplus = Intersect_indices_small(indexpairs_gplus,positions_j_gplus,npositions_j_gplus,
						/*diagterm1*/diagterm_j_gplus,
						accumulated_diagonals_gplus,accumulated_ndiagonals_gplus,
						/*diagterm2*/0);
#endif

    nseparated_gplus = 0;
    if (nindexpairs_gplus > 0 &&
	(nseparated_gplus = check_indexpairs(&separatedp_gplus,indexpairs_gplus,nindexpairs_gplus,qpos_j_gplus,
					     accumulated_qmin_gplus,accumulated_qmax_gplus,
					     /*min_separation*/index1part)) > 0) {
      donep = true;

    } else {
      FREE(indexpairs_gplus);
    }


    /* minus */
    qpos_j_gminus = order_gminus[k_gminus];
    npositions_j_gminus = stage1->minus_npositions[qpos_j_gminus]; /* Must be > 0 after initial loop */
#ifdef LARGE_GENOMES
    positions_high_j_gminus = stage1->minus_positions_high[qpos_j_gplus];
#endif
    positions_j_gminus = stage1->minus_positions[qpos_j_gminus];
    diagterm_j_gminus = qpos_j_gminus + index1part; /* minus */

    FREE(indexpairs_gminus);
    indexpairs_gminus = (int *) MALLOC(2 * npositions_j_gminus * sizeof(int));
    debug2(printf("Performing minus intersection of qpos %d (%d positions) with %d accumulated diagonals\n",
		  qpos_j_gminus,npositions_j_gminus,accumulated_ndiagonals_gminus));

#ifdef LARGE_GENOMES
    nindexpairs_gminus = Intersect_indices_large(indexpairs_gminus,positions_high_j_gminus,positions_j_gminus,npositions_j_gminus,
						/*diagterm1*/diagterm_j_gminus,
						accumulated_diagonals_gminus,accumulated_ndiagonals_gminus,
						/*diagterm2*/0);
#else
    nindexpairs_gminus = Intersect_indices_small(indexpairs_gminus,positions_j_gminus,npositions_j_gminus,
						/*diagterm1*/diagterm_j_gminus,
						accumulated_diagonals_gminus,accumulated_ndiagonals_gminus,
						/*diagterm2*/0);
#endif

    nseparated_gminus = 0;
    if (nindexpairs_gminus > 0 &&
	(nseparated_gminus = check_indexpairs(&separatedp_gminus,indexpairs_gminus,nindexpairs_gminus,qpos_j_gminus,
					      accumulated_qmin_gminus,accumulated_qmax_gminus,
					      /*min_separation*/index1part)) > 0) {
      donep = true;

    } else {
      FREE(indexpairs_gminus);
    }

    if (donep == false) {
      revise_accumulated(&accumulated_diagonals_gplus,&accumulated_qmin_gplus,&accumulated_qmax_gplus,
			 &accumulated_ndiagonals_gplus,
#ifdef LARGE_GENOMES
			 positions_high_j_gplus,
#endif
			 positions_j_gplus,npositions_j_gplus,qpos_j_gplus,diagterm_j_gplus);
      revise_accumulated(&accumulated_diagonals_gminus,&accumulated_qmin_gminus,&accumulated_qmax_gminus,
			 &accumulated_ndiagonals_gminus,
#ifdef LARGE_GENOMES
			 positions_high_j_gminus,
#endif
			 positions_j_gminus,npositions_j_gminus,
			 qpos_j_gminus,diagterm_j_gminus);
    }

    k_gplus++;
    k_gminus++;
  }


  while (donep == false && k_gplus < len) {
    /* plus */
    qpos_j_gplus = order_gplus[k_gplus];
    npositions_j_gplus = stage1->plus_npositions[qpos_j_gplus]; /* Must be > 0 after initial loop */
#ifdef LARGE_GENOMES
    positions_high_j_gplus = stage1->plus_positions_high[qpos_j_gplus];
#endif
    positions_j_gplus = stage1->plus_positions[qpos_j_gplus];
    diagterm_j_gplus = querylength - qpos_j_gplus; /* plus */

    FREE(indexpairs_gplus);
    indexpairs_gplus = (int *) MALLOC(2 * npositions_j_gplus * sizeof(int));
    debug2(printf("Performing plus intersection of qpos %d (%d positions) with %d accumulated diagonals\n",
		  qpos_j_gplus,npositions_j_gplus,accumulated_ndiagonals_gplus));
#ifdef LARGE_GENOMES
    nindexpairs_gplus = Intersect_indices_large(indexpairs_gplus,positions_high_j_gplus,positions_j_gplus,npositions_j_gplus,
						/*diagterm1*/diagterm_j_gplus,
						accumulated_diagonals_gplus,accumulated_ndiagonals_gplus,
						/*diagterm2*/0);
#else
    nindexpairs_gplus = Intersect_indices_small(indexpairs_gplus,positions_j_gplus,npositions_j_gplus,
						/*diagterm1*/diagterm_j_gplus,
						accumulated_diagonals_gplus,accumulated_ndiagonals_gplus,
						/*diagterm2*/0);
#endif

    nseparated_gplus = 0;
    if (nindexpairs_gplus > 0 &&
	(nseparated_gplus = check_indexpairs(&separatedp_gplus,indexpairs_gplus,nindexpairs_gplus,qpos_j_gplus,
					     accumulated_qmin_gplus,accumulated_qmax_gplus,
					     /*min_separation*/index1part)) > 0) {
      donep = true;

    } else {
      FREE(indexpairs_gplus);
      revise_accumulated(&accumulated_diagonals_gplus,&accumulated_qmin_gplus,
			 &accumulated_qmax_gplus,&accumulated_ndiagonals_gplus,
#ifdef LARGE_GENOMES
			 positions_high_j_gplus,
#endif
			 positions_j_gplus,npositions_j_gplus,
			 qpos_j_gplus,diagterm_j_gplus);
    }

    k_gplus++;
  }


  while (donep == false && k_gminus < len) {
    /* minus */
    qpos_j_gminus = order_gminus[k_gminus];
    npositions_j_gminus = stage1->minus_npositions[qpos_j_gminus]; /* Must be > 0 after initial loop */
#ifdef LARGE_GENOMES
    positions_high_j_gminus = stage1->minus_positions_high[qpos_j_gminus];
#endif
    positions_j_gminus = stage1->minus_positions[qpos_j_gminus];
    diagterm_j_gminus = qpos_j_gminus + index1part; /* minus */

    FREE(indexpairs_gminus);
    indexpairs_gminus = (int *) MALLOC(2 * npositions_j_gminus * sizeof(int));
    debug2(printf("Performing minus intersection of qpos %d (%d positions) with %d accumulated diagonals\n",
		  qpos_j_gminus,npositions_j_gminus,accumulated_ndiagonals_gminus));
#ifdef LARGE_GENOMES
    nindexpairs_gminus = Intersect_indices_large(indexpairs_gminus,positions_high_j_gminus,positions_j_gminus,npositions_j_gminus,
						/*diagterm1*/diagterm_j_gminus,
						accumulated_diagonals_gminus,accumulated_ndiagonals_gminus,
						/*diagterm2*/0);
#else
    nindexpairs_gminus = Intersect_indices_small(indexpairs_gminus,positions_j_gminus,npositions_j_gminus,
						/*diagterm1*/diagterm_j_gminus,
						accumulated_diagonals_gminus,accumulated_ndiagonals_gminus,
						/*diagterm2*/0);
#endif

    nseparated_gminus = 0;
    if (nindexpairs_gminus > 0 &&
	(nseparated_gminus = check_indexpairs(&separatedp_gminus,indexpairs_gminus,nindexpairs_gminus,qpos_j_gminus,
					      accumulated_qmin_gminus,accumulated_qmax_gminus,
					      /*min_separation*/index1part)) > 0) {
      donep = true;

    } else {
      FREE(indexpairs_gminus);
      revise_accumulated(&accumulated_diagonals_gminus,&accumulated_qmin_gminus,
			 &accumulated_qmax_gminus,&accumulated_ndiagonals_gminus,
#ifdef LARGE_GENOMES
			 positions_high_j_gminus,
#endif
			 positions_j_gminus,npositions_j_gminus,
			 qpos_j_gminus,diagterm_j_gminus);
    }

    k_gminus++;
  }

  FREE(order_gminus);
  FREE(order_gplus);


  /* Prepare output, gplus */
  if (nseparated_gplus == 0) {
    *univdiagonals_gplus = (Univcoord_T *) NULL;
    /* *qstart_gplus = (int *) NULL; */
    /* *qend_gplus = (int *) NULL; */
    *nunivdiagonals_gplus = 0;

  } else {
    *nunivdiagonals_gplus = nseparated_gplus;
    MALLOC_ALIGN(*univdiagonals_gplus,nseparated_gplus * sizeof(Univcoord_T));
    /* *qstart_gplus = (int *) MALLOC(nseparated_gplus * sizeof(int)); */
    /* *qend_gplus = (int *) MALLOC(nseparated_gplus * sizeof(int)); */
    
    j = 0;
    for (i = 0, k = 0; i < nindexpairs_gplus; i++, k += 2) {
      if (separatedp_gplus[i] == true) {
	index2 = indexpairs_gplus[k+1];
	(*univdiagonals_gplus)[j] = accumulated_diagonals_gplus[index2];
	/* (*qstart_gplus)[j] = accumulated_qmin_gplus[index2]; */
	/* (*qend_gplus)[j] = accumulated_qmax_gplus[index2] + index1part; */
	j++;
      }
    }
    assert(j == nseparated_gplus);
    FREE(separatedp_gplus);
  }
  FREE(indexpairs_gplus);

  FREE(accumulated_diagonals_gplus);
  FREE(accumulated_qmin_gplus);
  FREE(accumulated_qmax_gplus);


  /* Prepare output, gminus */
  if (nseparated_gminus == 0) {
    *univdiagonals_gminus = (Univcoord_T *) NULL;
    /* *qstart_gminus = (int *) NULL; */
    /* *qend_gminus = (int *) NULL; */
    *nunivdiagonals_gminus = 0;
  } else {
    *nunivdiagonals_gminus = nseparated_gminus;
    MALLOC_ALIGN(*univdiagonals_gminus,nseparated_gminus * sizeof(Univcoord_T));
    /* *qstart_gminus = (int *) MALLOC(nseparated_gminus * sizeof(int)); */
    /* *qend_gminus = (int *) MALLOC(nseparated_gminus * sizeof(int)); */
    
    j = 0;
    for (i = 0, k = 0; i < nindexpairs_gminus; i++, k += 2) {
      if (separatedp_gminus[i] == true) {
	index2 = indexpairs_gminus[k+1];
	(*univdiagonals_gminus)[j] = accumulated_diagonals_gminus[index2];
	/* (*qstart_gminus)[j] = querylength - (accumulated_qmax_gminus[index2] + index1part); */
	/* (*qend_gminus)[j] = querylength - accumulated_qmin_gminus[index2]; */
	j++;
      }
    }
    assert(j == nseparated_gminus);
    FREE(separatedp_gminus);
  }
  FREE(indexpairs_gminus);

  FREE(accumulated_diagonals_gminus);
  FREE(accumulated_qmin_gminus);
  FREE(accumulated_qmax_gminus);
  
#ifdef DEBUG
  printf("Kmer_anypair returning %d plus and %d minus hits\n",
	 (*nunivdiagonals_gplus),(*nunivdiagonals_gminus));

  for (i = 0; i < *nunivdiagonals_gplus; i++) {
    printf("plus %u\n",(*univdiagonals_gplus)[i]);
  }
  for (i = 0; i < *nunivdiagonals_gminus; i++) {
    printf("minus %u\n",(*univdiagonals_gminus)[i]);
  }
#endif

  return (*nunivdiagonals_gplus) + (*nunivdiagonals_gminus);
}


static int
most_prevalent (Univcoord_T *diagonals, int ndiagonals) {
  int most_prevalent = 0;
  int i, j;

  i = 0;
  while (i < ndiagonals) {
    j = i + 1;
    while (j < ndiagonals && diagonals[j] == diagonals[i]) {
      j++;
    }
    if (j - i > most_prevalent) {
      most_prevalent = j - i;
    }
    
    i = j;
  }

  return most_prevalent;
}

static Univcoord_T *
keep_prevalence (int *ndiagonals, Univcoord_T *all_diagonals, int all_ndiagonals,
		int prevalence) {
  Univcoord_T *diagonals;
  int k, i, j;

  /* Count number of diagonals with given prevalence */
  k = 0;
  i = 0;
  while (i < all_ndiagonals) {
    j = i + 1;
    while (j < all_ndiagonals && all_diagonals[j] == all_diagonals[i]) {
      j++;
    }
    if (j - i >= prevalence) {
      k++;
    }
    
    i = j;
  }

  /* Allocate new array.  Needs to be aligned for later merging with Stage1_T all_univdiagonals */
  MALLOC_ALIGN(diagonals,k*sizeof(Univcoord_T));

  /* Save diagonals if there are prevalence or more of them */
  k = 0;
  i = 0;
  while (i < all_ndiagonals) {
    j = i + 1;
    while (j < all_ndiagonals && all_diagonals[j] == all_diagonals[i]) {
      j++;
    }
    if (j - i >= prevalence) {
      diagonals[k++] = all_diagonals[i];
    }
    
    i = j;
  }

  *ndiagonals = k;
  return diagonals;
}


static Univcoord_T *
get_univdiagonals_gplus (int *ndiagonals, int *max_npositions,
			 Stage1_T this, int querylength) {
  int querypos, query_lastpos;

  query_lastpos = querylength - index1part;

  *max_npositions = 0;
  for (querypos = 0; querypos <= query_lastpos; querypos++) {
    this->plus_diagterms[querypos] = querylength - querypos;
    if (this->plus_npositions[querypos] > *max_npositions) {
      *max_npositions = this->plus_npositions[querypos];
    }
  }
  
#ifdef LARGE_GENOMES
  return Merge_diagonals_large(&(*ndiagonals),this->plus_positions_high,this->plus_positions,
			       this->plus_npositions,this->plus_diagterms,
			       /*nstreams*/query_lastpos+1,this->mergeinfo);
#else
  return Merge_diagonals(&(*ndiagonals),this->plus_positions,this->plus_npositions,
			 this->plus_diagterms,/*nstreams*/query_lastpos+1,this->mergeinfo);
#endif
}


static Univcoord_T *
get_univdiagonals_gminus (int *ndiagonals, int *max_npositions,
			  Stage1_T this, int querylength) {
  int querypos, query_lastpos;

  query_lastpos = querylength - index1part;

  *max_npositions = 0;
  for (querypos = 0; querypos <= query_lastpos; querypos++) {
    this->minus_diagterms[querypos] = querypos + index1part;
    if (this->minus_npositions[querypos] > *max_npositions) {
      *max_npositions = this->minus_npositions[querypos];
    }
  }
  
#ifdef LARGE_GENOMES
  return Merge_diagonals_large(&(*ndiagonals),this->minus_positions_high,this->minus_positions,
			       this->minus_npositions,this->minus_diagterms,
			       /*nstreams*/query_lastpos+1,this->mergeinfo);
#else
  return Merge_diagonals(&(*ndiagonals),this->minus_positions,this->minus_npositions,
			 this->minus_diagterms,/*nstreams*/query_lastpos+1,this->mergeinfo);
#endif
}


static int
make_unique (Univcoord_T *diagonals, int ndiagonals) {
  int k = 0, i, j;

  i = 0;
  while (i < ndiagonals) {
    j = i + 1;
    while (j < ndiagonals && diagonals[j] == diagonals[i]) {
      j++;
    }
    diagonals[k++] = diagonals[i];
    
    i = j;
  }

  return k;
}


/* Results are aligned because of Merge_diagonals */
/* Essentially a streamlined version of Kmer_search_complete */
/* Saves all_univdiagonals for later calls to Kmer_widest or Kmer_anchored */
int
Kmer_prevalent (Univcoord_T **all_univdiagonals_gplus, int *all_nunivdiagonals_gplus,
		int *max_npositions_gplus,
		Univcoord_T **all_univdiagonals_gminus, int *all_nunivdiagonals_gminus,
		int *max_npositions_gminus,

		Univcoord_T **univdiagonals_gplus, int *nunivdiagonals_gplus,
		Univcoord_T **univdiagonals_gminus, int *nunivdiagonals_gminus,

		Stage1_T stage1, int querylength) {

  int plus_nunivdiagonals, minus_nunivdiagonals, plus_nprevalent, minus_nprevalent;


  debug(printf("Entered Kmer_prevalent\n"));

  /* plus */
  *all_univdiagonals_gplus =
    get_univdiagonals_gplus(&plus_nunivdiagonals,&(*max_npositions_gplus),
			    stage1,querylength);
  plus_nprevalent = most_prevalent(*all_univdiagonals_gplus,plus_nunivdiagonals);

  /* minus */
  *all_univdiagonals_gminus =
    get_univdiagonals_gminus(&minus_nunivdiagonals,&(*max_npositions_gminus),
			     stage1,querylength);
  minus_nprevalent = most_prevalent(*all_univdiagonals_gminus,minus_nunivdiagonals);


  /* Pick best direction */
  if (plus_nprevalent > minus_nprevalent + index1interval) {
    *univdiagonals_gplus = keep_prevalence(&(*nunivdiagonals_gplus),
					  *all_univdiagonals_gplus,plus_nunivdiagonals,
					  plus_nprevalent - (index1interval - 1));
    *univdiagonals_gminus = (Univcoord_T *) NULL;
    *nunivdiagonals_gminus = 0;

  } else if (minus_nprevalent > plus_nprevalent + index1interval) {
    *univdiagonals_gminus = keep_prevalence(&(*nunivdiagonals_gminus),
					   *all_univdiagonals_gminus,minus_nunivdiagonals,
					   minus_nprevalent - (index1interval - 1));
    *univdiagonals_gplus = (Univcoord_T *) NULL;
    *nunivdiagonals_gplus = 0;

  } else {
    *univdiagonals_gplus = keep_prevalence(&(*nunivdiagonals_gplus),
					  *all_univdiagonals_gplus,plus_nunivdiagonals,
					  plus_nprevalent - (index1interval - 1));
    *univdiagonals_gminus = keep_prevalence(&(*nunivdiagonals_gminus),
					   *all_univdiagonals_gminus,minus_nunivdiagonals,
					   minus_nprevalent - (index1interval - 1));
  }
    
  /* Store a unique array of all_nunivdiagonals for use by Kmer_widest and Kmer_anchored */
  /* Later, could do this just before we call Kmer_widest or Kmer_anchored */
  *all_nunivdiagonals_gplus = make_unique(*all_univdiagonals_gplus,plus_nunivdiagonals);
  *all_nunivdiagonals_gminus = make_unique(*all_univdiagonals_gminus,minus_nunivdiagonals);

#ifdef DEBUG
  printf("Kmer_prevalent returning %d plus and %d minus hits\n",
	 (*nunivdiagonals_gplus),(*nunivdiagonals_gminus));

  for (int i = 0; i < *nunivdiagonals_gplus; i++) {
    printf("plus %u\n",(*univdiagonals_gplus)[i]);
  }
  for (int i = 0; i < *nunivdiagonals_gminus; i++) {
    printf("minus %u\n",(*univdiagonals_gminus)[i]);
  }
#endif

  return (*nunivdiagonals_gplus) + (*nunivdiagonals_gminus);
}



static int
revise_bounds_gplus (int **qmin, int **qmax, Univcoord_T *diagonals, int ndiagonals,
		     int max_npositions, Stage1_T this, int querylength) {
  int widest = 0;
  int index2;
  int i, k;

  int query_lastpos = querylength - index1part, querypos;
  int *indices;
  int nindices;

  *qmin = (int *) MALLOC(ndiagonals * sizeof(int));
  *qmax = (int *) CALLOC(ndiagonals,sizeof(int));
  for (i = 0; i < ndiagonals; i++) {
    (*qmin)[i] = querylength;
  }

  indices = (int *) MALLOC(2 * max_npositions * sizeof(int));
  for (querypos = 0; querypos <= query_lastpos; querypos++) {
#ifdef LARGE_GENOMES
    nindices = Intersect_indices_large(indices,this->plus_positions_high[querypos],
				       this->plus_positions[querypos],this->plus_npositions[querypos],
				       /*diagterm1*/querylength - querypos,
				       diagonals,ndiagonals,/*diagterm2*/0);
#else
    nindices = Intersect_indices_small(indices,this->plus_positions[querypos],this->plus_npositions[querypos],
				       /*diagterm1*/querylength - querypos,
				       diagonals,ndiagonals,/*diagterm2*/0);
#endif

    /* Revise qmin and qmax */
    for (i = 0, k = 1; i < nindices; i++, k += 2) {
      index2 = indices[k];	/* Effectively k + 1 */
      if (querypos < (*qmin)[index2]) {
	(*qmin)[index2] = querypos;
      }
      if (querypos > (*qmax)[index2]) {
	(*qmax)[index2] = querypos;
      }
      if ((*qmax)[index2] - (*qmax)[index2] > widest) {
	widest = (*qmax)[index2] - (*qmax)[index2];
      }
    }
  }
  FREE(indices);

  return widest;
}


static int
revise_bounds_gminus (int **qmin, int **qmax, Univcoord_T *diagonals, int ndiagonals,
		     int max_npositions, Stage1_T this, int querylength) {
  int widest = 0;
  int index2;
  int i, k;

  int query_lastpos = querylength - index1part, querypos;
  int *indices;
  int nindices;

  *qmin = (int *) MALLOC(ndiagonals * sizeof(int));
  *qmax = (int *) CALLOC(ndiagonals,sizeof(int));
  for (i = 0; i < ndiagonals; i++) {
    (*qmin)[i] = querylength;
  }

  indices = (int *) MALLOC(2 * max_npositions * sizeof(int));
  for (querypos = 0; querypos <= query_lastpos; querypos++) {
#ifdef LARGE_GENOMES
    nindices = Intersect_indices_large(indices,this->minus_positions_high[querypos],
				       this->minus_positions[querypos],this->minus_npositions[querypos],
				       /*diagterm1*/querypos + index1part,
				       diagonals,ndiagonals,/*diagterm2*/0);
#else
    nindices = Intersect_indices_small(indices,this->minus_positions[querypos],this->minus_npositions[querypos],
				       /*diagterm1*/querypos + index1part,
				       diagonals,ndiagonals,/*diagterm2*/0);
#endif

    /* Revise qmin and qmax */
    for (i = 0, k = 1; i < nindices; i++, k += 2) {
      index2 = indices[k];	/* Effectively k + 1 */
      if (querypos < (*qmin)[index2]) {
	(*qmin)[index2] = querypos;
      }
      if (querypos > (*qmax)[index2]) {
	(*qmax)[index2] = querypos;
      }
      if ((*qmax)[index2] - (*qmax)[index2] > widest) {
	widest = (*qmax)[index2] - (*qmax)[index2];
      }
    }
  }
  FREE(indices);

  return widest;
}


/* Follows keep_prevalence */
static Univcoord_T *
keep_width (int *ndiagonals, Univcoord_T *all_diagonals, int all_ndiagonals,
	    int *qmin, int *qmax, int width) {
  Univcoord_T *diagonals;
  int k, i;

  /* Count number of diagonals with given width */
  k = 0;
  for (i = 0; i < all_ndiagonals; i++) {
    if (qmax[i] - qmin[i] >= width) {
      k++;
    }
  }

  /* Allocate new array.  Needs to be aligned for later merging with Stage1_T all_univdiagonals */
  MALLOC_ALIGN(diagonals,k*sizeof(Univcoord_T));

  /* Save diagonals if they have given width */
  k = 0;
  for (i = 0; i < all_ndiagonals; i++) {
    if (qmax[i] - qmin[i] >= width) {
      diagonals[k++] = all_diagonals[i];
    }
  }

  *ndiagonals = k;
  return diagonals;
}


/* Re-uses all univdiagonals found by Kmer_prevalent */
/* Requires revising qmin/qmax bounds */
/* Attempting to find the same univdiagonals as Segment_search_all */
int
Kmer_widest (Univcoord_T **univdiagonals_gplus, int *nunivdiagonals_gplus,
	     Univcoord_T **univdiagonals_gminus, int *nunivdiagonals_gminus,

	     Univcoord_T *all_univdiagonals_gplus, int all_nunivdiagonals_gplus,
	     int max_npositions_gplus,
	     Univcoord_T *all_univdiagonals_gminus, int all_nunivdiagonals_gminus,
	     int max_npositions_gminus,

	     Stage1_T stage1, int querylength) {

  int plus_widest, minus_widest;
  int *plus_qmin, *plus_qmax, *minus_qmin, *minus_qmax;


  debug(printf("Entered Kmer_widest\n"));

  plus_widest = revise_bounds_gplus(&plus_qmin,&plus_qmax,
				    all_univdiagonals_gplus,all_nunivdiagonals_gplus,
				    max_npositions_gplus,stage1,querylength);
  minus_widest = revise_bounds_gminus(&minus_qmin,&minus_qmax,
				    all_univdiagonals_gminus,all_nunivdiagonals_gminus,
				    max_npositions_gminus,stage1,querylength);

  /* Pick best direction */
  if (plus_widest > minus_widest + index1interval) {
    *univdiagonals_gplus = keep_width(&(*nunivdiagonals_gplus),
				      all_univdiagonals_gplus,all_nunivdiagonals_gplus,
				      plus_qmin,plus_qmax,
				      /*width*/plus_widest - (index1interval - 1));
    *univdiagonals_gminus = (Univcoord_T *) NULL;
    *nunivdiagonals_gminus = 0;

  } else if (minus_widest > plus_widest + index1interval) {
    *univdiagonals_gminus = keep_width(&(*nunivdiagonals_gminus),
				       all_univdiagonals_gminus,all_nunivdiagonals_gminus,
				       minus_qmin,minus_qmax,
				       /*width*/minus_widest - (index1interval - 1));
    *univdiagonals_gplus = (Univcoord_T *) NULL;
    *nunivdiagonals_gplus = 0;

  } else {
    *univdiagonals_gplus = keep_width(&(*nunivdiagonals_gplus),
				      all_univdiagonals_gplus,all_nunivdiagonals_gplus,
				      plus_qmin,plus_qmax,
				      /*width*/plus_widest - (index1interval - 1));
    *univdiagonals_gminus = keep_width(&(*nunivdiagonals_gminus),
				       all_univdiagonals_gminus,all_nunivdiagonals_gminus,
				       minus_qmin,minus_qmax,
				       /*width*/minus_widest - (index1interval - 1));
  }

  FREE(minus_qmax);
  FREE(minus_qmin);
  FREE(plus_qmax);
  FREE(plus_qmin);

#ifdef DEBUG
  printf("Kmer_widest returning %d plus and %d minus hits\n",
	 (*nunivdiagonals_gplus),(*nunivdiagonals_gminus));

  for (int i = 0; i < *nunivdiagonals_gplus; i++) {
    printf("plus %u\n",(*univdiagonals_gplus)[i]);
  }
  for (int i = 0; i < *nunivdiagonals_gminus; i++) {
    printf("minus %u\n",(*univdiagonals_gminus)[i]);
  }
#endif

  return (*nunivdiagonals_gplus) + (*nunivdiagonals_gminus);
}


/* Need to have called Kmer_prevalent on 5' end first */
int
Kmer_anchored_5 (Univcoord_T **univdiagonals_gplus, int *nunivdiagonals_gplus,
		 Univcoord_T **univdiagonals_gminus, int *nunivdiagonals_gminus,

		 Univcoord_T *all_univdiagonals5_gplus, int all_nunivdiagonals5_gplus,
		 Univcoord_T *all_univdiagonals5_gminus, int all_nunivdiagonals5_gminus,

		 Univcoord_T *univdiagonals3_gplus, int nunivdiagonals3_gplus,
		 Univcoord_T *univdiagonals3_gminus, int nunivdiagonals3_gminus,

		 Univcoord_T slop) {

  Univcoord_T *diagpairs;
  int ndiagpairs, i, k;


  debug(printf("Entered Kmer_anchored_5\n"));

  /* plus */
#ifdef LARGE_GENOMES
  diagpairs = Intersect_approx_uint8(&ndiagpairs,
				     all_univdiagonals5_gplus,all_nunivdiagonals5_gplus,/*diagterm*/0,
				     univdiagonals3_gplus,nunivdiagonals3_gplus,/*diagterm*/0,
				     /*below_slop*/0,/*above_slop*/slop);
#else
  diagpairs = Intersect_approx_uint4(&ndiagpairs,
				     all_univdiagonals5_gplus,all_nunivdiagonals5_gplus,/*diagterm*/0,
				     univdiagonals3_gplus,nunivdiagonals3_gplus,/*diagterm*/0,
				     /*below_slop*/0,/*above_slop*/slop);
#endif

  if (ndiagpairs == 0) {
    *univdiagonals_gplus = (Univcoord_T *) NULL;
  } else {
    MALLOC_ALIGN(*univdiagonals_gplus, ndiagpairs * sizeof(Univcoord_T));
    for (i = 0, /*for 5'*/k = 0; i < ndiagpairs; i++, k += 2) {
      (*univdiagonals_gplus)[i] = diagpairs[k];
    }
  }
  FREE(diagpairs);
  *nunivdiagonals_gplus = ndiagpairs;


  /* minus */
#ifdef LARGE_GENOMES
  diagpairs = Intersect_approx_uint8(&ndiagpairs,
				     all_univdiagonals5_gminus,all_nunivdiagonals5_gminus,/*diagterm*/0,
				     univdiagonals3_gminus,nunivdiagonals3_gminus,/*diagterm*/0,
				     /*below_slop*/slop,/*above_slop*/0);
#else
  diagpairs = Intersect_approx_uint4(&ndiagpairs,
				     all_univdiagonals5_gminus,all_nunivdiagonals5_gminus,/*diagterm*/0,
				     univdiagonals3_gminus,nunivdiagonals3_gminus,/*diagterm*/0,
				     /*below_slop*/slop,/*above_slop*/0);
#endif

  if (ndiagpairs == 0) {
    *univdiagonals_gminus = (Univcoord_T *) NULL;
  } else {
    MALLOC_ALIGN(*univdiagonals_gminus, ndiagpairs * sizeof(Univcoord_T));
    for (i = 0, /*for 5'*/k = 0; i < ndiagpairs; i++, k += 2) {
      (*univdiagonals_gminus)[i] = diagpairs[k];
    }
  }
  FREE(diagpairs);
  *nunivdiagonals_gminus = ndiagpairs;

#ifdef DEBUG
  printf("Kmer_anchored_5 returning %d plus and %d minus hits\n",
	 (*nunivdiagonals_gplus),(*nunivdiagonals_gminus));

  for (i = 0; i < *nunivdiagonals_gplus; i++) {
    printf("plus %u\n",(*univdiagonals_gplus)[i]);
  }
  for (i = 0; i < *nunivdiagonals_gminus; i++) {
    printf("minus %u\n",(*univdiagonals_gminus)[i]);
  }
#endif

  return (*nunivdiagonals_gplus) + (*nunivdiagonals_gminus);
}



/* Need to have called Kmer_prevalent on 3' end first */
int
Kmer_anchored_3 (Univcoord_T **univdiagonals_gplus, int *nunivdiagonals_gplus,
		 Univcoord_T **univdiagonals_gminus, int *nunivdiagonals_gminus,

		 Univcoord_T *univdiagonals5_gplus, int nunivdiagonals5_gplus,
		 Univcoord_T *univdiagonals5_gminus, int nunivdiagonals5_gminus,

		 Univcoord_T *all_univdiagonals3_gplus, int all_nunivdiagonals3_gplus,
		 Univcoord_T *all_univdiagonals3_gminus, int all_nunivdiagonals3_gminus,

		 Univcoord_T slop) {

  Univcoord_T *diagpairs;
  int ndiagpairs, i, k;


  debug(printf("Entered Kmer_anchored_3\n"));

  /* plus */
#ifdef LARGE_GENOMES
  diagpairs = Intersect_approx_uint8(&ndiagpairs,
				     univdiagonals5_gplus,nunivdiagonals5_gplus,/*diagterm*/0,
				     all_univdiagonals3_gplus,all_nunivdiagonals3_gplus,/*diagterm*/0,
				     /*below_slop*/0,/*above_slop*/slop);
#else
  diagpairs = Intersect_approx_uint4(&ndiagpairs,
				     univdiagonals5_gplus,nunivdiagonals5_gplus,/*diagterm*/0,
				     all_univdiagonals3_gplus,all_nunivdiagonals3_gplus,/*diagterm*/0,
				     /*below_slop*/0,/*above_slop*/slop);
#endif

  if (ndiagpairs == 0) {
    *univdiagonals_gplus = (Univcoord_T *) NULL;
  } else {
    MALLOC_ALIGN(*univdiagonals_gplus, ndiagpairs * sizeof(Univcoord_T));
    for (i = 0, /*for 3'*/k = 1; i < ndiagpairs; i++, k += 2) {
      (*univdiagonals_gplus)[i] = diagpairs[k];
    }
  }
  FREE(diagpairs);
  *nunivdiagonals_gplus = ndiagpairs;


  /* minus */
#ifdef LARGE_GENOMES
  diagpairs = Intersect_approx_uint8(&ndiagpairs,
				     univdiagonals5_gminus,nunivdiagonals5_gminus,/*diagterm*/0,
				     all_univdiagonals3_gminus,all_nunivdiagonals3_gminus,/*diagterm*/0,
				     /*below_slop*/slop,/*above_slop*/0);
#else
  diagpairs = Intersect_approx_uint4(&ndiagpairs,
				     univdiagonals5_gminus,nunivdiagonals5_gminus,/*diagterm*/0,
				     all_univdiagonals3_gminus,all_nunivdiagonals3_gminus,/*diagterm*/0,
				     /*below_slop*/slop,/*above_slop*/0);
#endif

  if (ndiagpairs == 0) {
    *univdiagonals_gminus = (Univcoord_T *) NULL;
  } else {
    MALLOC_ALIGN(*univdiagonals_gminus, ndiagpairs * sizeof(Univcoord_T));
    for (i = 0, /*for 3'*/k = 1; i < ndiagpairs; i++, k += 2) {
      (*univdiagonals_gminus)[i] = diagpairs[k];
    }
  }
  FREE(diagpairs);
  *nunivdiagonals_gminus = ndiagpairs;


#ifdef DEBUG
  printf("Kmer_anchored_3 returning %d plus and %d minus hits\n",
	 (*nunivdiagonals_gplus),(*nunivdiagonals_gminus));

  for (i = 0; i < *nunivdiagonals_gplus; i++) {
    printf("plus %u\n",(*univdiagonals_gplus)[i]);
  }
  for (i = 0; i < *nunivdiagonals_gminus; i++) {
    printf("minus %u\n",(*univdiagonals_gminus)[i]);
  }
#endif

  return (*nunivdiagonals_gplus) + (*nunivdiagonals_gminus);
}


void
Kmer_exact_paths (bool *any_imperfect_ends_p, int *found_score,
		  List_T *sense_paths_gplus, List_T *sense_paths_gminus,
		  List_T *antisense_paths_gplus, List_T *antisense_paths_gminus,

		  List_T exact_univdiagonals_gplus, Intlist_T exact_nunivdiagonals_gplus,
		  List_T exact_univdiagonals_gminus, Intlist_T exact_nunivdiagonals_gminus,

		  Compress_T query_compress_fwd, Compress_T query_compress_rev,
		  Shortread_T queryseq, char *queryuc_ptr, char *queryrc, int querylength,
		  int genestrand, Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
		  Univcoordlistpool_T univcoordlistpool,
		  Listpool_T listpool, Pathpool_T pathpool, Vectorpool_T vectorpool,
		  Hitlistpool_T hitlistpool, Transcriptpool_T transcriptpool,
		  Method_T method) {
  List_T p;
  Intlist_T q;
  Univcoord_T *univdiagonals;
  int nunivdiagonals, i;

  int pos5, pos3;
  Chrnum_T chrnum;
  Univcoord_T chroffset, chrhigh;

  debug(printf("Entered Kmer_exact_paths with found score %d\n",*found_score));

  /* gplus */
  for (p = exact_univdiagonals_gplus, q = exact_nunivdiagonals_gplus; p != NULL;
       p = List_next(p), q = Intlist_next(q)) {
    univdiagonals = (Univcoord_T *) List_head(p);
    nunivdiagonals = Intlist_head(q);

    for (i = 0; i < nunivdiagonals; i++) {
      debug(printf("ULTRAFAST PLUS DIAGONAL %u\n",univdiagonals[i]));
      if (univdiagonals[i] < (Univcoord_T) querylength) {
	/* Skip */
      } else {
	chrnum = EF64_chrnum(&chroffset,&chrhigh,chromosome_ef64,
			     univdiagonals[i] - querylength,univdiagonals[i]);

#ifdef TRIM_AT_CHROMOSOME_BOUNDS
	pos5 = (univdiagonals[i] >= chroffset + (Univcoord_T) querylength) ? 0 : (int) (chroffset - left);
	pos3 = (univdiagonals[i] <= chrhigh) ? querylength : (int) (chrhigh - left);
#else
	pos5 = 0;
	pos3 = querylength;
#endif

	if (Path_solve_exact(&(*found_score),&(*sense_paths_gplus),&(*antisense_paths_gplus),
			     univdiagonals[i],pos5,pos3,/*plusp*/true,genestrand,
			     /*query_compress*/query_compress_fwd,
			     query_compress_fwd,query_compress_rev,
			     queryseq,queryuc_ptr,queryrc,querylength,
			     chrnum,chroffset,chrhigh,
			     intlistpool,uintlistpool,univcoordlistpool,
			     listpool,pathpool,vectorpool,hitlistpool,transcriptpool,
			     method) == false) {
	  *any_imperfect_ends_p = true;
	}
      }
    }
  }


  /* gminus */
  for (p = exact_univdiagonals_gminus, q = exact_nunivdiagonals_gminus; p != NULL;
       p = List_next(p), q = Intlist_next(q)) {
    univdiagonals = (Univcoord_T *) List_head(p);
    nunivdiagonals = Intlist_head(q);

    for (i = 0; i < nunivdiagonals; i++) {
      debug(printf("ULTRAFAST MINUS DIAGONAL %u\n",univdiagonals[i]));
      if (univdiagonals[i] < (Univcoord_T) querylength) {
	/* Skip */
      } else {
	chrnum = EF64_chrnum(&chroffset,&chrhigh,chromosome_ef64,
			     univdiagonals[i] - querylength,univdiagonals[i]);
	    
#ifdef TRIM_AT_CHROMOSOME_BOUNDS
	pos5 = (univdiagonals[i] >= chroffset + (Univcoord_T) querylength) ? 0 : (int) (chroffset - left);
	pos3 = (univdiagonals[i] <= chrhigh) ? querylength : (int) (chrhigh - left);
#else
	pos5 = 0;
	pos3 = querylength;
#endif

	if (Path_solve_exact(&(*found_score),&(*sense_paths_gminus),&(*antisense_paths_gminus),
			     univdiagonals[i],pos5,pos3,/*plusp*/false,genestrand,
			     /*query_compress*/query_compress_rev,
			     query_compress_fwd,query_compress_rev,
			     queryseq,queryuc_ptr,queryrc,querylength,
			     chrnum,chroffset,chrhigh,
			     intlistpool,uintlistpool,univcoordlistpool,
			     listpool,pathpool,vectorpool,hitlistpool,transcriptpool,
			     method) == false) {
	  *any_imperfect_ends_p = true;
	}
      }
    }
  }

  debug(printf("Kmer_exact_paths returning sense %d plus and %d minus paths, antisense %d plus and %d minus paths\n",
		List_length(*sense_paths_gplus),List_length(*sense_paths_gminus),
		List_length(*antisense_paths_gplus),List_length(*antisense_paths_gminus)));

  return;
}


#if 0
/* univdiagonal0/left0 is for the beginning of the query; univdiagonal1/left1 is for the end */
static void
combine_ends_plus (int *found_score,
		   List_T *unsolved_sense_paths, List_T *unsolved_antisense_paths,
		   List_T *sense_paths, List_T *antisense_paths,
		   Univcoord_T univdiagonal0, Univcoord_T univdiagonal1,
		   int pos5_0, int pos3_0, int pos5_1, int pos3_1,

		   Compress_T query_compress, Compress_T query_compress_fwd, Compress_T query_compress_rev,
		   Shortread_T queryseq, int querylength,
		   Chrnum_T chrnum, Univcoord_T chroffset, Univcoord_T chrhigh,

		   Indelinfo_T indelinfo, Spliceinfo_T spliceinfo, Knownsplicing_T knownsplicing,

		   int genestrand, int max_insertionlen, int max_deletionlen,
		   Intlistpool_T intlistpool, Uintlistpool_T uintlistpool, Univcoordlistpool_T univcoordlistpool,
		   Listpool_T listpool, Pathpool_T pathpool, Vectorpool_T vectorpool, Transcriptpool_T transcriptpool,
		   Hitlistpool_T hitlistpool, Method_T method) {

  debug(printf("Entered combine_ends_plus with univdiagonal0 %u and univdiagonal1 %u\n",
		univdiagonal0,univdiagonal1));

  /* Called checked for adj == 0 */
  assert(univdiagonal1 != univdiagonal0);

  if (univdiagonal0 < chroffset) {
    /* Skip.  Straddles chromosome bounds */
    debug(printf("Skipping because univdiagonal0 %u < chroffset %u\n",
		 univdiagonal0,chroffset));
    
  } else if (univdiagonal1 - querylength >= chrhigh) {
    /* Skip.  Straddles chromosome bounds */
    debug(printf("Skipping because univdiagonal1 %u - querylength %d >= chrhigh %u\n",
		 univdiagonal1,querylength,chrhigh));

  } else {
    Path_solve_from_ends(&(*found_score),&(*unsolved_sense_paths),&(*sense_paths),
			 univdiagonal0,pos5_0,pos3_0,univdiagonal1,pos5_1,pos3_1,
			 /*plusp*/true,/*sensedir*/SENSE_FORWARD,genestrand,
			 query_compress,query_compress_fwd,query_compress_rev,
			 queryseq,querylength,chrnum,chroffset,chrhigh,
			 indelinfo,spliceinfo,knownsplicing,
			 intlistpool,uintlistpool,univcoordlistpool,
			 listpool,pathpool,vectorpool,transcriptpool,
			 hitlistpool,max_insertionlen,max_deletionlen,method);

    if (splicingp == true) {
      Path_solve_from_ends(&(*found_score),&(*unsolved_antisense_paths),&(*antisense_paths),
			   univdiagonal0,pos5_0,pos3_0,univdiagonal1,pos5_1,pos3_1,
			   /*plusp*/true,/*sensedir*/SENSE_ANTI,genestrand,
			   query_compress,query_compress_fwd,query_compress_rev,
			   queryseq,querylength,chrnum,chroffset,chrhigh,
			   indelinfo,spliceinfo,knownsplicing,
			   intlistpool,uintlistpool,univcoordlistpool,
			   listpool,pathpool,vectorpool,transcriptpool,
			   hitlistpool,max_insertionlen,max_deletionlen,method);
    }
  }

  return;
}
#endif


#if 0
/* Genomic procedure */
/* left0 is for the beginning of the query; left1 is for the end */
static void
combine_ends_minus (int *found_score,
		    List_T *unsolved_sense_paths, List_T *unsolved_antisense_paths,
		    List_T *sense_paths, List_T *antisense_paths,
		    Univcoord_T univdiagonal0, Univcoord_T univdiagonal1,
		    int pos5_0, int pos3_0, int pos5_1, int pos3_1,

		    Compress_T query_compress, Compress_T query_compress_fwd, Compress_T query_compress_rev,
		    Shortread_T queryseq, int querylength,
		    Chrnum_T chrnum, Univcoord_T chroffset, Univcoord_T chrhigh,

		    Indelinfo_T indelinfo, Spliceinfo_T spliceinfo, Knownsplicing_T knownsplicing,

		    int genestrand, int max_insertionlen, int max_deletionlen,
		    Intlistpool_T intlistpool, Uintlistpool_T uintlistpool, Univcoordlistpool_T univcoordlistpool,
		    Listpool_T listpool, Pathpool_T pathpool, Vectorpool_T vectorpool, Transcriptpool_T transcriptpool,
		    Hitlistpool_T hitlistpool, Method_T method) {

  debug(printf("Entered combine_ends_minus with univdiagonal %u and univdiagonal %u\n",
	       univdiagonal0,univdiagonal1));

  /* Caller checked for adj == 0 */
  assert(univdiagonal1 != univdiagonal0);

  if (univdiagonal0 < chroffset) {
    /* Skip.  Straddles chromosome bounds */
    debug(printf("Skipping because univdiagonal0 %u < chroffset %u\n",
		 univdiagonal0,chroffset));
    
  } else if (univdiagonal1 - querylength >= chrhigh) {
    /* Skip.  Straddles chromosome bounds */
    debug(printf("Skipping because univdiagonal1 %u - querylength %d >= chrhigh %u\n",
		 univdiagonal1,querylength,chrhigh));

  } else {
    Path_solve_from_ends(&(*found_score),&(*unsolved_sense_paths),&(*sense_paths),
			 univdiagonal0,pos5_0,pos3_0,univdiagonal1,pos5_1,pos3_1,
			 /*plusp*/false,/*sensedir*/SENSE_FORWARD,genestrand,
			 query_compress,query_compress_fwd,query_compress_rev,
			 queryseq,querylength,chrnum,chroffset,chrhigh,
			 indelinfo,spliceinfo,knownsplicing,
			 intlistpool,uintlistpool,univcoordlistpool,
			 listpool,pathpool,vectorpool,transcriptpool,
			 hitlistpool,max_insertionlen,max_deletionlen,method);

    if (splicingp == true) {
      Path_solve_from_ends(&(*found_score),&(*unsolved_antisense_paths),&(*antisense_paths),
			   univdiagonal0,pos5_0,pos3_0,univdiagonal1,pos5_1,pos3_1,
			   /*plusp*/false,/*sensedir*/SENSE_ANTI,genestrand,
			   query_compress,query_compress_fwd,query_compress_rev,
			   queryseq,querylength,chrnum,chroffset,chrhigh,
			   indelinfo,spliceinfo,knownsplicing,
			   intlistpool,uintlistpool,univcoordlistpool,
			   listpool,pathpool,vectorpool,transcriptpool,
			   hitlistpool,max_insertionlen,max_deletionlen,method);
    }
  }

  return;
}
#endif


#if 0
/* Works well if there are only a few positions to check, but generally slow */
void
Kmer_search_end (int *found_score,

		 List_T *sense_paths_gplus, List_T *sense_paths_gminus,
		 List_T *antisense_paths_gplus, List_T *antisense_paths_gminus,

		 Stage1_T stage1, Shortread_T queryseq,
		 char *queryuc_ptr, char *queryrc, int querylength,
		 Compress_T query_compress_fwd, Compress_T query_compress_rev,
		 Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
		 Univcoordlistpool_T univcoordlistpool,
		 Listpool_T listpool, Pathpool_T pathpool, Vectorpool_T vectorpool,
		 Hitlistpool_T hitlistpool, Transcriptpool_T transcriptpool,
		 Method_T method) {
		 
#ifdef LARGE_GENOMES
  unsigned char *positions_high;
  UINT4 *positions;
#else
  Univcoord_T *positions;
#endif
  Univcoord_T univdiagonal;
  int npositions, i;
  int diagterm;
  int mod;


#if 0
  /* Previously limited this method for RNA-seq data */
  int total_positions = 0;
  for (mod = 0; mod < index1interval; mod++) {
    total_positions += stage1->plus_npositions_end5[mod];
    total_positions += stage1->minus_npositions_end5[mod];
    total_positions += stage1->plus_npositions_end3[mod];
    total_positions += stage1->minus_npositions_end3[mod];
  }
  if (total_positions > max_positions) {
    return;
  }
#endif


  for (mod = 0; mod < index1interval; mod++) {
    /* qstart (5'), gplus */
#ifdef LARGE_GENOMES
    positions_high = stage1->plus_positions_high_end5[mod];
#endif
    positions = stage1->plus_positions_end5[mod];
    npositions = stage1->plus_npositions_end5[mod];
    diagterm = stage1->plus_diagterms_end5[mod];

    for (i = 0; i < npositions; i++) {
#ifdef LARGE_GENOMES
      univdiagonal = GETPOS(positions[i],positions[i]) + (Univcoord_T) diagterm;
#else
      univdiagonal = positions[i] + (Univcoord_T) diagterm;
#endif
      if (univdiagonal /*+qstart:0*/ < (Univcoord_T) querylength) {
	/* Skip */
      } else {
	Path_solve_from_qstart(&(*found_score),
			       &(*sense_paths_gplus),&(*antisense_paths_gplus),
			       univdiagonal,mod,
			       queryseq,/*queryptr*/queryuc_ptr,
			       /*query_compress*/query_compress_fwd,
			       query_compress_fwd,query_compress_rev,
			       /*plusp*/true,querylength,
			       intlistpool,uintlistpool,univcoordlistpool,
			       listpool,pathpool,vectorpool,hitlistpool,
			       transcriptpool,method);
      }
    }

    /* qend, gminus */
#ifdef LARGE_GENOMES
    positions_high = stage1->minus_positions_high_end5[mod];
#endif
    positions = stage1->minus_positions_end5[mod];
    npositions = stage1->minus_npositions_end5[mod];
    diagterm = stage1->minus_diagterms_end5[mod];

    for (i = 0; i < npositions; i++) {
#ifdef LARGE_GENOMES
      univdiagonal = GETPOS(positions[i],positions[i]) + (Univcoord_T) diagterm;
#else
      univdiagonal = positions[i] + (Univcoord_T) diagterm;
#endif
      if (univdiagonal /*+qstart:0*/ < (Univcoord_T) querylength) {
	/* Skip */
      } else {
	Path_solve_from_qend(&(*found_score),
			     &(*sense_paths_gminus),&(*antisense_paths_gminus),
			     univdiagonal,mod,
			     queryseq,/*queryptr*/queryrc,
			     /*query_compress*/query_compress_rev,
			     query_compress_fwd,query_compress_rev,
			     /*plusp*/false,querylength,
			     intlistpool,uintlistpool,univcoordlistpool,
			     listpool,pathpool,vectorpool,hitlistpool,
			     transcriptpool,method);
      }
    }
  }


  for (mod = 0; mod < index1interval; mod++) {
    /* qend, gplus */
#ifdef LARGE_GENOMES
    positions_high = stage1->plus_positions_high_end3[mod];
#endif
    positions = stage1->plus_positions_end3[mod];
    npositions = stage1->plus_npositions_end3[mod];
    diagterm = stage1->plus_diagterms_end3[mod];

    for (i = 0; i < npositions; i++) {
#ifdef LARGE_GENOMES
      univdiagonal = GETPOS(positions[i],positions[i]) + (Univcoord_T) diagterm;
#else
      univdiagonal = positions[i] + (Univcoord_T) diagterm;
#endif
      if (univdiagonal /*+qstart:0*/ < (Univcoord_T) querylength) {
	/* Skip */
      } else {
	Path_solve_from_qend(&(*found_score),
			     &(*sense_paths_gplus),&(*antisense_paths_gplus),
			     univdiagonal,mod,
			     queryseq,/*queryptr*/queryuc_ptr,
			     /*query_compress*/query_compress_fwd,
			     query_compress_fwd,query_compress_rev,
			     /*plusp*/true,querylength,
			     intlistpool,uintlistpool,univcoordlistpool,
			     listpool,pathpool,vectorpool,hitlistpool,
			     transcriptpool,method);
      }
    }


    /* qstart gminus */
#ifdef LARGE_GENOMES
    positions_high = stage1->minus_positions_high_end3[mod];
#endif
    positions = stage1->minus_positions_end3[mod];
    npositions = stage1->minus_npositions_end3[mod];
    diagterm = stage1->minus_diagterms_end3[mod];

    for (i = 0; i < npositions; i++) {
#ifdef LARGE_GENOMES
      univdiagonal = GETPOS(positions[i],positions[i]) + (Univcoord_T) diagterm;
#else
      univdiagonal = positions[i] + (Univcoord_T) diagterm;
#endif
      if (univdiagonal /*+qstart:0*/ < (Univcoord_T) querylength) {
	/* Skip */
      } else {
	Path_solve_from_qstart(&(*found_score),
			       &(*sense_paths_gminus),&(*antisense_paths_gminus),
			       univdiagonal,mod,
			       queryseq,/*queryptr*/queryrc,
			       /*query_compress*/query_compress_rev,
			       query_compress_fwd,query_compress_rev,
			       /*plusp*/false,querylength,
			       intlistpool,uintlistpool,univcoordlistpool,
			       listpool,pathpool,vectorpool,hitlistpool,
			       transcriptpool,method);
      }
    }
  }

  return;
}
#endif


#if 0
/* Can be very slow in some cases, so no longer using this method */
/* Performs a merge.  About 10% faster than the alternative of trying all combinations of mod5 and mod3. */
/* Need max_hits, because repetitive reads can give many exact matches in a genome */
/* Approx distance can cover a typical splice distance */
void
Kmer_search_approx (int *found_score,
		    List_T *unsolved_sense_paths_gplus, List_T *unsolved_sense_paths_gminus,
		    List_T *unsolved_antisense_paths_gplus, List_T *unsolved_antisense_paths_gminus,

		    List_T *sense_paths_gplus, List_T *sense_paths_gminus,
		    List_T *antisense_paths_gplus, List_T *antisense_paths_gminus,
				
		    Compress_T query_compress_fwd, Compress_T query_compress_rev,
		    Shortread_T queryseq, int querylength,
		    Stage1_T stage1, Knownsplicing_T knownsplicing,
		    int genestrand, int max_insertionlen, int max_deletionlen, int max_gaplen, int sizelimit,

		    Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
		    Univcoordlistpool_T univcoordlistpool,
		    Listpool_T listpool, Pathpool_T pathpool, Vectorpool_T vectorpool,
		    Hitlistpool_T hitlistpool, Transcriptpool_T transcriptpool, Method_T method) {

  int gplus_streami_5 = 0, gminus_streami_5 = 0, gplus_streami_3 = 0, gminus_streami_3 = 0;

  int total_npositions_gplus_5 = 0, total_npositions_gminus_5 = 0,
    total_npositions_gplus_3 = 0, total_npositions_gminus_3 = 0;

  Univcoord_T *gplus_diagonals_5, *gminus_diagonals_5, *gplus_diagonals_3, *gminus_diagonals_3;
  int n_gplus_diagonals_5, n_gminus_diagonals_5, n_gplus_diagonals_3, n_gminus_diagonals_3;

  Univcoord_T *gplus_diagpairs, *gminus_diagpairs;
  int n_gplus_diagpairs, n_gminus_diagpairs;

  int i, k;
  Univcoord_T univdiagonala, univdiagonalb, univdiagonal0, univdiagonal1, low, high;

  int pos5_0, pos3_0, pos5_1, pos3_1, pos5a, pos3a, pos5b, pos3b, adj;
  int trim_a5, trim_a3, trim_b5, trim_b3;
  int nmismatches_a5, nmismatches_a3, nmismatches_b5, nmismatches_b3;

  Chrnum_T chrnum;
  Univcoord_T chroffset, chrhigh;


  /* max_hits = 1000000; */

  for (i = 0; i < index1interval; i++) {
    if (stage1->plus_npositions_end5[i] > 0) {
#ifdef LARGE_GENOMES
      stage1->gplus_stream_high_array_5[gplus_streami_5] = stage1->plus_positions_high_end5[i];
      stage1->gplus_stream_low_array_5[gplus_streami_5] = stage1->plus_positions_end5[i];
#else
      stage1->gplus_stream_array_5[gplus_streami_5] = stage1->plus_positions_end5[i];
#endif
      stage1->gplus_streamsize_array_5[gplus_streami_5] = stage1->plus_npositions_end5[i];
      stage1->gplus_diagterm_array_5[gplus_streami_5] = stage1->plus_diagterms_end5[i];
      total_npositions_gplus_5 += stage1->plus_npositions_end5[i];
      gplus_streami_5++;
    }

    if (stage1->minus_npositions_end5[i] > 0) {
#ifdef LARGE_GENOMES
      stage1->gminus_stream_high_array_5[gminus_streami_5] = stage1->minus_positions_high_end5[i];
      stage1->gminus_stream_low_array_5[gminus_streami_5] = stage1->minus_positions_end5[i];
#else
      stage1->gminus_stream_array_5[gminus_streami_5] = stage1->minus_positions_end5[i];
#endif
      stage1->gminus_streamsize_array_5[gminus_streami_5] = stage1->minus_npositions_end5[i];
      stage1->gminus_diagterm_array_5[gminus_streami_5] = stage1->minus_diagterms_end5[i];
      total_npositions_gminus_5 += stage1->minus_npositions_end5[i];
      gminus_streami_5++;
    }

    if (stage1->plus_npositions_end3[i] > 0) {
#ifdef LARGE_GENOMES
      stage1->gplus_stream_high_array_3[gplus_streami_3] = stage1->plus_positions_high_end3[i];
      stage1->gplus_stream_low_array_3[gplus_streami_3] = stage1->plus_positions_end3[i];
#else
      stage1->gplus_stream_array_3[gplus_streami_3] = stage1->plus_positions_end3[i];
#endif
      stage1->gplus_streamsize_array_3[gplus_streami_3] = stage1->plus_npositions_end3[i];
      stage1->gplus_diagterm_array_3[gplus_streami_3] = stage1->plus_diagterms_end3[i];
      total_npositions_gplus_3 += stage1->plus_npositions_end3[i];
      gplus_streami_3++;
    }

    if (stage1->minus_npositions_end3[i] > 0) {
#ifdef LARGE_GENOMES
      stage1->gminus_stream_high_array_3[gminus_streami_3] = stage1->minus_positions_high_end3[i];
      stage1->gminus_stream_low_array_3[gminus_streami_3] = stage1->minus_positions_end3[i];
#else
      stage1->gminus_stream_array_3[gminus_streami_3] = stage1->minus_positions_end3[i];
#endif
      stage1->gminus_streamsize_array_3[gminus_streami_3] = stage1->minus_npositions_end3[i];
      stage1->gminus_diagterm_array_3[gminus_streami_3] = stage1->minus_diagterms_end3[i];
      total_npositions_gminus_3 += stage1->minus_npositions_end3[i];
      gminus_streami_3++;
    }
  }

  debug(printf("Comparing total_npositions_gplus_5 %d against sizelmit %d\n",total_npositions_gplus_5,sizelimit));
  if (0 && total_npositions_gplus_5 > sizelimit) {
    gplus_diagonals_5 = (Univcoord_T *) NULL;
    n_gplus_diagonals_5 = 0;
  } else {
#ifdef LARGE_GENOMES
    gplus_diagonals_5 = Merge_diagonals_large(&n_gplus_diagonals_5,
					      stage1->gplus_stream_high_array_5,stage1->gplus_stream_low_array_5,
					      stage1->gplus_streamsize_array_5,stage1->gplus_diagterm_array_5,
					      /*nstreams*/gplus_streami_5,stage1->mergeinfo);
#else
    gplus_diagonals_5 = Merge_diagonals(&n_gplus_diagonals_5,stage1->gplus_stream_array_5,
					stage1->gplus_streamsize_array_5,stage1->gplus_diagterm_array_5,
					/*nstreams*/gplus_streami_5,stage1->mergeinfo);
#endif
  }


  debug(printf("Comparing total_npositions_gminus_5 %d against sizelmit %d\n",total_npositions_gminus_5,sizelimit));
  if (0 && total_npositions_gminus_5 > sizelimit) {
    gminus_diagonals_5 = (Univcoord_T *) NULL;
    n_gminus_diagonals_5 = 0;
  } else {
#ifdef LARGE_GENOMES
    gminus_diagonals_5 = Merge_diagonals_large(&n_gminus_diagonals_5,
					       stage1->gminus_stream_high_array_5,stage1->gminus_stream_low_array_5,
					       stage1->gminus_streamsize_array_5,stage1->gminus_diagterm_array_5,
					       /*nstreams*/gminus_streami_5,stage1->mergeinfo);
#else
    gminus_diagonals_5 = Merge_diagonals(&n_gminus_diagonals_5,stage1->gminus_stream_array_5,
					 stage1->gminus_streamsize_array_5,stage1->gminus_diagterm_array_5,
					 /*nstreams*/gminus_streami_5,stage1->mergeinfo);
#endif
  }


  debug(printf("Comparing total_npositions_gplus_3 %d against sizelmit %d\n",total_npositions_gplus_3,sizelimit));
  if (0 && total_npositions_gplus_3 > sizelimit) {
    gplus_diagonals_3 = (Univcoord_T *) NULL;
    n_gplus_diagonals_3 = 0;
  } else {
#ifdef LARGE_GENOMES
    gplus_diagonals_3 = Merge_diagonals_large(&n_gplus_diagonals_3,
					      stage1->gplus_stream_high_array_3,stage1->gplus_stream_low_array_3,
					      stage1->gplus_streamsize_array_3,stage1->gplus_diagterm_array_3,
					      /*nstreams*/gplus_streami_3,stage1->mergeinfo);
#else
    gplus_diagonals_3 = Merge_diagonals(&n_gplus_diagonals_3,stage1->gplus_stream_array_3,
					stage1->gplus_streamsize_array_3,stage1->gplus_diagterm_array_3,
					/*nstreams*/gplus_streami_3,stage1->mergeinfo);
#endif

  }


  debug(printf("Comparing total_npositions_gminus_3 %d against sizelimit %d\n",total_npositions_gminus_3,sizelimit));
  if (0 && total_npositions_gminus_3 > sizelimit) {
    gminus_diagonals_3 = (Univcoord_T *) NULL;
    n_gminus_diagonals_3 = 0;
  } else {
#ifdef LARGE_GENOMES
    gminus_diagonals_3 = Merge_diagonals_large(&n_gminus_diagonals_3,
					       stage1->gminus_stream_high_array_3,stage1->gminus_stream_low_array_3,
					       stage1->gminus_streamsize_array_3,stage1->gminus_diagterm_array_3,
					       /*nstreams*/gminus_streami_3,stage1->mergeinfo);
#else
    gminus_diagonals_3 = Merge_diagonals(&n_gminus_diagonals_3,stage1->gminus_stream_array_3,
					 stage1->gminus_streamsize_array_3,stage1->gminus_diagterm_array_3,
					 /*nstreams*/gminus_streami_3,stage1->mergeinfo);
#endif
  }

#ifdef LARGE_GENOMES
  gplus_diagpairs = Intersect_approx_uint8(&n_gplus_diagpairs,
					   /*set1*/gplus_diagonals_5,/*length1*/n_gplus_diagonals_5,/*diagterm1*/0,
					   /*set2*/gplus_diagonals_3,/*length2*/n_gplus_diagonals_3,/*diagterm2*/0,
					    /*maxdistance*/max_gaplen);
  gminus_diagpairs = Intersect_approx_uint8(&n_gminus_diagpairs,
					    /*set1*/gminus_diagonals_5,/*length1*/n_gminus_diagonals_5,/*diagterm1*/0,
					    /*set2*/gminus_diagonals_3,/*length2*/n_gminus_diagonals_3,/*diagterm2*/0,
					    /*maxdistance*/max_gaplen);
#else
  gplus_diagpairs = Intersect_approx_uint4(&n_gplus_diagpairs,
					   /*set1*/gplus_diagonals_5,/*length1*/n_gplus_diagonals_5,/*diagterm1*/0,
					   /*set2*/gplus_diagonals_3,/*length2*/n_gplus_diagonals_3,/*diagterm2*/0,
					   /*below_slop*/max_insertionlen,/*above_slop*/max_gaplen);
  gminus_diagpairs = Intersect_approx_uint4(&n_gminus_diagpairs,
					    /*set1*/gminus_diagonals_5,/*length1*/n_gminus_diagonals_5,/*diagterm1*/0,
					    /*set2*/gminus_diagonals_3,/*length2*/n_gminus_diagonals_3,/*diagterm2*/0,
					    /*below_slop*/max_gaplen,/*above_slop*/max_insertionlen);
#endif

  debug(printf("***Intersect ends approx: exactp %d and %d.  %d plus and %d minus diagpairs\n",
		gplus_exactp,gminus_exactp,n_gplus_diagpairs,n_gminus_diagpairs));

#if !defined(LARGE_GENOMES) || defined(HAVE_AVX512) || defined(HAVE_AVX2)
  FREE_ALIGN(gplus_diagonals_5);
  FREE_ALIGN(gminus_diagonals_5);
  FREE_ALIGN(gplus_diagonals_3);
  FREE_ALIGN(gminus_diagonals_3);
#else
  FREE(gplus_diagonals_5);
  FREE(gminus_diagonals_5);
  FREE(gplus_diagonals_3);
  FREE(gminus_diagonals_3);
#endif

  i = k = 0;
  while (/*nhits <= max_hits && */ i < n_gplus_diagpairs) {
    univdiagonala = gplus_diagpairs[k];
    univdiagonalb = gplus_diagpairs[k+1];

#ifdef TRIM_AT_CHROMOSOME_BOUNDS
    pos5a = (univdiagonala >= chroffset + (Univcoord_T) querylength) ? 0 : (int) (chroffset - lefta);
    pos3a = (univdiagonala <= chrhigh) ? querylength : (int) (chrhigh - lefta);
#else
    pos5a = 0;
    pos3a = querylength;
#endif
    trim_a5 = Genomebits_first_kmer_left(&nmismatches_a5,genomebits,query_compress_fwd,
					 univdiagonala,querylength,pos5a,pos3a,
					 /*plusp*/true,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
    trim_a3 = querylength - Genomebits_first_kmer_right(&nmismatches_a3,genomebits,query_compress_fwd,
							univdiagonala,querylength,pos5a,pos3a,
							/*plusp*/true,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
      
#ifdef TRIM_AT_CHROMOSOME_BOUNDS
    pos5b = (univdiagonalb >= chroffset + (Univcoord_T) querylength) ? 0 : (int) (chroffset - leftb);
    pos3b = (univdiagonalb <= chrhigh) ? querylength : (int) (chrhigh - leftb);
#else
    pos5b = 0;
    pos3b = querylength;
#endif
    trim_b5 = Genomebits_first_kmer_left(&nmismatches_b5,genomebits,query_compress_fwd,
					 univdiagonalb,querylength,pos5b,pos3b,
					 /*plusp*/true,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
    trim_b3 = querylength - Genomebits_first_kmer_right(&nmismatches_b3,genomebits,query_compress_fwd,
							univdiagonalb,querylength,pos5b,pos3b,
							/*plusp*/true,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
      
    debug(printf("trimmed a %d..%d, trimmed b %d..%d\n",
		 trim_a5,querylength - trim_a3,trim_b5,querylength - trim_b3));
    if (trim_a5 == pos5a || trim_b3 == pos3b) {
      univdiagonal0 = univdiagonala; pos5_0 = pos5a; pos3_0 = pos3a;
      univdiagonal1 = univdiagonalb; pos5_1 = pos5b; pos3_1 = pos3b;
      /* low0 = lefta + pos5a; high0 = lefta + pos3a; */
      /* low1 = leftb + pos5b; high1 = leftb + pos3b; */
      adj = (int) (univdiagonalb - univdiagonala);
      debug(printf("Setting a first, b second, adj %d\n",adj));
	
    } else if (trim_a3 == pos3a || trim_b5 == pos5b) {
      univdiagonal0 = univdiagonalb; pos5_0 = pos5b; pos3_0 = pos3b;
      univdiagonal1 = univdiagonala; pos5_1 = pos5a; pos3_1 = pos3a;
      /* low0 = leftb + pos5b; high0 = leftb + pos3b; */
      /* low1 = lefta + pos5a; high1 = lefta + pos3a; */
      adj = (int) (univdiagonala - univdiagonalb);
      debug(printf("Setting b first, a second, adj %d\n",adj));
	
    } else {
      adj = 0;
    }
	  
    /* printf("plus adj %d, univdiagonal0 %llu, univdiagonal1 %llu\n",adj,univdiagonal0,univdiagonal1); */
    if (adj == 0) {
      /* Skip, since would have been found by Kmer exact */

    } else if (adj > max_gaplen) {
      /* Too far apart */

    } else if (adj < -max_insertionlen) {
      /* Impossible */

    } else {
      /* Doesn't matter if univdiagonal0 is less than or greater than univdiagonal1 */
#ifdef TRIM_AT_GENOME_BOUNDS
      low = (univdiagonal0 >= (Univcoord_T) querylength) ? univdiagonal0 - querylength : 0;
      high = (univdiagonal1 <= genomelength) ? univdiagonal1 : genomelength;
#endif
#ifdef NEED_CHRNUM
      chrnum = EF64_chrnum(&chroffset,&chrhigh,chromosome_ef64,low,high);
#endif

      combine_ends_plus(&(*found_score),
			&(*unsolved_sense_paths_gplus),&(*unsolved_antisense_paths_gplus),
			&(*sense_paths_gplus),&(*antisense_paths_gplus),
			univdiagonal0,univdiagonal1,pos5_0,pos3_0,pos5_1,pos3_1,
			/*query_compress*/query_compress_fwd,query_compress_fwd,query_compress_rev,
			queryseq,querylength,chrnum,chroffset,chrhigh,

			stage1->indelinfo,stage1->spliceinfo,knownsplicing,
			genestrand,max_insertionlen,max_deletionlen,
			intlistpool,uintlistpool,univcoordlistpool,listpool,
			pathpool,vectorpool,transcriptpool,hitlistpool,method);
    }

    i++;
    k += 2;
  }
      

  i = k = 0;
  while (/*nhits <= max_hits && */ i < n_gminus_diagpairs) {
    univdiagonala = gminus_diagpairs[k];
    univdiagonalb = gminus_diagpairs[k+1];

#ifdef TRIM_AT_CHROMOSOME_BOUNDS
    pos5a = (univdiagonala >= chroffset + (Univcoord_T) querylength) ? 0 : (int) (chroffset - lefta);
    pos3a = (univdiagonala <= chrhigh) ? querylength : (int) (chrhigh - lefta);
#else
    pos5a = 0;
    pos3a = querylength;
#endif
    trim_a5 = Genomebits_first_kmer_left(&nmismatches_a5,genomebits,query_compress_rev,
					 univdiagonala,querylength,pos5a,pos3a,
					 /*plusp*/false,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
    trim_a3 = querylength - Genomebits_first_kmer_right(&nmismatches_a3,genomebits,query_compress_rev,
							univdiagonala,querylength,pos5a,pos3a,
							/*plusp*/false,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
      
#ifdef TRIM_AT_CHROMOSOME_BOUNDS
    pos5b = (univdiagonalb >= chroffset + (Univcoord_T) querylength) ? 0 : (int) (chroffset - leftb);
    pos3b = (univdiagonalb <= chrhigh) ? querylength : (int) (chrhigh - leftb);
#else
    pos5b = 0;
    pos3b = querylength;
#endif
    trim_b5 = Genomebits_first_kmer_left(&nmismatches_b5,genomebits,query_compress_rev,
					 univdiagonalb,querylength,pos5b,pos3b,
					 /*plusp*/false,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
    trim_b3 = querylength - Genomebits_first_kmer_right(&nmismatches_b3,genomebits,query_compress_rev,
							univdiagonalb,querylength,pos5b,pos3b,
							/*plusp*/false,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/index1part);
      
    debug(printf("trimmed a %d..%d, trimmed b %d..%d\n",
		 trim_a5,querylength - trim_a3,trim_b5,querylength - trim_b3));
    if (trim_a5 == pos5a || trim_b3 == pos3b) {
      univdiagonal0 = univdiagonala; pos5_0 = pos5a; pos3_0 = pos3a;
      univdiagonal1 = univdiagonalb; pos5_1 = pos5b; pos3_1 = pos3b;
      /* low0 = lefta + pos5a; high0 = lefta + pos3a; */
      /* low1 = leftb + pos5b; high1 = leftb + pos3b; */
      adj = (int) (univdiagonalb - univdiagonala);
      debug(printf("Setting a first, b second, adj %d\n",adj));
	
    } else if (trim_a3 == pos3a || trim_b5 == pos5b) {
      univdiagonal0 = univdiagonalb; pos5_0 = pos5b; pos3_0 = pos3b;
      univdiagonal1 = univdiagonala; pos5_1 = pos5a; pos3_1 = pos3a;
      /* low0 = leftb + pos5b; high0 = leftb + pos3b; */
      /* low1 = lefta + pos5a; high1 = lefta + pos3a; */
      adj = (int) (univdiagonala - univdiagonalb);
      debug(printf("Setting b first, a second, adj %d\n",adj));
	
    } else {
      adj = 0;
    }
	  
    /* printf("minus adj %d, univdiagonal0 %llu, univdiagonal1 %llu\n",adj,univdiagonal0,univdiagonal1); */
    if (adj == 0) {
      /* Skip, since would have been found by Kmer exact */

    } else if (adj > max_gaplen) {
      /* Too far apart */

    } else if (adj < -max_insertionlen) {
      /* Impossible */

    } else {
      /* Doesn't matter if univdiagonal1 is less than or greater than univdiagonal0 */
#ifdef TRIM_AT_GENOME_BOUNDS
      low = (univdiagonal0 >= (Univcoord_T) querylength) ? univdiagonal0 - querylength : 0;
      high = (univdiagonal1 <= genomelength) ? univdiagonal1 : genomelength;
#endif
#ifdef NEED_CHRNUM
      chrnum = EF64_chrnum(&chroffset,&chrhigh,chromosome_ef64,low,high);
#endif

      combine_ends_minus(&(*found_score),
			 &(*unsolved_sense_paths_gminus),&(*unsolved_antisense_paths_gminus),
			 &(*sense_paths_gminus),&(*antisense_paths_gminus),
			 univdiagonal0,univdiagonal1,pos5_0,pos3_0,pos5_1,pos3_1,
			 /*query_compress*/query_compress_rev,query_compress_fwd,query_compress_rev,
			 queryseq,querylength,chrnum,chroffset,chrhigh,

			 stage1->indelinfo,stage1->spliceinfo,knownsplicing,
			 genestrand,max_insertionlen,max_deletionlen,
			 intlistpool,uintlistpool,univcoordlistpool,listpool,
			 pathpool,vectorpool,transcriptpool,hitlistpool,method);
    }

    i++;
    k += 2;
  }

  FREE(gminus_diagpairs);
  FREE(gplus_diagpairs);

  debug(printf("Done with Kmer_search_approx\n"));
  return;
}
#endif


/* Copied from transcriptome-search.c */
/* Overwrite values array */
/* Repetitive oligos can calse false indel diagonals */
static int
most_prevalent_univcoord (int *nloci, int **counts, Univcoord_T *values, int nvalues) {
  int max_count, count, *count_ptr;
  Univcoord_T *out, *ptr, *end, *first;

  assert(nvalues > 0);
  count_ptr = *counts = (int *) MALLOC(nvalues*sizeof(int));

  ptr = out = &(values[0]);	/* Reset */
  end = &(values[nvalues]);

  max_count = 1;
  while (ptr < end) {
    first = ptr;
    debug0(printf("DIAGONAL %u",*first));
    if (ptr + max_count - 1 >= end) {
      /* End of list fails */
      debug0(printf(" => Goes beyond end of list\n"));
      ptr = end;

    } else if (ptr[max_count - 1] != *first) {
      /* Fails */
      debug0(printf(" => Fails\n"));
      if (ptr[max_count - 1] != ptr[max_count - 2]) {
	/* Can jump immediately */
	ptr += max_count - 1;
      } else {
	/* Advance forward until we see a new value */
	while (ptr < end && *ptr == *first) {
	  ptr++;
	}
      }

    } else {
      /* Contender */
      ptr += max_count;
      while (ptr < end && *ptr == *first) {
	ptr++;
      }
      debug0(printf(" => Count %d, index %d => printing\n",ptr - first,first - values));
      if ((count = ptr - first) > max_count + SUBOPT) {
	out = &(values[0]);	/* Reset */
	count_ptr = *counts;	/* Reset */
      } else if (count > max_count) {
	max_count = count;
      }
      *out++ = *first;
      *count_ptr++ = count;
    }
  }

  *nloci = out - &(values[0]);
  return max_count;
}


/* Use oligos computed by Stage1_fill_all_oligos_gen */
/* Assigns own positions, so does not need a procedure for Stage1_fill_all_positions_gen */
void
Kmer_search_complete (int *found_score,
		      List_T *unextended_sense_paths_gplus, List_T *unextended_sense_paths_gminus,
		      List_T *unextended_antisense_paths_gplus, List_T *unextended_antisense_paths_gminus,

		      List_T *sense_paths_gplus, List_T *sense_paths_gminus,
		      List_T *antisense_paths_gplus, List_T *antisense_paths_gminus,

		      Stage1_T stage1, Shortread_T queryseq, char *queryuc_ptr, char *queryrc,
		      int querylength, int genestrand, 

		      Knownsplicing_T knownsplicing, Knownindels_T knownindels,

#ifdef LARGE_GENOMES
		      unsigned char **plus_stream_high_array, UINT4 **plus_stream_low_array,
		      unsigned char **minus_stream_high_array, UINT4 **minus_stream_low_array,
#else
		      Univcoord_T **plus_stream_array, Univcoord_T **minus_stream_array,
#endif
		      int *plus_streamsize_array, int *plus_diagterm_array,
		      int *minus_streamsize_array, int *minus_diagterm_array,

		      int *mismatch_positions_alloc, Univcoord_T *novel_diagonals_alloc, unsigned short *localdb_alloc,
		      
		      Compress_T query_compress_fwd, Compress_T query_compress_rev,
		      int max_insertionlen, int max_deletionlen,
		      Intlistpool_T intlistpool, Uintlistpool_T uintlistpool, Univcoordlistpool_T univcoordlistpool,
		      Listpool_T listpool, Pathpool_T pathpool,
		      Transcriptpool_T transcriptpool, Vectorpool_T vectorpool, Hitlistpool_T hitlistpool,
		      Spliceendsgen_T spliceendsgen, bool paired_end_p, bool first_read_p,
		      Method_T method, bool find_splices_p) {

  int max_count, max_plus_count, max_minus_count, *plus_counts, *minus_counts;

  Univcoord_T univdiagonal;
  Univcoord_T *plus_diagonals, *minus_diagonals;
  int n_plus_loci, n_minus_loci, n_plus_diagonals, n_minus_diagonals,
    total_ndiagonals_plus = 0, total_ndiagonals_minus = 0, i;
  
#ifdef LARGE_GENOMES
  unsigned char *positions_high;
  UINT4 *positions;
#else
  Univcoord_T *positions;
#endif
  int npositions;
  int plus_streami = 0, minus_streami = 0;

  int querypos, query_lastpos;

  Chrnum_T chrnum;
  Univcoord_T chroffset, chrhigh;


  query_lastpos = querylength - index1part;
  for (querypos = 0; querypos <= query_lastpos; querypos++) {
    if (stage1->validp[querypos] == false) {
      /* Skip */
    } else {
      if (stage1->plus_retrievedp[querypos] == true) {
#ifdef LARGE_GENOMES
	positions_high = stage1->plus_positions_high[querypos];
#endif
	positions = stage1->plus_positions[querypos];
	npositions = stage1->plus_npositions[querypos];
      } else {
#ifdef LARGE_GENOMES
	npositions = stage1->plus_npositions[querypos] =
	  Indexdb_largeptr(&stage1->plus_positions_high[querypos],&stage1->plus_positions[querypos],
			   indexdb,stage1->forward_oligos[querypos]);
	positions_high = stage1->plus_positions_high[querypos];
#else
	npositions = stage1->plus_npositions[querypos] =
	  Indexdb_ptr(&stage1->plus_positions[querypos],indexdb,stage1->forward_oligos[querypos]);
	positions = stage1->plus_positions[querypos];
#endif
	stage1->plus_retrievedp[querypos] = true;
      }
      if (npositions > 0) {
#ifdef LARGE_GENOMES
	plus_stream_high_array[plus_streami] = positions_high;
	plus_stream_low_array[plus_streami] = positions;
#else
	plus_stream_array[plus_streami] = positions;
#endif
	plus_streamsize_array[plus_streami] = npositions;
	plus_diagterm_array[plus_streami] = querylength - querypos;
	total_ndiagonals_plus += npositions;
	plus_streami++;
      }

      if (stage1->minus_retrievedp[querypos] == true) {
#ifdef LARGE_GENOMES
	positions_high = stage1->minus_positions_high[querypos];
#endif
	positions = stage1->minus_positions[querypos];
	npositions = stage1->minus_npositions[querypos];
      } else {
#ifdef LARGE_GENOMES
	npositions = stage1->minus_npositions[querypos] =
	  Indexdb_largeptr(&stage1->minus_positions_high[querypos],&stage1->minus_positions[querypos],
			   indexdb,stage1->revcomp_oligos[querypos]);
	positions_high = stage1->minus_positions_high[querypos];
#else
	npositions = stage1->minus_npositions[querypos] =
	  Indexdb_ptr(&stage1->minus_positions[querypos],indexdb,stage1->revcomp_oligos[querypos]);
#endif
	positions = stage1->minus_positions[querypos];
	stage1->minus_retrievedp[querypos] = true;
      }
      if (npositions > 0) {
#ifdef LARGE_GENOMES
	minus_stream_high_array[minus_streami] = positions_high;
	minus_stream_low_array[minus_streami] = positions;
#else
	minus_stream_array[minus_streami] = positions;
#endif
	minus_streamsize_array[minus_streami] = npositions;
	minus_diagterm_array[minus_streami] = querypos + index1part;
	total_ndiagonals_minus += npositions;
	minus_streami++;
      }
    }
  }

#ifdef LARGE_GENOMES
  plus_diagonals = Merge_diagonals_large(&n_plus_diagonals,plus_stream_high_array,plus_stream_low_array,
					 plus_streamsize_array,plus_diagterm_array,/*nstreams*/plus_streami,
					 stage1->mergeinfo);
#else
  plus_diagonals = Merge_diagonals(&n_plus_diagonals,plus_stream_array,
				   plus_streamsize_array,plus_diagterm_array,/*nstreams*/plus_streami,
				   stage1->mergeinfo);
#endif

  if (n_plus_diagonals == 0) {
    max_plus_count = 0;
    n_plus_loci = 0;
    plus_counts = (int *) NULL;
  } else {
    max_plus_count = most_prevalent_univcoord(&n_plus_loci,&plus_counts,plus_diagonals,n_plus_diagonals);
  }

#ifdef DEBUG9
  printf("max_plus_count: %d\n",max_plus_count);
  for (i = 0; i < n_plus_loci; i++) {
    printf("PLUS_LOCUS %u\n",plus_diagonals[i]);
  }
#endif

#ifdef LARGE_GENOMES
  minus_diagonals = Merge_diagonals_large(&n_minus_diagonals,minus_stream_high_array,minus_stream_low_array,
					  minus_streamsize_array,minus_diagterm_array,/*nstreams*/minus_streami,
					  stage1->mergeinfo);
#else
  minus_diagonals = Merge_diagonals(&n_minus_diagonals,minus_stream_array,
				    minus_streamsize_array,minus_diagterm_array,/*nstreams*/minus_streami,
				    stage1->mergeinfo);
#endif

  if (n_minus_diagonals == 0) {
    max_minus_count = 0;
    n_minus_loci = 0;
    minus_counts = (int *) NULL;
  } else {
    max_minus_count = most_prevalent_univcoord(&n_minus_loci,&minus_counts,minus_diagonals,n_minus_diagonals);
  }


#ifdef DEBUG9
  printf("max_minus_count: %d\n",max_minus_count);
  for (i = 0; i < n_minus_loci; i++) {
    printf("MINUS_LOCUS %u\n",minus_diagonals[i]);
  }
#endif


  if (max_plus_count > max_minus_count) {
    max_count = max_plus_count;
  } else {
    max_count = max_minus_count;
  }

  /* plusp == true */
  debug(printf("plus loci %d\n",n_plus_loci));
  if (max_plus_count < max_count - SUBOPT) {
    debug(printf("plus not good enough\n"));
  } else {
    for (i = 0; i < n_plus_loci; i++) {
      univdiagonal = plus_diagonals[i];
      if (univdiagonal /*+qstart:0*/ < (Univcoord_T) querylength) {
	/* Skip */
      } else if (plus_counts[i] < max_count - SUBOPT) {
	debug(printf("Skipping univdiagonal %u with %d counts\n",univdiagonal,plus_counts[i]));
      } else {
	debug(printf("Complete case 1: plus\n"));
	chrnum = EF64_chrnum(&chroffset,&chrhigh,chromosome_ef64,univdiagonal,univdiagonal+querylength);
	Path_solve_from_diagonals(&(*found_score),
				  &(*unextended_sense_paths_gplus),&(*unextended_antisense_paths_gplus),
				  &(*sense_paths_gplus),&(*antisense_paths_gplus),
				  /*middle_univdiagonal*/univdiagonal,
				  /*middle_univdiagonal_qstart*/0,/*middle_univdiagonal_qend*/querylength,
				  /*middle_nmismatches*/-1,/*qstart_univdiags*/NULL,/*qend_univdiags*/NULL,
				    
				  queryseq,/*queryptr*/queryuc_ptr,querylength,mismatch_positions_alloc,
				  novel_diagonals_alloc,localdb_alloc,stage1,knownsplicing,knownindels,
				    
				  /*query_compress*/query_compress_fwd,query_compress_fwd,query_compress_rev,
				  chrnum,chroffset,chrhigh,/*plusp*/true,genestrand,
				  max_insertionlen,max_deletionlen,/*localdb_nmismatches_allowed*/0,
				  /*overall_end_distance*/0,paired_end_p,first_read_p,
				  intlistpool,uintlistpool,univcoordlistpool,listpool,
				  pathpool,transcriptpool,vectorpool,hitlistpool,spliceendsgen,
				  method,find_splices_p);
      }
    }
  }
    
  FREE(plus_counts);
#if !defined(LARGE_GENOMES) || defined(HAVE_AVX512) || defined(HAVE_AVX2)
  FREE_ALIGN(plus_diagonals);
#else
  FREE(plus_diagonals);
#endif

  /* plusp == false */
  debug(printf("minus loci %d\n",n_minus_loci));
  if (max_minus_count < max_count - SUBOPT) {
    debug(printf("minus not good enough\n"));
  } else {
    for (i = 0; i < n_minus_loci; i++) {
      univdiagonal = minus_diagonals[i];
      if (univdiagonal /*+qstart:0*/ < (Univcoord_T) querylength) {
	/* Skip */
      } else if (minus_counts[i] < max_count - SUBOPT) {
	debug(printf("Skipping univdiagonal %u with %d counts\n",univdiagonal,minus_counts[i]));
      } else {
	debug(printf("Complete case 2: minus\n"));
	chrnum = EF64_chrnum(&chroffset,&chrhigh,chromosome_ef64,univdiagonal,univdiagonal+querylength);
	Path_solve_from_diagonals(&(*found_score),
				  &(*unextended_sense_paths_gminus),&(*unextended_antisense_paths_gminus),
				  &(*sense_paths_gminus),&(*antisense_paths_gminus),
				  /*middle_univdiagonal*/univdiagonal,
				  /*middle_univdiagonal_qstart*/0,/*middle_univdiagonal_qend*/querylength,
				  /*middle_nmismatches*/-1,/*qstart_univdiags*/NULL,/*qend_univdiags*/NULL,
				    
				  queryseq,/*queryptr*/queryrc,querylength,mismatch_positions_alloc,
				  novel_diagonals_alloc,localdb_alloc,stage1,knownsplicing,knownindels,
				    
				  /*query_compress*/query_compress_rev,query_compress_fwd,query_compress_rev,
				  chrnum,chroffset,chrhigh,/*plusp*/false,genestrand,
				  max_insertionlen,max_deletionlen,/*localdb_nmismatches_allowed*/0,
				  /*overall_end_distance*/0,paired_end_p,first_read_p,
				  intlistpool,uintlistpool,univcoordlistpool,listpool,
				  pathpool,transcriptpool,vectorpool,hitlistpool,spliceendsgen,
				  method,find_splices_p);
      }
    }
  }

  FREE(minus_counts);
#if !defined(LARGE_GENOMES) || defined(HAVE_AVX512) || defined(HAVE_AVX2)
  FREE_ALIGN(minus_diagonals);
#else
  FREE(minus_diagonals);
#endif

  return;
}


void
Kmer_search_setup (int index1part_in, int index1interval_in,
		   Indexdb_T indexdb_in, EF64_T chromosome_ef64_in,
		   Genomebits_T genomebits_in, Genomebits_T genomebits_alt_in,
		   Univcoord_T genomelength_in, bool splicingp_in) {

  index1part = index1part_in;
  index1interval = index1interval_in;
  indexdb = indexdb_in;

  chromosome_ef64 = chromosome_ef64_in;
  genomebits = genomebits_in;
  genomebits_alt = genomebits_alt_in;
  genomelength = genomelength_in;

  splicingp = splicingp_in;

  return;
}


