
#include "merge-search.h"

#include <stdio.h>
#include <stdlib.h>

#include "assert.h"
#include "univdiagdef.h"

#include "intlist.h"
#ifdef LARGE_GENOMES
#include "merge-diagonals-simd-uint8.h"
#include "intersect-indices-large.h"
#else
#include "merge-diagonals-simd-uint4.h"
#include "intersect-indices-small.h"
#endif
#include "path-solve.h"


static int index1part;
static EF64_T chromosome_ef64;


struct Record_T {
  int qmin;
  int qmax;

  Chrnum_T chrnum;
  Univcoord_T chroffset;
  Univcoord_T chrhigh;

  bool fwd_long_segment_p;
  int fwd_score;
  int fwd_segmenti;
  Intlist_T previ;

  bool rev_long_segment_p;
  int rev_score;
  int rev_segmenti;
  Intlist_T nexti;
};

typedef struct Record_T *Record_T;


static void
record_print (Record_T this) {
  printf("  #%d %d..%d fwd %d:%d %s  rev %d:%d %s.",
	 this->chrnum,this->qmin,this->qmax,
	 this->fwd_segmenti,this->fwd_score,Intlist_to_string(this->previ),
	 this->rev_segmenti,this->rev_score,Intlist_to_string(this->nexti));
  return;	 
}


static int
make_unique (Univcoord_T *diagonals, int ndiagonals) {
  int k = 0, i, j;

  i = 0;
  while (i < ndiagonals) {
    j = i + 1;
    while (j < ndiagonals && diagonals[j] == diagonals[i]) {
      j++;
    }
    diagonals[k++] = diagonals[i];
    
    i = j;
  }

  return k;
}


static void
assign_chrinfo (struct Record_T *records, Univcoord_T *diagonals, int n,
		int querylength) {
  int i = 0;
  Record_T recordi;
  Univcoord_T genomichigh, left;
  Chrnum_T chrnum;
  Univcoord_T chroffset, chrhigh;

  while (i < n && diagonals[i] + records[i].qmax < (Univcoord_T) querylength) {
    /* Before beginning of genome */
    recordi = &(records[i]);
    recordi->chrnum = 0;
    recordi->chroffset = 0;
    recordi->chrhigh = 0;
    i++;
  }

  chrnum = 1;
  EF64_chrbounds(&chroffset,&chrhigh,chromosome_ef64,/*chrnum*/1);
  while (i < n) {
    recordi = &(records[i]);

    left = diagonals[i] - querylength;
    genomichigh = left + recordi->qmax;

    if (genomichigh < chrhigh) {
      recordi->chrnum = chrnum;
      recordi->chroffset = chroffset;
      recordi->chrhigh = chrhigh;
    } else {
      chrnum = EF64_chrnum(&chroffset,&chrhigh,chromosome_ef64,
			   /*low*/left + recordi->qmin,/*high*/genomichigh);
      recordi->chrnum = chrnum;
      recordi->chroffset = chroffset;
      recordi->chrhigh = chrhigh;
    }

    i++;
  }
      
  return;
}


static Intlist_T
forward_dynprog (int *max_score, struct Record_T *records, Univcoord_T *diagonals, int n,
		 int querylength, int max_insertionlen, Chrpos_T overall_end_distance,
		 Intlistpool_T intlistpool) {
  Intlist_T best_indices = NULL;

  int i, j;
  Record_T recordi, recordj;
  int best_score;
  int best_segmenti;
  bool long_segment_p = false;

  *max_score = 0;
  for (i = 0; i < n; i++) {
    recordi = &(records[i]);

    best_score = 0;
    best_segmenti = querylength;
    recordi->previ = (Intlist_T) NULL;

    j = i - 1;
    while (j >= 0 && diagonals[j] + max_insertionlen >= diagonals[i]) {
      recordj = &(records[j]);

      /* TODO: Need a criterion for insertions */
      if (recordj->qmax > recordi->qmin /*&& recordj->qmin < recordi->qmax*/) {
	/* Skip */

      } else if (recordj->chrnum != recordi->chrnum) {
	/* Skip */

      } else if (recordj->fwd_score > best_score ||
		 (recordj->fwd_score == best_score && recordj->fwd_segmenti < best_segmenti)) {
	Intlistpool_free_list(&recordi->previ,intlistpool
			      intlistpool_trace(__FILE__,__LINE__));
	recordi->previ = Intlistpool_push(NULL,intlistpool,j
					  intlistpool_trace(__FILE__,__LINE__));
	best_score = recordj->fwd_score;
	best_segmenti = recordj->fwd_segmenti;
	long_segment_p = recordj->fwd_long_segment_p;

      } else if (recordj->fwd_score == best_score && recordj->fwd_segmenti == best_segmenti) {
	recordi->previ = Intlistpool_push(recordi->previ,intlistpool,j
					  intlistpool_trace(__FILE__,__LINE__));
	long_segment_p = recordj->fwd_long_segment_p;
      }

      j--;
    }
    
    while (j >= 0 && diagonals[j] + overall_end_distance >= diagonals[i]) {
      recordj = &(records[j]);
      if (recordj->qmax > recordi->qmin) {
	/* Skip */

      } else if (recordj->chrnum != recordi->chrnum) {
	/* Skip */

      } else if (recordj->fwd_score > best_score ||
		 (recordj->fwd_score == best_score && recordj->fwd_segmenti < best_segmenti)) {
	Intlistpool_free_list(&recordi->previ,intlistpool
			      intlistpool_trace(__FILE__,__LINE__));
	recordi->previ = Intlistpool_push(NULL,intlistpool,j
					  intlistpool_trace(__FILE__,__LINE__));
	best_score = recordj->fwd_score;
	best_segmenti = recordj->fwd_segmenti;
	long_segment_p = recordj->fwd_long_segment_p;

      } else if (recordj->fwd_score == best_score && recordj->fwd_segmenti == best_segmenti) {
	recordi->previ = Intlistpool_push(recordi->previ,intlistpool,j
					  intlistpool_trace(__FILE__,__LINE__));
	long_segment_p = recordj->fwd_long_segment_p;
      }

      j--;
    }

    if (recordi->previ == (Intlist_T) NULL) {
      recordi->fwd_score = recordi->qmax - recordi->qmin;
      recordi->fwd_segmenti = 0;
      if (recordi->qmax - recordi->qmin >= index1part + index1part) {
	recordi->fwd_long_segment_p = true;
      }

    } else {
      recordi->fwd_score = best_score + /*score*/(recordi->qmax - recordi->qmin);
      recordi->fwd_segmenti = best_segmenti + 1;

      if (long_segment_p == true) {
	recordi->fwd_long_segment_p = true;
      } else if (recordi->qmax - recordi->qmin >= index1part + index1part) {
	recordi->fwd_long_segment_p = true;
      }
    }

    if (recordi->fwd_score > *max_score) {
      Intlistpool_free_list(&best_indices,intlistpool
			    intlistpool_trace(__FILE__,__LINE__));
      best_indices = Intlistpool_push(NULL,intlistpool,i
				      intlistpool_trace(__FILE__,__LINE__));
      *max_score = recordi->fwd_score;

    } else if (recordi->fwd_score == *max_score) {
      best_indices = Intlistpool_push(best_indices,intlistpool,i
				      intlistpool_trace(__FILE__,__LINE__));
    }
  }

  return best_indices;
}


static Intlist_T
reverse_dynprog (int *max_score, struct Record_T *records, Univcoord_T *diagonals, int n,
		 int querylength, int max_insertionlen, Chrpos_T overall_end_distance,
		 Intlistpool_T intlistpool) {
  Intlist_T best_indices = NULL;

  int i, j;
  Record_T recordi, recordj;
  int best_score;
  int best_segmenti;
  bool long_segment_p = false;

  *max_score = 0;
  for (i = n - 1; i >= 0; i--) {
    recordi = &(records[i]);

    best_score = 0;
    best_segmenti = querylength;
    recordi->nexti = (Intlist_T) NULL;

    j = i + 1;
    while (j < n && diagonals[i] + max_insertionlen >= diagonals[j]) {
      recordj = &(records[j]);

      /* TODO: Need a criterion for insertions */
      if (recordi->qmax > recordj->qmin /*&& recordi->qmin < recordj->qmax*/) {
	/* Skip */

      } else if (recordj->chrnum != recordi->chrnum) {
	/* Skip */

      } else if (recordj->rev_score > best_score ||
		 (recordj->rev_score == best_score && recordj->rev_segmenti < best_segmenti)) {
	Intlistpool_free_list(&recordi->nexti,intlistpool
			      intlistpool_trace(__FILE__,__LINE__));
	recordi->nexti = Intlistpool_push(NULL,intlistpool,j
					  intlistpool_trace(__FILE__,__LINE__));
	best_score = recordj->rev_score;
	best_segmenti = recordj->rev_segmenti;
	long_segment_p = recordj->rev_long_segment_p;

      } else if (recordj->rev_score == best_score && recordj->rev_segmenti == best_segmenti) {
	recordi->nexti = Intlistpool_push(recordi->nexti,intlistpool,j
					  intlistpool_trace(__FILE__,__LINE__));
	long_segment_p = recordj->rev_long_segment_p;
      }

      j++;
    }
    
    while (j < n && diagonals[i] + overall_end_distance >= diagonals[j]) {
      recordj = &(records[j]);
      if (recordi->qmax > recordj->qmin) {
	/* Skip */

      } else if (recordj->chrnum != recordi->chrnum) {
	/* Skip */

      } else if (recordj->rev_score > best_score ||
		 (recordj->rev_score == best_score && recordj->rev_segmenti < best_segmenti)) {
	Intlistpool_free_list(&recordi->nexti,intlistpool
			      intlistpool_trace(__FILE__,__LINE__));
	recordi->nexti = Intlistpool_push(NULL,intlistpool,j
					  intlistpool_trace(__FILE__,__LINE__));
	best_score = recordj->rev_score;
	best_segmenti = recordj->rev_segmenti;
	long_segment_p = recordj->rev_long_segment_p;

      } else if (recordj->rev_score == best_score && recordj->rev_segmenti == best_segmenti) {
	recordi->nexti = Intlistpool_push(recordi->nexti,intlistpool,j
					  intlistpool_trace(__FILE__,__LINE__));
	long_segment_p = recordj->rev_long_segment_p;
      }

      j++;
    }

    if (recordi->nexti == (Intlist_T) NULL) {
      recordi->rev_score = recordi->qmax - recordi->qmin;
      recordi->rev_segmenti = 0;
      if (recordi->qmax - recordi->qmin >= index1part + index1part) {
	recordi->rev_long_segment_p = true;
      }
    } else {
      recordi->rev_score = best_score + /*score*/(recordi->qmax - recordi->qmin);
      recordi->rev_segmenti = best_segmenti + 1;

      if (long_segment_p == true) {
	recordi->rev_long_segment_p = true;
      } else if (recordi->qmax - recordi->qmin >= index1part + index1part) {
	recordi->rev_long_segment_p = true;
      }
    }

    if (recordi->rev_score > *max_score) {
      Intlistpool_free_list(&best_indices,intlistpool
			    intlistpool_trace(__FILE__,__LINE__));
      best_indices = Intlistpool_push(NULL,intlistpool,i
				      intlistpool_trace(__FILE__,__LINE__));
      *max_score = recordi->rev_score;

    } else if (recordi->rev_score == *max_score) {
      best_indices = Intlistpool_push(best_indices,intlistpool,i
				      intlistpool_trace(__FILE__,__LINE__));
    }
  }

  return best_indices;
}


static Univcoord_T *
univdiagonals_gplus (int *nunique, int *max_npositions, Stage1_T this, int querylength) {
  Univcoord_T *diagonals;
  int ndiagonals;
  int querypos, query_lastpos;

  query_lastpos = querylength - index1part;

  *max_npositions = 0;
  for (querypos = 0; querypos <= query_lastpos; querypos++) {
    this->plus_diagterms[querypos] = querylength - querypos;
    if (this->plus_npositions[querypos] > *max_npositions) {
      *max_npositions = this->plus_npositions[querypos];
    }
  }
  
#ifdef LARGE_GENOMES
  diagonals = Merge_diagonals_large(&ndiagonals,this->plus_positions_high,this->plus_positions,
				    this->plus_npositions,this->plus_diagterms,
				    /*nstreams*/query_lastpos+1,this->mergeinfo);
#else
  diagonals = Merge_diagonals(&ndiagonals,this->plus_positions,this->plus_npositions,
			      this->plus_diagterms,/*nstreams*/query_lastpos+1,this->mergeinfo);
#endif

  *nunique = make_unique(diagonals,ndiagonals);

  return diagonals;
}


static Univcoord_T *
univdiagonals_gminus (int *nunique, int *max_npositions, Stage1_T this, int querylength) {
  Univcoord_T *diagonals;
  int ndiagonals;
  int querypos, query_lastpos;

  query_lastpos = querylength - index1part;

  *max_npositions = 0;
  for (querypos = 0; querypos <= query_lastpos; querypos++) {
    this->minus_diagterms[querypos] = querypos + index1part;
    if (this->minus_npositions[querypos] > *max_npositions) {
      *max_npositions = this->minus_npositions[querypos];
    }
  }
  
#ifdef LARGE_GENOMES
  diagonals = Merge_diagonals_large(&ndiagonals,this->minus_positions_high,this->minus_positions,
				    this->minus_npositions,this->minus_diagterms,
				    /*nstreams*/query_lastpos+1,this->mergeinfo);
#else
  diagonals = Merge_diagonals(&ndiagonals,this->minus_positions,this->minus_npositions,
			      this->minus_diagterms,/*nstreams*/query_lastpos+1,this->mergeinfo);
#endif

  *nunique = make_unique(diagonals,ndiagonals);

  return diagonals;
}


struct Record_T *
make_records_gplus (Univcoord_T *diagonals, int ndiagonals, int max_npositions,
		    Stage1_T this, int querylength) {

  struct Record_T *records = (struct Record_T *) CALLOC(ndiagonals,sizeof(struct Record_T));
  Record_T recordi;
  int index2;
  int i, k;

  int query_lastpos = querylength - index1part, querypos;
  int *indices;
  int nindices;


  /* Need to compute qmin and qmax to perform dynamic programming */
  for (i = 0; i < ndiagonals; i++) {
    recordi = &(records[i]);
    recordi->qmin = querylength;
    /* recordi->qmax = 0; */
  }

  indices = (int *) MALLOC(2 * max_npositions * sizeof(int));
  for (querypos = 0; querypos <= query_lastpos; querypos++) {
#ifdef LARGE_GENOMES
    nindices = Intersect_indices_large(indices,this->plus_positions_high[querypos],
				       this->plus_positions[querypos],this->plus_npositions[querypos],
				       /*diagterm1*/this->plus_diagterms[querypos],
				       diagonals,ndiagonals,/*diagterm2*/0);
#else
    nindices = Intersect_indices_small(indices,this->plus_positions[querypos],this->plus_npositions[querypos],
				       /*diagterm1*/this->plus_diagterms[querypos],
				       diagonals,ndiagonals,/*diagterm2*/0);
#endif

    /* Revise qmin and qmax */
    for (i = 0, k = 1; i < nindices; i++, k += 2) {
      index2 = indices[k];	/* Effectively k + 1 */
      recordi = &(records[index2]);
      if (querypos < recordi->qmin) {
	recordi->qmin = querypos;
      }
      if (querypos + index1part > recordi->qmax) {
	recordi->qmax = querypos + index1part;
      }
    }
  }
  FREE(indices);

  return records;
}


struct Record_T *
make_records_gminus (Univcoord_T *diagonals, int ndiagonals, int max_npositions,
		    Stage1_T this, int querylength) {

  struct Record_T *records = (struct Record_T *) CALLOC(ndiagonals,sizeof(struct Record_T));
  Record_T recordi;
  int index2;
  int i, k;

  int query_lastpos = querylength - index1part, querypos;
  int *indices;
  int nindices;


  /* Need to compute qmin and qmax to perform dynamic programming */
  for (i = 0; i < ndiagonals; i++) {
    recordi = &(records[i]);
    recordi->qmin = querylength;
    /* recordi->qmax = 0; */
  }

  indices = (int *) MALLOC(2 * max_npositions * sizeof(int));
  for (querypos = 0; querypos <= query_lastpos; querypos++) {
#ifdef LARGE_GENOMES
    nindices = Intersect_indices_large(indices,this->minus_positions_high[querypos],
				       this->minus_positions[querypos],this->minus_npositions[querypos],
				       /*diagterm1*/this->minus_diagterms[querypos],
				       diagonals,ndiagonals,/*diagterm2*/0);
#else
    nindices = Intersect_indices_small(indices,this->minus_positions[querypos],this->minus_npositions[querypos],
				       /*diagterm1*/this->minus_diagterms[querypos],
				       diagonals,ndiagonals,/*diagterm2*/0);
#endif

    /* Revise qmin and qmax */
    for (i = 0, k = 1; i < nindices; i++, k += 2) {
      index2 = indices[k];	/* Effectively k + 1 */
      recordi = &(records[index2]);
      if (querypos < recordi->qmin) {
	recordi->qmin = querypos;
      }
      if (querypos + index1part > recordi->qmax) {
	recordi->qmax = querypos + index1part;
      }
    }
  }
  FREE(indices);

  return records;
}


struct Solution_T {
  Chrnum_T chrnum;
  Univcoord_T chroffset;
  Univcoord_T chrhigh;

  Intlist_T qstart_alts;
  List_T univdiags;
  Intlist_T qend_alts;
};

typedef struct Solution_T *Solution_T;


static Solution_T
Solution_new (Intlist_T qstart_alts, Intlist_T middle_indices, Intlist_T qend_alts,
	      Univcoord_T *diagonals, struct Record_T *records,
	      Chrnum_T chrnum, Univcoord_T chroffset, Univcoord_T chrhigh,
	      Univdiagpool_T univdiagpool) {
  Solution_T new = (Solution_T) MALLOC(sizeof(*new));
  Intlist_T p;
  Univcoord_T univdiagonal;
  Record_T recordi;
  int index;

  new->chrnum = chrnum;
  new->chroffset = chroffset;
  new->chrhigh = chrhigh;

  new->qstart_alts = qstart_alts;

  new->univdiags = (List_T) NULL;
  for (p = middle_indices; p != NULL; p = Intlist_next(p)) {
    index = Intlist_head(p);
    univdiagonal = diagonals[index];
    recordi = &(records[index]);
    new->univdiags = Univdiagpool_push(new->univdiags,univdiagpool,
				       recordi->qmin,recordi->qmax,/*nmismatches*/-1,univdiagonal
				       univdiagpool_trace(__FILE__,__LINE__));
  }
  new->univdiags = List_reverse(new->univdiags);

  new->qend_alts = qend_alts;
  return new;
}

static void
Solution_print (Solution_T this) {
  List_T p;
  Univdiag_T univdiag;

  printf("qstart_alts: %s\n",Intlist_to_string(this->qstart_alts));
  printf("univdiags:\n");
  for (p = this->univdiags; p != NULL; p = List_next(p)) {
    univdiag = (Univdiag_T) List_head(p);
    printf("%u, %d..%d\n",univdiag->univdiagonal,univdiag->qstart,univdiag->qend);
  }
  printf("qend_alts: %s\n",Intlist_to_string(this->qend_alts));

  return;
}


/* A solution is an Intlist_T of indices */
static void
traceback (List_T *all_solutions, Intlist_T middle_indices, Intlist_T qend_alts,
	   int index, Univcoord_T *diagonals, struct Record_T *records,
	   Univdiagpool_T univdiagpool) {
  Solution_T solution;
  Intlist_T qstart_alts;

  Intlist_T p;
  Record_T recordi;
  int prev_index;
  
  recordi = &(records[index]);
  if (recordi->fwd_segmenti == 0) {
    assert(recordi->previ == NULL);
    solution = Solution_new(/*qstart_alts*/NULL,middle_indices,qend_alts,
			    diagonals,records,
			    recordi->chrnum,recordi->chroffset,recordi->chrhigh,
			    univdiagpool);
    *all_solutions = List_push(*all_solutions,(void *) solution);

  } else if (recordi->fwd_segmenti == 1 &&
	     Intlist_length(recordi->previ) > 1) {
    qstart_alts = recordi->previ;
    solution = Solution_new(qstart_alts,middle_indices,qend_alts,
			    diagonals,records,
			    recordi->chrnum,recordi->chroffset,recordi->chrhigh,
			    univdiagpool);
    *all_solutions = List_push(*all_solutions,(void *) solution);

  } else {
    for (p = recordi->previ; p != NULL; p = Intlist_next(p)) {
      prev_index = Intlist_head(p);
      middle_indices = Intlist_push(middle_indices,prev_index);
      traceback(&(*all_solutions),middle_indices,qend_alts,
		prev_index,diagonals,records,univdiagpool);
    }
  }

  return;
}


static int
int_cmp (const void *x, const void *y) {
  int a = * (int *) x;
  int b = * (int *) y;

  if (a < b) {
    return -1;
  } else if (b < a) {
    return +1;
  } else {
    return 0;
  }
}


static List_T
gather_solutions (Intlist_T fwd_indices, Univcoord_T *diagonals, struct Record_T *records,
		  Univdiagpool_T univdiagpool) {
  List_T all_solutions = NULL;
  Intlist_T singleton_indices, prev_indices;

  Record_T recordi;
  Intlist_T middle_indices, end_alts, p, q;
  int *index_array, index;
  int n, i, j;

  singleton_indices = (Intlist_T) NULL;
  prev_indices = (Intlist_T) NULL;
  for (p = fwd_indices; p != NULL; p = Intlist_next(p)) {
    index = Intlist_head(p);
    recordi = &(records[index]);
    if (recordi->previ == NULL) {
      singleton_indices = Intlist_push(singleton_indices,index);
    } else {
      for (q = recordi->previ; q != NULL; q = Intlist_next(q)) {
	prev_indices = Intlist_push(prev_indices,Intlist_head(q));
      }
    }
  }
    
  /* Handle singleton indices */
  printf("Singleton indices: %s\n",Intlist_to_string(singleton_indices));
  for (p = singleton_indices; p != NULL; p = Intlist_next(p)) {
    index = Intlist_head(p);
    middle_indices = Intlist_push(NULL,index);
    traceback(&all_solutions,middle_indices,/*end_alts*/NULL,index,
	      diagonals,records,univdiagpool);
  }

  if (prev_indices != NULL) {
    /* Handle multisegment solutions */
    index_array = Intlist_to_array(&n,prev_indices);
    qsort(index_array,n,sizeof(int),int_cmp);

    i = 0;
    while (i < n) {
      index = index_array[i];
      recordi = &(records[index]);

      if (Intlist_length(recordi->nexti) == 0) {
	abort();
      } else if (Intlist_length(recordi->nexti) == 1) {
	middle_indices = Intlist_push(NULL,Intlist_head(recordi->nexti));
	middle_indices = Intlist_push(middle_indices,index);
	traceback(&all_solutions,middle_indices,/*end_alts*/NULL,index,
		  diagonals,records,univdiagpool);
      } else {
	end_alts = recordi->nexti;
	traceback(&all_solutions,/*middle_indices*/NULL,end_alts,index,
		  diagonals,records,univdiagpool);
      }
      
      /* Skip duplicates */
      j = i + 1;
      while (j < n && index_array[j] == index) {
	j++;
      }

      i = j;
    }
  }

  return all_solutions;
}



/* IDEA: Instead of dynamic programming, just take any univdiagonal
   with >= 30 separation, and treat as KMER_MERGE */

void
Merge_search (int *found_score,

	      List_T *unextended_sense_paths_gplus, List_T *unextended_sense_paths_gminus,
	      List_T *unextended_antisense_paths_gplus, List_T *unextended_antisense_paths_gminus,
		    
	      List_T *sense_paths_gplus, List_T *sense_paths_gminus,
	      List_T *antisense_paths_gplus, List_T *antisense_paths_gminus,

	      Stage1_T stage1, Knownsplicing_T knownsplicing, int querylength,
	      Compress_T query_compress_fwd, Compress_T query_compress_rev,
	      int max_insertionlen, int max_deletionlen, Chrpos_T overall_end_distance,
	      Univdiagpool_T univdiagpool, Intlistpool_T intlistpool,
	      Univcoordlistpool_T univcoordlistpool, Listpool_T listpool,
	      Pathpool_T pathpool, Vectorpool_T vectorpool, Hitlistpool_T hitlistpool,
	      Method_T method) {

  Path_T sense_path, antisense_path;

  Univcoord_T *plus_univdiagonals, *minus_univdiagonals;
  int plus_nunivdiagonals, minus_nunivdiagonals, plus_max_npositions, minus_max_npositions;
  struct Record_T *plus_records, *minus_records;

  Intlist_T plus_fwd_indices, plus_rev_indices, minus_fwd_indices, minus_rev_indices;
  int plus_fwd_max_score, plus_rev_max_score, minus_fwd_max_score, minus_rev_max_score;

  List_T all_solutions, s;
  Solution_T solution;
  Record_T recordi;

  /* Stage1_dump(stage1,querylength); */

  plus_univdiagonals = univdiagonals_gplus(&plus_nunivdiagonals,&plus_max_npositions,
					   stage1,querylength);
  plus_records = make_records_gplus(plus_univdiagonals,plus_nunivdiagonals,plus_max_npositions,
				    stage1,querylength);

  assign_chrinfo(plus_records,plus_univdiagonals,/*n*/plus_nunivdiagonals,querylength);

  plus_fwd_indices = forward_dynprog(&plus_fwd_max_score,plus_records,plus_univdiagonals,/*n*/plus_nunivdiagonals,
				     querylength,max_insertionlen,overall_end_distance,intlistpool);
  plus_rev_indices = reverse_dynprog(&plus_rev_max_score,plus_records,plus_univdiagonals,/*n*/plus_nunivdiagonals,
				     querylength,max_insertionlen,overall_end_distance,intlistpool);

  printf("%d plus fwd indices: %s\n",plus_fwd_max_score,Intlist_to_string(plus_fwd_indices));
  printf("%d plus rev indices: %s\n",plus_rev_max_score,Intlist_to_string(plus_rev_indices));
#if 0
  for (int i = 0; i < plus_nunivdiagonals; i++) {
    printf("plus %d: %u ",i,plus_univdiagonals[i]);
    record_print(&((plus_records)[i]));
    printf("\n");
  }
#endif

  all_solutions = gather_solutions(plus_fwd_indices,plus_univdiagonals,plus_records,
				   univdiagpool);
  for (s = all_solutions; s != NULL; s = List_next(s)) {
    solution = (Solution_T) List_head(s);
    Solution_print(solution);

    if ((sense_path = Path_add_junctions(&(*found_score),solution->univdiags,
					 /*plusp*/true,querylength,
					 /*sensedir*/SENSE_FORWARD,/*genestrand*/0,
					 solution->chrnum,solution->chroffset,solution->chrhigh,
					 max_insertionlen,max_deletionlen,
					 /*query_compress*/query_compress_fwd,
					 query_compress_fwd,query_compress_rev,
					 stage1,knownsplicing,intlistpool,univcoordlistpool,
					 listpool,pathpool,vectorpool,method)) == NULL) {
      /* Skip */

    } else if (Path_unextendedp(sense_path,/*endtrim_allowed*/8,/*allow_ambig_p*/false) == true) {
      *unextended_sense_paths_gplus =
	Hitlist_push(*unextended_sense_paths_gplus,hitlistpool,(void *) Path_expect_fwd(sense_path)
		     hitlistpool_trace(__FILE__,__LINE__));
    } else {
      sense_path->completep = true;
      *sense_paths_gplus =
	Hitlist_push(*sense_paths_gplus,hitlistpool,(void *) Path_expect_fwd(sense_path)
		     hitlistpool_trace(__FILE__,__LINE__));
    }

    if ((antisense_path = Path_add_junctions(&(*found_score),solution->univdiags,
					 /*plusp*/true,querylength,
					 /*sensedir*/SENSE_ANTI,/*genestrand*/0,
					 solution->chrnum,solution->chroffset,solution->chrhigh,
					 max_insertionlen,max_deletionlen,
					 /*query_compress*/query_compress_fwd,
					 query_compress_fwd,query_compress_rev,
					 stage1,knownsplicing,intlistpool,univcoordlistpool,
					 listpool,pathpool,vectorpool,method)) == NULL) {
      /* Skip */

    } else if (Path_unextendedp(antisense_path,/*endtrim_allowed*/8,/*allow_ambig_p*/false) == true) {
      *unextended_antisense_paths_gplus =
	Hitlist_push(*unextended_antisense_paths_gplus,hitlistpool,(void *) Path_expect_fwd(antisense_path)
		     hitlistpool_trace(__FILE__,__LINE__));
    } else {
      antisense_path->completep = true;
      *antisense_paths_gplus =
	Hitlist_push(*antisense_paths_gplus,hitlistpool,(void *) Path_expect_fwd(antisense_path)
		     hitlistpool_trace(__FILE__,__LINE__));
    }
  }


  minus_univdiagonals = univdiagonals_gminus(&minus_nunivdiagonals,&minus_max_npositions,
					     stage1,querylength);

  minus_records = make_records_gminus(minus_univdiagonals,minus_nunivdiagonals,minus_max_npositions,
				    stage1,querylength);
  assign_chrinfo(minus_records,minus_univdiagonals,/*n*/minus_nunivdiagonals,querylength);

  minus_fwd_indices = forward_dynprog(&minus_fwd_max_score,minus_records,minus_univdiagonals,/*n*/minus_nunivdiagonals,
				      querylength,max_insertionlen,overall_end_distance,intlistpool);
  minus_rev_indices = reverse_dynprog(&minus_rev_max_score,minus_records,minus_univdiagonals,/*n*/minus_nunivdiagonals,
				      querylength,max_insertionlen,overall_end_distance,intlistpool);

  printf("%d minus fwd indices: %s\n",minus_fwd_max_score,Intlist_to_string(minus_fwd_indices));
  printf("%d minus rev indices: %s\n",minus_rev_max_score,Intlist_to_string(minus_rev_indices));
#if 1
  for (int i = 0; i < minus_nunivdiagonals; i++) {
    recordi = &(minus_records)[i];
    if (recordi->nexti == NULL && recordi->fwd_long_segment_p == true) {
      printf(" * ");
    }
    printf("minus %d: %u ",i,minus_univdiagonals[i]);
    record_print(recordi);
    printf("\n");
  }
#endif

  all_solutions = gather_solutions(minus_fwd_indices,minus_univdiagonals,minus_records,
				   univdiagpool);
  for (s = all_solutions; s != NULL; s = List_next(s)) {
    solution = (Solution_T) List_head(s);
    Solution_print(solution);

    if ((sense_path = Path_add_junctions(&(*found_score),solution->univdiags,
					 /*plusp*/false,querylength,
					 /*sensedir*/SENSE_FORWARD,/*genestrand*/0,
					 solution->chrnum,solution->chroffset,solution->chrhigh,
					 max_insertionlen,max_deletionlen,
					 /*query_compress*/query_compress_rev,
					 query_compress_fwd,query_compress_rev,
					 stage1,knownsplicing,intlistpool,univcoordlistpool,
					 listpool,pathpool,vectorpool,method)) == NULL) {
      /* Skip */

    } else if (Path_unextendedp(sense_path,/*endtrim_allowed*/8,/*allow_ambig_p*/false) == true) {
      *unextended_sense_paths_gminus =
	Hitlist_push(*unextended_sense_paths_gminus,hitlistpool,(void *) Path_expect_fwd(sense_path)
		     hitlistpool_trace(__FILE__,__LINE__));
    } else {
      sense_path->completep = true;
      *sense_paths_gminus =
	Hitlist_push(*sense_paths_gminus,hitlistpool,(void *) Path_expect_fwd(sense_path)
		     hitlistpool_trace(__FILE__,__LINE__));
    }

    if ((antisense_path = Path_add_junctions(&(*found_score),solution->univdiags,
					 /*plusp*/false,querylength,
					 /*sensedir*/SENSE_ANTI,/*genestrand*/0,
					 solution->chrnum,solution->chroffset,solution->chrhigh,
					 max_insertionlen,max_deletionlen,
					 /*query_compress*/query_compress_rev,
					 query_compress_fwd,query_compress_rev,
					 stage1,knownsplicing,intlistpool,univcoordlistpool,
					 listpool,pathpool,vectorpool,method)) == NULL) {
      /* Skip */

    } else if (Path_unextendedp(antisense_path,/*endtrim_allowed*/8,/*allow_ambig_p*/false) == true) {
      *unextended_antisense_paths_gminus =
	Hitlist_push(*unextended_antisense_paths_gminus,hitlistpool,(void *) Path_expect_fwd(antisense_path)
		     hitlistpool_trace(__FILE__,__LINE__));
    } else {
      antisense_path->completep = true;
      *antisense_paths_gminus =
	Hitlist_push(*antisense_paths_gminus,hitlistpool,(void *) Path_expect_fwd(antisense_path)
		     hitlistpool_trace(__FILE__,__LINE__));
    }
  }

  return;
}


void
Merge_search_setup (int index1part_in, EF64_T chromosome_ef64_in) {
  index1part = index1part_in;
  chromosome_ef64 = chromosome_ef64_in;
  return;
}
