/***************************** LICENSE START ***********************************

 Copyright 2022 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "MvLog.h"

#include <cassert>
#include <iostream>
#include <regex>
#include <vector>

#include "MvAbstractApplication.h"

MvAbstractApplication* MvLog::app_ = nullptr;

//---------------------------------
// MvLog
//---------------------------------

MvLog::MvLog() = default;

MvLog::~MvLog()
{
    output(os_.str());
}

void MvLog::registerApp(MvAbstractApplication* app)
{
    app_ = app;
}

MvLog& MvLog::popup()
{
    popup_ = true;
    return *this;
}

std::ostringstream& MvLog::info()
{
    level_ = MvLogLevel::INFO;
    return os_;
}

std::ostringstream& MvLog::err()
{
    level_ = MvLogLevel::ERROR;
    return os_;
}

std::ostringstream& MvLog::errNoExit()
{
    level_       = MvLogLevel::ERROR;
    exitOnError_ = false;
    return os_;
}

std::ostringstream& MvLog::warn()
{
    level_ = MvLogLevel::WARN;
    return os_;
}

std::ostringstream& MvLog::dbg()
{
    level_ = MvLogLevel::DBG;
    return os_;
}

// to be used in MV_FN_INFO with __PRETTY_FUNCTION__
std::string MvLog::formatFuncInfo(const std::string& t)
{
    // extracts the function name from the __PRETTY_FUNCTION__
    // string (as t) and returns either:
    // - "classsame::function() "
    // - "function() "

    static std::regex rx(R"((\w+(::)?\w+\())");
    auto it = std::sregex_iterator(t.begin(), t.end(), rx);
    if (it != std::sregex_iterator()) {
        return it->str() + ") ";
    }
    return t + " ";
}

// to be used in MV_FN_INFO with __func__
std::string MvLog::formatFuncInfoBasic(const std::string& t)
{
    return t + "() ";
}

void MvLog::output(const std::string& msg)
{
    if (app_) {
        if (exitOnError_ && level_ == MvLogLevel::ERROR) {
            std::string t = msg + "\n\nThe program will terminate!";
            app_->toLog(t, level_, popup_);
            app_->exitWithError();
        }
        else {
            app_->toLog(msg, level_, popup_);
        }
    }
}

//--------------------------------------------
// Overload ostringstream for various objects
//--------------------------------------------

//std::ostream& operator <<(std::ostream &stream,const std::vector<std::string>& vec)
//{
//    stream << "[";
//    for(size_t i=0; i < vec.size(); i++)
//    {
//        if(i > 0) stream << ",";
//        stream << vec[i];
//    }
//    stream << "]";

//    return stream;
//}
