#include <sys/types.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <sys/un.h>
#include <sys/ioctl.h>

#include <netinet/in.h>

#include <arpa/inet.h>

#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <assert.h>

#include "buffer.h"
#include "server.h"
#include "keyvalue.h"
#include "log.h"

#include "http_chunk.h"
#include "fdevent.h"
#include "connections.h"
#include "response.h"
#include "joblist.h"

#include "plugin.h"

#include "inet_ntop_cache.h"

#include <fastcgi.h>
#include <stdio.h>

#ifdef HAVE_SYS_FILIO_H
# include <sys/filio.h>
#endif


#ifndef UNIX_PATH_MAX
# define UNIX_PATH_MAX 108
#endif

/*
 * 
 * TODO:
 * 
 * - add timeout for a connect to a non-fastcgi process
 *   (use state_timestamp + state)
 * 
 */

typedef struct {
	array *extensions;
	int debug;
} plugin_config;

typedef struct {
	size_t *ptr;
	size_t used;
	size_t size;
} buffer_uint;

/* generic plugin data, shared between all connections */
typedef struct {
	PLUGIN_DATA;
	buffer_uint fcgi_request_id;
	
	buffer *fcgi_env;
	buffer *fcgi_master;
	
	buffer *path;
	buffer *parse_response;
	
	plugin_config **config_storage;
	
	plugin_config conf; /* this is only used as long as no handler_ctx is setup */
} plugin_data;

/* connection specific data */
typedef enum { FCGI_STATE_INIT, FCGI_STATE_CONNECT, FCGI_STATE_PREPARE_WRITE, FCGI_STATE_WRITE, FCGI_STATE_READ, FCGI_STATE_ERROR } fcgi_connection_state_t;
typedef struct {
	buffer *response;
	size_t response_len;
	int response_type;
	int response_padding;
	size_t response_request_id;
	
	data_fastcgi *host;
	
	fcgi_connection_state_t state;
	time_t state_timestamp;
	
	buffer *write_buffer;
	size_t  write_offset;
	
	read_buffer *rb;
	
	buffer *response_header;
	
	int delayed;

	size_t request_id;
	int fd; /* fd to the fastcgi process */
	int fde_ndx; /* index into the fd-event buffer */

	size_t path_info_offset; /* start of path_info in uri.path */
	
	plugin_config conf;
	
	connection *remote_conn;  /* dumb pointer */
	plugin_data *plugin_data; /* dumb pointer */
} handler_ctx;


/* ok, we need a prototype */
static handler_t fcgi_handle_fdevent(void *s, void *ctx, int revents);

static handler_ctx * handler_ctx_init() {
	handler_ctx * hctx;
	
	hctx = calloc(1, sizeof(*hctx));
	
	hctx->fde_ndx = -1;
	
	hctx->response = buffer_init();
	hctx->response_header = buffer_init();
	hctx->write_buffer = buffer_init();
	
	hctx->request_id = 0;
	hctx->state = FCGI_STATE_INIT;
	hctx->host = NULL;
	
	hctx->response_len = 0;
	hctx->response_type = 0;
	hctx->response_padding = 0;
	hctx->response_request_id = 0;
	hctx->fd = -1;
	
	return hctx;
}

static void handler_ctx_free(handler_ctx *hctx) {
	buffer_free(hctx->response);
	buffer_free(hctx->response_header);
	buffer_free(hctx->write_buffer);
	
	if (hctx->rb) {
		if (hctx->rb->ptr) free(hctx->rb->ptr);
		free(hctx->rb);
	}
	
	free(hctx);
}

INIT_FUNC(mod_fastcgi_init) {
	plugin_data *p;
	
	p = calloc(1, sizeof(*p));
	
	p->fcgi_env = buffer_init();
	p->fcgi_master = buffer_init();
	
	p->path = buffer_init();
	p->parse_response = buffer_init();
	
	return p;
}


FREE_FUNC(mod_fastcgi_free) {
	plugin_data *p = p_d;
	buffer_uint *r = &(p->fcgi_request_id);
	
	UNUSED(srv);

	if (r->ptr) free(r->ptr);
	
	buffer_free(p->fcgi_env);
	buffer_free(p->fcgi_master);
	buffer_free(p->path);
	buffer_free(p->parse_response);
	
	if (p->config_storage) {
		size_t i;
		for (i = 0; i < srv->config_context->used; i++) {
			plugin_config *s = p->config_storage[i];
			
			array_free(s->extensions);
			
			free(s);
		}
		free(p->config_storage);
	}
	
	free(p);
	
	return HANDLER_GO_ON;
}

SETDEFAULTS_FUNC(mod_fastcgi_set_defaults) {
	plugin_data *p = p_d;
	data_unset *du;
	data_config *dc;
	size_t i = 0;
	int ret = HANDLER_GO_ON;
	
	config_values_t cv[] = { 
		{ "fastcgi.server",              NULL, T_CONFIG_LOCAL, T_CONFIG_SCOPE_CONNECTION },       /* 0 */
		{ "fastcgi.debug",               NULL, T_CONFIG_SHORT, T_CONFIG_SCOPE_CONNECTION },       /* 1 */
		{ NULL,                          NULL, T_CONFIG_UNSET, T_CONFIG_SCOPE_UNSET }
	};
	
	p->config_storage = malloc(srv->config_context->used * sizeof(specific_config *));
	
	for (i = 0; i < srv->config_context->used; i++) {
		plugin_config *s;
		
		s = malloc(sizeof(plugin_config));
		s->extensions    = array_init();
		s->debug         = 0;
		
		cv[0].destination = s->extensions;
		cv[1].destination = &(s->debug);
		
		p->config_storage[i] = s;
		srv->config = ((data_config *)srv->config_context->data[i])->value;
	
		if (0 != (ret = config_insert_values(srv, cv))) {
			ret = HANDLER_ERROR;
			
			break;
		}
		
		ret = HANDLER_GO_ON;
		
		if (NULL == (du = array_get_element(srv->config, "fastcgi.server"))) {
			/* no fastcgi.server defined */
			continue;
		}
		
		if (du->type == TYPE_ARRAY) {
			size_t j;
			data_array *da = (data_array *)du;
			
			/* get the extension from -> key */
			
			for (j = 0; j < da->value->used; j++) {
				size_t n;
				data_array *da_ext = (data_array *)da->value->data[j];
				
				if (da->value->data[j]->type != TYPE_ARRAY) {
					log_error_write(srv, __FILE__, __LINE__, "sssbs", 
							"unexpected type for key: ", "fastcgi.server", 
							"[", da->value->data[j]->key, "](string)");
					
					return HANDLER_ERROR;
				}
				
				/* 
				 * da_ext->key == name of the extension 
				 */
					
				for (n = 0; n < da_ext->value->used; n++) {
					size_t m;
					data_array *da_host = (data_array *)da_ext->value->data[n];
					const char *host = NULL, *docroot = NULL, *mode = NULL, *sock = NULL;
					unsigned short port = 0, check_local = 2;
					
					data_fastcgi *df;
					data_array *dfa; 
					
					if (da_ext->value->data[n]->type != TYPE_ARRAY) {
						log_error_write(srv, __FILE__, __LINE__, "ssSBS", 
								"unexpected type for key:", 
								"fastcgi.server", 
								"[", da_ext->value->data[n]->key, "](string)");
						
						return HANDLER_ERROR;
					}
							
							
					for (m = 0; m < da_host->value->used; m++) {
						data_unset *da_unset_entry = da_host->value->data[m];
						
						switch(da_unset_entry->type) {
						case TYPE_STRING: {
							data_string *da_entry = (data_string *)da_unset_entry;
							
							if (0 == strcmp(da_entry->key->ptr, "host")) {
								host = da_entry->value->ptr;
							} else if (0 == strcmp(da_entry->key->ptr, "check-local")) {
								if (0 == strcmp(da_entry->value->ptr, "enable")) {
									check_local = 1;
								} else if (0 == strcmp(da_entry->value->ptr, "disable")) {
									check_local = 0;
								} else {
									log_error_write(srv, __FILE__, __LINE__, "sbbbb", 
											"(warning) unexpected value for boolean:", 
											da->key,
											da_ext->key,
											da_host->key,
											da_entry->key);
									
									return HANDLER_ERROR;
								}
							} else if (0 == strcmp(da_entry->key->ptr, "docroot")) {
								docroot = da_entry->value->ptr;
							} else if (0 == strcmp(da_entry->key->ptr, "mode")) {
								mode = da_entry->value->ptr;
							} else if (0 == strcmp(da_entry->key->ptr, "socket")) {
								sock = da_entry->value->ptr;
							} else {
								log_error_write(srv, __FILE__, __LINE__, "sbbbb", 
										"(warning) unexpected key (string):", 
										da->key,
										da_ext->key,
										da_host->key,
										da_entry->key);
							}
							break;
						}
						case TYPE_INTEGER: {
							data_integer *da_entry = (data_integer *)da_unset_entry;
							if (0 == strcmp(da_entry->key->ptr, "port")) {
								port = da_entry->value;
							} else {
								log_error_write(srv, __FILE__, __LINE__, "sbbbb", 
										"(warning) unexpected key (short):", 
										da->key,
										da_ext->key,
										da_host->key,
										da_entry->key);
							}
							break;
						}
						default:
							log_error_write(srv, __FILE__, __LINE__, "sbbbb", 
									"(warning) unexpected key (short):", 
									da->key,
									da_ext->key,
									da_host->key,
									da_unset_entry->key);
						}
					}
					
					/* do a local stat by default */
					if (check_local == 2) {
						check_local = 1;
					}
					
					if ((host || port) && sock) {
						log_error_write(srv, __FILE__, __LINE__, "s", 
								"either host+port or socket");
						
						return HANDLER_ERROR;
					}
					
					if (sock) {
						/* unix domain socket */
						
						if (strlen(sock) > UNIX_PATH_MAX - 1) {
							log_error_write(srv, __FILE__, __LINE__, "s", 
									"path of the unixdomain socket is too large");
							return HANDLER_ERROR;
						}
					} else {
						/* tcp/ip */
						if (host == NULL) {
							log_error_write(srv, __FILE__, __LINE__, "sbbbs", 
									"missing key (string):", 
									da->key,
									da_ext->key,
									da_host->key,
									"host");
							
							return HANDLER_ERROR;
						} else if (port == 0) {
							log_error_write(srv, __FILE__, __LINE__, "sbbbs", 
									"missing key (short):", 
									da->key,
									da_ext->key,
									da_host->key,
									"port");
							return HANDLER_ERROR;
						}
					}
							
					df = data_fastcgi_init();
					
					buffer_copy_string_buffer(df->key, da_host->key);
					
					if (sock) {
						buffer_copy_string(df->unixsocket, sock);
					} else {
						buffer_copy_string(df->host, host);
						df->port = port;
					}
					df->check_local = check_local;
					
					if (docroot) buffer_copy_string(df->docroot, docroot);
					if (mode) {
						if (strcmp(mode, "responder") == 0) df->mode = FCGI_RESPONDER;
						else if (strcmp(mode, "authorizer") == 0) {
							df->mode = FCGI_AUTHORIZER;
							if (docroot == NULL) {
								log_error_write(srv, __FILE__, __LINE__, "s",
										"ERROR: docroot is required for authorizer mode.");
								return HANDLER_ERROR;
							}
						} else {
							log_error_write(srv, __FILE__, __LINE__, "sss",
									"WARNING: unknown fastcgi mode:",
									mode, "(ignored, mode set to responder)");
						}
					}
					
					/* if extension already exists, take it */
							
					if (NULL == (dfa = (data_array *)array_get_element(s->extensions, da_ext->key->ptr))) {
						dfa = data_array_init();
						
						buffer_copy_string_buffer(dfa->key, da_ext->key);
						
						array_insert_unique(dfa->value, (data_unset *)df);
						array_insert_unique(s->extensions, (data_unset *)dfa);
					} else {
						array_insert_unique(dfa->value, (data_unset *)df);
					}
				}
			}
		}
	}
	
	if (NULL != (dc = (data_config *)array_get_element(srv->config_context, "global"))) {
		srv->config = dc->value;
	}
	
	return ret;
}


static size_t fcgi_requestid_new(server *srv, plugin_data *p) {
	size_t m = 0;
	size_t i;
	buffer_uint *r = &(p->fcgi_request_id);
	
	UNUSED(srv);

	for (i = 0; i < r->used; i++) {
		if (r->ptr[i] > m) m = r->ptr[i];
	}
	
	if (r->size == 0) {
		r->size = 16;
		r->ptr = malloc(sizeof(*r->ptr) * r->size);
	} else if (r->used == r->size) {
		r->size += 16;
		r->ptr = realloc(r->ptr, sizeof(*r->ptr) * r->size);
	}
	
	r->ptr[r->used++] = ++m;
	
	return m;
}

static int fcgi_requestid_del(server *srv, plugin_data *p, size_t request_id) {
	size_t i;
	buffer_uint *r = &(p->fcgi_request_id);
	
	UNUSED(srv);

	for (i = 0; i < r->used; i++) {
		if (r->ptr[i] == request_id) break;
	}
	
	if (i != r->used) {
		/* found */
		
		if (i != r->used - 1) {
			r->ptr[i] = r->ptr[r->used - 1];
		}
		r->used--;
	}
	
	return 0;
}

void fcgi_connection_cleanup(server *srv, handler_ctx *hctx) {
	plugin_data *p;
	connection  *con;
	
	if (NULL == hctx) return;
	
	p    = hctx->plugin_data;
	con  = hctx->remote_conn;
	
	if (con->mode != p->id) return;
	
	fdevent_event_del(srv->ev, &(hctx->fde_ndx), hctx->fd);
	fdevent_unregister(srv->ev, hctx->fd);
	fcgi_requestid_del(srv, p, hctx->request_id);
	if (hctx->fd != -1) close(hctx->fd);
	
	handler_ctx_free(hctx);
	con->plugin_ctx[p->id] = NULL;	
}

static handler_t fcgi_connection_reset(server *srv, connection *con, void *p_d) {
	plugin_data *p = p_d;
	
	fcgi_connection_cleanup(srv, con->plugin_ctx[p->id]);
	
	return HANDLER_GO_ON;
}


static int fcgi_env_add(buffer *env, const char *key, size_t key_len, const char *val) {
	size_t val_len, len;
	
	if (!key || !val) return -1;
	
	val_len = strlen(val);
	
	len = key_len + val_len;
	
	len += key_len > 127 ? 4 : 1;
	len += val_len > 127 ? 4 : 1;
	
	buffer_prepare_append(env, len);
	
	if (key_len > 127) {
		env->ptr[env->used++] = ((key_len >> 24) & 0xff) | 0x80;
		env->ptr[env->used++] = (key_len >> 16) & 0xff;
		env->ptr[env->used++] = (key_len >> 8) & 0xff;
		env->ptr[env->used++] = (key_len >> 0) & 0xff;
	} else {
		env->ptr[env->used++] = (key_len >> 0) & 0xff;
	}
	
	if (val_len > 127) {
		env->ptr[env->used++] = ((val_len >> 24) & 0xff) | 0x80;
		env->ptr[env->used++] = (val_len >> 16) & 0xff;
		env->ptr[env->used++] = (val_len >> 8) & 0xff;
		env->ptr[env->used++] = (val_len >> 0) & 0xff;
	} else {
		env->ptr[env->used++] = (val_len >> 0) & 0xff;
	}
	
	memcpy(env->ptr + env->used, key, key_len);
	env->used += key_len;
	memcpy(env->ptr + env->used, val, val_len);
	env->used += val_len;
	
	return 0;
}

static int fcgi_header(FCGI_Header * header, unsigned char type, size_t request_id, int contentLength, unsigned char paddingLength) {
	header->version = FCGI_VERSION_1;
	header->type = type;
	header->requestIdB0 = request_id & 0xff;
	header->requestIdB1 = (request_id >> 8) & 0xff;
	header->contentLengthB0 = contentLength & 0xff;
	header->contentLengthB1 = (contentLength >> 8) & 0xff;
	header->paddingLength = paddingLength;
	header->reserved = 0;
	
	return 0;
}
/**
 * 
 * returns
 *   -1 error
 *    0 connected
 *    1 not connected yet
 */

static int fcgi_establish_connection(server *srv, handler_ctx *hctx) {
	struct sockaddr *fcgi_addr;
	struct sockaddr_in fcgi_addr_in;
	struct sockaddr_un fcgi_addr_un;
	socklen_t servlen;
	
	data_fastcgi *host= hctx->host;
	int fcgi_fd       = hctx->fd;
	
	memset(&fcgi_addr, 0, sizeof(fcgi_addr));
	
	if (host->unixsocket->used) {
		/* use the unix domain socket */
		fcgi_addr_un.sun_family = AF_UNIX;
		strcpy(fcgi_addr_un.sun_path, host->unixsocket->ptr);
#ifdef SUN_LEN
		servlen = SUN_LEN(&fcgi_addr_un);
#else
		/* stevens says: */
		servlen = strlen(fcgi_addr_un.sun_path) + sizeof(fcgi_addr_un.sun_family);
#endif
		fcgi_addr = (struct sockaddr *) &fcgi_addr_un;
	} else {
		fcgi_addr_in.sin_family = AF_INET;
		fcgi_addr_in.sin_addr.s_addr = inet_addr(host->host->ptr);
		fcgi_addr_in.sin_port = htons(host->port);
		servlen = sizeof(fcgi_addr_in);
		
		fcgi_addr = (struct sockaddr *) &fcgi_addr_in;
	}
	
	if (-1 == connect(fcgi_fd, fcgi_addr, servlen)) {
		if (errno == EINPROGRESS || 
		    errno == EALREADY ||
		    errno == EINTR) {
			if (hctx->conf.debug) {
				log_error_write(srv, __FILE__, __LINE__, "sd", 
						"connect delayed:", fcgi_fd);
			}
			
			return 1;
		} else {
			
			log_error_write(srv, __FILE__, __LINE__, "sdsd", 
					"connect failed:", fcgi_fd, strerror(errno), errno);
			
			return -1;
		}
	}
	if (hctx->conf.debug) {
		log_error_write(srv, __FILE__, __LINE__, "sd", 
				"connect succeeded: ", fcgi_fd);
	}
	return 0;
}

static int fcgi_env_add_request_headers(server *srv, connection *con, plugin_data *p) {
	size_t i;
	
	for (i = 0; i < con->request.headers->used; i++) {
		data_string *ds;
		
		ds = (data_string *)con->request.headers->data[i];
		
		if (ds->value->used && ds->key->used) {
			size_t j;
			buffer_reset(srv->tmp_buf);
			
			/* don't forward the Authorization: Header */
			if (0 == strcasecmp(ds->key->ptr, "AUTHORIZATION")) {
				continue;
			}
			
			if (0 != strcasecmp(ds->key->ptr, "CONTENT-TYPE")) {
				buffer_copy_string(srv->tmp_buf, "HTTP_");
				srv->tmp_buf->used--;
			}
			
			buffer_prepare_append(srv->tmp_buf, ds->key->used + 2);
			for (j = 0; j < ds->key->used - 1; j++) {
				srv->tmp_buf->ptr[srv->tmp_buf->used++] = 
					isalpha((unsigned char)ds->key->ptr[j]) ? 
					toupper((unsigned char)ds->key->ptr[j]) : '_';
			}
			srv->tmp_buf->ptr[srv->tmp_buf->used] = '\0';
			
			fcgi_env_add(p->fcgi_env, srv->tmp_buf->ptr, srv->tmp_buf->used, ds->value->ptr);
		}
	}
	
	return 0;
}


static int fcgi_create_env(server *srv, handler_ctx *hctx, size_t request_id) {
	FCGI_BeginRequestRecord beginRecord;
	FCGI_Header header;
	
	char buf[32];
	size_t offset;
	
	plugin_data *p    = hctx->plugin_data;
	data_fastcgi *host= hctx->host;
	connection *con   = hctx->remote_conn;
	server_socket *srv_sock = con->srv_socket;
	
#define CONST_STRING(x) \
		x, sizeof(x)-1
	
	/* send FCGI_BEGIN_REQUEST */
	
	fcgi_header(&(beginRecord.header), FCGI_BEGIN_REQUEST, request_id, sizeof(beginRecord.body), 0);
	beginRecord.body.roleB0 = host->mode;
	beginRecord.body.roleB1 = 0;
	beginRecord.body.flags = 0;
	memset(beginRecord.body.reserved, 0, sizeof(beginRecord.body.reserved));
	
	buffer_copy_memory(hctx->write_buffer, (const char *)&beginRecord, sizeof(beginRecord));
	
	/* send FCGI_PARAMS */
	buffer_prepare_copy(p->fcgi_env, 1024);

	if (p->fcgi_master->used == 0) {
#ifdef HAVE_IPV6
		char b2[INET6_ADDRSTRLEN + 1];
#endif
#ifdef PACKAGE_NAME
		fcgi_env_add(p->fcgi_env, CONST_STRING("SERVER_SOFTWARE"), PACKAGE_NAME"/"PACKAGE_VERSION);
#else
		fcgi_env_add(p->fcgi_env, CONST_STRING("SERVER_SOFTWARE"), PACKAGE"/"VERSION);
#endif
		fcgi_env_add(p->fcgi_env, CONST_STRING("SERVER_NAME"),
#ifdef HAVE_IPV6
			     inet_ntop(srv_sock->addr.plain.sa_family, 
				       srv_sock->addr.plain.sa_family == AF_INET6 ? 
				       (const void *) &(srv_sock->addr.ipv6.sin6_addr) :
				       (const void *) &(srv_sock->addr.ipv4.sin_addr),
				       b2, sizeof(b2)-1)
#else
			     inet_ntoa(srv_sock->addr.ipv4.sin_addr)
#endif
			     );
		fcgi_env_add(p->fcgi_env, CONST_STRING("GATEWAY_INTERFACE"), "CGI/1.1");
		
		buffer_prepare_copy(p->fcgi_master, p->fcgi_env->used);
		
		memcpy(p->fcgi_master->ptr, p->fcgi_env->ptr, p->fcgi_env->used);
		p->fcgi_master->used = p->fcgi_env->used;
	} else {
		memcpy(p->fcgi_env->ptr, p->fcgi_master->ptr, p->fcgi_master->used);
		p->fcgi_env->used = p->fcgi_master->used;
	}
	
	ltostr(buf, 
#ifdef HAVE_IPV6
	       ntohs(srv_sock->addr.plain.sa_family ? srv_sock->addr.ipv6.sin6_port : srv_sock->addr.ipv4.sin_port)
#else
	       ntohs(srv_sock->addr.ipv4.sin_port)
#endif
	       );
	
	fcgi_env_add(p->fcgi_env, CONST_STRING("SERVER_PORT"), buf);
	
	fcgi_env_add(p->fcgi_env, CONST_STRING("REMOTE_ADDR"),
		     inet_ntop_cache_get_ip(srv, &(con->dst_addr)));
	
	if (con->authed_user->used) {
		fcgi_env_add(p->fcgi_env, CONST_STRING("REMOTE_USER"),
			     con->authed_user->ptr);
	}
	
	if (con->request.content_length > 0 && host->mode != FCGI_AUTHORIZER ) {
		/* CGI-SPEC 6.1.2 and FastCGI spec 6.3 */
		
		/* request.content_length < SSIZE_MAX, see request.c */
		ltostr(buf, con->request.content_length);
		fcgi_env_add(p->fcgi_env, CONST_STRING("CONTENT_LENGTH"), buf);
	}

	if (host->mode != FCGI_AUTHORIZER) {
		/*
		 * SCRIPT_NAME, PATH_INFO and PATH_TRANSLATED according to
		 * http://cgi-spec.golux.com/draft-coar-cgi-v11-03-clean.html
		 * (6.1.14, 6.1.6, 6.1.7)
		 * For AUTHORIZER mode these headers should be omitted.
		 */

		if (hctx->path_info_offset == 0) {  /* no pathinfo */
			fcgi_env_add(p->fcgi_env, CONST_STRING("SCRIPT_NAME"), con->uri.path->ptr);
			fcgi_env_add(p->fcgi_env, CONST_STRING("PATH_INFO"), "");
		} else {                                /* pathinfo */
			*(con->uri.path->ptr + hctx->path_info_offset) = '\0'; /* get sctipt_name part */
			fcgi_env_add(p->fcgi_env, CONST_STRING("SCRIPT_NAME"), con->uri.path->ptr);
			
			*(con->uri.path->ptr + hctx->path_info_offset) = '/';  /* restore uri.path */
			fcgi_env_add(p->fcgi_env, CONST_STRING("PATH_INFO"), con->uri.path->ptr + hctx->path_info_offset);
			
			if (host->docroot->used) {
				buffer_copy_string_buffer(p->path, host->docroot);
				buffer_append_string(p->path, con->uri.path->ptr + hctx->path_info_offset);
				fcgi_env_add(p->fcgi_env, CONST_STRING("PATH_TRANSLATED"), p->path->ptr);
			} else {
				fcgi_env_add(p->fcgi_env, CONST_STRING("PATH_TRANSLATED"), con->uri.path->ptr + hctx->path_info_offset);
			}
		}
	}

	/*
	 * SCRIPT_FILENAME and DOCUMENT_ROOT for php. The PHP manual
	 * http://www.php.net/manual/en/reserved.variables.php
	 * treatment of PATH_TRANSLATED is different from the one of CGI specs.
	 * TODO: this code should be checked against cgi.fix_pathinfo php
	 * parameter.
	 */

	if (host->docroot->used) {
		/* 
		 * rewrite SCRIPT_FILENAME 
		 * 
		 */
		
		buffer_copy_string_buffer(p->path, host->docroot);
		buffer_append_string_buffer(p->path, con->uri.path);
		
		fcgi_env_add(p->fcgi_env, CONST_STRING("SCRIPT_FILENAME"), p->path->ptr);
		fcgi_env_add(p->fcgi_env, CONST_STRING("DOCUMENT_ROOT"), host->docroot->ptr);
	} else {
		if (con->request.pathinfo->used) {
			fcgi_env_add(p->fcgi_env, CONST_STRING("PATH_INFO"), con->request.pathinfo->ptr);
		}
		
		fcgi_env_add(p->fcgi_env, CONST_STRING("SCRIPT_FILENAME"), con->physical.path->ptr);
		fcgi_env_add(p->fcgi_env, CONST_STRING("DOCUMENT_ROOT"), con->physical.doc_root->ptr);
	}
	fcgi_env_add(p->fcgi_env, CONST_STRING("REQUEST_URI"), con->request.uri->ptr);
	fcgi_env_add(p->fcgi_env, CONST_STRING("QUERY_STRING"), con->uri.query->used ? con->uri.query->ptr : "");
	fcgi_env_add(p->fcgi_env, CONST_STRING("REQUEST_METHOD"), get_http_method_name(con->request.http_method));
	fcgi_env_add(p->fcgi_env, CONST_STRING("REDIRECT_STATUS"), "200");
	fcgi_env_add(p->fcgi_env, CONST_STRING("SERVER_PROTOCOL"), get_http_version_name(con->request.http_version));
	
	fcgi_env_add_request_headers(srv, con, p);
	
	fcgi_header(&(header), FCGI_PARAMS, request_id, p->fcgi_env->used, 0);
	buffer_append_memory(hctx->write_buffer, (const char *)&header, sizeof(header));
	buffer_append_memory(hctx->write_buffer, (const char *)p->fcgi_env->ptr, p->fcgi_env->used);
	
	fcgi_header(&(header), FCGI_PARAMS, request_id, 0, 0);
	buffer_append_memory(hctx->write_buffer, (const char *)&header, sizeof(header));
	
	/* send FCGI_STDIN */
	
	/* something to send ? */
	for (offset = 0; offset != con->request.content_length; ) {
		/* send chunks of 1024 bytes */
		size_t toWrite = con->request.content_length - offset > 4096 ? 4096 : con->request.content_length - offset;
		
		fcgi_header(&(header), FCGI_STDIN, request_id, toWrite, 0);
		buffer_append_memory(hctx->write_buffer, (const char *)&header, sizeof(header));
		buffer_append_memory(hctx->write_buffer, (const char *)(con->request.content->ptr + offset), toWrite);
		
		offset += toWrite;
	}
	
	/* terminate STDIN */
	fcgi_header(&(header), FCGI_STDIN, request_id, 0, 0);
	buffer_append_memory(hctx->write_buffer, (const char *)&header, sizeof(header));

#if 0
	for (i = 0; i < hctx->write_buffer->used; i++) {
		fprintf(stderr, "%02x ", hctx->write_buffer->ptr[i]);
		if ((i+1) % 16 == 0) {
			size_t j;
			for (j = i-15; j <= i; j++) {
				fprintf(stderr, "%c", 
					isprint((unsigned char)hctx->write_buffer->ptr[j]) ? hctx->write_buffer->ptr[j] : '.');
			}
			fprintf(stderr, "\n");
		}
	}
#endif
	
	return 0;
}

static int fcgi_set_state(server *srv, handler_ctx *hctx, fcgi_connection_state_t state) {
	hctx->state = state;
	hctx->state_timestamp = srv->cur_ts;
	
	return 0;
}


static int fcgi_response_parse(server *srv, connection *con, plugin_data *p, buffer *in) {
	char *s, *ns;
	
	UNUSED(srv);

	/* \r\n -> \0\0 */
	
	buffer_copy_string_buffer(p->parse_response, in);
	
	for (s = p->parse_response->ptr; NULL != (ns = strstr(s, "\r\n")); s = ns + 2) {
		char *key, *value;
		int key_len;
		data_string *ds;
		
		ns[0] = '\0';
		ns[1] = '\0';
		
		key = s;
		if (NULL == (value = strchr(s, ':'))) {
			/* we expect: "<key>: <value>\n" */
			continue;
		}
		
		key_len = value - key;
		
		value++;
		/* strip WS */
		while (*value == ' ' || *value == '\t') value++;
		
		
		if (NULL == (ds = (data_string *)array_get_unused_element(con->response.headers, TYPE_STRING))) {
			ds = data_response_init();
		}
		buffer_copy_string_len(ds->key, key, key_len);
		buffer_copy_string(ds->value, value);
			
		array_insert_unique(con->response.headers, (data_unset *)ds);
		
		switch(key_len) {
		case 4:
			if (0 == strncasecmp(key, "Date", key_len)) {
				con->parsed_response |= HTTP_DATE;
			}
			break;
		case 6:
			if (0 == strncasecmp(key, "Status", key_len)) {
				con->http_status = strtol(value, NULL, 10);
				con->parsed_response |= HTTP_STATUS;
			}
			break;
		case 8:
			if (0 == strncasecmp(key, "Location", key_len)) {
				con->parsed_response |= HTTP_LOCATION;
			}
			break;
		case 10:
			if (0 == strncasecmp(key, "Connection", key_len)) {
				con->response.keep_alive = (0 == strcasecmp(value, "Keep-Alive")) ? 1 : 0;
				con->parsed_response |= HTTP_CONNECTION;
			}
			break;
		case 14:
			if (0 == strncasecmp(key, "Content-Length", key_len)) {
				con->response.content_length = strtol(value, NULL, 10);
				con->parsed_response |= HTTP_CONTENT_LENGTH;
			}
			break;
		default:
			break;
		}
	}
	
	/* CGI/1.1 rev 03 - 7.2.1.2 */
	if ((con->parsed_response & HTTP_LOCATION) &&
	    !(con->parsed_response & HTTP_STATUS)) {
		con->http_status = 302;
	}
	
	return 0;
}


static int fcgi_demux_response(server *srv, handler_ctx *hctx) {
	ssize_t len;
	int fin = 0;
	int b;
	ssize_t r;
	
	plugin_data *p    = hctx->plugin_data;
	connection *con   = hctx->remote_conn;
	int fcgi_fd       = hctx->fd;
	
	/* check how much we have to read */
	if (ioctl(hctx->fd, FIONREAD, &b)) {
		log_error_write(srv, __FILE__, __LINE__, "sd", "unexpected end-of-file (perhaps the fastcgi process died): ",
				fcgi_fd
				);
		return -1;
	}
	
	/* init read-buffer */
	if (hctx->rb == NULL) {
		hctx->rb = calloc(1, sizeof(*hctx->rb));
	}
	
	if (b > 0) {
		if (hctx->rb->size == 0) {
			hctx->rb->size = b;
			hctx->rb->ptr = malloc(hctx->rb->size * sizeof(*hctx->rb->ptr));
		} else if (hctx->rb->size < hctx->rb->used + b) {
			hctx->rb->size += b;
			hctx->rb->ptr = realloc(hctx->rb->ptr, hctx->rb->size * sizeof(*hctx->rb->ptr));
		}
		
		/* append to read-buffer */
		if (-1 == (r = read(hctx->fd, hctx->rb->ptr + hctx->rb->used, b))) {
			log_error_write(srv, __FILE__, __LINE__, "sds", 
					"unexpected end-of-file (perhaps the fastcgi process died):",
					fcgi_fd, strerror(errno));
			return -1;
		}
		
		/* this should be catched by the b > 0 above */
		assert(r);
		
		hctx->rb->used += r;
	} else {
		fcgi_connection_cleanup(srv, hctx);
		
		log_error_write(srv, __FILE__, __LINE__, "sd", "unexpected end-of-file (perhaps the fastcgi process died): ",
				fcgi_fd);
		
		return -1;
	}

	/* parse all fcgi packets 
	 * 
	 *   start: hctx->rb->ptr 
	 *   end  : hctx->rb->ptr + hctx->rb->used
	 * 
	 */
	while (fin == 0) {
		size_t request_id;
		
		if (hctx->response_len == 0) {
			FCGI_Header *header;
			
			if (hctx->rb->used - hctx->rb->offset < sizeof(*header)) {
				/* didn't get the full header packet (most often 0),
				 * but didn't recieved the final packet either
				 * 
				 * we will come back later and finish everything
				 * 
				 */
				
				hctx->delayed = 1;
#if 0
				log_error_write(srv, __FILE__, __LINE__, "sddd", "didn't get the full header: ",
						hctx->rb->used - hctx->rb->offset, sizeof(*header),
						fcgi_fd
						);
#endif
				break;
			}
#if 0
			fprintf(stderr, "fcgi-version: %02x\n", hctx->rb->ptr[hctx->rb->offset]);
#endif
			
			header = (FCGI_Header *)(hctx->rb->ptr + hctx->rb->offset);
			hctx->rb->offset += sizeof(*header);
			
			len = (header->contentLengthB0 | (header->contentLengthB1 << 8)) + header->paddingLength;
			request_id = (header->requestIdB0 | (header->requestIdB1 << 8));

			hctx->response_len = len;
			hctx->response_request_id = request_id;
			hctx->response_type = header->type;
			hctx->response_padding = header->paddingLength;
			
#if 0
			log_error_write(srv, __FILE__, __LINE__, "sddd", "offset: ",
					fcgi_fd, hctx->rb->offset, header->type
					);
#endif
			
		} else {
			len = hctx->response_len;
		}
		
		if (hctx->rb->used - hctx->rb->offset < hctx->response_len) {
			/* we are not finished yet */
			break;
		}
		
		hctx->response->ptr = hctx->rb->ptr + hctx->rb->offset;
		hctx->rb->offset += hctx->response_len;
#if 0
		log_error_write(srv, __FILE__, __LINE__, "sdd", "offset: ",
				fcgi_fd, hctx->rb->offset
				);
#endif
		
		/* remove padding */
#if 0
		hctx->response->ptr[hctx->response_len - hctx->response_padding] = '\0';
#endif
		hctx->response->used = hctx->response_len - hctx->response_padding + 1;
		
		/* mark the fast-cgi packet as finished */
		hctx->response_len = 0;
		
		switch(hctx->response_type) {
		case FCGI_STDOUT:
			if (len) {
#if 0
				log_error_write(srv, __FILE__, __LINE__, "sds", "len", len, hctx->response->ptr);
#endif
				
				if (0 == con->got_response) {
					con->got_response = 1;
					buffer_prepare_copy(hctx->response_header, 128);
				}
				
				if (0 == con->file_started) {
					char *c;
					
					/* search for the \r\n\r\n in the string */
					if (NULL != (c = buffer_search_string_len(hctx->response, "\r\n\r\n", 4))) {
						size_t hlen = c - hctx->response->ptr + 4;
						size_t blen = hctx->response->used - hlen - 1;
						/* found */
						
						buffer_append_string_len(hctx->response_header, hctx->response->ptr, c - hctx->response->ptr + 4);
#if 0
						log_error_write(srv, __FILE__, __LINE__, "ss", "Header:", hctx->response_header->ptr);
#endif
						/* parse the response header */
						fcgi_response_parse(srv, con, p, hctx->response_header);
						
						/* enable chunked-transfer-encoding */
						if (con->request.http_version == HTTP_VERSION_1_1 &&
						    !(con->parsed_response & HTTP_CONTENT_LENGTH)) {
							con->response.transfer_encoding = HTTP_TRANSFER_ENCODING_CHUNKED;
						}

						con->file_started = 1;
						
						if (blen) {
							http_chunk_append_mem(srv, con, c + 4, blen + 1);
							joblist_append(srv, con);
#if 0
							log_error_write(srv, __FILE__, __LINE__, "sd", "body-len", blen);
#endif
						}
					} else {
						/* copy everything */
						buffer_append_string_buffer(hctx->response_header, hctx->response);
					}
				} else {
					http_chunk_append_mem(srv, con, hctx->response->ptr, hctx->response->used);
					joblist_append(srv, con);
#if 0
					log_error_write(srv, __FILE__, __LINE__, "sd", "body-len", hctx->response->used);
#endif
				}
			} else {
				/* finished */
			}
			break;
		case FCGI_END_REQUEST:
			con->file_finished = 1;
			
			http_chunk_append_mem(srv, con, NULL, 0);
			joblist_append(srv, con);
			
			fin = 1;
			break;
		default:
			log_error_write(srv, __FILE__, __LINE__, "sd", "fcgi: header.type not handled: ", hctx->response_type);
			break;
		}
	}
	
	hctx->response->ptr = NULL;

	return fin;
}


static int fcgi_write_request(server *srv, handler_ctx *hctx) {
	plugin_data *p    = hctx->plugin_data;
	data_fastcgi *host= hctx->host;
	
	int r;
	
	if (!host || 
	    ((!host->host->used || !host->port) && !host->unixsocket->used)) return -1;
	
	switch(hctx->state) {
	case FCGI_STATE_INIT:
		r = host->unixsocket->used ? AF_UNIX : AF_INET;
		
		if (-1 == (hctx->fd = socket(r, SOCK_STREAM, 0))) {
			log_error_write(srv, __FILE__, __LINE__, "ss", "socket failed: ", strerror(errno));
			return -1;
		}
		hctx->fde_ndx = -1;
		
		fdevent_register(srv->ev, hctx->fd, fcgi_handle_fdevent, hctx);
		
		if (-1 == fdevent_fcntl_set(srv->ev, hctx->fd)) {
			log_error_write(srv, __FILE__, __LINE__, "ss", "fcntl failed: ", strerror(errno));
			
			fcgi_connection_cleanup(srv, hctx);
			
			return -1;
		}
		
		/* fall through */
	case FCGI_STATE_CONNECT:
		if (hctx->state == FCGI_STATE_INIT) {
			switch (fcgi_establish_connection(srv, hctx)) {
			case 1:
				/* comeback here */
				fcgi_set_state(srv, hctx, FCGI_STATE_CONNECT);
				return 0;
			case -1:
				hctx->fde_ndx = -1;
				return -1;
			default:
				/* everything is ok, go on */
				break;
			}
		} else {
			int socket_error;
			socklen_t socket_error_len = sizeof(socket_error);
			
			/* try to finish the connect() */
			getsockopt(hctx->fd, SOL_SOCKET, SO_ERROR, &socket_error, &socket_error_len);
			if (socket_error != 0) {
				log_error_write(srv, __FILE__, __LINE__, "ss", "getsockopt failed:", strerror(socket_error));
				
				return -1;
			}
		}
		
		if (hctx->request_id == 0) {
			hctx->request_id = fcgi_requestid_new(srv, p);
		} else {
			log_error_write(srv, __FILE__, __LINE__, "sd", "fcgi-request is already in use:", hctx->request_id);
		}
		
		fcgi_set_state(srv, hctx, FCGI_STATE_PREPARE_WRITE);
		/* fall through */
	case FCGI_STATE_PREPARE_WRITE:
		fcgi_create_env(srv, hctx, hctx->request_id);
		
		fcgi_set_state(srv, hctx, FCGI_STATE_WRITE);
		hctx->write_offset = 0;
		
		/* fall through */
	case FCGI_STATE_WRITE:
		/* continue with the code after the switch */
		if (-1 == (r = write(hctx->fd, 
				     hctx->write_buffer->ptr + hctx->write_offset, 
				     hctx->write_buffer->used - hctx->write_offset))) {
			if ((errno != EAGAIN) &&
			    (errno != EINTR)) {
				log_error_write(srv, __FILE__, __LINE__, "ssd", "write failed:", strerror(errno), r);
				
				return -1;
			} else {
				return 0;
			}
		}
		
		hctx->write_offset += r;
		
		if (hctx->write_offset == hctx->write_buffer->used) {
			fcgi_set_state(srv, hctx, FCGI_STATE_READ);
		}
		
		break;
	case FCGI_STATE_READ:
		/* waiting for a response */
		break;
	default:
		log_error_write(srv, __FILE__, __LINE__, "s", "(debug) unknown state");
		return -1;
	}
	
	return 0;
}


SUBREQUEST_FUNC(mod_fastcgi_handle_subrequest) {
	plugin_data *p = p_d;
	
	handler_ctx *hctx = con->plugin_ctx[p->id];
	data_fastcgi *host;
	
	if (NULL == hctx) return HANDLER_GO_ON;
	
	host = hctx->host;
	
	/* not my job */
	if (con->mode != p->id) return HANDLER_GO_ON;
	
	/* ok, create the request */
	if (-1 == fcgi_write_request(srv, hctx)) {
		log_error_write(srv, __FILE__, __LINE__,  "sbdd", "fcgi-server disabled:", 
				host->host,
				host->port,
				hctx->fd);
		
		/* disable this server */
		host->usage = -1;
		host->disable_ts = srv->cur_ts;
		
		hctx->state = FCGI_STATE_ERROR;
		fcgi_connection_cleanup(srv, hctx);
		
		con->mode = DIRECT;
		con->http_status = 503;
		return HANDLER_FINISHED;
	}
	
	if (con->file_started == 1) {
		return HANDLER_FINISHED;
	} else {
		return HANDLER_WAIT_FOR_EVENT;
	}
}

static handler_t fcgi_connection_close(server *srv, handler_ctx *hctx) {
	plugin_data *p;
	connection  *con;
	
	if (NULL == hctx) return HANDLER_GO_ON;
	
	p    = hctx->plugin_data;
	con  = hctx->remote_conn;
	
	if (con->mode != p->id) return HANDLER_GO_ON;
	
	log_error_write(srv, __FILE__, __LINE__, "ssdsd", 
			"emergency exit: fastcgi:", 
			"connection-fd:", con->fd,
			"fcgi-fd:", hctx->fd);
	
	
	
	fcgi_connection_cleanup(srv, hctx);
	
	return HANDLER_FINISHED;
}


static handler_t fcgi_handle_fdevent(void *s, void *ctx, int revents) {
	server      *srv  = (server *)s;
	handler_ctx *hctx = ctx;
	connection  *con  = hctx->remote_conn;
	plugin_data *p    = hctx->plugin_data;
	
	joblist_append(srv, con);
	
	if ((revents & FDEVENT_IN) &&
	    hctx->state == FCGI_STATE_READ) {
		switch (fcgi_demux_response(srv, hctx)) {
		case 0:
			break;
		case 1:
			hctx->host->usage--;
			
			if (hctx->host->mode == FCGI_AUTHORIZER && con->http_status == 200) {
				/*
				 * If we are here in AUTHORIZER mode then a request for autorizer
				 * was proceeded already, and status 200 has been returned. We need
				 * now to handle autorized request.
				 */

				buffer_copy_string_buffer(con->physical.path, hctx->host->docroot);
				buffer_append_string_buffer(con->physical.path, con->uri.path);
				con->mode = DIRECT;
			}
			
			/* we are done */
			fcgi_connection_cleanup(srv, hctx);
			
			return HANDLER_FINISHED;
		case -1:
			if (con->file_started == 0) {
				/* nothing has been send out yet, send a 500 */
				connection_set_state(srv, con, HANDLE_REQUEST);
				con->http_status = 500;
				con->mode = DIRECT;
			} else {
				/* response might have been already started, kill the connection */
				connection_set_state(srv, con, ERROR);
			}
			
			return HANDLER_FINISHED;
		}
	}
	
	if (revents & FDEVENT_OUT) {
		if (hctx->state == FCGI_STATE_CONNECT ||
		    hctx->state == FCGI_STATE_WRITE) {
			/* we are allowed to send something out
			 * 
			 * 1. in a unfinished connect() call
			 * 2. in a unfinished write() call (long POST request)
			 */
			return mod_fastcgi_handle_subrequest(srv, con, p);
		} else {
			log_error_write(srv, __FILE__, __LINE__, "sd", "fcgi: out", hctx->state);
		}
	}
	
	/* perhaps this issue is already handled */
	if (revents & FDEVENT_HUP) {
		
		if (hctx->state == FCGI_STATE_CONNECT) {
			/* getoptsock will catch this one (right ?)
			 * 
			 * if we are in connect we might get a EINPROGRESS in the first call 
			 * a FDEVENT_HUP in the second round
			 * 
			 * FIXME: as it is a bit ugly.
			 * 
			 */
			return mod_fastcgi_handle_subrequest(srv, con, p);
		}
		log_error_write(srv, __FILE__, __LINE__, "sbSBSDS", 
				"error: unexpected close of fastcgi connection for", 
				con->uri.path,
				"(no fastcgi process on host: ", 
				hctx->host->host,
				", port: ", 
				hctx->host->port,
				" ?)" );
		
#ifndef USE_LINUX_SIGIO
		fcgi_connection_close(srv, hctx);
# if 0
		log_error_write(srv, __FILE__, __LINE__, "sd", "fcgi-FDEVENT_HUP", con->fd);
# endif			
		return HANDLER_ERROR;
#endif
	} else if (revents & FDEVENT_ERR) {
		log_error_write(srv, __FILE__, __LINE__, "s", "fcgi: err");
		/* kill all connections to the fastcgi process */
		
		fcgi_connection_close(srv, hctx);
#if 1
		log_error_write(srv, __FILE__, __LINE__, "s", "fcgi-FDEVENT_ERR");
#endif			
		return HANDLER_ERROR;
	}
	
	return HANDLER_FINISHED;
}

static int fcgi_patch_connection(server *srv, connection *con, plugin_data *p, const char *stage, size_t stage_len) {
	size_t i, j;
	
	/* skip the first, the global context */
	for (i = 1; i < srv->config_context->used; i++) {
		data_config *dc = (data_config *)srv->config_context->data[i];
		plugin_config *s = p->config_storage[i];
		
		/* not our stage */
		if (!buffer_is_equal_string(dc->comp_key, stage, stage_len)) continue;
		
		/* condition didn't match */
		if (!config_check_cond(srv, con, dc)) continue;
		
		/* merge config */
		for (j = 0; j < dc->value->used; j++) {
			data_unset *du = dc->value->data[j];
			
			if (buffer_is_equal_string(du->key, CONST_STR_LEN("fastcgi.server"))) {
				p->conf.extensions = s->extensions;
			} else if (buffer_is_equal_string(du->key, CONST_STR_LEN("fastcgi.debug"))) {
				p->conf.debug = s->debug;
			}
		}
	}
	
	return 0;
}

static int fcgi_setup_connection(server *srv, connection *con, plugin_data *p) {
	plugin_config *s = p->config_storage[0];
	UNUSED(srv);
	UNUSED(con);
		
	p->conf.extensions = s->extensions;
	p->conf.debug = s->debug;
	
	return 0;
}



static handler_t fcgi_check_extension(server *srv, connection *con, void *p_d, int uri_path_handler) {
	plugin_data *p = p_d;
	size_t s_len;
	int used = -1;
	int ndx;
	size_t k, i;
	buffer *fn;
	data_array *extension = NULL;
	size_t path_info_offset;
	
	/* Possibly, we processed already this request */
	if (con->file_started == 1) return HANDLER_GO_ON;
	
	fn = uri_path_handler ? con->uri.path : con->physical.path;

	if (fn->used == 0) {
		return HANDLER_ERROR;
	}
	
	s_len = fn->used - 1;
	
	/* select the right config */
	fcgi_setup_connection(srv, con, p);
	for (i = 0; i < srv->config_patches->used; i++) {
		buffer *patch = srv->config_patches->ptr[i];
		
		fcgi_patch_connection(srv, con, p, CONST_BUF_LEN(patch));
	}
	
	path_info_offset = 0;
	
	/* check if extension matches */
	for (k = 0; k < p->conf.extensions->used; k++) {
		size_t ct_len;
		
		extension = (data_array *)p->conf.extensions->data[k];
		
		if (extension->key->used == 0) continue;
		
		ct_len = extension->key->used - 1;
		
		if (s_len < ct_len) continue;
		
		/* check extension in the form "/fcgi_pattern" */
		if (*(extension->key->ptr) == '/' && strncmp(fn->ptr, extension->key->ptr, ct_len) == 0) {
			if (s_len > ct_len + 1) {
				char *pi_offset;
				
				if (0 != (pi_offset = strchr(fn->ptr + ct_len + 1, '/'))) {
					path_info_offset = pi_offset - fn->ptr;
				}
			}
			break;
		} else if (0 == strncmp(fn->ptr + s_len - ct_len, extension->key->ptr, ct_len)) {
			/* check extension in the form ".fcg" */
			break;
		}
	}
	
	/* extension doesn't match */
	if (k == p->conf.extensions->used) {
		return HANDLER_GO_ON;
	}
	
	/* get best server */
	for (k = 0, ndx = -1; k < extension->value->used; k++) {
		data_fastcgi *host = (data_fastcgi *)extension->value->data[k];
		
		/* enable the server again, perhaps it is back again */
		if ((host->usage == -1) &&
		    (srv->cur_ts - host->disable_ts > FCGI_RETRY_TIMEOUT)) {
			host->usage = 0;
			
			log_error_write(srv, __FILE__, __LINE__,  "sbd", "fcgi-server re-enabled:", 
					host->host, host->port);
		}
		
		if (used == -1 || host->usage < used) {
			used = host->usage;
			
			ndx = k;
		}
	}
	
	/* found a server */
	if (ndx != -1) {
		data_fastcgi *host = (data_fastcgi *)extension->value->data[ndx];
		
		/* 
		 * if check-local is disabled, use the uri.path handler 
		 * 
		 */
		
		/* init handler-context */
		if (uri_path_handler) {
			if (host->check_local == 0) {
				handler_ctx *hctx;
				hctx = handler_ctx_init();
				
				hctx->path_info_offset = path_info_offset;
				hctx->remote_conn      = con;
				hctx->plugin_data      = p;
				hctx->host             = host;
				
				hctx->conf.extensions  = p->conf.extensions;
				hctx->conf.debug       = p->conf.debug;
				
				con->plugin_ctx[p->id] = hctx;
				
				host->usage++;
				
				con->mode = p->id;
			}
			return HANDLER_GO_ON;
		} else {
			handler_ctx *hctx;
			hctx = handler_ctx_init();
			
			hctx->path_info_offset = path_info_offset;
			hctx->remote_conn      = con;
			hctx->plugin_data      = p;
			hctx->host             = host;
			
			hctx->conf.extensions  = p->conf.extensions;
			hctx->conf.debug       = p->conf.debug;
			
			con->plugin_ctx[p->id] = hctx;
			
			host->usage++;
			
			con->mode = p->id;
			
			return HANDLER_FINISHED;
		}
	} else {
		/* no handler found */
		con->http_status = 500;
		
		log_error_write(srv, __FILE__, __LINE__,  "sb", 
				"no fcgi-handler found for:", 
				fn);
		
		return HANDLER_FINISHED;
	}
	return HANDLER_GO_ON;
}

/* uri-path handler */
static handler_t fcgi_check_extension_1(server *srv, connection *con, void *p_d) {
	return fcgi_check_extension(srv, con, p_d, 1);
}

/* start request handler */
static handler_t fcgi_check_extension_2(server *srv, connection *con, void *p_d) {
	return fcgi_check_extension(srv, con, p_d, 0);
}

JOBLIST_FUNC(mod_fastcgi_handle_joblist) {
	plugin_data *p = p_d;
	handler_ctx *hctx = con->plugin_ctx[p->id];
	
	if (hctx == NULL) return HANDLER_GO_ON;

	if (hctx->fd != -1) {
		switch (hctx->state) {
		case FCGI_STATE_READ:
			fdevent_event_add(srv->ev, &(hctx->fde_ndx), hctx->fd, FDEVENT_IN);
			
			break;
		case FCGI_STATE_CONNECT:
		case FCGI_STATE_WRITE:
			fdevent_event_add(srv->ev, &(hctx->fde_ndx), hctx->fd, FDEVENT_OUT);
			
			break;
		default:
			log_error_write(srv, __FILE__, __LINE__, "sd", "unhandled fcgi.state", hctx->state);
			break;
		}
	}

	return HANDLER_GO_ON;
}


static handler_t fcgi_connection_close_callback(server *srv, connection *con, void *p_d) {
	plugin_data *p = p_d;
	
	return fcgi_connection_close(srv, con->plugin_ctx[p->id]);
}

TRIGGER_FUNC(mod_fastcgi_handle_trigger) {
	plugin_data *p = p_d;
	
	UNUSED(srv);
	UNUSED(p);
	
	/* perhaps we should kill a connect attempt after 10-15 seconds
	 * 
	 * currently we wait for the TCP timeout which is on Linux 180 seconds
	 * 
	 * 
	 * 
	 */

	return HANDLER_GO_ON;
}


int mod_fastcgi_plugin_init(plugin *p) {
	p->name         = buffer_init_string("fastcgi");

	p->init         = mod_fastcgi_init;
	p->cleanup      = mod_fastcgi_free;
	p->set_defaults = mod_fastcgi_set_defaults;
	p->connection_reset        = fcgi_connection_reset;
	p->handle_connection_close = fcgi_connection_close_callback;
	p->handle_uri_clean        = fcgi_check_extension_1;
	p->handle_subrequest_start = fcgi_check_extension_2;
	p->handle_subrequest       = mod_fastcgi_handle_subrequest;
	p->handle_joblist          = mod_fastcgi_handle_joblist;
	p->handle_trigger          = mod_fastcgi_handle_trigger;
	
	p->data         = NULL;
	
	return 0;
}
