#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "base.h"
#include "log.h"
#include "buffer.h"

#include "plugin.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

typedef struct {
	array *access_deny;
} plugin_config;

typedef struct {
	PLUGIN_DATA;
	
	plugin_config **config_storage;
	
	plugin_config conf; 
} plugin_data;

INIT_FUNC(mod_access_init) {
	plugin_data *p;
	
	p = calloc(1, sizeof(*p));
	
	return p;
}

FREE_FUNC(mod_access_free) {
	plugin_data *p = p_d;
	
	UNUSED(srv);

	if (!p) return HANDLER_GO_ON;
	
	if (p->config_storage) {
		size_t i;
		for (i = 0; i < srv->config_context->used; i++) {
			plugin_config *s = p->config_storage[i];
			
			array_free(s->access_deny);
			
			free(s);
		}
		free(p->config_storage);
	}
	
	free(p);
	
	return HANDLER_GO_ON;
}

SETDEFAULTS_FUNC(mod_access_set_defaults) {
	plugin_data *p = p_d;
	data_config *dc;
	size_t i = 0;
	int ret = HANDLER_GO_ON;
	
	
	config_values_t cv[] = { 
		{ "url.access-deny",             NULL, T_CONFIG_ARRAY, T_CONFIG_SCOPE_CONNECTION },
		{ NULL,                          NULL, T_CONFIG_UNSET, T_CONFIG_SCOPE_UNSET }
	};
	
	p->config_storage = malloc(srv->config_context->used * sizeof(specific_config *));
	
	for (i = 0; i < srv->config_context->used; i++) {
		plugin_config *s;
		
		s = malloc(sizeof(plugin_config));
		s->access_deny    = array_init();
		
		cv[0].destination = s->access_deny;
		
		p->config_storage[i] = s;
		srv->config = ((data_config *)srv->config_context->data[i])->value;
	
		if (0 != (ret = config_insert_values(srv, cv))) {
			ret = HANDLER_ERROR;
			
			break;
		}
		
		ret = HANDLER_GO_ON;
	}
	
	if (NULL != (dc = (data_config *)array_get_element(srv->config_context, "global"))) {
		srv->config = dc->value;
	}
	
	return ret;
}

static int mod_access_patch_connection(server *srv, connection *con, plugin_data *p, const char *stage, size_t stage_len) {
	size_t i, j;
	
	/* skip the first, the global context */
	for (i = 1; i < srv->config_context->used; i++) {
		data_config *dc = (data_config *)srv->config_context->data[i];
		plugin_config *s = p->config_storage[i];
		
		/* not our stage */
		if (!buffer_is_equal_string(dc->comp_key, stage, stage_len)) continue;
		
		/* condition didn't match */
		if (!config_check_cond(srv, con, dc)) continue;
		
		/* merge config */
		for (j = 0; j < dc->value->used; j++) {
			data_unset *du = dc->value->data[j];
			
			if (buffer_is_equal_string(du->key, CONST_STR_LEN("url.access-deny"))) {
				p->conf.access_deny = s->access_deny;
			}
		}
	}
	
	return 0;
}

static int mod_access_setup_connection(server *srv, connection *con, plugin_data *p) {
	plugin_config *s = p->config_storage[0];
	UNUSED(srv);
	UNUSED(con);
		
	p->conf.access_deny = s->access_deny;
	
	return 0;
}


static handler_t mod_access_uri_handler(server *srv, connection *con, void *p_data) {
	plugin_data *p = p_data;
	int s_len;
	size_t k, i;
	
	UNUSED(srv);

	if (con->uri.path->used == 0) return HANDLER_GO_ON;
	
	mod_access_setup_connection(srv, con, p);
	for (i = 0; i < srv->config_patches->used; i++) {
		buffer *patch = srv->config_patches->ptr[i];
		
		mod_access_patch_connection(srv, con, p, CONST_BUF_LEN(patch));
	}
	
	s_len = con->uri.path->used - 1;
	
	for (k = 0; k < p->conf.access_deny->used; k++) {
		data_string *ds = (data_string *)p->conf.access_deny->data[k];
		int ct_len = ds->value->used - 1;
		
		if (ct_len > s_len) continue;
		if (ds->value->used == 0) continue;
		
		if (0 == strncmp(con->uri.path->ptr + s_len - ct_len, ds->value->ptr, ct_len)) {
			con->http_status = 403;
	
			return HANDLER_FINISHED;
		}
	}
	
	/* not found */
	return HANDLER_GO_ON;
}


int mod_access_plugin_init(plugin *p) {
	p->name        = buffer_init_string("access");
	
	p->init        = mod_access_init;
	p->set_defaults = mod_access_set_defaults;
	p->handle_uri_clean  = mod_access_uri_handler;
	p->cleanup     = mod_access_free;
	
	p->data        = NULL;
	
	return 0;
}
