#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "base.h"
#include "log.h"
#include "buffer.h"

#include "plugin.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/**
 * this is a httptls for a lighttpd plugin
 * 
 * just replaces every occurance of 'httptls' by your plugin name
 * 
 * e.g. in vim:
 * 
 *   :%s/httptls/myhandler/
 * 
 */



/* plugin config for all request/connections */

typedef struct {
	PLUGIN_DATA;
	buffer *match_buf;
	array *conf_require_tls;
} plugin_data;

/* init the plugin data */
INIT_FUNC(mod_httptls_init) {
	plugin_data *p;
	
	p = calloc(1, sizeof(*p));
	
	p->match_buf = buffer_init();
	p->conf_require_tls = array_init();
	
	return p;
}

/* detroy the plugin data */
FREE_FUNC(mod_httptls_free) {
	plugin_data *p = p_d;
	
	UNUSED(srv);

	if (!p) return HANDLER_GO_ON;
	
	buffer_free(p->match_buf);
	array_free(p->conf_require_tls);
	
	free(p);
	
	return HANDLER_GO_ON;
}

/* handle plugin config and check values */

SETDEFAULTS_FUNC(mod_httptls_set_defaults) {
	plugin_data *p = p_d;
	size_t i = 0;
	
	config_values_t cv[] = { 
		{ "httptls.require",            NULL, T_CONFIG_ARRAY, T_CONFIG_SCOPE_CONNECTION },       /* 0 */
		{ NULL,                         NULL, T_CONFIG_UNSET, T_CONFIG_SCOPE_UNSET }
	};
	
	if (!p) return HANDLER_ERROR;
	
	/* 0 */
	cv[i++].destination = p->conf_require_tls;
	
	if (0 != config_insert_values(srv, cv)) {
		return HANDLER_ERROR;
	}
	
	return HANDLER_GO_ON;
}

URIHANDLER_FUNC(mod_httptls_uri_handler) {
	plugin_data *p = p_d;
	int s_len;
	size_t k;
	
	UNUSED(srv);

	if (con->uri.path->used == 0) return HANDLER_GO_ON;
	
	s_len = con->uri.path->used - 1;
	
	for (k = 0; k < p->conf_require_tls->used; k++) {
		data_string *ds = (data_string *)p->conf_require_tls->data[k];
		int ct_len = ds->value->used - 1;
		
		if (ct_len > s_len) continue;
		if (ds->value->used == 0) continue;
		
		if (0 == strncmp(con->uri.path->ptr, ds->value->ptr, ct_len)) {
			if (con->request.http_version == HTTP_VERSION_1_1) {
				/* update required */
				con->http_status = 426;
				
				if (NULL == (ds = (data_string *)array_get_unused_element(con->request.headers, TYPE_STRING))) {
					ds = data_string_init();
				}
				buffer_copy_string(ds->key, "Upgrade");
				buffer_copy_string(ds->value, "TLS/1.0, HTTP/1.1");
				
				array_insert_unique(con->response.headers, (data_unset *)ds);
				
				if (NULL == (ds = (data_string *)array_get_unused_element(con->response.headers, TYPE_STRING))) {
					ds = data_string_init();
				}
				
				buffer_copy_string(ds->key, "Connection");
				buffer_copy_string(ds->value, "Upgrade");
				
				array_insert_unique(con->response.headers, (data_unset *)ds);
			} else {
				/* access denied */
				con->http_status = 403;
			}
			
			return HANDLER_FINISHED;
		}
	}
	
	/* not found */
	return HANDLER_GO_ON;
}

/* this function is called at dlopen() time and inits the callbacks */

int mod_httptls_plugin_init(plugin *p) {
	p->name        = buffer_init_string("httptls");
	
	p->init        = mod_httptls_init;
	p->handle_uri_clean  = mod_httptls_uri_handler;
	p->set_defaults  = mod_httptls_set_defaults;
	p->cleanup     = mod_httptls_free;
	
	p->data        = NULL;
	
	return 0;
}
