#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "base.h"
#include "log.h"
#include "buffer.h"

#include "plugin.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef USE_OPENSSL
# include <openssl/md5.h>
#else
# include "md5_global.h"
# include "md5.h"
#endif

#define HASHLEN 16
typedef unsigned char HASH[HASHLEN];
#define HASHHEXLEN 32
typedef char HASHHEX[HASHHEXLEN+1];
#ifdef USE_OPENSSL
#define IN const
#else
#define IN 
#endif
#define OUT

/**
 * this is a secdownload for a lighttpd plugin
 * 
 * just replaces every occurance of 'secdownload' by your plugin name
 * 
 * e.g. in vim:
 * 
 *   :%s/secdownload/myhandler/
 * 
 */



/* plugin config for all request/connections */

typedef struct {
	PLUGIN_DATA;
	buffer *conf_doc_root;
	buffer *conf_secret;
	buffer *conf_uri_prefix;
	buffer *md5;
	
	unsigned short conf_timeout;
} plugin_data;

/* init the plugin data */
INIT_FUNC(mod_secdownload_init) {
	plugin_data *p;
	
	p = calloc(1, sizeof(*p));
	
	p->conf_secret = buffer_init();
	p->conf_doc_root = buffer_init();
	p->conf_uri_prefix = buffer_init();
	p->md5 = buffer_init();
	
	return p;
}

/* detroy the plugin data */
FREE_FUNC(mod_secdownload_free) {
	plugin_data *p = p_d;
	UNUSED(srv);
	
	if (!p) return HANDLER_GO_ON;
	
	buffer_free(p->conf_secret);
	buffer_free(p->conf_doc_root);
	buffer_free(p->conf_uri_prefix);
	buffer_free(p->md5);
	
	free(p);
	
	return HANDLER_GO_ON;
}

/* handle plugin config and check values */

SETDEFAULTS_FUNC(mod_secdownload_set_defaults) {
	plugin_data *p = p_d;
	size_t i = 0;
	
	config_values_t cv[] = { 
		{ "secdownload.secret",            NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },       /* 0 */
		{ "secdownload.document-root",     NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },       /* 1 */
		{ "secdownload.uri-prefix",        NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },       /* 2 */
		{ "secdownload.timeout",           NULL, T_CONFIG_SHORT, T_CONFIG_SCOPE_CONNECTION },        /* 3 */
		{ NULL,                            NULL, T_CONFIG_UNSET, T_CONFIG_SCOPE_UNSET }
	};
	
	if (!p) return HANDLER_ERROR;
	
	/* 0 */
	cv[i++].destination = p->conf_secret;
	cv[i++].destination = p->conf_doc_root;
	cv[i++].destination = p->conf_uri_prefix;
	cv[i++].destination = &(p->conf_timeout);
	
	if (0 != config_insert_values(srv, cv)) {
		return HANDLER_ERROR;
	}
	
	if (p->conf_uri_prefix->used == 0) {
		buffer_copy_string(p->conf_uri_prefix, "/");
	}
	
	if (p->conf_secret->used == 0) {
		log_error_write(srv, __FILE__, __LINE__, "s",
				"secdownload.secret has to be set");
		return HANDLER_ERROR;
	}
	
	if (p->conf_doc_root->used == 0) {
		log_error_write(srv, __FILE__, __LINE__, "s",
				"secdownload.document-root has to be set");
		return HANDLER_ERROR;
	}
	
	if (p->conf_timeout == 0) {
		p->conf_timeout = 60;
	}
	
	return HANDLER_GO_ON;
}

/**
 * checks if the supplied string is a MD5 string
 * 
 * @param str a possible MD5 string
 * @return if the supplied string is a valid MD5 string 1 is returned otherwise 0
 */

int is_hex_len(const char *str, size_t len) {
	size_t i;
	
	if (NULL == str) return 0;
	
	for (i = 0; i < len && *str; i++, str++) {
		/* illegal characters */
		if (!((*str >= '0' && *str <= '9') ||
		      (*str >= 'a' && *str <= 'f') ||
		      (*str >= 'A' && *str <= 'F')) 
		    ) {
			return 0;
		}
	}
	
	return i == len;
}


URIHANDLER_FUNC(mod_secdownload_uri_handler) {
	plugin_data *p = p_d;
	MD5_CTX Md5Ctx;
	HASH HA1;
	const char *rel_uri, *ts_str, *md5_str;
	time_t ts = 0;
	size_t i;
	
	if (con->uri.path->used == 0) return HANDLER_GO_ON;
	
	/* 
	 *  /<uri-prefix>[a-f0-9]{32}/[a-f0-9]{8}/<rel-path>
	 */
	
	if (0 != strncmp(con->uri.path->ptr, p->conf_uri_prefix->ptr, p->conf_uri_prefix->used - 1)) return HANDLER_GO_ON;
	
	md5_str = con->uri.path->ptr + p->conf_uri_prefix->used - 1;
	
	if (!is_hex_len(md5_str, 32)) return HANDLER_GO_ON;
	if (*(md5_str + 32) != '/') return HANDLER_GO_ON;
	
	ts_str = md5_str + 32 + 1;
	
	if (!is_hex_len(ts_str, 8)) return HANDLER_GO_ON;
	if (*(ts_str + 8) != '/') return HANDLER_GO_ON;
	
	for (i = 0; i < 8; i++) {
		ts = (ts << 4) + hex2int(*(ts_str + i));
	}
	
	/* timed-out */
	if (srv->cur_ts - ts > p->conf_timeout || 
	    srv->cur_ts - ts < -p->conf_timeout) {
		con->http_status = 408;
		
		return HANDLER_FINISHED;
	}
	
	rel_uri = ts_str + 8;
	
	/* checking MD5 
	 * 
	 * <secret><rel-path><timestamp-hex>
	 */
	
	buffer_copy_string_buffer(p->md5, p->conf_secret);
	buffer_append_string(p->md5, rel_uri);
	buffer_append_string_len(p->md5, ts_str, 8);
	
	MD5_Init(&Md5Ctx);
	MD5_Update(&Md5Ctx, (unsigned char *)p->md5->ptr, p->md5->used - 1);
	MD5_Final(HA1, &Md5Ctx);
	
	buffer_copy_string_hex(p->md5, (char *)HA1, 16);
	
	if (0 != strncmp(md5_str, p->md5->ptr, 32)) {
		con->http_status = 403;
		
		log_error_write(srv, __FILE__, __LINE__, "sss", 
				"md5 invalid:",
				md5_str, p->md5->ptr);
		
		return HANDLER_FINISHED;
	}
	
	/* starting with the last / we should have relative-path to the docroot
	 */
	
	buffer_copy_string_buffer(con->physical.path, p->conf_doc_root);
	buffer_append_string(con->physical.path, rel_uri);
	
	return HANDLER_COMEBACK;
}

/* this function is called at dlopen() time and inits the callbacks */

int mod_secdownload_plugin_init(plugin *p) {
	p->name        = buffer_init_string("secdownload");
	
	p->init        = mod_secdownload_init;
	p->handle_uri_clean  = mod_secdownload_uri_handler;
	p->set_defaults  = mod_secdownload_set_defaults;
	p->cleanup     = mod_secdownload_free;
	
	p->data        = NULL;
	
	return 0;
}
