#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "base.h"
#include "log.h"
#include "buffer.h"
#include "file_cache.h"

#include "plugin.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

typedef struct {
	PLUGIN_DATA;
	buffer *conf_server_root;
	buffer *conf_default_host;
	buffer *conf_document_root;
	
	buffer *doc_root;
} plugin_data;

INIT_FUNC(mod_simple_vhost_init) {
	plugin_data *p;
	
	p = calloc(1, sizeof(*p));
	
	p->conf_server_root = buffer_init();
	p->conf_default_host = buffer_init();
	p->conf_document_root = buffer_init();
	
	p->doc_root = buffer_init();
	
	return p;
}

FREE_FUNC(mod_simple_vhost_free) {
	plugin_data *p = p_d;
	
	UNUSED(srv);

	if (!p) return HANDLER_GO_ON;
	
	buffer_free(p->conf_document_root);
	buffer_free(p->conf_default_host);
	buffer_free(p->conf_server_root);
	
	buffer_free(p->doc_root);
	
	free(p);
	
	return HANDLER_GO_ON;
}

SETDEFAULTS_FUNC(mod_simple_vhost_set_defaults) {
	plugin_data *p = p_d;
	
	config_values_t cv[] = { 
		{ "simple-vhost.server-root",       NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },
		{ "simple-vhost.default-host",      NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },
		{ "simple-vhost.document-root",     NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },
		{ NULL,                             NULL, T_CONFIG_UNSET, T_CONFIG_SCOPE_UNSET }
	};
	
	cv[0].destination = p->conf_server_root;
	cv[1].destination = p->conf_default_host;
	cv[2].destination = p->conf_document_root;
	
	if (0 != config_insert_values(srv, cv)) {
		return HANDLER_ERROR;
	}
	
	return HANDLER_GO_ON;
}

static int build_doc_root(server *srv, connection *con, plugin_data *p, buffer *out, buffer *host) {
	buffer_prepare_copy(out, 128);

	if (p->conf_server_root->used) {
		buffer_copy_string_buffer(out, p->conf_server_root);
		
		if (host->used) {
			/* a hostname has to start with a alpha-numerical character
			 * and must not contain a slash "/"
			 */
			char *dp;
			
			BUFFER_APPEND_SLASH(out);
			
			if (NULL == (dp = strchr(host->ptr, ':'))) {
				buffer_append_string_buffer(out, host);
			} else {
				buffer_append_string_len(out, host->ptr, dp - host->ptr);
			}
		}
		BUFFER_APPEND_SLASH(out);
		
		if (p->conf_document_root->used > 2 && p->conf_document_root->ptr[0] == '/') {
			buffer_append_string_len(out, p->conf_document_root->ptr + 1, p->conf_document_root->used - 2);
		} else {
			buffer_append_string_buffer(out, p->conf_document_root);
			BUFFER_APPEND_SLASH(out);
		}
	} else {
		buffer_copy_string_buffer(out, con->conf.document_root);
		BUFFER_APPEND_SLASH(out);
	}
	
	if (NULL == (con->fce = file_cache_get_entry(srv, con, out, con->fce))) {
		log_error_write(srv, __FILE__, __LINE__, "sb",
				strerror(errno), out);
		return -1;
	}
	
	if (!S_ISDIR(con->fce->st.st_mode)) {
		return -1;
	}
	
	return 0;
}


static handler_t mod_simple_vhost_docroot(server *srv, connection *con, void *p_data) {
	plugin_data *p = p_data;

	/*
	 * cache the last successfull translation from hostname (authority) to docroot
	 * - this saves us a stat() call
	 * 
	 */
	
	if (con->docroot_cache_key->used && 
	    con->uri.authority->used &&
	    buffer_is_equal(con->docroot_cache_key, con->uri.authority)) {
		/* cache hit */
		buffer_copy_string_buffer(con->physical.doc_root, con->docroot_cache_value);
		buffer_copy_string_buffer(con->server_name, con->uri.authority);
	} else {
		/* build document-root */
		if ((con->uri.authority->used == 0) ||
		    build_doc_root(srv, con, p, p->doc_root, con->uri.authority)) {
			/* not found, fallback the default-host */
			if (build_doc_root(srv, con, p, 
					   p->doc_root, 
					   p->conf_default_host)) {
				return HANDLER_GO_ON;
			}
		}
		
		/* copy to cache */
		buffer_copy_string_buffer(con->docroot_cache_key, con->uri.authority);
		buffer_copy_string_buffer(con->docroot_cache_value, p->doc_root);
		buffer_copy_string_buffer(con->physical.doc_root, p->doc_root);
		buffer_copy_string_buffer(con->server_name, con->uri.authority);
	}
	
	return HANDLER_GO_ON;
}


int mod_simple_vhost_plugin_init(plugin *p) {
	p->name        = buffer_init_string("simple_vhost");
	
	p->init        = mod_simple_vhost_init;
	p->set_defaults = mod_simple_vhost_set_defaults;
	p->handle_docroot  = mod_simple_vhost_docroot;
	p->cleanup     = mod_simple_vhost_free;
	
	p->data        = NULL;
	
	return 0;
}
