/*
 * path.cpp
 *
 * Copyright (C) 2003 Ed Cogburn <ecogburn@xtn.net>
 *
 * This file is part of the program "d: The Directory Lister".
 *
 * "d: The Directory Lister" is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.
 *
 * "d: The Directory Lister" is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with "d: The Directory Lister" in a file named "COPYING".
 * If not, visit their website at http://www.gnu.org for a copy, or request a copy by writing to:
 *     Free Software Foundation, Inc.
 *     59 Temple Place, Suite 330
 *     Boston, MA 02111-1307 USA
*/



/*
	To use path(), simply pass it the name of a file or directory.  path() will parse it into its components.  '/dir1/dir2/name.ext'
	becomes dir='/dir1/dir2', name='name', and ext='ext'.  Note that path() doesn't differentiate between a file and dir when it
	comes to (sub)components.  '/dir1/dir2/dir3/'  will get parsed into dir='/dir1/dir2' and name='dir3'.  If you pass it something
	which doesn't start with '/', path() will prepend the current working directory to its input and then normalize that.  If the
	input starts with '~' then path() prepends the current user's home directory to the input.  path() normalizes its input in order
	to remove '.' and '..' from the path.
*/



#include "globals.hpp"
#include <pwd.h>
#include <unistd.h>
#include <sys/types.h>



//-----------------------------------------------------------------------------
string path::get_home_dir()
{
	char buf[1024];
	string s;
	uid_t uid;
	struct passwd pw;
	struct passwd* pwptr;

	// This is what we'll return if error
	s = "";

	// Get real user id for this process
	uid = getuid();

	// Get pointer to password entry for this user
	getpwuid_r(uid, &pw, buf, sizeof(buf), &pwptr);

	// Return home dir if no error
	if (pwptr == &pw)
		s = pw.pw_dir;
	return s;
}



//-----------------------------------------------------------------------------
string path::get_cw_dir()
{
	string s;
	char buf[PATH_MAX];

	getcwd(buf, PATH_MAX);

	s = buf;

	return s;
}



//-----------------------------------------------------------------------------
void path::normalize_auxillary(const char* source_string)
{
	unsigned idx1;
	unsigned idx2;

	fdir = "";
	fname = "";
	fext = "";

	string src = source_string;

	if (src == "")
		throw runtime_error("path():  Null string as input!");

	// Get home directory if '~' is used
	if (src[0] == '~') {
		string s;

		src.erase(0, 1);
		s = src;
		src = get_home_dir();
		if (src == "")
			throw runtime_error("path():  Can't read user's home directory!");
		src += s;
	}

	// Force to an absolute path
	if (src[0] != '/') {
		string s;

		s = get_cw_dir();
		s += '/';
		s += src;
		src = s;
	}

	// Temporarily force a '/' at the end of the dir string to allow tests for "/."
	// and "/.." to work for the case where the "/." or "/.." is at the end of the string.
	if (src[src.size() - 1] != '/')
		src += '/';

	if (src.size() == 1) {  // It can only be '/', so we're done
		fdir = "/";
		return;
	}

	// Now remove duplicate separators, i.e. "//", from the dir string.
	while (1) {
		idx1 = src.find("//", 0);
		if (idx1 == src.npos)
			break;
		src.erase(idx1, 1);
	}

	// Now remove "/." (reference to the current directory) from the dir string.
	while (1) {
		idx1 = src.find("/./", 0);
		if (idx1 == src.npos)
			break;
		src.erase(idx1, 2);
	}

	// Now remove "/.." (reference to the previous directory) from the dir string along with the previous directory.
	while (1) {
		idx1 = src.find("/../", 0);
		if (idx1 == src.npos)
			break;
		if (idx1 == 0) {
			src.erase(idx1, 3);  // Keep last '/' to show absolute path
		} else {
			// Search for previous directory by looking for its preceding '/'
			idx2 = src.find_last_of('/', idx1 - 1);
			if (idx2 == src.npos) {
				stringstream ss;
				ss << "path():  Parse failure at checkpoint 1 of path \"" << source_string << "\"!";
				throw logic_error(ss.str());
			} else
				src.erase(idx2, idx1 - idx2 + 3);  // Remove starting from the '/' at the beginning of previous directory
		}
	}

	// Now remove the trailing '/' from the directory string.
	if (src.size() > 1 && src[src.size() - 1] == '/')
		src.erase(src.size() - 1);

	// Root by itself is a valid "directory".  We're done.
	if (src == "/") {
		fdir = src;
		return;
	}

	// Now get extension
	idx1 = src.find_last_of('.');
	idx2 = src.find_last_of('/');
	if (idx1 != src.npos) {
		if (idx2 != src.npos && idx1 > idx2) {  // Only in the filename portion.
			if (idx1 != src.npos && src[idx1 - 1] != '/') {  // if period follows a '/' then its a hidden file/dir, so skip
				fext = src.substr(idx1 + 1);
				src.erase(idx1);
			}		
		}
	}

	// Now get filename
	idx1 = src.find_last_of('/');
	if (idx1 == src.npos) {
		stringstream ss;
		ss << "path():  Parse failure at checkpoint 2 of path \"" << source_string << "\"!";
		throw logic_error(ss.str());
	} else {
		fname = src.substr(idx1 + 1);
		if (idx1 == 0)  
			fdir = '/';  // A file/subdir in the root directory
		else
			fdir = src.substr(0, idx1);  // Everything before fname is fdir
	}

	if (fdir == "") {
		stringstream ss;
		ss << "path():  Parse failure at checkpoint 3 of path \"" << source_string << "\"!";
		throw logic_error(ss.str());
	}
}



//-----------------------------------------------------------------------------
void path::normalize(const string& src)
{
	normalize_auxillary(src.c_str());
}



//-----------------------------------------------------------------------------
void path::normalize(const char* src)
{
	normalize_auxillary(src);
}



//-----------------------------------------------------------------------------
string path::full() const
{
	string s = "";

	if (fdir != "") {
		s = fdir;
		if (s != "/")
			s += '/';
	}

	if (fname != "")
		s += fname;

	if (fext != "") {
		s += '.';
		s += fext;
	}

	return s;
}



//-----------------------------------------------------------------------------
string path::dir() const
{
	string s = fdir;

	return s;
}



//-----------------------------------------------------------------------------
string path::name() const
{
	string s = fname;

	return s;
}



//-----------------------------------------------------------------------------
string path::name_ext() const
{
	string s = fname;

	if (fext != "") {
		s += '.';
		s += fext;
	}

	return s;
}



//-----------------------------------------------------------------------------
string path::ext() const
{
	string s = fext;

	return s;
}



//-----------------------------------------------------------------------------
path& path::operator= (const path& rhs)
{
	fdir = rhs.dir();
	fname = rhs.name();
	fext = rhs.ext();

	return *this;
}



//-----------------------------------------------------------------------------
path& path::operator= (const string& rhs)
{
	if (rhs == "") {
		fdir = "";
		fname = "";
		fext = "";
	} else
		normalize_auxillary(rhs.c_str());

	return *this;
}



//-----------------------------------------------------------------------------
path& path::operator= (const char* rhs)
{
	if (rhs == NULL || rhs[0] == '\0') {
		fdir = "";
		fname = "";
		fext = "";
	} else
		normalize_auxillary(rhs);

	return *this;
}



//-----------------------------------------------------------------------------
bool path::operator< (const path& rhs)
{
	return (full() < rhs.full());
}



//-----------------------------------------------------------------------------
bool path::operator< (const string& rhs)
{
	path tmp = rhs;

	return (full() < tmp.full());
}



//-----------------------------------------------------------------------------
bool path::operator< (const char* rhs)
{
	path tmp = rhs;

	return (full() < tmp.full());
}



//-----------------------------------------------------------------------------
bool path::operator> (const path& rhs)
{
	return (full() > rhs.full());
}



//-----------------------------------------------------------------------------
bool path::operator> (const string& rhs)
{
	path tmp = rhs;

	return (full() > tmp.full());
}



//-----------------------------------------------------------------------------
bool path::operator> (const char* rhs)
{
	path tmp = rhs;

	return (full() > tmp.full());
}



//-----------------------------------------------------------------------------
bool path::operator== (const path& rhs)
{
	return (full() == rhs.full());
}



//-----------------------------------------------------------------------------
bool path::operator== (const string& rhs)
{
	path tmp = rhs;

	return (full() == tmp.full());
}



//-----------------------------------------------------------------------------
bool path::operator== (const char* rhs)
{
	path tmp = rhs;

	return (full() == tmp.full());
}



//-----------------------------------------------------------------------------
path::path(const char* srcdir, const char* srcname, const char* srcext)
{
	string src;

	src = srcdir;
	src += '/';
	src += srcname;
	src += '/';
	src += srcext;

	normalize_auxillary(src.c_str());
}



//-----------------------------------------------------------------------------
path::path(const string srcdir, const string srcname)
{
	string src;

	src = srcdir;
	src += '/';
	src += srcname;

	normalize_auxillary(src.c_str());
}



//-----------------------------------------------------------------------------
path::path(const string& src)
{
	normalize_auxillary(src.c_str());
}



//-----------------------------------------------------------------------------
path::path(const char* src)
{
	normalize_auxillary(src);
}



//-----------------------------------------------------------------------------
path::path()
{
	fdir = "";
	fname = "";
	fext = "";
}
