#ifndef GCC_DCMPLR_CODEGEN_H
#define GCC_DCMPLR_CODEGEN_H

#include "d-irstate.h"

typedef integer_t xdmd_integer_t;

class ModuleInfo {
public:
    // Assuming only one module is compiled per program instances...
    Array imports; // Array of Module*
    Array classes; // Array of ClassDeclaration*
    Array ctors; // Arrays of FuncDeclaration*
    Array dtors;
    Array unitTests;
};

typedef enum {
    LIBCALL_ASSERT,
    LIBCALL_ARRAY_BOUNDS,
    LIBCALL_SWITCH_ERROR,
    LIBCALL_INVARIANT,
    LIBCALL_NEWCLASS,
    LIBCALL_NEW,
    LIBCALL_NEWARRAYI,
    LIBCALL_NEWBITARRAY,
    LIBCALL_DELCLASS,
    LIBCALL_DELARRAY,
    LIBCALL_CALLFINALIZER,
    LIBCALL_ARRAYSETLENGTH,
    LIBCALL_ARRAYSETLENGTH_B,
    LIBCALL_DYNAMIC_CAST,
    LIBCALL_ADEQ,
    LIBCALL_ADEQBIT,
    LIBCALL_ADCMP,
    LIBCALL_ADCMPCHAR,
    LIBCALL_ADCMPBIT,
    LIBCALL_AAIN,
    LIBCALL_AAGET,
    LIBCALL_AADEL,
    LIBCALL_ARRAYCAST,
    LIBCALL_ARRAYCAST_FROMBIT,
    LIBCALL_ARRAYCOPY,
    LIBCALL_ARRAYCOPYBIT,
    LIBCALL_ARRAYCAT,
    LIBCALL_ARRAYCATN,
    LIBCALL_ARRAYAPPEND,
    LIBCALL_ARRAYAPPENDC,
    LIBCALL_ARRAYSETBIT,
    LIBCALL_MONITORENTER,
    LIBCALL_MONITOREXIT,
    LIBCALL_CRITICALENTER,
    LIBCALL_CRITICALEXIT,
    LIBCALL_THROW,
    LIBCALL_SWITCH_STRING,
    LIBCALL_SWITCH_USTRING,
    LIBCALL_GNU_BITARRAYSLICE, // %% todo: inline bounds check and just use 
    LIBCALL_GNU_BITARRAYSLICEP, // the pointer form
    LIBCALL_GNU_COPYTOBITARRAYSLICE,
    LIBCALL_count
} LibCall;

// Code generation routines should be in a separate namespace, but so many
// routines need a reference to an IRState to expand Expressions.  Solution
// is to make IRState the code generation namespace with the actual IR state
// routines as a mixin.  There is still the global 'gen' which can be used when
// it is known the full function IR state is not needed. Also a lot of routines
// are static (don't need IR state or expand Expressions), but are still called
// using . or ->.

// Functions without a verb create trees
// Functions with 'do' affect the current instruction stream (or output assembler code.)
// functions with other names are for attribute manipulate, etc.

struct IRState : IRBase
{
    typedef enum  {
	INTRINSIC_BSF, INTRINSIC_BSR,
	INTRINSIC_BT, INTRINSIC_BTC, INTRINSIC_BTR, INTRINSIC_BTS,
	INTRINSIC_BSWAP,
	INTRINSIC_INP, INTRINSIC_INPW, INTRINSIC_INPL,
	INTRINSIC_OUTP, INTRINSIC_OUTPW, INTRINSIC_OUTPL,
	INTRINSIC_count
    } Intrinsic;

    static void doLineNote(const Loc & loc);

    // ** Declaration maninpulation
    static void setDeclLoc(tree t, const Loc & loc)
    {
	// Backend will often crash if the DECL_SOURCE_FILE is not set...
	//assert(loc.filename);TODO
	DECL_SOURCE_FILE (t) = loc.filename; // %%line number?
	DECL_SOURCE_LINE (t) = loc.linnum;
    }

    // Some DMD Declarations don't have the loc set, this search decl's parents
    // until a valid loc is found %%TODO: update for 0.81
    static void setDeclLoc(tree t, Dsymbol * decl);

    // Set a DECL's STATIC and EXTERN based on the decl's storage class
    // and if it is to be emitted in this module.
    static void setupSymbolStorage(Dsymbol * decl, tree decl_tree, bool force_static_public = false);
    
    // Definitely in static data, but not neccessarily this module.
    // Assumed to be public data.
    static void setupStaticStorage(Dsymbol * dsym, tree decl_tree);

    // Called just before symbol compilation to handle 'once only' and
    // 'static constructor' attributes.
    static void prepareSymbolOutput(Dsymbol * dsym, tree decl_tree);
    static void outputStaticSymbol(tree t) {
	// This may be needed for debugging, but doesn't seem to do anything yet...
	d_add_global_function(t);
	
	// %%TODO: flags
	rest_of_decl_compilation(t, NULL, 1, 0);
    }

    static void declareType(tree t, Dsymbol * d_sym);

    // Hack for systems without linkonce support
    static bool emitTemplates;
    static bool shouldEmit(Dsymbol * d_sym);
    
    // ** Type conversion
    
    // 'convertTo' just to give it a different name from the extern "C" convert
    tree convertTo(Expression * exp, Type * target_type);
    tree convertTo(tree exp, Type * exp_type, Type * target_type);
    
    tree convertForAssignment(Expression * exp, Type * target_type);
    tree convertForAssignment(tree exp_tree, Type * exp_type, Type * target_type);

    tree convertForInitialization(Expression * exp, Declaration * target_decl); // handle refs %% needed for constructing init value for _d_newarrayi
    tree convertForInitialization(tree exp_tree, Type * exp_type, Declaration * target_decl);
    tree convertForArgument(Expression * exp, Argument * arg);
protected:
    tree doConvertForInit(Expression * exp, tree in_exp_tree, Type * in_exp_type, Declaration * target_decl);
public:
    tree convertForCondition(Expression * exp);
    tree convertForVarArg(Expression * exp);
    tree rawArray(Expression * exp);

    // Can't use VAR_DECLs for the DECL_INITIAL of static varibles or in CONSTRUCTORSs
    static tree stripVarDecl(tree value);

    
    // ** Type management
    static Type * getDType(tree t) {
	// %% TODO: assert that its a type node..
	struct lang_type * l = TYPE_LANG_SPECIFIC( t );
	return l ? l->d_type : 0;
    }

    unsigned getPointerSize()
    {
	return int_size_in_bytes( TREE_TYPE( null_pointer_node ));
    }

    static Type * getObjectType() {
	return ClassDeclaration::classinfo->baseClass->type;
    }

    // Routines to handle variables that are references.
    static bool isDeclarationReferenceType(Declaration * decl);
    static tree trueDeclarationType(Declaration * decl);
    static bool isArgumentReferenceType(Argument * arg);
    static tree trueArgumentType(Argument * arg);

    static tree arrayType(Type * d_type, xdmd_integer_t size) // %% use of integer_t
    { return arrayType(d_type->toCtype(), size); }
    static tree arrayType(tree type_node, xdmd_integer_t size);

    // Can't call this until common types have been built
    static bool haveLongDouble() {
	return TYPE_MODE(long_double_type_node) != TYPE_MODE(double_type_node);
    }

    // ** Simple constants

    static tree nullPointer() { return d_null_pointer; }
    static tree integerConstant(xdmd_integer_t value, tree type = 0) {
	// Assuming xdmd_integer_t is 64 bits
#if HOST_BITS_PER_WIDE_INT == 32
	tree tree_value = build_int_2(value & 0xffffffff, (value >> 32) & 0xffffffff);
#elif HOST_BITS_PER_WIDE_INT == 64
	tree tree_value = build_int_2(value, 0);
#else
#error Fix This
#endif
	if (type)
	    TREE_TYPE( tree_value ) = type;
	return tree_value;
    }
    static tree integerConstant(xdmd_integer_t value, Type * type) {
	return integerConstant(value, type->toCtype());
    }

    static Expression * sizeInt(xdmd_integer_t value) {
	Loc loc;
	return new IntegerExp(loc, value, Type::tsize_t);
    }
    tree floatConstant( real_t value, TypeBasic * target_type );

    // ** Routines for built in structured types

    static tree realPart(tree c);
    static tree imagPart(tree c);
    
    // Dynamic arrays
    static tree darrayPtrRef(tree exp);
           tree darrayPtrRef(Expression * e) { return darrayPtrRef(e->toElem(this)); }
    static tree darrayLenRef(tree exp);
    
    static tree darrayVal(tree type, tree len, tree data);
    // data may be NULL for a null pointer value
    static tree darrayVal(tree type, unsigned len, tree data);
    static tree darrayVal(Type * type, unsigned len, tree data) {
	return darrayVal(type->toCtype(), len, data);
    }
    static tree darrayString(const char * str);

    // Length of either a static or dynamic array
    tree arrayLength(Expression * exp)
    { return arrayLength(exp->toElem(this), exp->type); }
    static tree arrayLength(tree exp, Type * exp_type);

    // Delegates
    static tree delegateMethodRef(tree exp);
    static tree delegateObjectRef(tree exp);
    static tree delegateVal(tree method_exp, tree object_exp, Type * d_type);
    // These are for references to nested functions/methods as opposed to a variable of
    // type Tdelegate
    tree methodCallExpr(tree callee, tree object, Type * d_type) {
	tree t = delegateVal(callee, object, d_type);
	D_IS_METHOD_CALL_EXPR( t ) = 1;
	return t;
    }
    void extractMethodCallExpr(tree mcr, tree & callee_out, tree & object_out);
    tree objectInstanceMethod(Expression * obj_exp, FuncDeclaration * func, Type * d_type);

    static tree twoFieldType(Type * ft1, Type * ft2, Type * d_type = 0);
    
    // ** Various expressions
    static tree aggDeclStaticInit(AggregateDeclaration * agg_decl);

    static tree addressOf(tree exp) {
	d_mark_addressable(exp);
	// %% build a special type that deals with aliasing
	// maybe in convert addresses -- carry alias set of original
	// addressed type (also need to consider type that an
	// address field is a member of, etc.)
	return build1(ADDR_EXPR, build_pointer_type(TREE_TYPE(exp)), exp);
    }
    static tree addressOf(Dsymbol *d) {
	return addressOf( d->toSymbol()->Stree );
    }

    tree pointerIntSum(Expression * ptr_exp, Expression * idx_exp)
    { return pointerIntSum(ptr_exp->toElem(this), idx_exp->toElem(this)); }
    tree pointerIntSum(tree ptr_node, tree idx_exp);

    // DMD allows { void[] a; & a[3]; }
    static tree
    pvoidOkay(tree t) {
	if ( VOID_TYPE_P( TREE_TYPE( TREE_TYPE( t ))) ) {
	    // ::warning("indexing array of void"); 
	    return convert(Type::tuns8->pointerTo()->toCtype(), t);
	} else
	    return t;
    }

    tree boolOp(enum tree_code code, tree a, tree b) {
	return build(code, boolean_type_node, a, b);
    }

    tree checkedIndex(Loc loc, tree index, tree upper_bound, bool inclusive);
    tree boundsCond(tree index, tree upper_bound, bool inclusive);
    
    // Returns something that can be used as an l-value or an r-value unless
    // the array is of type bit in which case it can only be used as an r-value
    tree arrayElemRef(IndexExp * aer_exp);

    // Returns something that can be used as an l-value.  For bit arrays, also
    // creates expressions needed to modify or select bits

    // Only one of or_mask_out or shift_count_out is needed .. not
    // sure which is better to use...
    tree arrayElemRef(IndexExp * aer_exp,
	tree * and_mask_out, tree * or_mask_out, tree * shift_count_out);
    
    bool isBitArrayAccess(Expression * e) {
	if (e->op == TOKarray) {
	    Type * e1_basetype = e1_basetype = ((IndexExp *) e)->e1->type->toBasetype();
	    // assuming != Taarray is good enough for == Tarray,Tsarray,Tpointer
	    return e1_basetype->ty != Taarray && e1_basetype->next->isbit();
	} else {
	    return false;
	}
    }
    
    static void extractSliceExp(Expression * slice_exp,
	Expression * & array_out, Expression * & lwr_out, Expression * & upr_out) {
	// %% assumes TOKrange // assert(slice_exp->op == TOKrange);
	array_out = ((SliceExp*) slice_exp)->e1;
	lwr_out = ((SliceExp*) slice_exp)->lwr;
	upr_out = ((SliceExp*) slice_exp)->upr;
    }
    typedef enum {
	GS_CreateBounds = 0x1,
	GS_BoundsCheck = 0x2,
	GS_ExtractPointer = 0x4, // return pointer to start of array/pointer
	GS_ExtractDArray = 0x8, // unimpl: return darray with start of array; fails for pointer
	GS_MakePointer = 0xc, // unimpl: return pointer to array[lwr]
	GS_MakeDArray = 010, // unimpl: return array[lwr..upr]
	GS_ProcessMask = 0x1c
    } GrokSliceFlags;
    void grokSlice(Expression * slice_exp, tree & array_out, tree & lwr_out, tree & upr_out, int flags = GS_CreateBounds);
    
    // ** Function calls
    tree call(FuncDeclaration * func_decl, Array * args);
    tree call(Expression * expr, Array * arguments);
    tree call(tree callable, tree object, Array * arguments);
    
    tree libCall(LibCall lib_call, unsigned n_args, Expression ** args, tree result_type = 0);
    tree libCall(LibCall lib_call, Array * args, tree result_type = 0);
    tree assertCall(Loc loc, LibCall libcall = LIBCALL_ASSERT);
    static FuncDeclaration * getLibCallDecl(LibCall lib_call);
    static void replaceLibCallDecl(FuncDeclaration * d_decl);
    // This does not perform conversions on the arguments.  This allows
    // arbitrary data to be passed through varargs without going through the
    // usual conversions.
    static tree libCall(LibCall lib_call, unsigned n_args, tree *args, tree force_result_type = 0);

    // GCC 3.3 does not set TREE_SIDE_EFFECTS call by default. GCC 3.4
    // sets it depending on the const/pure attributes of the funcion
    // and the SIDE_EFFECTS flags of the arguments.

    // We don't have a way to indicate const/pure functions in D, so
    // always set this flag
    static tree buildCall(tree type, tree callee, tree args) {
	tree t = build(CALL_EXPR, type, callee, args);
#if D_GCC_VER <= 33
	TREE_SIDE_EFFECTS(t) = 1;
#endif
	return t;
    }
    
protected:
    static tree maybeExpandSpecialCall(tree call_exp);
public:
    tree floatMod(tree a, tree b, Type * d_type);

    Expression * typeinfoReferenceExp(Type * t, const Loc loc = 0);
    tree typeinfoReference(Type * t, const Loc loc = 0);
    
    // ** Other

    static Module * builtinsModule;
    static Module * intrinsicModule;
    static void setBuiltinsModule(Module * mod) { builtinsModule = mod; }
    static void setIntrinsicModule(Module * mod) { intrinsicModule = mod; }
    //static Module * getBuiltinsModule() { return builtinsModule; }
    static bool maybeSetUpBuiltin(Declaration * decl);

    static tree functionPointer(FuncDeclaration * func_decl);
    
    // Returns the address of the thunk function (not sure if this can
    // disturb the instruction stream -- %%TODO)  for use in the current module.
    // Thunk function bodies are only created once and only in the module
    // of the method's class.
    tree doThunk(unsigned offset, FuncDeclaration * target);
protected:
    tree doThunk1(unsigned offset, FuncDeclaration * target);
public:

    tree doSimpleFunction(Module * mod, const char * name, tree expr, bool static_ctor);
    tree doFunctionToCallFunctions(Module * mod, const char * name, Array * functions);
	
    // Create a tree node to set multiple elements to a single value
    static tree arraySet(tree ptr, tree src, tree count) {
	tree exp = build( (enum tree_code) D_ARRAY_SET_EXPR, void_type_node,
	    ptr, src, count);
	TREE_SIDE_EFFECTS( exp ) = 1;
	TREE_READONLY( exp ) = 1;
	return exp;
    }

    // Common code for ClassDeclaration and StructDeclaration
    // 1. Do type compilation/STUB_DECL stuff on the type.
    // 2. emit child declarations
    // 3. emit static initializer
    void aggregateToObjFileCommon(AggregateDeclaration * agg_decl, tree type_node);

    tree aggregateInitializer(AggregateDeclaration * agg_decl,
	Array * ini_fields, Array * ini_values, tree ini_nodes);

    // Create an array of pointers to objects declared by a Declaration
    //   vtbl:        array of Declaration
    //   ci_override: overrides element zero -- handles ClassDeclaration.vtbl.data[0] which
    //                seems to always be Object's ClassDeclaration
    tree vtable(Array * vtbl, ClassDeclaration * ci_override = 0);

    // Returns the D object that was thrown.  Different from the generic exception pointer
    tree exceptionObject();

    // ** Instruction stream manipulation

    static void
    doJump(Statement * stmt, tree label_tree)
    {
	// %%TODO: c-semantics.c:expand_stmt GOTO_STMT branch prediction
	TREE_USED( label_tree ) = 1 ;
	doLineNote( stmt->loc );
	expand_goto( label_tree );
    }

    // ** Callback statement evalutation

    static Array stmtExprList;
    
    static tree makeStmtExpr(Statement * statement, IRState * irs);
    static void retrieveStmtExpr(tree t, Statement ** s_out, IRState ** i_out);
    
    // ** Module info.  Assuming only one module per run of the compiler.
    static ModuleInfo moduleInfo;
    ModuleInfo & getModuleInfo() { return moduleInfo; }

};

struct BitConfig {
    Type * elemType;
    unsigned shift;
    unsigned mask;
    tree mask_tree;
    tree shift_tree;
    unsigned bitCountToBytes(unsigned bc) {
	return ((bc + mask) >> shift) * elemType->size();
    }
    unsigned bitCountToWords(unsigned bc) {
	return (bc + mask) >> shift;
    }
    void setType(Type * type);
};

extern BitConfig bitConfig;

extern IRState gen;

// Various helpers that need extra state

struct WrappedExp : Expression
{
    tree exp_node;
    WrappedExp(Loc loc, enum TOK op, tree exp_node, Type * type);
    void toCBuffer(OutBuffer *buf);
    elem *toElem(IRState *irs);
};

class AggLayout {
public:
    AggregateDeclaration * aggDecl;
    tree aggType;
    AggLayout(AggregateDeclaration * ini_agg_decl, tree ini_agg_type) :
	aggDecl(ini_agg_decl),
	aggType(ini_agg_type) { }
    void addFields(Array * fields); // array of VarDeclaration
    void addField(tree field_decl, unsigned offset); // FIELD_DECL
    void finish();
};

// It might be possible to make this more simple by using targetm stuff
// target.h,output.h
class CtorMaker {
    IRState * irs;
    AggregateDeclaration * aggDecl;
    int storageClass;
    tree curField;
    tree elemList;
public:
    CtorMaker(AggregateDeclaration * agg_decl, int storage_class, IRState * irs);
    CtorMaker & add(tree value);
    CtorMaker & add(tree field, tree value);
    CtorMaker & addArray(unsigned count, tree data);
    tree finish();
};

class ArrayMaker {
    Type * elemType;
    int storageClass;
    tree elemList;
    unsigned count;
public:
    ArrayMaker(Type * elem_type, int storage_class);
    ArrayMaker & add(tree value);
    tree finish();
    
};


#endif


