/*
 * mt.c: implementation of mt(1) as part of a Cygwin environment
 *
 * Copyright 1998,1999,2000,2001  Corinna Vinschen,
 * bug reports to  cygwin@cygwin.com
 *
 * Caution: This Programm runs only under Cygwin since it uses
 *	    Cygwin specific struct mtget members.
 *
 *	    Remote devices are NOT supported!
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <errno.h>
#include <getopt.h>
#include <mntent.h>
#include <sys/mount.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/mtio.h>
#include <sys/cygwin.h>
#include <sys/utsname.h>

#include <windows.h>

static char *SCCSid = "@(#)mt V2.1, Corinna Vinschen, " __DATE__ "\n";

char *myname;

struct mt_tape_info mt_tape_info[] = MT_TAPE_INFO;

struct 
{
  int code;
  char *name;
} densities[] =
{
  {0x00, "default"},
  {0x01, "NRZI (800 bpi)"},
  {0x02, "PE (1600 bpi)"},
  {0x03, "GCR (6250 bpi)"},
  {0x05, "QIC-45/60 (GCR, 8000 bpi)"},
  {0x06, "PE (3200 bpi)"},
  {0x07, "IMFM (6400 bpi)"},
  {0x08, "GCR (8000 bpi)"},
  {0x09, "GCR /37871 bpi)"},
  {0x0a, "MFM (6667 bpi)"},
  {0x0b, "PE (1600 bpi)"},
  {0x0c, "GCR (12960 bpi)"},
  {0x0d, "GCR (25380 bpi)"},
  {0x0f, "QIC-120 (GCR 10000 bpi)"},
  {0x10, "QIC-150/250 (GCR 10000 bpi)"},
  {0x11, "QIC-320/525 (GCR 16000 bpi)"},
  {0x12, "QIC-1350 (RLL 51667 bpi)"},
  {0x13, "DDS (61000 bpi)"},
  {0x14, "EXB-8200 (RLL 43245 bpi)"},
  {0x15, "EXB-8500 (RLL 45434 bpi)"},
  {0x16, "MFM 10000 bpi"},
  {0x17, "MFM 42500 bpi"},
  {0x21, "QIC-5010-DC"},
  {0x22, "QIC-2GB-DC"},
  {0x24, "DDS-2"},
  {0x26, "QIC-4GB-DC"},
  {0x30, "MLR3"},
  {0x32, "ALRF-2"},
  {0x33, "SLR6"},
  {0x34, "ALRF-1"},
  {0x36, "ALRF-6"},
  {0x81, "EXB-V23 compressed"},
  {0x8c, "EXB-8505 compressed"},
  {0x90, "EXB-8205 compressed"},
  {0, NULL}
};

int
errprintf (int ret, char *format, ...)
{
  va_list ap;

  fprintf (stderr, "%s: ", myname);
  va_start (ap, format);
  vfprintf (stderr, format, ap);
  va_end (ap);
  putc ('\n', stderr);
  return ret;
}

static int 
check_cygwin_api_version (int major, int minor)
{
  struct utsname uts;
  char *c;
  int api_major = 0, api_minor = 0;

  if (!uname(&uts) && (c = strchr(uts.release, '(')) != NULL)
    sscanf(c + 1, "%d.%d", &api_major, &api_minor);
  if (api_major < major
      || (api_major == major && api_minor < minor))
    return -1;
  if (api_major == major && api_minor == minor)
    return 0;
  return 1;
}

int
tape_status (int fd, struct mtget *get, long type)
{
  long cygwin_api_minor =
  	strtol (strstr ((char *) cygwin_internal (CW_GETVERSIONINFO),
			"api minor: ") + 11, NULL, 10);

  int has_tape = !GMT_DR_OPEN (get->mt_gstat);

  if (!has_tape)
    {
      printf ("No media\n");
      if (type < 2)
	type = 2;
    }

  if (has_tape)
    {
      int i;
      char *ti_name = "Generic tape device";
      for (i = 0; mt_tape_info[i].t_name; ++i)
        if (get->mt_type == mt_tape_info[i].t_type)
	  {
	    ti_name = mt_tape_info[i].t_name;
	    break;
	  }
	printf ("drive type       :       %02x (%s)\n", get->mt_type, ti_name);
      if (get->mt_featureslow & TAPE_DRIVE_TAPE_CAPACITY)
	printf ("tape capacity    : %8lu KB          ", get->mt_capacity >> 10);
      if (get->mt_featureslow & TAPE_DRIVE_TAPE_REMAINING)
	printf ("remaining        : %8lu KB\n", get->mt_remaining >> 10);
      else if (get->mt_featureslow & TAPE_DRIVE_TAPE_CAPACITY)
	putc ('\n', stdout);
      if (get->mt_featureslow
          & (TAPE_DRIVE_GET_ABSOLUTE_BLK | TAPE_DRIVE_GET_LOGICAL_BLK))
	printf ("current block    : %8lu             ", get->mt_blkno);
      if (check_cygwin_api_version (0, 112) >= 0)
	printf ("active partition : %8lu", get->mt_resid);
      puts ("");
    }

  if (has_tape && (get->mt_featureslow & TAPE_DRIVE_WRITE_PROTECT))
    printf ("write protected  :      %s             ",
	    GMT_WR_PROT (get->mt_gstat) ? "yes" : " no");
  if (get->mt_featureslow & TAPE_DRIVE_COMPRESSION)
    printf ("datcompression   :      %s\n",
	    GMT_HW_COMP (get->mt_gstat) ? " on" : "off");
  else if (has_tape && (get->mt_featureslow & TAPE_DRIVE_WRITE_PROTECT))
    putc ('\n', stdout);
  if (type > 1)
    {
      if (get->mt_featureshigh & TAPE_DRIVE_SET_BLOCK_SIZE)
	{
	  printf
	    ("min block size   : %8lu             max block size   : %8lu\n",
	     get->mt_minblksize, get->mt_maxblksize);
	  printf ("def block size   : %8lu             ",
		  get->mt_defblksize);
	}
      if (has_tape)
	printf ("cur block size   : %8lu\n",
		(get->mt_dsreg & MT_ST_BLKSIZE_MASK) >> MT_ST_BLKSIZE_SHIFT);
      else if (get->mt_featureshigh & TAPE_DRIVE_SET_BLOCK_SIZE)
	putc ('\n', stdout);
      if (has_tape)
        {
	  int ds = (get->mt_dsreg & MT_ST_DENSITY_MASK) >> MT_ST_DENSITY_SHIFT;
	  char *ds_name = "unknown";
	  int i;
	  for (i = 0; densities[i].name; ++i)
	    if (ds == densities[i].code)
	      {
	        ds_name = densities[i].name;
		break;
	      }
	  printf ("density code     :       %02x (%s)\n",
		  (get->mt_dsreg & MT_ST_DENSITY_MASK) >> MT_ST_DENSITY_SHIFT,
		  ds_name);
        }
      if (get->mt_featureslow & TAPE_DRIVE_ECC)
	printf ("hw err correction:      %s                  ",
		GMT_HW_ECC (get->mt_gstat) ? " on" : "off");
      if (get->mt_featureslow & TAPE_DRIVE_PADDING)
	printf ("data padding     :      %s\n",
		GMT_PADDING (get->mt_gstat) ? " on" : "off");
      else if (get->mt_featureslow & TAPE_DRIVE_ECC)
	putc ('\n', stdout);
      if (get->mt_featureslow & TAPE_DRIVE_REPORT_SMKS)
	printf ("report setmarks  :      %s                  ",
		GMT_IM_REP_EN (get->mt_gstat) ? " on" : "off");

      if (cygwin_api_minor >= 47 /* That struct member didn't exist prior
				    to Cygwin's API minor version 47. */
          && get->mt_featureslow & TAPE_DRIVE_SET_EOT_WZ_SIZE)
        printf ("EOT zone size    : %8lu\n", get->mt_eotwarningzonesize);
      else if (get->mt_featureslow & TAPE_DRIVE_REPORT_SMKS)
	putc ('\n', stdout);
    }
  if (type > 2)
    {
      printf ("\nFeatures:\n");
      printf ("---------\n");
      printf
	("hw compression   : %s                  erase on bop only: %s\n",
	 (get->mt_featureslow & TAPE_DRIVE_COMPRESSION) ? "yes" : " no",
	 (get->mt_featureslow & TAPE_DRIVE_ERASE_BOP_ONLY) ? "yes" : " no");
      printf
	("hw err correction: %s                  erase immediately: %s\n",
	 (get->mt_featureslow & TAPE_DRIVE_ECC) ? "yes" : " no",
	 (get->mt_featureslow & TAPE_DRIVE_ERASE_IMMEDIATE) ? "yes" : " no");
      printf
	("sw eject media   : %s                  long erase op    : %s\n",
	 (get->mt_featureslow & TAPE_DRIVE_EJECT_MEDIA) ? "yes" : " no",
	 (get->mt_featureslow & TAPE_DRIVE_ERASE_LONG) ? "yes" : " no");
      printf
	("write protection : %s                  short erase op   : %s\n",
	 (get->mt_featureslow & TAPE_DRIVE_WRITE_PROTECT) ? "yes" : " no",
	 (get->mt_featureslow & TAPE_DRIVE_ERASE_SHORT) ? "yes" : " no");
      printf
	("fixed length blks: %s                  fixed partitions : %s\n",
	 (get->mt_featureslow & TAPE_DRIVE_FIXED_BLOCK) ? "yes" : " no",
	 (get->mt_featureslow & TAPE_DRIVE_FIXED) ? "yes" : " no");
      printf
	("var length blks  : %s                  select partitions: %s\n",
	 (get->mt_featureslow & TAPE_DRIVE_VARIABLE_BLOCK) ? "yes" : " no",
	 (get->mt_featureslow & TAPE_DRIVE_SELECT) ? "yes" : " no");
      printf
	("get abs blockaddr: %s                  returns capacity : %s\n",
	 (get->mt_featureslow & TAPE_DRIVE_GET_ABSOLUTE_BLK) ? "yes" : " no",
	 (get->mt_featureslow & TAPE_DRIVE_TAPE_CAPACITY) ? "yes" : " no");
      printf
	("get log blockaddr: %s                  returns remaining: %s\n",
	 (get->mt_featureslow & TAPE_DRIVE_GET_LOGICAL_BLK) ? "yes" : " no",
	 (get->mt_featureslow & TAPE_DRIVE_TAPE_REMAINING) ? "yes" : " no");
      printf
	("data padding     : %s                  report setmarks  : %s\n",
	 (get->mt_featureslow & TAPE_DRIVE_PADDING) ? "yes" : " no",
	 (get->mt_featureslow & TAPE_DRIVE_REPORT_SMKS) ? "yes" : " no");
      printf
	("rep EOT warn size: %s                  initiator partit.: %s\n",
	 (get->mt_featureslow & TAPE_DRIVE_EOT_WZ_SIZE) ? "yes" : " no",
	 (get->mt_featureslow & TAPE_DRIVE_INITIATOR) ? "yes" : " no");

      printf
	("abs blk position : %s                  abs blk immediate: %s\n",
	 (get->mt_featureshigh & TAPE_DRIVE_ABSOLUTE_BLK) ? "yes" : " no",
	 (get->mt_featureshigh & TAPE_DRIVE_ABS_BLK_IMMED) ? "yes" : " no");
      printf
	("log blk position : %s                  log blk immediate: %s\n",
	 (get->mt_featureshigh & TAPE_DRIVE_LOGICAL_BLK) ? "yes" : " no",
	 (get->mt_featureshigh & TAPE_DRIVE_LOG_BLK_IMMED) ? "yes" : " no");
      printf
	("filemark position: %s                  rel blk position : %s\n",
	 (get->mt_featureshigh & TAPE_DRIVE_FILEMARKS) ? "yes" : " no",
	 (get->mt_featureshigh & TAPE_DRIVE_RELATIVE_BLKS) ? "yes" : " no");
      printf
	("end data position: %s                  reverse position : %s\n",
	 (get->mt_featureshigh & TAPE_DRIVE_END_OF_DATA) ? "yes" : " no",
	 (get->mt_featureshigh & TAPE_DRIVE_REVERSE_POSITION) ? "yes" : " no");
      printf
	("immediate spacing: %s                  immediate rewind : %s\n",
	 (get->mt_featureshigh & TAPE_DRIVE_SPACE_IMMEDIATE) ? "yes" : " no",
	 (get->mt_featureshigh & TAPE_DRIVE_REWIND_IMMEDIATE) ? "yes" : " no");
      printf
	("sequential filem.: %s                  sequential setm. : %s\n",
	 (get->mt_featureshigh & TAPE_DRIVE_SEQUENTIAL_FMKS) ? "yes" : " no",
	 (get->mt_featureshigh & TAPE_DRIVE_SEQUENTIAL_SMKS) ? "yes" : " no");
      printf
	("set block size   : %s                  set compression. : %s\n",
	 (get->mt_featureshigh & TAPE_DRIVE_SET_BLOCK_SIZE) ? "yes" : " no",
	 (get->mt_featureshigh & TAPE_DRIVE_SET_COMPRESSION) ? "yes" : " no");
      printf
	("set hw compress. : %s                  set data padding : %s\n",
	 (get->mt_featureshigh & TAPE_DRIVE_SET_ECC) ? "yes" : " no",
	 (get->mt_featureshigh & TAPE_DRIVE_SET_PADDING) ? "yes" : " no");
      printf
	("setmark position : %s                  set report setm. : %s\n",
	 (get->mt_featureshigh & TAPE_DRIVE_SETMARKS) ? "yes" : " no",
	 (get->mt_featureshigh & TAPE_DRIVE_SET_REPORT_SMKS) ? "yes" : " no");
      printf
	("load and unload  : %s                  un/load immediate: %s\n",
	 (get->mt_featureshigh & TAPE_DRIVE_LOAD_UNLOAD) ? "yes" : " no",
	 (get->mt_featureshigh & TAPE_DRIVE_LOAD_UNLD_IMMED) ? "yes" : " no");
      printf
	("lock and unlock  : %s                  un/lock immediate: %s\n",
	 (get->mt_featureshigh & TAPE_DRIVE_LOCK_UNLOCK) ? "yes" : " no",
	 (get->mt_featureshigh & TAPE_DRIVE_LOCK_UNLK_IMMED) ? "yes" : " no");
      printf
	("tape retension   : %s                  retens. immediate: %s\n",
	 (get->mt_featureshigh & TAPE_DRIVE_TENSION) ? "yes" : " no",
	 (get->mt_featureshigh & TAPE_DRIVE_TENSION_IMMED) ? "yes" : " no");
      printf
	("write filemarks  : %s                  write setmarks   : %s\n",
	 (get->mt_featureshigh & TAPE_DRIVE_WRITE_FILEMARKS) ? "yes" : " no",
	 (get->mt_featureshigh & TAPE_DRIVE_WRITE_SETMARKS) ? "yes" : " no");
      printf
	("write long filem.: %s                  write marks immed: %s\n",
	 (get->mt_featureshigh & TAPE_DRIVE_WRITE_LONG_FMKS) ? "yes" : " no",
	 (get->mt_featureshigh & TAPE_DRIVE_WRITE_MARK_IMMED) ? "yes" : " no");
      printf
	("write shrt filem.: %s                  set EOT warn size: %s\n",
	 (get->mt_featureshigh & TAPE_DRIVE_WRITE_SHORT_FMKS) ? "yes" : " no",
	 (get->mt_featureslow & TAPE_DRIVE_SET_EOT_WZ_SIZE) ? "yes" : " no");
    }
  return NO_ERROR;
}

int
usage ()
{
  fprintf (stderr, "usage: %s [-V] [-f device] operation [count]\n", myname);
  return 1;
}

int
main (int argc, char *argv[])
{
  char *cmd;
  char path[256];
  long count = 1;
  int c;
  int ret = 0;
  int fd;
  struct stat st;
  struct mtop op;

  myname = argv[0];

  setvbuf (stdout, NULL, _IONBF, 0);

  strcpy (path, DEFTAPE);
  if (getenv ("TAPE"))
    strcpy (path, getenv ("TAPE"));

  while ((c = getopt (argc, argv, "f:V")) != EOF)
    switch (c)
      {
      case 'f':
	if (!optarg)
	  {
	    usage ();
	    return 1;
	  }
	strcpy (path, optarg);
	break;
      case 'V':
	fprintf (stderr, "%s", SCCSid + 4);
	return 0;
      case '?':
	return usage ();
      }

  if (optind >= argc)
    return usage ();
  cmd = argv[optind++];
  if (optind < argc && (count = atol (argv[optind])) < 0)
    return errprintf (1, "count must be positive.");

  if ((fd = open (path, O_RDWR)) < 0)
    return errprintf (1, "%s: %s", path, strerror (errno));

  if (fstat (fd, &st))
    return errprintf (1, "%s: %s", path, strerror (errno));

  if (!S_ISCHR (st.st_mode))
    return errprintf (1, "%s is not a character special file", path);

  op.mt_count = count;

  /* get status of tape */
  if (!strcmp (cmd, "status"))
    {
      struct mtget get;
      if (ioctl (fd, MTIOCGET, &get) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
      else
        tape_status (fd, &get, count);
      return ret;
    }
  /* write count EOF marks */
  else if (!strcmp (cmd, "eof") || !strcmp (cmd, "weof"))
    {
      op.mt_op = MTWEOF;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* forward count files */
  else if (!strcmp (cmd, "fsf"))
    {
      op.mt_op = MTFSF;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* backward count files */
  else if (!strcmp (cmd, "bsf"))
    {
      op.mt_op = MTBSF;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* forward count records */
  else if (!strcmp (cmd, "fsr"))
    {
      op.mt_op = MTFSR;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* backward count records */
  else if (!strcmp (cmd, "bsr"))
    {
      op.mt_op = MTBSR;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* forward count filemarks */
  else if (!strcmp (cmd, "fsfm"))
    {
      op.mt_op = MTFSFM;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* backward count filemarks */
  else if (!strcmp (cmd, "bsfm"))
    {
      op.mt_op = MTBSFM;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* move to absolut file number count */
  else if (!strcmp (cmd, "asf"))
    {
      op.mt_op = MTREW;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
      else
        {
	  op.mt_op = MTFSF;
	  op.mt_count = count;
	  if (ioctl (fd, MTIOCTOP, &op) < 0)
	    ret = errprintf (2, "%s: %s", path, strerror (errno));
	}
    }
  /* move to end of data */
  else if (!strcmp (cmd, "eom"))
    {
      op.mt_op = MTEOM;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* rewind tape */
  else if (!strcmp (cmd, "rewind"))
    {
      op.mt_op = MTREW;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* offline tape */
  else if (!strcmp (cmd, "rewoffl") || !strcmp (cmd, "offline"))
    {
      op.mt_op = MTOFFL;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* retension tape */
  else if (!strcmp (cmd, "retension"))
    {
      op.mt_op = MTREW;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
      else
        {
	  op.mt_op = MTRETEN;
	  if (ioctl (fd, MTIOCTOP, &op) < 0)
	    ret = errprintf (2, "%s: %s", path, strerror (errno));
	  else
	    {
	      op.mt_op = MTREW;
	      if (ioctl (fd, MTIOCTOP, &op) < 0)
		ret = errprintf (2, "%s: %s", path, strerror (errno));
	    }
	}
    }
  /* erase tape */
  else if (!strcmp (cmd, "erase"))
    {
      op.mt_op = MTERASE;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* forward count setmarks */
  else if (!strcmp (cmd, "fss"))
    {
      op.mt_op = MTFSS;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* backward count setmarks */
  else if (!strcmp (cmd, "bss"))
    {
      op.mt_op = MTBSS;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* write count setmarks */
  else if (!strcmp (cmd, "wset"))
    {
      op.mt_op = MTWSM;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* move to end of data */
  else if (!strcmp (cmd, "eod") || !strcmp (cmd, "seod"))
    {
      op.mt_op = MTEOM;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* set block size */
  else if (!strcmp (cmd, "setblk"))
    {
      op.mt_op = MTSETBLK;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* set density */
  else if (!strcmp (cmd, "setdensity"))
    {
      op.mt_op = MTSETDENSITY;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* set drive buffer size */
  else if (!strcmp (cmd, "drvbuffer"))
    {
      op.mt_op = MTSETDRVBUFFER;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* set driver options */
  else if (!strcmp (cmd, "stoptions"))
    {
      ret = errprintf (1, "operation '%s' not implemented", cmd);
    }
  /* set write threshold */
  else if (!strcmp (cmd, "stwrthreshold"))
    {
      ret = errprintf (1, "operation '%s' not implemented", cmd);
    }
  /* seek to count block */
  else if (!strcmp (cmd, "seek"))
    {
      op.mt_op = MTSEEK;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  /* tell block count */
  else if (!strcmp (cmd, "tell"))
    {
      op.mt_op = MTTELL;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
      else
	printf ("At block %d.\n", op.mt_count);
    }
  /* write densities */
  else if (!strcmp (cmd, "densities"))
    {

      puts ("Some SCSI tape density codes:\n"
	    "code   explanation");
      for (count = 0; densities[count].name; ++count)
        printf ("0x%02x   %s\n", densities[count].code, densities[count].name);
    }
  /* get/set compression */
  else if (!strcmp (cmd, "datcompression"))
    {
      op.mt_op = MTCOMPRESSION;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
	ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  else if (!strcmp (cmd, "mkpart"))
    {
      op.mt_op = MTMKPART;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
        ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  else if (!strcmp (cmd, "setpart"))
    {
      op.mt_op = MTSETPART;
      if (ioctl (fd, MTIOCTOP, &op) < 0)
        ret = errprintf (2, "%s: %s", path, strerror (errno));
    }
  else
    ret = errprintf (1, "unknown operation '%s'", cmd);

  close (fd);
  return ret;
}
