/*
 *  LinKT - the Linux Kde pr-Terminal
 *  Copyright (C) 1997-2001 Jochen Sarrazin, DG6VJ. All rights reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 *   some parts Copyright (C) by Jonathan Naylor
 */
//---------------------------------------------------------------------------
#include "yapp.h"
#include "yapp.moc"

#include "channel.h"
#include "toolbox.h"
#include "main.h"
#include "dostime.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
//---------------------------------------------------------------------------
#define YAPP_TIMEOUT 10000		// Retransmission-Timeout in ms
//---------------------------------------------------------------------------
#define	NUL		0x00
#define	SOH		0x01
#define	STX		0x02
#define	ETX		0x03
#define	EOT		0x04
#define	ENQ		0x05
#define	ACK		0x06
#define	DLE		0x10
#define	NAK		0x15
#define	CAN		0x18
//---------------------------------------------------------------------------
YAPPBase::YAPPBase()
{
}
//---------------------------------------------------------------------------
void YAPPBase::SendRR()
{
   char buf[2];

	buf[0] = ACK;
	buf[1] = 0x01;

	sendData( buf, 2 );
}
//---------------------------------------------------------------------------
void YAPPBase::SendRF()
{
   char buf[2];

	buf[0] = ACK;
	buf[1] = 0x02;

	sendData( buf, 2 );
}
//---------------------------------------------------------------------------
void YAPPBase::SendRT()
{
   char buf[2];

	buf[0] = ACK;
	buf[1] = ACK;

	sendData( buf, 2 );
}
//---------------------------------------------------------------------------
void YAPPBase::SendAF()
{
   char buf[2];

	buf[0] = ACK;
	buf[1] = 0x03;

	sendData( buf, 2 );
}
//---------------------------------------------------------------------------
void YAPPBase::SendAT()
{
   char buf[2];

	buf[0] = ACK;
	buf[1] = 0x04;

	sendData( buf, 2 );
}
//---------------------------------------------------------------------------
void YAPPBase::SendSI()
{
   char buf[2];

	buf[0] = ENQ;
	buf[1] = 0x01;

	sendData( buf, 2 );
}
//---------------------------------------------------------------------------
void YAPPBase::SendHD( const char *shortname, long filelen )
{
   char buffer[259];
   char size_buffer[10];
   int len, len_filename;


   sprintf( size_buffer, "%ld", filelen );
   len_filename = strlen(shortname)+1;
   len = len_filename+strlen(size_buffer)+1; // +1, weil die \0 am Ende mitzaehlt

   buffer[0] = SOH;
   buffer[1] = len;

   memcpy(buffer+2, shortname, len_filename);
   memcpy(buffer+2+len_filename, size_buffer, strlen(size_buffer)+1);

   sendData( buffer, len+2 );
}
//---------------------------------------------------------------------------
void YAPPBase::SendDT( int length )
{
	char buf[2];
	
	if (length > 255) length = 0;
	
	buf[0] = STX;
	buf[1] = length;
	
	sendData( buf, 2 );
}
//---------------------------------------------------------------------------
void YAPPBase::SendEF()
{
   char buf[2];

	buf[0] = ETX;
	buf[1] = 0x01;

	sendData( buf, 2 );
}
//---------------------------------------------------------------------------
void YAPPBase::SendET()
{
   char buf[2];

	buf[0] = EOT;
	buf[1] = 0x01;

	sendData( buf, 2 );
}
//---------------------------------------------------------------------------
void YAPPBase::SendNR( const char *reason )
{
	char buf[257];
	int length;

	if ((length = strlen(reason)) > 255)
		length = 255;
	
	buf[0] = NAK;
	buf[1] = length;
	memcpy(buf + 2, reason, length);
	
	sendData( buf, length+2 );
}
//---------------------------------------------------------------------------
void YAPPBase::SendRE( int length )
{
	char buf[256];
	int len;

	buf[0] = NAK;
	buf[1] = 'R';
	buf[3] = 0;

	len = sprintf(buf + 4, "%d", length) + 5;
	
	buf[len]     = 'C';
	buf[len + 1] = 0;
	buf[1]       = len;
	
	sendData( buf, len+2 );
}
//---------------------------------------------------------------------------
void YAPPBase::SendCN( const char *reason )
{
	char buf[257];
	int length;

	if ((length = strlen(reason)) > 255)
		length = 255;
	
	buf[0] = CAN;
	buf[1] = length;
	memcpy(buf + 2, reason, length);
	
	sendData( buf, length+2 );
}
//---------------------------------------------------------------------------
// Resume
void YAPPBase::SendRS( int length )
{
	char buffer[256];
	int len;

	buffer[0] = NAK;
	buffer[2] = 'R';
	buffer[3] = 0;

	len = sprintf(buffer + 4, "%d", length) + 5;
	
	buffer[len]     = 'C';
	buffer[len + 1] = 0;
	buffer[1]       = len;

   sendData( buffer, len+2 );
}
//---------------------------------------------------------------------------
unsigned char YAPPBase::checksum( const char *data, int len )
{
	int i;
	unsigned char sum = 0;

	for (i=0; i<len; i++)
		sum += data[i];

	return sum;
}
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
YAPP_TX::YAPP_TX( Channel *chan, const QString & filename )
		: FileTransfer( chan, filename ), YAPPBase()
{
	QString tmp;


   // Datenfile oeffnen
   if ((fd = open(filename.latin1(), O_RDONLY)) == -1)
   {
	   SendCN( "Cannot open sourcefile" );
		tmp.sprintf( "<LinKT>: Cannot open sourcefile.\r(%s)\r", strerror(errno) );
      sendMessage( tmp.latin1() );
		txready = SQTRSTAT_EMERGSTOP;
      return;
	}

   timer = new QTimer( this );
   connect(timer, SIGNAL(timeout()), this, SLOT(sendBlock()));

   savelen = 0;
   state = 0;
   totalsent = 0;
   filelen = filesize( filename.latin1() );
   starttime = 0;
   yappc = false;
	txready = SQTRSTAT_NOSTOP;
}
//---------------------------------------------------------------------------
YAPP_TX::~YAPP_TX()
{
	if (fd != -1)
   	::close( fd );

	if (savelen != 0)
   	free( savedata );

   emit unshowStatus();
}
//---------------------------------------------------------------------------
void YAPP_TX::getFileData()
{
   int len, lenread;
   char data[300];	// 256-byte-bloecke + evtl. der ende-block


   if (state != 3) return;

   lenread = read(fd, data+2, 256);

   data[0] = STX;
   if (lenread == 256)
   	data[1] = 0;
	else
	   data[1] = lenread;
   len = lenread+2;

   if (yappc)
   {
      data[len] = checksum( data+2, len-2 );
      len++;
   }

   totalsent += lenread;
	emit statusRxBytes( totalsent );

   if (lenread != 256)
   {
      // Ende Gelaende. EF schicken und auf Antwort warten.
      data[len] =	ETX;
      data[len+1] = 0x01;
      len += 2;

      state = 4;
   }

   emit sendString( data, len, false );
}
//---------------------------------------------------------------------------
void YAPP_TX::startSending()
{
	const char uploadstart[] = "<LinKT>: YAPP-Upload starting...\r";
	chan->outText( uploadstart, strlen(uploadstart), OUTCOLOR_TXTEXT );

	emit setLinemode( false );
   SendSI();
   state = 1;
   timer->start( YAPP_TIMEOUT, false );
}
//---------------------------------------------------------------------------
int YAPP_TX::sentReady()
{
   return txready;
}
//---------------------------------------------------------------------------
void YAPP_TX::abortTransmission()
{
	char tmp[100];

   emit deleteFTData();
	txready = SQTRSTAT_ABORT;

   SendCN("aborted by sender");

   strcpy( tmp, "<LinKT>: YAPP-TX aborted.\r" );
	sendMessage( tmp );
}
//---------------------------------------------------------------------------
void YAPP_TX::slotReceivedString( const char *data, int len )
{
	if (savelen == 0)
   {
   	savedata = (char *) malloc( len );
      memcpy( savedata, data, len );
      savelen = len;
   }
   else
   {
	   savedata = (char *) realloc( savedata, savelen+len );
	   memcpy( savedata+savelen, data, len );
	   savelen += len;
   }

   if (savelen < 2) return;

	switch (state)
   {
		case 1: // SI abgeschickt
			if (savedata[0] == ACK && savedata[1] == 0x01)
			{
				SendHD( shortname.latin1(), filelen );
				timer->start( YAPP_TIMEOUT, false );
				state = 2;  // HD abgeschickt
				break;
			}
			if (savedata[0] == ACK && savedata[1] == 0x02)
			{
			   emit showStatus( this, shortname.latin1(), filelen, 0, TRANSART_YAPPTX );
				timer->stop();
				state = 3;  // Daten abschicken/abgeschickt
				break;
			}
			if (!lookForAbort(savedata))
				unknownCode();
			txready = SQTRSTAT_EMERGSTOP;
         return;
		case 2: // HD (Header) abgeschickt
			if (savelen >= 3 && savedata[0] == NAK && savedata[2] == 'R') // resume
			{
				int len;
				off_t rpos;

				len = savedata[1];
            if (savelen < len+2) return;
				if (savedata[len] == 'C') yappc = true;
				rpos = atol((char *)savedata + 4);
				lseek( fd, rpos, SEEK_SET );
            totalsent = rpos;

			   emit showStatus( this, shortname.latin1(), filelen, rpos, TRANSART_YAPPTX );
			   starttime = time(NULL);

				state = 3;
				timer->stop();
				break;
			}
			if (savedata[0] == ACK && (savedata[1] == 0x02 || savedata[1] == ACK))
			{
				if (savedata[1] == ACK) yappc = true;

			   emit showStatus( this, shortname.latin1(), filelen, 0, TRANSART_YAPPTX );
			   starttime = time(NULL);

				state = 3;
				timer->stop();
				break;
			}
			if (!lookForAbort(savedata))
				unknownCode();
			txready = SQTRSTAT_EMERGSTOP;
         return;
		case 3: // Daten abschicken
			if (!lookForAbort(savedata))
				unknownCode();
			txready = SQTRSTAT_EMERGSTOP;
         return;
		case 4: // EF gesendet, auf Antwort warten
			if (savedata[0] == ACK && savedata[1] == 0x03)
			{
				SendET();
				timer->start( YAPP_TIMEOUT, false );
				state = 5;  // ET gesendet
				break;
			}
			txready = SQTRSTAT_EMERGSTOP;
         return;
		case 5:
			if (savedata[0] == ACK && savedata[1] == 0x04)
			{
				sendTransferInfo();
				txready = SQTRSTAT_STDSTOP;
	         return;
			}
			if (!lookForAbort(savedata))
				unknownCode();
			txready = SQTRSTAT_EMERGSTOP;
         return;
	}

   if (savelen != 0)
   {
   	free( savedata );
      savelen = 0;
	}
}
//---------------------------------------------------------------------------
void YAPP_TX::sendBlock()
{
   switch (state)
   {
      case 1: SendSI(); break;
      case 2: SendHD( shortname.latin1(), filelen ); break;
      case 4: SendEF(); break;
      case 5: SendET(); break;
   }
}
//---------------------------------------------------------------------------
void YAPP_TX::sendData( const char *data, int len, bool show )
{
	emit sendString( data, len, show );
}
//---------------------------------------------------------------------------
bool YAPP_TX::lookForAbort( const char *data )
{
	char str[500], tmp[260];

	if (data[0] == CAN)
	{
		memcpy(tmp, data+2, (unsigned char) data[1]);
		tmp[(unsigned char) data[1]] = '\0';
		sprintf(str, "\r<LinKT>: YAPP-TX aborted by peer.\r" \
                        "         (%s)\r", tmp);
		sendMessage( str );
		return true;
	}

	return false;
}
//---------------------------------------------------------------------------
void YAPP_TX::unknownCode()
{
   char str[100];


   SendCN( "Unknown code" );
	strcpy( str, "\r<LinKT>: YAPP-TX aborted (Unknown YAPP-Code)\r" );

   sendMessage( str );
}
//---------------------------------------------------------------------------
void YAPP_TX::sendTransferInfo()
{
   time_t timediff;
   char *timeptr;
   char text[200];

   timediff = time(NULL) - starttime;
   if (timediff == 0) timediff = 1;
   timeptr = (char *)spec_time(timediff);

   sprintf(text, "<LinKT>: YAPP-TX OK. (time: %s, %li bit/s)\xD",
                 timeptr,(filelen*8)/timediff);

	sendMessage( text );

   free(timeptr);
}
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
YAPP::YAPP( Channel *chan, const QString & filename )
			: QObject(), YAPPBase(), SendData( chan )
{
	int i;


	state = 1;
   savedatalen = 0;
   file_time = -1;
   yappc = false;
   shortname = "";
   filelen = 0;
   offset = 0;
   totalrxlen = 0;
   this->chan = chan;
   transInfo = NULL;

   this->filename = filename;

   shortname = filename;
	if ((i = shortname.findRev('\\')) != -1)
		shortname.remove( 0, i+1 );
	if ((i = shortname.findRev('/')) == -1)
		shortname.remove( 0, i+1 );

   timer = new QTimer( this );
   connect(timer, SIGNAL(timeout()), this, SLOT(sendBlock()));
}
//---------------------------------------------------------------------------
YAPP::~YAPP()
{
	if (savedatalen > 0)
   	free( savedata );

	if (transInfo != NULL)
   {
		chan->withoutTransInfo();
      delete transInfo;
   }
}
//---------------------------------------------------------------------------
// Rueckgabe: true, wenn fertig, false wenn nicht.
bool YAPP::proceed( const char *data, int len )
{
   int rxlen;

	// Die empfangenen Daten hinten an savedata anhaengen.
   if (savedatalen == 0)
   {
   	savedata = (char *) malloc( len );
      memcpy( savedata, data, len );
      savedatalen = len;
   }
   else
   {
	   savedata = (char *) realloc( savedata, savedatalen+len );
	   memcpy( savedata+savedatalen, data, len );
	   savedatalen += len;
	}

	while (true)
	{
   	// Eventuell noch vorhandene Daten vor der Antwort der Gegenseite
      // verwerfen.
		if (state == 1)
		{
			// Alles vor "ENQ 0x01" loeschen
			if (!lookForStart())
			{
				// Kein  ENQ 0x01  entdeckt - den verbleibenden String (ein
				// Byte) gespeichert lassen und aus dieser Funktion 'raus.
				return false;
			}
		}

		switch (state)
		{
			case 1:
				if (savedata[0] == ENQ && savedata[1] == 0x01)
				{
					SendRR();
//					timer->start( YAPP_TIMEOUT, false );
					state = 2;
					savedatalen -= 2;
					memmove( savedata, savedata+2, len );
					break;
				}
				if (!lookForAbort(savedata))
					unknownCode();
				return true;
			case 2: // Erwarte Header
				if (savedata[0] == SOH)
				{
					if (savedatalen < (unsigned char)savedata[1])
               {
               	// Der Header ist noch nicht vollstaendig
						return false;
					}

					if (readHeader())
						return true;
					break;
				}
				if (savedata[0] == ENQ && savedata[1] == 0x01)
				{
					savedatalen -= 2;
					memmove( savedata, savedata+2, savedatalen );
					break;
				}
				if (savedata[0] == EOT && savedata[1] == 0x01)
				{
					savedatalen -= 2;
					memmove( savedata, savedata+2, savedatalen );
					SendAT();
					return true;
				}
				if (!lookForAbort(savedata))
					unknownCode();
				return true;
			case 3: // Empfange Daten
				if (savedata[0] == STX)
				{
					if ((rxlen = (unsigned char)savedata[1]) == 0)
						rxlen = 256;
					if ((yappc && (savedatalen < rxlen+3)) ||
						(!yappc && (savedatalen < rxlen+2)))
					{
               	// Noch nicht alles empfangen
						return false;
					}
					if (readData())
						return true;
					break;
				}
				if (savedata[0] == ETX && savedata[1] == 0x01)
				{
					SendAF();
					state = 2;
					::close(fd);
					fd = -1;
					if (transInfo != NULL)
					{
						delete transInfo;
						transInfo = NULL;
						chan->withoutTransInfo();
					}
					sendTransferInfo();
					savedatalen -= 2;
					memmove( savedata, savedata+2, savedatalen );
					break;
				}
				if (!lookForAbort(savedata))
					unknownCode();
				return true;
		}
		if (savedatalen == 0)
		{
			free( savedata );
			return false;
		}
		if (savedatalen == 1)
      	return false;
	}
}
//---------------------------------------------------------------------------
//   bool YAPP::lookForStart()
//
// Sorgt dafuer, dass  ENQ 0x01  am Anfang des Strings savedata
// zurueckgegeben wird.
//
// Ruckgabe: true, wenn  ENQ 0x01  entdeckt wurde,
//           false, wenn nicht.
//
bool YAPP::lookForStart()
{
   int i;

   for (i=0; i<savedatalen-1; i++)
   {
      if (savedata[i] == ENQ)
         if (savedata[i+1] == 0x01)
         {
            savedatalen -= i;
            memmove(savedata, savedata+i, savedatalen);
            return true;
         }
   }

   return false;
}
//---------------------------------------------------------------------------
void YAPP::sendData( const char *data, int len, bool show )
{
	SendData::sendData( data, len, show );
}
//---------------------------------------------------------------------------
bool YAPP::lookForAbort( const char *data )
{
	char str[500], tmp[260];

	if (data[0] == CAN)
	{
		memcpy(tmp, str+2, (unsigned char) str[1]);
		tmp[(unsigned char) str[1]] = '\0';
		sprintf(str, "\r<LinKT>: YAPP-RX aborted by peer.\r" \
                        "         (%s)\r", tmp);
		sendMessage( str );
		return true;
	}

	return false;
}
//---------------------------------------------------------------------------
bool YAPP::readHeader()
{
   char *hptr, *hfield[3];
   int length;
   int k=0,i, savelen;
   QString str;


   if ((length = savedata[1]) == 0) length = 256;
   savelen = length;
   hptr = (char *)savedata + 2;

   while (length > 0)
   {
      int hlen;
      hlen = strlen(hptr) + 1;
      hfield[(int)k++] = hptr;
      hptr   += hlen;
      length -= hlen;
   }

   if (k < 3)
      yappc = false;
   else
   {
      file_time = yapp2unix(hfield[2]);
      yappc = true;
   }

   str = hfield[0];
   if ((i = str.findRev('/')) != -1)
      str.remove( 0, i+1 );
   if ((i = str.findRev('\\')) != -1)
      str.remove( 0, i+1 );

   if (filename.isEmpty())
   {
      shortname = str;
      filename = conf->getDirABin()+"/"+shortname;
   }

   if ((fd = open(filename.latin1(), O_RDWR | O_APPEND | O_CREAT)) == -1)
   {
      SendNR("Invalid filename");
      sendMessage( "<LinKT>: YAPP-RX: Cannot open datafile. Aborted.\r" );
      return true;
   }

   // Zugriffsrechte einstellen
   fchmod( fd, S_IRUSR|S_IWUSR );

   filelen = atoi(hfield[1]);
   totalrxlen = 0;
   offset = 0;

   if (yappc)
   {
      // Wenn der resume-Mode explizit aktiviert wurde (Preferences), wird
      // nach resume gesucht, sonst *nicht*

      if (conf->getFlag(CFG_YAPPRESUME))
      {
         struct stat sb;

         if (!fstat(fd, &sb) && sb.st_size)
         {
            SendRS( sb.st_size );
            totalrxlen = sb.st_size;
            offset = sb.st_size;
            lseek( fd, sb.st_size, SEEK_SET );
         }
         else
            SendRT();
      }
      else
         SendRT();
   }
   else
      SendRF();
   timer->stop();

   state = 3;

   str.sprintf( "<LinKT>: YAPP-Download startet at offset %li.\r" \
                "         filename: %s, size: %li bytes\r",
             offset, shortname.latin1(), filelen );
	chan->outText( str.latin1(), str.length(), OUTCOLOR_TXTEXT );

	transInfo = new TransferInfo( ((Channel *)chan)->centralWidget, TRANSART_YAPPRX, shortname, filelen, offset );
	chan->withTransInfo( transInfo );
   starttime = time(NULL);

   savedatalen -= savelen+2;
   memmove( savedata, savedata+2+savelen, savedatalen);
   return false;
}
//---------------------------------------------------------------------------
bool YAPP::readData()
{
   int length;
   unsigned char i;


   if ((length = (unsigned char)savedata[1]) == 0) length = 256;

   totalrxlen += length;
   transInfo->setReceivedBytes( totalrxlen );

   if (yappc)
   {
      if ((i = checksum( savedata+2, length )) != (unsigned char)savedata[length + 2])
      {
      	SendCN("Bad Checksum");
         sendMessage( "<LinKT>: Bad checksum in YAPP-transfer. Aborted.\r" );
         return true;
      }
   }

   write(fd, savedata+2, length);

   if (yappc)
   {
      savedatalen -= length+3;
      memmove( savedata, savedata+length+3, savedatalen );
   }
   else
   {
      savedatalen -= length+2;
      memmove( savedata, savedata+length+2, savedatalen );
   }
   return false;
}
//---------------------------------------------------------------------------
void YAPP::sendBlock()
{
	switch (state)
   {
//		case 101: SendRR(); break;
	}
}
//---------------------------------------------------------------------------
void YAPP::unknownCode()
{
   char str[100];


   SendCN( "Unknown code" );
	strcpy( str, "\r<LinKT>: YAPP-RX aborted (Unknown YAPP-Code)\r" );

   sendMessage( str );
}
//---------------------------------------------------------------------------
void YAPP::sendTransferInfo()
{
   time_t timediff;
   char *timeptr;
   char text[200];

   timediff = time(NULL) - starttime;
   if (timediff == 0) timediff = 1;
   timeptr = (char *)spec_time(timediff);

   sprintf(text, "<LinKT>: YAPP-RX OK. (time: %s, %li bit/s)\xD",
                 timeptr,(filelen*8)/timediff);

	chan->outText( text, strlen(text), OUTCOLOR_TXTEXT );
//	sendMessage( text );

   free(timeptr);
}
//---------------------------------------------------------------------------

