/*
 * Copyright (c) 2012 Stefan Walter
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *     * Redistributions of source code must retain the above
 *       copyright notice, this list of conditions and the
 *       following disclaimer.
 *     * Redistributions in binary form must reproduce the
 *       above copyright notice, this list of conditions and
 *       the following disclaimer in the documentation and/or
 *       other materials provided with the distribution.
 *     * The names of contributors to this software may not be
 *       used to endorse or promote products derived from this
 *       software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 * Author: Stef Walter <stef@thewalter.net>, Daiki Ueno
 */

#include "config.h"

#define CRYPTOKI_EXPORTS 1
#include "pkcs11.h"

#include "attrs.h"
#include "debug.h"
#include "mock.h"
#include "test.h"
#include <stdint.h>

static const CK_TOKEN_INFO MOCK_TOKEN_INFO = {
	"PUBKEY LABEL                    ",
	"PUBKEY MANUFACTURER             ",
	"PUBKEY MODEL    ",
	"PUBKEY SERIAL   ",
	CKF_LOGIN_REQUIRED | CKF_USER_PIN_INITIALIZED | CKF_CLOCK_ON_TOKEN | CKF_TOKEN_INITIALIZED,
	1,
	2,
	3,
	4,
	5,
	6,
	7,
	8,
	9,
	10,
	{ 75, 175 },
	{ 85, 185 },
	{ '1', '9', '9', '9', '0', '5', '2', '5', '0', '9', '1', '9', '5', '9', '0', '0' }
};

static uint8_t rsa_modulus[] = {
	0xc0, 0xae, 0x21, 0x2e, 0x0b, 0x11, 0xc6, 0x5b, 0x55, 0xd0, 0xcb, 0x9d,
	0x61, 0x6b, 0x76, 0xc9, 0x96, 0x16, 0x82, 0x31, 0xe1, 0xd4, 0x1d, 0x86,
	0xe4, 0x4b, 0x89, 0x0d, 0x1b, 0xe3, 0x91, 0x82, 0x06, 0x88, 0x59, 0x7a,
	0x9c, 0x39, 0x3d, 0x6a, 0x45, 0xb1, 0x6b, 0x08, 0x09, 0xa1, 0x59, 0xa7,
	0x66, 0x0d, 0xd8, 0xd1, 0xae, 0x28, 0x82, 0x4d, 0x07, 0xf3, 0xef, 0x96,
	0x5c, 0xe6, 0x63, 0x6d, 0x52, 0xf5, 0xbd, 0x8a, 0xbd, 0x7f, 0x14, 0x26,
	0xd8, 0xa3, 0x0d, 0x64, 0xf5, 0xf4, 0x4c, 0xcf, 0xde, 0x7e, 0x1d, 0x0f,
	0xf1, 0x83, 0x3c, 0x94, 0x0c, 0xb1, 0x8a, 0x20, 0xa7, 0x0b, 0x01, 0xa4,
	0x7d, 0xf6, 0xd4, 0x16, 0xa3, 0x9a, 0xfb, 0xee, 0x13, 0xff, 0x7d, 0x05,
	0x45, 0xf7, 0x97, 0x0f, 0x56, 0x1a, 0x35, 0x81, 0xf5, 0x64, 0xf7, 0xf6,
	0x08, 0xa4, 0xa5, 0x9b, 0xab, 0x94, 0x23, 0x2a, 0x85, 0x39, 0xa9, 0x31,
	0x2d, 0xc6, 0x93, 0x56, 0x3e, 0xa8, 0x5e, 0x27, 0xf8, 0x35, 0x6b, 0x60,
	0xe8, 0x47, 0x6e, 0xe8, 0x8d, 0x6b, 0x11, 0x2d, 0x6c, 0x04, 0xc9, 0xec,
	0x29, 0x04, 0x04, 0x94, 0xb5, 0xe8, 0x28, 0xb2, 0x68, 0xbe, 0x9c, 0xd9,
	0x80, 0x87, 0x89, 0x7d, 0xac, 0x57, 0x86, 0x74, 0x31, 0x7d, 0x61, 0x47,
	0xe1, 0x87, 0xb4, 0x04, 0x01, 0x78, 0x62, 0xdc, 0xe8, 0x0d, 0x75, 0xe7,
	0x49, 0xf2, 0x95, 0x8a, 0x88, 0xb0, 0x35, 0x2d, 0x3a, 0x8b, 0xa6, 0x47,
	0xf5, 0x9b, 0x5d, 0x3f, 0x6d, 0x13, 0x7b, 0x4f, 0xc1, 0x0e, 0x3c, 0x4f,
	0x32, 0x64, 0xf0, 0x9a, 0x5f, 0xe3, 0xe4, 0xd5, 0xbf, 0x9c, 0xcd, 0x0d,
	0xe2, 0xd9, 0x5d, 0x97, 0x61, 0xf1, 0x6f, 0x08, 0x04, 0x92, 0xaf, 0xa6,
	0xe1, 0xa5, 0x45, 0x60, 0x6a, 0xe9, 0xd0, 0xa3, 0xa2, 0x86, 0x00, 0xb9,
	0x90, 0x31, 0x8e, 0x7d
};

static uint8_t rsa_public_exponent[] = {
	0x01, 0x00, 0x01
};

static uint8_t ec_params[] = {
	0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07
};

static uint8_t ec_point[] = {
	0x04, 0x41, 0x04, 0xb1, 0xa4, 0xc9, 0xb7, 0x47, 0x5e, 0x6d, 0x76, 0x96,
	0xec, 0x7a, 0x5c, 0xae, 0x99, 0xfb, 0x5f, 0x4e, 0xd2, 0xb1, 0x39, 0x3d,
	0xe8, 0xe2, 0x85, 0x6a, 0x5f, 0x7b, 0x47, 0x67, 0xfd, 0xc1, 0xeb, 0x66,
	0x2b, 0xf1, 0x15, 0xc9, 0x92, 0x23, 0x0a, 0xcb, 0x61, 0xe2, 0xdb, 0xe4,
	0x56, 0xdb, 0xf2, 0x56, 0xc6, 0xb0, 0x04, 0x5d, 0x00, 0xda, 0x22, 0x3b,
	0x37, 0xeb, 0x99, 0x76, 0x23, 0xec, 0x87
};

static uint8_t spki[] = {
	0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02,
	0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03,
	0x42, 0x00, 0x04, 0xb1, 0xa4, 0xc9, 0xb7, 0x47, 0x5e, 0x6d, 0x76, 0x96,
	0xec, 0x7a, 0x5c, 0xae, 0x99, 0xfb, 0x5f, 0x4e, 0xd2, 0xb1, 0x39, 0x3d,
	0xe8, 0xe2, 0x85, 0x6a, 0x5f, 0x7b, 0x47, 0x67, 0xfd, 0xc1, 0xeb, 0x66,
	0x2b, 0xf1, 0x15, 0xc9, 0x92, 0x23, 0x0a, 0xcb, 0x61, 0xe2, 0xdb, 0xe4,
	0x56, 0xdb, 0xf2, 0x56, 0xc6, 0xb0, 0x04, 0x5d, 0x00, 0xda, 0x22, 0x3b,
	0x37, 0xeb, 0x99, 0x76, 0x23, 0xec, 0x87
};

static CK_RV
override_C_Initialize (CK_VOID_PTR init_args)
{
	CK_OBJECT_CLASS klass = CKO_PUBLIC_KEY;
	CK_KEY_TYPE rsa_type = CKK_RSA;
	char rsa_label[] = "RSA";
	CK_ATTRIBUTE rsa_pubkey[] = {
		{ CKA_CLASS, &klass, sizeof (klass) },
		{ CKA_LABEL, rsa_label, sizeof (rsa_label) - 1 },
		{ CKA_KEY_TYPE, &rsa_type, sizeof (rsa_type) },
		{ CKA_PUBLIC_KEY_INFO, NULL, 0 },
		{ CKA_MODULUS, rsa_modulus, sizeof (rsa_modulus) },
		{ CKA_PUBLIC_EXPONENT, rsa_public_exponent, sizeof (rsa_public_exponent) },
		{ CKA_INVALID },
	};
	CK_KEY_TYPE ec_type = CKK_EC;
	char ec_label[] = "EC";
	CK_ATTRIBUTE ec_pubkey[] = {
		{ CKA_CLASS, &klass, sizeof (klass) },
		{ CKA_LABEL, ec_label, sizeof (ec_label) - 1 },
		{ CKA_KEY_TYPE, &ec_type, sizeof (ec_type) },
		{ CKA_PUBLIC_KEY_INFO, NULL, 0 },
		{ CKA_EC_PARAMS, ec_params, sizeof (ec_params) },
		{ CKA_EC_POINT, ec_point, sizeof (ec_point) },
		{ CKA_INVALID },
	};
	char spki_label[] = "SPKI";
	CK_ATTRIBUTE spki_pubkey[] = {
		{ CKA_CLASS, &klass, sizeof (klass) },
		{ CKA_LABEL, spki_label, sizeof (spki_label) - 1 },
		{ CKA_KEY_TYPE, &ec_type, sizeof (ec_type) },
		{ CKA_PUBLIC_KEY_INFO, spki, sizeof (spki) },
		{ CKA_INVALID },
	};
	CK_RV rv;

	rv = mock_C_Initialize (init_args);
	mock_module_add_object (MOCK_SLOT_ONE_ID, rsa_pubkey);
	mock_module_add_object (MOCK_SLOT_ONE_ID, ec_pubkey);
	mock_module_add_object (MOCK_SLOT_ONE_ID, spki_pubkey);
	return rv;
}

static CK_RV
override_C_GetTokenInfo (CK_SLOT_ID slot_id,
			 CK_TOKEN_INFO_PTR info)
{
	return_val_if_fail (info != NULL, CKR_ARGUMENTS_BAD);

	if (slot_id == MOCK_SLOT_ONE_ID) {
		memcpy (info, &MOCK_TOKEN_INFO, sizeof (*info));
		return CKR_OK;
	} else if (slot_id == MOCK_SLOT_TWO_ID) {
		return CKR_TOKEN_NOT_PRESENT;
	} else {
		return CKR_SLOT_ID_INVALID;
	}
}

#ifdef OS_WIN32
__declspec(dllexport)
#endif
CK_RV
C_GetFunctionList (CK_FUNCTION_LIST_PTR_PTR list)
{
	mock_module_init ();
	mock_module.C_GetFunctionList = C_GetFunctionList;
	if (list == NULL)
		return CKR_ARGUMENTS_BAD;
	mock_module.C_Initialize = override_C_Initialize;
	mock_module.C_GetTokenInfo = override_C_GetTokenInfo;
	*list = &mock_module;
	return CKR_OK;
}
