#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
# This file is part of mutter.
#
# Copyright (C) 2024 Alberto Ruiz <aruiz@gnome.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# Author: Alberto Ruiz <aruiz@gnome.org>

# Python port of <https://gitlab.gnome.org/GNOME/gtk/-/blob/main/gdk/gen-keyname-table.pl>

from dataclasses import dataclass
import sys
import re
import time


@dataclass
class Key:
    name: str
    keyval: int
    offset: int = 0


if __name__ == "__main__":
    if len(sys.argv) != 2:
        sys.exit(f"Usage: {sys.argv[0]} keynames.txt > keyname-table.h")

    keys = []
    with open(sys.argv[1], "r", encoding="utf8") as keynames:
        for line in keynames.readlines():
            if line.startswith("!"):
                continue

            match = re.match(r"^\s*(0x[0-9a-f]+)\s+([\w_]*\S)\s+(1)?\s*$", line)
            if not match:
                sys.exit(f"Failed to parse line: {line}")

            groups = match.groups()
            keys.append(Key(groups[1], int(groups[0])))

    formatted_time = time.strftime("%a %b %d %H:%M:%S %Y", time.gmtime())
    print(f"""/* clutter-keyname-table.h: Generated by gen-keyname-table.py from keynames.txt
 *
 *  Date: {formatted_time}
 *
 * Do not edit.
 */
static const char keynames[] =""")

    offset = 0
    for key in keys:
        if offset != 0:
            print("")

        print(f'  "{key.name}\\0"', end="")

        key.offset = offset
        offset += len(key.name) + 1

    print(";")

    print("""
typedef struct {
    unsigned int keyval;
    unsigned int offset;
} clutter_key;

static const clutter_key clutter_keys_by_keyval[] = {""")

    for i, key in enumerate(keys):
        if i != 0:
            print(",")

        print("".join(f"  {{ {key.keyval}, {key.offset} }}"), end="")

    print("\n};\n")
