/*
* Copyright (c) 2022, Intel Corporation
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
* THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
* OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*/
//!
//! \file     mhw_vebox_cmdpar.h
//! \brief    MHW command parameters
//! \details
//!

#ifndef __MHW_VEBOX_CMDPAR_H__
#define __MHW_VEBOX_CMDPAR_H__

#include "mhw_vebox.h"

#define HDR_OETF_1DLUT_POINT_NUMBER          256

//!
//! \brief Macro for Vebox Scalable
//!
#define MHW_VEBOX_MAX_PIPE_SIZE                         4096
#define MHW_VEBOX_MAX_SEMAPHORE_NUM                     4
#define MHW_VEBOX_TIMESTAMP_CNTS_PER_SEC                12000048
#define MHW_VEBOX_4K_PIC_WIDTH                          3840
#define MHW_VEBOX_4K_PIC_HEIGHT                         2160
#define MHW_VEBOX_TIMESTAMP_PER_TICK_IN_NS              83.333f
#define MHW_LACE_COLOR_COMPENSATION_LUT_POINT_NUMBER    16
#define MHW_VEBOX_SLIM_IPU_DN_CMD_SIZE_INUSE            34

namespace mhw
{
namespace vebox
{
struct MHW_LACE_COLOR_WEIGHT_LUT
{
    int32_t iPoint[MHW_LACE_COLOR_COMPENSATION_LUT_POINT_NUMBER] = {};  // U5.3
    int32_t iBias[MHW_LACE_COLOR_COMPENSATION_LUT_POINT_NUMBER]  = {};  // U5.8
    int32_t iSlope[MHW_LACE_COLOR_COMPENSATION_LUT_POINT_NUMBER] = {};  // U1.10
};

struct MHW_LACE_COLOR_CORRECTION
{
    bool                      bColorCorrectionEnable        = false;        // Color correction enable from Gen12;
    bool                      bYUVFullRange                 = false;        // Color correction need YUV offset from Gen12
    float                     fColorCompensationPowerFactor = 0.0;          // Color Compensation Power Factor from Gen12
    MHW_LACE_COLOR_WEIGHT_LUT colorWeightLut                = {};
};

// H2S Manual Mode Coef
static const uint16_t g_Hdr_ColorCorrect_EOTF_SMPTE_ST2084_Input[HDR_OETF_1DLUT_POINT_NUMBER] =
{
       0,     257,     514,     771,    1028,    1285,    1542,    1799,    2056,    2313,    2570,    2827,    3084,    3341,    3598,    3855,
    4112,    4369,    4626,    4883,    5140,    5397,    5654,    5911,    6168,    6425,    6682,    6939,    7196,    7453,    7710,    7967,
    8224,    8481,    8738,    8995,    9252,    9509,    9766,   10023,   10280,   10537,   10794,   11051,   11308,   11565,   11822,   12079,
   12336,   12593,   12850,   13107,   13364,   13621,   13878,   14135,   14392,   14649,   14906,   15163,   15420,   15677,   15934,   16191,
   16448,   16705,   16962,   17219,   17476,   17733,   17990,   18247,   18504,   18761,   19018,   19275,   19532,   19789,   20046,   20303,
   20560,   20817,   21074,   21331,   21588,   21845,   22102,   22359,   22616,   22873,   23130,   23387,   23644,   23901,   24158,   24415,
   24672,   24929,   25186,   25443,   25700,   25957,   26214,   26471,   26728,   26985,   27242,   27499,   27756,   28013,   28270,   28527,
   28784,   29041,   29298,   29555,   29812,   30069,   30326,   30583,   30840,   31097,   31354,   31611,   31868,   32125,   32382,   32639,
   32896,   33153,   33410,   33667,   33924,   34181,   34438,   34695,   34952,   35209,   35466,   35723,   35980,   36237,   36494,   36751,
   37008,   37265,   37522,   37779,   38036,   38293,   38550,   38807,   39064,   39321,   39578,   39835,   40092,   40349,   40606,   40863,
   41120,   41377,   41634,   41891,   42148,   42405,   42662,   42919,   43176,   43433,   43690,   43947,   44204,   44461,   44718,   44975,
   45232,   45489,   45746,   46003,   46260,   46517,   46774,   47031,   47288,   47545,   47802,   48059,   48316,   48573,   48830,   49087,
   49344,   49601,   49858,   50115,   50372,   50629,   50886,   51143,   51400,   51657,   51914,   52171,   52428,   52685,   52942,   53199,
   53456,   53713,   53970,   54227,   54484,   54741,   54998,   55255,   55512,   55769,   56026,   56283,   56540,   56797,   57054,   57311,
   57568,   57825,   58082,   58339,   58596,   58853,   59110,   59367,   59624,   59881,   60138,   60395,   60652,   60909,   61166,   61423,
   61680,   61937,   62194,   62451,   62708,   62965,   63222,   63479,   63736,   63993,   64250,   64507,   64764,   65021,   65278,   65535
};

static const uint16_t g_Hdr_ColorCorrect_EOTF_SMPTE_ST2084_Output[HDR_OETF_1DLUT_POINT_NUMBER] =
{
       0,       0,       0,       0,       0,       0,       0,       0,       0,       0,       0,       0,       0,       0,       0,       1,
       1,       1,       1,       1,       1,       1,       1,       2,       2,       2,       2,       2,       3,       3,       3,       4,
       4,       4,       5,       5,       5,       6,       6,       7,       7,       8,       9,       9,      10,      11,      11,      12,
      13,      14,      15,      16,      17,      18,      19,      20,      22,      23,      24,      26,      27,      29,      31,      32,
      34,      36,      38,      40,      43,      45,      47,      50,      52,      55,      58,      61,      64,      67,      71,      74,
      78,      82,      86,      90,      95,      99,     104,     109,     114,     119,     125,     131,     137,     143,     150,     157,
     164,     171,     179,     187,     195,     204,     213,     222,     232,     242,     252,     263,     274,     286,     298,     311,
     324,     338,     352,     367,     382,     398,     414,     431,     449,     467,     486,     506,     527,     548,     570,     593,
     617,     641,     667,     693,     721,     749,     779,     809,     841,     874,     908,     944,     980,    1018,    1058,    1099,
    1141,    1185,    1231,    1278,    1327,    1377,    1430,    1484,    1541,    1599,    1660,    1722,    1787,    1855,    1925,    1997,
    2072,    2150,    2230,    2314,    2400,    2490,    2583,    2679,    2778,    2882,    2989,    3099,    3214,    3333,    3457,    3584,
    3717,    3854,    3996,    4143,    4296,    4454,    4618,    4787,    4963,    5146,    5335,    5530,    5733,    5943,    6161,    6387,
    6621,    6863,    7115,    7375,    7645,    7925,    8215,    8515,    8827,    9150,    9485,    9832,   10192,   10565,   10952,   11353,
   11769,   12200,   12647,   13110,   13591,   14089,   14606,   15142,   15698,   16275,   16873,   17494,   18138,   18805,   19498,   20217,
   20963,   21736,   22539,   23372,   24237,   25134,   26066,   27032,   28036,   29077,   30158,   31281,   32446,   33656,   34912,   36217,
   37572,   38979,   40441,   41959,   43536,   45174,   46876,   48645,   50482,   52392,   54376,   56438,   58582,   60810,   63127,   65535
};

static const uint16_t g_Hdr_ColorCorrect_OETF_Rec709_Input[HDR_OETF_1DLUT_POINT_NUMBER] =
{
       0,     257,     514,     771,    1028,    1285,    1542,    1799,    2056,    2313,    2570,    2827,    3084,    3341,    3598,    3855,
    4112,    4369,    4626,    4883,    5140,    5397,    5654,    5911,    6168,    6425,    6682,    6939,    7196,    7453,    7710,    7967,
    8224,    8481,    8738,    8995,    9252,    9509,    9766,   10023,   10280,   10537,   10794,   11051,   11308,   11565,   11822,   12079,
   12336,   12593,   12850,   13107,   13364,   13621,   13878,   14135,   14392,   14649,   14906,   15163,   15420,   15677,   15934,   16191,
   16448,   16705,   16962,   17219,   17476,   17733,   17990,   18247,   18504,   18761,   19018,   19275,   19532,   19789,   20046,   20303,
   20560,   20817,   21074,   21331,   21588,   21845,   22102,   22359,   22616,   22873,   23130,   23387,   23644,   23901,   24158,   24415,
   24672,   24929,   25186,   25443,   25700,   25957,   26214,   26471,   26728,   26985,   27242,   27499,   27756,   28013,   28270,   28527,
   28784,   29041,   29298,   29555,   29812,   30069,   30326,   30583,   30840,   31097,   31354,   31611,   31868,   32125,   32382,   32639,
   32896,   33153,   33410,   33667,   33924,   34181,   34438,   34695,   34952,   35209,   35466,   35723,   35980,   36237,   36494,   36751,
   37008,   37265,   37522,   37779,   38036,   38293,   38550,   38807,   39064,   39321,   39578,   39835,   40092,   40349,   40606,   40863,
   41120,   41377,   41634,   41891,   42148,   42405,   42662,   42919,   43176,   43433,   43690,   43947,   44204,   44461,   44718,   44975,
   45232,   45489,   45746,   46003,   46260,   46517,   46774,   47031,   47288,   47545,   47802,   48059,   48316,   48573,   48830,   49087,
   49344,   49601,   49858,   50115,   50372,   50629,   50886,   51143,   51400,   51657,   51914,   52171,   52428,   52685,   52942,   53199,
   53456,   53713,   53970,   54227,   54484,   54741,   54998,   55255,   55512,   55769,   56026,   56283,   56540,   56797,   57054,   57311,
   57568,   57825,   58082,   58339,   58596,   58853,   59110,   59367,   59624,   59881,   60138,   60395,   60652,   60909,   61166,   61423,
   61680,   61937,   62194,   62451,   62708,   62965,   63222,   63479,   63736,   63993,   64250,   64507,   64764,   65021,   65278,   65535
};

static const uint16_t g_Hdr_ColorCorrect_OETF_Rec709_Output[HDR_OETF_1DLUT_POINT_NUMBER] =
{
       0,    1157,    2313,    3469,    4626,    5788,    6838,    7795,    8680,    9505,   10282,   11017,   11716,   12383,   13023,   13639,
   14232,   14805,   15359,   15898,   16420,   16929,   17424,   17907,   18379,   18840,   19291,   19733,   20165,   20589,   21006,   21414,
   21816,   22211,   22599,   22981,   23357,   23727,   24092,   24451,   24806,   25155,   25500,   25841,   26177,   26509,   26837,   27161,
   27481,   27798,   28111,   28421,   28727,   29030,   29330,   29627,   29921,   30213,   30501,   30786,   31069,   31350,   31628,   31903,
   32176,   32447,   32715,   32982,   33246,   33507,   33767,   34025,   34281,   34535,   34787,   35037,   35285,   35531,   35776,   36019,
   36260,   36500,   36738,   36974,   37209,   37443,   37674,   37905,   38134,   38361,   38587,   38812,   39035,   39257,   39478,   39697,
   39915,   40132,   40348,   40562,   40776,   40988,   41199,   41409,   41617,   41825,   42031,   42237,   42441,   42645,   42847,   43048,
   43249,   43448,   43646,   43844,   44040,   44236,   44430,   44624,   44817,   45009,   45200,   45390,   45580,   45768,   45956,   46143,
   46329,   46514,   46699,   46882,   47065,   47248,   47429,   47610,   47790,   47969,   48147,   48325,   48502,   48679,   48854,   49029,
   49204,   49378,   49551,   49723,   49895,   50066,   50236,   50406,   50575,   50744,   50912,   51080,   51246,   51413,   51578,   51743,
   51908,   52072,   52235,   52398,   52560,   52722,   52883,   53044,   53204,   53364,   53523,   53682,   53840,   53997,   54154,   54311,
   54467,   54623,   54778,   54932,   55086,   55240,   55393,   55546,   55699,   55850,   56002,   56153,   56303,   56453,   56603,   56752,
   56901,   57049,   57197,   57345,   57492,   57639,   57785,   57931,   58076,   58221,   58366,   58510,   58654,   58798,   58941,   59083,
   59226,   59368,   59509,   59651,   59792,   59932,   60072,   60212,   60351,   60490,   60629,   60768,   60906,   61043,   61181,   61318,
   61454,   61591,   61727,   61862,   61998,   62133,   62267,   62402,   62536,   62669,   62803,   62936,   63069,   63201,   63333,   63465,
   63597,   63728,   63859,   63990,   64120,   64250,   64380,   64509,   64638,   64767,   64896,   65024,   65152,   65280,   65408,   65535
};

const int32_t g_Vebox_BT2020_Inverse_Pixel_Value[256] =
{
    0x0000, 0x14bc, 0x15a8, 0x1694, 0x1780, 0x1870, 0x195c, 0x1a48, 0x1b34, 0x1c24, 0x1d10, 0x1dfc, 0x1eec, 0x1fd8, 0x20c4, 0x21b0,
    0x22a0, 0x238c, 0x2478, 0x2568, 0x2654, 0x2740, 0x282c, 0x291c, 0x2a08, 0x2af4, 0x2be0, 0x2cd0, 0x2dbc, 0x2ea8, 0x2f98, 0x3084,
    0x3170, 0x325c, 0x334c, 0x3438, 0x3524, 0x3614, 0x3700, 0x37ec, 0x38d8, 0x39c8, 0x3ab4, 0x3ba0, 0x3c8c, 0x3d7c, 0x3e68, 0x3f54,
    0x4044, 0x4130, 0x421c, 0x4308, 0x43f8, 0x44e4, 0x45d0, 0x46c0, 0x47ac, 0x4898, 0x4984, 0x4a74, 0x4b60, 0x4c4c, 0x4d38, 0x4e28,
    0x4f14, 0x5000, 0x50f0, 0x51dc, 0x52c8, 0x53b4, 0x54a4, 0x5590, 0x567c, 0x576c, 0x5858, 0x5944, 0x5a30, 0x5b20, 0x5c0c, 0x5cf8,
    0x5de8, 0x5ed4, 0x5fc0, 0x60ac, 0x619c, 0x6288, 0x6374, 0x6460, 0x6550, 0x663c, 0x6728, 0x6818, 0x6904, 0x69f0, 0x6adc, 0x6bcc,
    0x6cb8, 0x6da4, 0x6e94, 0x6f80, 0x706c, 0x7158, 0x7248, 0x7334, 0x7420, 0x750c, 0x75fc, 0x76e8, 0x77d4, 0x78c4, 0x79b0, 0x7a9c,
    0x7b88, 0x7c78, 0x7d64, 0x7e50, 0x7f40, 0x802c, 0x8118, 0x8204, 0x82f4, 0x83e0, 0x84cc, 0x85b8, 0x86a8, 0x8794, 0x8880, 0x8970,
    0x8a5c, 0x8b48, 0x8c34, 0x8d24, 0x8e10, 0x8efc, 0x8fec, 0x90d8, 0x91c4, 0x92b0, 0x93a0, 0x948c, 0x9578, 0x9664, 0x9754, 0x9840,
    0x992c, 0x9a1c, 0x9b08, 0x9bf4, 0x9ce0, 0x9dd0, 0x9ebc, 0x9fa8, 0xa098, 0xa184, 0xa270, 0xa35c, 0xa44c, 0xa538, 0xa624, 0xa714,
    0xa800, 0xa8ec, 0xa9d8, 0xaac8, 0xabb4, 0xaca0, 0xad8c, 0xae7c, 0xaf68, 0xb054, 0xb144, 0xb230, 0xb31c, 0xb408, 0xb4f8, 0xb5e4,
    0xb6d0, 0xb7c0, 0xb8ac, 0xb998, 0xba84, 0xbb74, 0xbc60, 0xbd4c, 0xbe38, 0xbf28, 0xc014, 0xc100, 0xc1f0, 0xc2dc, 0xc3c8, 0xc4b4,
    0xc5a4, 0xc690, 0xc77c, 0xc86c, 0xc958, 0xca44, 0xcb30, 0xcc20, 0xcd0c, 0xcdf8, 0xcee4, 0xcfd4, 0xd0c0, 0xd1ac, 0xd29c, 0xd388,
    0xd474, 0xd560, 0xd650, 0xd73c, 0xd828, 0xd918, 0xda04, 0xdaf0, 0xdbdc, 0xdccc, 0xddb8, 0xdea4, 0xdf94, 0xe080, 0xe16c, 0xe258,
    0xe348, 0xe434, 0xe520, 0xe60c, 0xe6fc, 0xe7e8, 0xe8d4, 0xe9c4, 0xeab0, 0xeb9c, 0xec88, 0xed78, 0xee64, 0xef50, 0xf040, 0xf12c,
    0xf218, 0xf304, 0xf3f4, 0xf4e0, 0xf5cc, 0xf6b8, 0xf7a8, 0xf894, 0xf980, 0xfa70, 0xfb5c, 0xfc48, 0xfd34, 0xfe24, 0xff10, 0xffff
};

const int32_t g_Vebox_BT2020_Forward_Pixel_Value[256] =
{
    0x0000, 0x049c, 0x0598, 0x0694, 0x0794, 0x0890, 0x098c, 0x0a8c, 0x0b88, 0x0c84, 0x0d84, 0x0e80, 0x0f7c, 0x107c, 0x1178, 0x1274,
    0x1374, 0x1470, 0x156c, 0x166c, 0x1768, 0x1864, 0x1964, 0x1a60, 0x1b5c, 0x1c5c, 0x1d58, 0x1e54, 0x1f54, 0x2050, 0x214c, 0x224c,
    0x2348, 0x2444, 0x2544, 0x2640, 0x273c, 0x283c, 0x2938, 0x2a34, 0x2b34, 0x2c30, 0x2d30, 0x2e2c, 0x2f28, 0x3028, 0x3124, 0x3220,
    0x3320, 0x341c, 0x3518, 0x3618, 0x3714, 0x3810, 0x3910, 0x3a0c, 0x3b08, 0x3c08, 0x3d04, 0x3e00, 0x3f00, 0x3ffc, 0x40f8, 0x41f8,
    0x42f4, 0x43f0, 0x44f0, 0x45ec, 0x46e8, 0x47e8, 0x48e4, 0x49e0, 0x4ae0, 0x4bdc, 0x4cd8, 0x4dd8, 0x4ed4, 0x4fd0, 0x50d0, 0x51cc,
    0x52c8, 0x53c8, 0x54c4, 0x55c4, 0x56c0, 0x57bc, 0x58bc, 0x59b8, 0x5ab4, 0x5bb4, 0x5cb0, 0x5dac, 0x5eac, 0x5fa8, 0x60a4, 0x61a4,
    0x62a0, 0x639c, 0x649c, 0x6598, 0x6694, 0x6794, 0x6890, 0x698c, 0x6a8c, 0x6b88, 0x6c84, 0x6d84, 0x6e80, 0x6f7c, 0x707c, 0x7178,
    0x7274, 0x7374, 0x7470, 0x756c, 0x766c, 0x7768, 0x7864, 0x7964, 0x7a60, 0x7b5c, 0x7c5c, 0x7d58, 0x7e58, 0x7f54, 0x8050, 0x8150,
    0x824c, 0x8348, 0x8448, 0x8544, 0x8640, 0x8740, 0x883c, 0x8938, 0x8a38, 0x8b34, 0x8c30, 0x8d30, 0x8e2c, 0x8f28, 0x9028, 0x9124,
    0x9220, 0x9320, 0x941c, 0x9518, 0x9618, 0x9714, 0x9810, 0x9910, 0x9a0c, 0x9b08, 0x9c08, 0x9d04, 0x9e00, 0x9f00, 0x9ffc, 0xa0f8,
    0xa1f8, 0xa2f4, 0xa3f0, 0xa4f0, 0xa5ec, 0xa6ec, 0xa7e8, 0xa8e4, 0xa9e4, 0xaae0, 0xabdc, 0xacdc, 0xadd8, 0xaed4, 0xafd4, 0xb0d0,
    0xb1cc, 0xb2cc, 0xb3c8, 0xb4c4, 0xb5c4, 0xb6c0, 0xb7bc, 0xb8bc, 0xb9b8, 0xbab4, 0xbbb4, 0xbcb0, 0xbdac, 0xbeac, 0xbfa8, 0xc0a4,
    0xc1a4, 0xc2a0, 0xc39c, 0xc49c, 0xc598, 0xc694, 0xc794, 0xc890, 0xc98c, 0xca8c, 0xcb88, 0xcc84, 0xcd84, 0xce80, 0xcf80, 0xd07c,
    0xd178, 0xd278, 0xd374, 0xd470, 0xd570, 0xd66c, 0xd768, 0xd868, 0xd964, 0xda60, 0xdb60, 0xdc5c, 0xdd58, 0xde58, 0xdf54, 0xe050,
    0xe150, 0xe24c, 0xe348, 0xe448, 0xe544, 0xe640, 0xe740, 0xe83c, 0xe938, 0xea38, 0xeb34, 0xec30, 0xed30, 0xee2c, 0xef28, 0xf028,
    0xf124, 0xf220, 0xf320, 0xf41c, 0xf518, 0xf618, 0xf714, 0xf814, 0xf910, 0xfa0c, 0xfb0c, 0xfc08, 0xfd04, 0xfe04, 0xff00, 0xffff
};

const int32_t g_Vebox_BT2020_Inverse_Gamma_LUT[256] =
{
    0x0000, 0x049c, 0x04cc, 0x0503, 0x053a, 0x0574, 0x05ae, 0x05e9, 0x0626, 0x0665, 0x06a5, 0x06e5, 0x0729, 0x076c, 0x07b1, 0x07f7,
    0x083f, 0x0888, 0x08d2, 0x091f, 0x096c, 0x09bb, 0x0a0a, 0x0a5d, 0x0aaf, 0x0b03, 0x0b58, 0x0bb0, 0x0c09, 0x0c62, 0x0cbf, 0x0d1b,
    0x0d79, 0x0dd8, 0x0e3a, 0x0e9c, 0x0f00, 0x0f66, 0x0fcd, 0x1035, 0x109e, 0x110b, 0x1177, 0x11e5, 0x1254, 0x12c6, 0x1339, 0x13ac,
    0x1423, 0x149a, 0x1512, 0x158c, 0x1609, 0x1685, 0x1704, 0x1785, 0x1806, 0x1889, 0x190d, 0x1995, 0x1a1c, 0x1aa5, 0x1b2f, 0x1bbe,
    0x1c4b, 0x1cda, 0x1d6d, 0x1dff, 0x1e92, 0x1f27, 0x1fc0, 0x2059, 0x20f2, 0x2190, 0x222d, 0x22cc, 0x236c, 0x2410, 0x24b3, 0x2558,
    0x2601, 0x26a9, 0x2752, 0x27fe, 0x28ad, 0x295b, 0x2a0b, 0x2abd, 0x2b73, 0x2c28, 0x2cde, 0x2d99, 0x2e53, 0x2f0e, 0x2fcb, 0x308c,
    0x314c, 0x320e, 0x32d5, 0x339a, 0x3460, 0x3528, 0x35f6, 0x36c1, 0x378e, 0x385d, 0x3931, 0x3a03, 0x3ad7, 0x3bb0, 0x3c87, 0x3d60,
    0x3e3a, 0x3f1a, 0x3ff8, 0x40d7, 0x41bc, 0x429f, 0x4383, 0x4469, 0x4555, 0x463e, 0x4729, 0x4816, 0x4909, 0x49f9, 0x4aeb, 0x4be3,
    0x4cd8, 0x4dcf, 0x4ec8, 0x4fc7, 0x50c3, 0x51c1, 0x52c5, 0x53c6, 0x54c9, 0x55ce, 0x56d9, 0x57e1, 0x58eb, 0x59f6, 0x5b08, 0x5c17,
    0x5d28, 0x5e3f, 0x5f54, 0x606a, 0x6181, 0x62a0, 0x63bb, 0x64d8, 0x65fc, 0x671c, 0x683e, 0x6962, 0x6a8d, 0x6bb5, 0x6cde, 0x6e0e,
    0x6f3b, 0x706a, 0x719a, 0x72d1, 0x7405, 0x753b, 0x7672, 0x77b1, 0x78ec, 0x7a29, 0x7b6d, 0x7cad, 0x7def, 0x7f33, 0x807e, 0x81c6,
    0x830f, 0x8460, 0x85ad, 0x86fb, 0x884c, 0x89a4, 0x8af8, 0x8c4e, 0x8da6, 0x8f05, 0x9061, 0x91be, 0x9323, 0x9483, 0x95e6, 0x974a,
    0x98b7, 0x9a1f, 0x9b89, 0x9cfb, 0x9e68, 0x9fd8, 0xa149, 0xa2c2, 0xa437, 0xa5ae, 0xa726, 0xa8a7, 0xaa23, 0xaba2, 0xad28, 0xaeaa,
    0xb02d, 0xb1b3, 0xb341, 0xb4ca, 0xb655, 0xb7e9, 0xb977, 0xbb08, 0xbc9b, 0xbe36, 0xbfcc, 0xc164, 0xc305, 0xc4a1, 0xc63e, 0xc7de,
    0xc986, 0xcb2a, 0xcccf, 0xce76, 0xd026, 0xd1d1, 0xd37e, 0xd533, 0xd6e4, 0xd896, 0xda4a, 0xdc08, 0xddc0, 0xdf7a, 0xe13d, 0xe2fb,
    0xe4bb, 0xe67c, 0xe847, 0xea0c, 0xebd4, 0xed9d, 0xef6f, 0xf13c, 0xf30b, 0xf4e4, 0xf6b6, 0xf88b, 0xfa62, 0xfc42, 0xfe1c, 0xffff,
};

const int32_t g_Vebox_BT2020_Forward_Gamma_LUT[256] =
{
    0x0000, 0x14bc, 0x1901, 0x1cd0, 0x2060, 0x23a3, 0x26b2, 0x29a2, 0x2c60, 0x2eff, 0x318a, 0x33f3, 0x3644, 0x388a, 0x3ab5, 0x3cce,
    0x3ee0, 0x40db, 0x42c9, 0x44b3, 0x4689, 0x4854, 0x4a1c, 0x4bd4, 0x4d82, 0x4f2f, 0x50cd, 0x5264, 0x53f9, 0x5582, 0x5703, 0x5885,
    0x59fb, 0x5b6a, 0x5cdb, 0x5e40, 0x5fa0, 0x6100, 0x6257, 0x63a9, 0x64fc, 0x6646, 0x6790, 0x68d2, 0x6a10, 0x6b4f, 0x6c86, 0x6db9,
    0x6eee, 0x701a, 0x7144, 0x726f, 0x7393, 0x74b3, 0x75d6, 0x76f1, 0x780a, 0x7924, 0x7a37, 0x7b48, 0x7c5b, 0x7d68, 0x7e72, 0x7f7e,
    0x8083, 0x8187, 0x828d, 0x838c, 0x8489, 0x8589, 0x8683, 0x877a, 0x8874, 0x8968, 0x8a5b, 0x8b4f, 0x8c3e, 0x8d2c, 0x8e1b, 0x8f06,
    0x8fee, 0x90d9, 0x91bf, 0x92a6, 0x9389, 0x946a, 0x954e, 0x962c, 0x970a, 0x97e9, 0x98c3, 0x999d, 0x9a78, 0x9b4f, 0x9c24, 0x9cfc,
    0x9dcf, 0x9ea1, 0x9f76, 0xa045, 0xa114, 0xa1e5, 0xa2b1, 0xa37d, 0xa44a, 0xa514, 0xa5dc, 0xa6a6, 0xa76d, 0xa832, 0xa8f9, 0xa9bd,
    0xaa7f, 0xab44, 0xac05, 0xacc4, 0xad86, 0xae44, 0xaf02, 0xafc1, 0xb07c, 0xb137, 0xb1f4, 0xb2ad, 0xb368, 0xb41f, 0xb4d6, 0xb58e,
    0xb643, 0xb6f8, 0xb7ae, 0xb861, 0xb913, 0xb9c7, 0xba78, 0xbb28, 0xbbda, 0xbc89, 0xbd36, 0xbde6, 0xbe93, 0xbf3f, 0xbfed, 0xc097,
    0xc141, 0xc1ed, 0xc296, 0xc33e, 0xc3e8, 0xc48f, 0xc535, 0xc5de, 0xc683, 0xc727, 0xc7ce, 0xc871, 0xc914, 0xc9b9, 0xca5a, 0xcafc,
    0xcb9f, 0xcc3f, 0xccde, 0xcd80, 0xce1e, 0xcebf, 0xcf5c, 0xcff9, 0xd098, 0xd134, 0xd1cf, 0xd26d, 0xd307, 0xd3a1, 0xd43d, 0xd4d6,
    0xd56e, 0xd609, 0xd6a0, 0xd738, 0xd7d1, 0xd867, 0xd8fd, 0xd994, 0xda29, 0xdabe, 0xdb54, 0xdbe8, 0xdc7b, 0xdd10, 0xdda2, 0xde34,
    0xdec8, 0xdf59, 0xdfea, 0xe07d, 0xe10c, 0xe19c, 0xe22d, 0xe2bc, 0xe34a, 0xe3db, 0xe468, 0xe4f5, 0xe584, 0xe611, 0xe69f, 0xe72b,
    0xe7b6, 0xe843, 0xe8ce, 0xe958, 0xe9e4, 0xea6e, 0xeaf7, 0xeb82, 0xec0b, 0xec93, 0xed1d, 0xeda4, 0xee2c, 0xeeb5, 0xef3b, 0xefc1,
    0xf049, 0xf0cf, 0xf154, 0xf1db, 0xf25f, 0xf2e4, 0xf36a, 0xf3ed, 0xf471, 0xf4f6, 0xf579, 0xf5fb, 0xf67f, 0xf701, 0xf783, 0xf806,
    0xf887, 0xf907, 0xf98a, 0xfa0a, 0xfa8a, 0xfb0b, 0xfb8a, 0xfc0b, 0xfc8a, 0xfd08, 0xfd89, 0xfe06, 0xfe84, 0xff03, 0xff80, 0xffff
};

//!
//! Vertext Table for BT601
//!
const unsigned short g_VeboxVertexTableBT601[512][2] =
{
    //{  Cv,   Lv}
    { 1936, 666 },{ 1932, 673 },{ 1928, 680 },{ 1924, 687 },
    { 1920, 694 },{ 1917, 701 },{ 1913, 708 },{ 1910, 715 },
    { 1906, 722 },{ 1903, 729 },{ 1900, 736 },{ 1897, 743 },
    { 1894, 749 },{ 1891, 756 },{ 1888, 763 },{ 1885, 770 },
    { 1882, 776 },{ 1880, 783 },{ 1877, 790 },{ 1875, 796 },
    { 1872, 803 },{ 1870, 810 },{ 1868, 816 },{ 1866, 823 },
    { 1864, 830 },{ 1862, 836 },{ 1860, 843 },{ 1858, 849 },
    { 1856, 856 },{ 1854, 862 },{ 1853, 869 },{ 1851, 875 },
    { 1849, 882 },{ 1848, 888 },{ 1847, 895 },{ 1845, 901 },
    { 1844, 908 },{ 1843, 914 },{ 1842, 921 },{ 1841, 927 },
    { 1840, 933 },{ 1839, 940 },{ 1838, 946 },{ 1838, 953 },
    { 1837, 959 },{ 1836, 965 },{ 1836, 972 },{ 1835, 978 },
    { 1835, 985 },{ 1835, 991 },{ 1834, 997 },{ 1834, 1004 },
    { 1834, 1010 },{ 1834, 1017 },{ 1834, 1023 },{ 1834, 1029 },
    { 1834, 1036 },{ 1835, 1042 },{ 1835, 1048 },{ 1835, 1055 },
    { 1836, 1061 },{ 1836, 1068 },{ 1837, 1074 },{ 1837, 1080 },
    { 1838, 1087 },{ 1839, 1093 },{ 1840, 1100 },{ 1841, 1106 },
    { 1842, 1112 },{ 1843, 1119 },{ 1844, 1125 },{ 1845, 1132 },
    { 1847, 1138 },{ 1848, 1145 },{ 1849, 1151 },{ 1851, 1158 },
    { 1852, 1164 },{ 1854, 1171 },{ 1856, 1177 },{ 1858, 1184 },
    { 1859, 1190 },{ 1861, 1197 },{ 1863, 1203 },{ 1866, 1210 },
    { 1868, 1217 },{ 1870, 1223 },{ 1872, 1230 },{ 1875, 1236 },
    { 1877, 1243 },{ 1880, 1250 },{ 1882, 1257 },{ 1885, 1263 },
    { 1888, 1270 },{ 1891, 1277 },{ 1894, 1284 },{ 1897, 1290 },
    { 1900, 1297 },{ 1903, 1304 },{ 1906, 1311 },{ 1910, 1318 },
    { 1913, 1325 },{ 1916, 1332 },{ 1920, 1339 },{ 1924, 1346 },
    { 1928, 1353 },{ 1931, 1360 },{ 1935, 1367 },{ 1939, 1374 },
    { 1944, 1381 },{ 1948, 1388 },{ 1952, 1396 },{ 1957, 1403 },
    { 1961, 1410 },{ 1966, 1417 },{ 1970, 1425 },{ 1975, 1432 },
    { 1980, 1439 },{ 1985, 1447 },{ 1990, 1454 },{ 1995, 1462 },
    { 2001, 1470 },{ 2006, 1477 },{ 2011, 1485 },{ 2017, 1492 },
    { 2023, 1500 },{ 2029, 1508 },{ 2035, 1516 },{ 2041, 1524 },
    { 2047, 1532 },{ 2053, 1540 },{ 2059, 1548 },{ 2066, 1556 },
    { 2073, 1564 },{ 2079, 1572 },{ 2086, 1580 },{ 2093, 1588 },
    { 2100, 1597 },{ 2108, 1605 },{ 2115, 1614 },{ 2123, 1622 },
    { 2130, 1631 },{ 2138, 1639 },{ 2146, 1648 },{ 2154, 1657 },
    { 2162, 1666 },{ 2171, 1674 },{ 2179, 1683 },{ 2186, 1691 },
    { 2179, 1688 },{ 2172, 1684 },{ 2164, 1681 },{ 2157, 1678 },
    { 2151, 1674 },{ 2144, 1671 },{ 2137, 1668 },{ 2131, 1664 },
    { 2124, 1661 },{ 2118, 1658 },{ 2112, 1655 },{ 2106, 1651 },
    { 2100, 1648 },{ 2094, 1645 },{ 2088, 1642 },{ 2083, 1639 },
    { 2077, 1636 },{ 2072, 1633 },{ 2067, 1629 },{ 2061, 1626 },
    { 2056, 1623 },{ 2051, 1620 },{ 2046, 1617 },{ 2042, 1614 },
    { 2037, 1611 },{ 2032, 1608 },{ 2028, 1605 },{ 2024, 1602 },
    { 2019, 1599 },{ 2015, 1596 },{ 2011, 1593 },{ 2007, 1590 },
    { 2003, 1588 },{ 1999, 1585 },{ 1996, 1582 },{ 1992, 1579 },
    { 1988, 1576 },{ 1985, 1573 },{ 1981, 1570 },{ 1978, 1568 },
    { 1975, 1565 },{ 1972, 1562 },{ 1969, 1559 },{ 1966, 1556 },
    { 1963, 1554 },{ 1960, 1551 },{ 1957, 1548 },{ 1955, 1545 },
    { 1952, 1542 },{ 1950, 1540 },{ 1947, 1537 },{ 1945, 1534 },
    { 1943, 1531 },{ 1941, 1529 },{ 1939, 1526 },{ 1937, 1523 },
    { 1935, 1521 },{ 1933, 1518 },{ 1931, 1515 },{ 1929, 1513 },
    { 1928, 1510 },{ 1926, 1507 },{ 1925, 1504 },{ 1923, 1502 },
    { 1922, 1499 },{ 1921, 1496 },{ 1919, 1494 },{ 1918, 1491 },
    { 1917, 1488 },{ 1916, 1486 },{ 1916, 1483 },{ 1915, 1481 },
    { 1914, 1478 },{ 1913, 1475 },{ 1913, 1473 },{ 1912, 1470 },
    { 1912, 1467 },{ 1911, 1465 },{ 1911, 1462 },{ 1911, 1459 },
    { 1911, 1457 },{ 1911, 1454 },{ 1911, 1451 },{ 1911, 1449 },
    { 1911, 1446 },{ 1911, 1444 },{ 1911, 1441 },{ 1912, 1438 },
    { 1912, 1436 },{ 1912, 1433 },{ 1913, 1430 },{ 1914, 1428 },
    { 1914, 1425 },{ 1915, 1422 },{ 1916, 1420 },{ 1917, 1417 },
    { 1918, 1414 },{ 1919, 1412 },{ 1920, 1409 },{ 1921, 1406 },
    { 1923, 1404 },{ 1924, 1401 },{ 1925, 1398 },{ 1927, 1396 },
    { 1929, 1393 },{ 1930, 1390 },{ 1932, 1388 },{ 1934, 1385 },
    { 1936, 1382 },{ 1938, 1380 },{ 1940, 1377 },{ 1942, 1374 },
    { 1944, 1371 },{ 1946, 1369 },{ 1949, 1366 },{ 1951, 1363 },
    { 1954, 1360 },{ 1956, 1358 },{ 1959, 1355 },{ 1962, 1352 },
    { 1964, 1349 },{ 1967, 1347 },{ 1970, 1344 },{ 1973, 1341 },
    { 1977, 1338 },{ 1980, 1335 },{ 1983, 1332 },{ 1987, 1330 },
    { 1990, 1327 },{ 1994, 1324 },{ 1998, 1321 },{ 2001, 1318 },
    { 2005, 1315 },{ 2009, 1312 },{ 2013, 1309 },{ 2017, 1306 },
    { 2022, 1303 },{ 2026, 1300 },{ 2030, 1298 },{ 2035, 1295 },
    { 2040, 1292 },{ 2044, 1289 },{ 2049, 1285 },{ 2054, 1282 },
    { 2059, 1279 },{ 2064, 1276 },{ 2069, 1273 },{ 2075, 1270 },
    { 2080, 1267 },{ 2086, 1264 },{ 2091, 1261 },{ 2097, 1258 },
    { 2103, 1254 },{ 2109, 1251 },{ 2115, 1248 },{ 2121, 1245 },
    { 2128, 1241 },{ 2134, 1238 },{ 2141, 1235 },{ 2147, 1232 },
    { 2154, 1228 },{ 2161, 1225 },{ 2153, 1241 },{ 2145, 1259 },
    { 2137, 1276 },{ 2128, 1293 },{ 2120, 1309 },{ 2113, 1326 },
    { 2105, 1343 },{ 2097, 1359 },{ 2090, 1375 },{ 2083, 1392 },
    { 2075, 1408 },{ 2068, 1424 },{ 2062, 1440 },{ 2055, 1455 },
    { 2048, 1471 },{ 2042, 1486 },{ 2035, 1502 },{ 2029, 1517 },
    { 2023, 1533 },{ 2016, 1548 },{ 2011, 1563 },{ 2005, 1578 },
    { 1999, 1593 },{ 1993, 1608 },{ 1988, 1622 },{ 1982, 1637 },
    { 1977, 1652 },{ 1972, 1666 },{ 1967, 1681 },{ 1962, 1695 },
    { 1957, 1709 },{ 1952, 1723 },{ 1947, 1738 },{ 1942, 1752 },
    { 1938, 1766 },{ 1933, 1780 },{ 1929, 1793 },{ 1925, 1807 },
    { 1921, 1821 },{ 1917, 1835 },{ 1913, 1848 },{ 1909, 1862 },
    { 1905, 1875 },{ 1901, 1889 },{ 1897, 1902 },{ 1894, 1916 },
    { 1890, 1929 },{ 1887, 1942 },{ 1884, 1956 },{ 1881, 1969 },
    { 1877, 1982 },{ 1874, 1995 },{ 1871, 2008 },{ 1868, 2021 },
    { 1866, 2034 },{ 1863, 2047 },{ 1860, 2060 },{ 1858, 2073 },
    { 1855, 2086 },{ 1853, 2098 },{ 1850, 2111 },{ 1848, 2124 },
    { 1846, 2136 },{ 1844, 2149 },{ 1842, 2162 },{ 1840, 2174 },
    { 1838, 2187 },{ 1836, 2200 },{ 1834, 2212 },{ 1832, 2225 },
    { 1831, 2237 },{ 1829, 2250 },{ 1828, 2262 },{ 1826, 2274 },
    { 1825, 2287 },{ 1824, 2299 },{ 1822, 2312 },{ 1821, 2324 },
    { 1820, 2336 },{ 1819, 2349 },{ 1818, 2361 },{ 1818, 2373 },
    { 1817, 2385 },{ 1816, 2398 },{ 1815, 2410 },{ 1815, 2422 },
    { 1814, 2435 },{ 1814, 2447 },{ 1813, 2459 },{ 1813, 2471 },
    { 1813, 2483 },{ 1813, 2496 },{ 1813, 2508 },{ 1813, 2520 },
    { 1813, 2532 },{ 1813, 2545 },{ 1813, 2557 },{ 1813, 2569 },
    { 1814, 2581 },{ 1814, 2594 },{ 1814, 2606 },{ 1815, 2618 },
    { 1816, 2630 },{ 1816, 2643 },{ 1817, 2655 },{ 1818, 2667 },
    { 1819, 2679 },{ 1819, 2692 },{ 1820, 2704 },{ 1822, 2716 },
    { 1823, 2729 },{ 1824, 2741 },{ 1825, 2754 },{ 1827, 2766 },
    { 1828, 2778 },{ 1829, 2791 },{ 1831, 2803 },{ 1833, 2816 },
    { 1834, 2828 },{ 1836, 2841 },{ 1838, 2853 },{ 1840, 2866 },
    { 1842, 2879 },{ 1844, 2891 },{ 1846, 2904 },{ 1848, 2917 },
    { 1851, 2929 },{ 1853, 2942 },{ 1856, 2955 },{ 1858, 2968 },
    { 1861, 2981 },{ 1863, 2994 },{ 1866, 3006 },{ 1869, 3019 },
    { 1872, 3032 },{ 1875, 3046 },{ 1878, 3059 },{ 1881, 3072 },
    { 1884, 3085 },{ 1888, 3098 },{ 1891, 3111 },{ 1895, 3125 },
    { 1898, 3138 },{ 1902, 3152 },{ 1906, 3165 },{ 1909, 3179 },
    { 1913, 3192 },{ 1917, 3206 },{ 1921, 3220 },{ 1926, 3233 },
    { 1930, 3247 },{ 1934, 3261 },{ 1939, 3275 },{ 1943, 3289 },
    { 1948, 3303 },{ 1953, 3317 },{ 1958, 3332 },{ 1962, 3346 },
    { 1968, 3360 },{ 1973, 3375 },{ 1978, 3389 },{ 1983, 3404 },
    { 1989, 3418 },{ 1994, 3433 },{ 2000, 3448 },{ 2006, 3463 },
    { 2012, 3478 },{ 2018, 3493 },{ 2024, 3508 },{ 2030, 3524 },
    { 2036, 3539 },{ 2043, 3554 },{ 2049, 3570 },{ 2056, 3586 },
    { 2063, 3601 },{ 2070, 3617 },{ 2073, 3627 },{ 2066, 3619 },
    { 2060, 3611 },{ 2054, 3603 },{ 2047, 3595 },{ 2041, 3587 },
    { 2035, 3579 },{ 2029, 3571 },{ 2023, 3563 },{ 2017, 3556 },
    { 2012, 3548 },{ 2006, 3540 },{ 2001, 3533 },{ 1996, 3525 },
    { 1990, 3517 },{ 1985, 3510 },{ 1980, 3503 },{ 1975, 3495 },
    { 1971, 3488 },{ 1966, 3480 },{ 1961, 3473 },{ 1957, 3466 },
    { 1952, 3459 },{ 1948, 3451 },{ 1944, 3444 },{ 1940, 3437 }
};

//!
//! Vertex Table
//!
const unsigned short g_VeboxVertexTableBT709[512][2] =
{
    //{  Cv,   Lv}
    { 2005, 375 },{ 2002, 381 },{ 2000, 386 },{ 1997, 391 },
    { 1994, 396 },{ 1992, 401 },{ 1990, 407 },{ 1987, 412 },
    { 1985, 417 },{ 1983, 422 },{ 1981, 427 },{ 1979, 432 },
    { 1977, 437 },{ 1975, 442 },{ 1973, 447 },{ 1972, 453 },
    { 1970, 458 },{ 1969, 463 },{ 1967, 468 },{ 1966, 473 },
    { 1965, 478 },{ 1963, 483 },{ 1962, 488 },{ 1961, 493 },
    { 1960, 498 },{ 1959, 503 },{ 1959, 508 },{ 1958, 513 },
    { 1957, 518 },{ 1957, 523 },{ 1956, 528 },{ 1956, 533 },
    { 1955, 538 },{ 1955, 543 },{ 1955, 548 },{ 1954, 553 },
    { 1954, 558 },{ 1954, 563 },{ 1954, 568 },{ 1955, 572 },
    { 1955, 577 },{ 1955, 582 },{ 1955, 587 },{ 1956, 592 },
    { 1956, 597 },{ 1957, 602 },{ 1957, 607 },{ 1958, 612 },
    { 1959, 617 },{ 1960, 622 },{ 1961, 627 },{ 1962, 632 },
    { 1963, 637 },{ 1964, 642 },{ 1965, 647 },{ 1967, 652 },
    { 1968, 657 },{ 1970, 662 },{ 1971, 668 },{ 1973, 673 },
    { 1974, 678 },{ 1976, 683 },{ 1978, 688 },{ 1980, 693 },
    { 1982, 698 },{ 1984, 703 },{ 1986, 708 },{ 1989, 713 },
    { 1991, 719 },{ 1993, 724 },{ 1996, 729 },{ 1998, 734 },
    { 2001, 739 },{ 2004, 744 },{ 2007, 750 },{ 2010, 755 },
    { 2012, 760 },{ 2016, 766 },{ 2019, 771 },{ 2022, 776 },
    { 2025, 781 },{ 2029, 787 },{ 2032, 792 },{ 2036, 798 },
    { 2040, 803 },{ 2043, 808 },{ 2047, 814 },{ 2051, 819 },
    { 2055, 825 },{ 2059, 830 },{ 2064, 836 },{ 2068, 841 },
    { 2072, 847 },{ 2077, 853 },{ 2082, 858 },{ 2086, 864 },
    { 2091, 869 },{ 2096, 875 },{ 2101, 881 },{ 2106, 887 },
    { 2112, 892 },{ 2117, 898 },{ 2122, 904 },{ 2128, 910 },
    { 2134, 916 },{ 2139, 922 },{ 2145, 928 },{ 2151, 934 },
    { 2157, 940 },{ 2164, 946 },{ 2170, 952 },{ 2177, 958 },
    { 2183, 964 },{ 2190, 971 },{ 2197, 977 },{ 2204, 983 },
    { 2211, 989 },{ 2218, 996 },{ 2225, 1002 },{ 2233, 1009 },
    { 2241, 1015 },{ 2248, 1022 },{ 2256, 1028 },{ 2264, 1035 },
    { 2272, 1042 },{ 2281, 1049 },{ 2289, 1055 },{ 2298, 1062 },
    { 2307, 1069 },{ 2316, 1076 },{ 2325, 1083 },{ 2334, 1090 },
    { 2344, 1097 },{ 2353, 1104 },{ 2363, 1112 },{ 2373, 1119 },
    { 2383, 1126 },{ 2393, 1134 },{ 2404, 1141 },{ 2414, 1149 },
    { 2425, 1156 },{ 2436, 1164 },{ 2433, 1165 },{ 2423, 1162 },
    { 2412, 1160 },{ 2402, 1157 },{ 2393, 1154 },{ 2383, 1152 },
    { 2374, 1149 },{ 2364, 1147 },{ 2355, 1145 },{ 2346, 1142 },
    { 2337, 1140 },{ 2329, 1137 },{ 2320, 1135 },{ 2312, 1132 },
    { 2304, 1130 },{ 2296, 1128 },{ 2288, 1125 },{ 2280, 1123 },
    { 2272, 1121 },{ 2265, 1119 },{ 2257, 1116 },{ 2250, 1114 },
    { 2243, 1112 },{ 2236, 1110 },{ 2229, 1107 },{ 2223, 1105 },
    { 2216, 1103 },{ 2209, 1101 },{ 2203, 1099 },{ 2197, 1097 },
    { 2191, 1095 },{ 2185, 1092 },{ 2179, 1090 },{ 2173, 1088 },
    { 2167, 1086 },{ 2162, 1084 },{ 2156, 1082 },{ 2151, 1080 },
    { 2146, 1078 },{ 2141, 1076 },{ 2136, 1074 },{ 2131, 1072 },
    { 2126, 1070 },{ 2121, 1068 },{ 2117, 1066 },{ 2112, 1064 },
    { 2108, 1062 },{ 2103, 1060 },{ 2099, 1058 },{ 2095, 1056 },
    { 2091, 1054 },{ 2087, 1052 },{ 2083, 1050 },{ 2079, 1048 },
    { 2076, 1046 },{ 2072, 1045 },{ 2069, 1043 },{ 2065, 1041 },
    { 2062, 1039 },{ 2059, 1037 },{ 2056, 1035 },{ 2053, 1033 },
    { 2050, 1031 },{ 2047, 1030 },{ 2044, 1028 },{ 2041, 1026 },
    { 2039, 1024 },{ 2036, 1022 },{ 2034, 1020 },{ 2031, 1019 },
    { 2029, 1017 },{ 2027, 1015 },{ 2025, 1013 },{ 2023, 1011 },
    { 2021, 1009 },{ 2019, 1008 },{ 2017, 1006 },{ 2015, 1004 },
    { 2014, 1002 },{ 2012, 1001 },{ 2010, 999 },{ 2009, 997 },
    { 2008, 995 },{ 2006, 993 },{ 2005, 992 }, { 2004, 990  },
    { 2003, 988 },{ 2002, 986 },{ 2001, 984 }, { 2000, 983  },
    { 2000, 981 },{ 1999, 979 },{ 1999, 977 },{ 1998, 976  },
    { 1998, 974 },{ 1997, 972 },{ 1997, 970 },{ 1997, 969  },
    { 1997, 967 },{ 1997, 965 },{ 1997, 963 },{ 1997, 962  },
    { 1997, 960 },{ 1997, 958 },{ 1998, 956 },{ 1998, 955  },
    { 1998, 953 },{ 1999, 951 },{ 2000, 949 },{ 2000, 947  },
    { 2001, 946 },{ 2002, 944 },{ 2003, 942 },{ 2004, 940  },
    { 2005, 939 },{ 2006, 937 },{ 2007, 935 },{ 2009, 933  },
    { 2010, 931 },{ 2012, 930 },{ 2013, 928 },{ 2015, 926  },
    { 2016, 924 },{ 2018, 923 },{ 2020, 921 },{ 2022, 919  },
    { 2024, 917 },{ 2026, 915 },{ 2028, 913 },{ 2031, 912  },
    { 2033, 910 },{ 2035, 908 },{ 2038, 906 },{ 2041, 904  },
    { 2043, 902 },{ 2046, 901 },{ 2049, 899 },{ 2052, 897  },
    { 2055, 895 },{ 2058, 893 },{ 2061, 891 },{ 2064, 889  },
    { 2068, 888 },{ 2071, 886 },{ 2075, 884 },{ 2078, 882  },
    { 2082, 880 },{ 2086, 878 },{ 2090, 876 },{ 2094, 874  },
    { 2098, 872 },{ 2098, 876 },{ 2089, 895 },{ 2080, 914  },
    { 2070, 933 },{ 2061, 952 },{ 2052, 970 },{ 2044, 989  },
    { 2035, 1007 },{ 2027, 1025 },{ 2019, 1043 },{ 2010, 1061 },
    { 2002, 1078 },{ 1995, 1096 },{ 1987, 1113 },{ 1979, 1130 },
    { 1972, 1148 },{ 1964, 1164 },{ 1957, 1181 },{ 1950, 1198 },
    { 1943, 1215 },{ 1936, 1231 },{ 1930, 1247 },{ 1923, 1264 },
    { 1917, 1280 },{ 1910, 1296 },{ 1904, 1312 },{ 1898, 1328 },
    { 1892, 1343 },{ 1886, 1359 },{ 1880, 1374 },{ 1875, 1390 },
    { 1869, 1405 },{ 1863, 1420 },{ 1858, 1436 },{ 1853, 1451 },
    { 1848, 1466 },{ 1843, 1481 },{ 1838, 1495 },{ 1833, 1510 },
    { 1828, 1525 },{ 1823, 1539 },{ 1819, 1554 },{ 1814, 1568 },
    { 1810, 1583 },{ 1805, 1597 },{ 1801, 1611 },{ 1797, 1625 },
    { 1793, 1640 },{ 1789, 1654 },{ 1785, 1668 },{ 1781, 1681 },
    { 1777, 1695 },{ 1774, 1709 },{ 1770, 1723 },{ 1767, 1737 },
    { 1763, 1750 },{ 1760, 1764 },{ 1756, 1777 },{ 1753, 1791 },
    { 1750, 1804 },{ 1747, 1818 },{ 1744, 1831 },{ 1741, 1844 },
    { 1739, 1857 },{ 1736, 1871 },{ 1733, 1884 },{ 1731, 1897 },
    { 1728, 1910 },{ 1726, 1923 },{ 1723, 1936 },{ 1721, 1949 },
    { 1719, 1962 },{ 1717, 1975 },{ 1715, 1988 },{ 1713, 2001 },
    { 1711, 2014 },{ 1709, 2026 },{ 1707, 2039 },{ 1705, 2052 },
    { 1703, 2065 },{ 1702, 2077 },{ 1700, 2090 },{ 1699, 2103 },
    { 1697, 2115 },{ 1696, 2128 },{ 1695, 2140 },{ 1694, 2153 },
    { 1693, 2165 },{ 1691, 2178 },{ 1690, 2191 },{ 1689, 2203 },
    { 1689, 2216 },{ 1688, 2228 },{ 1687, 2240 },{ 1686, 2253 },
    { 1686, 2265 },{ 1685, 2278 },{ 1685, 2290 },{ 1684, 2303 },
    { 1684, 2315 },{ 1684, 2327 },{ 1683, 2340 },{ 1683, 2352 },
    { 1683, 2365 },{ 1683, 2377 },{ 1683, 2389 },{ 1683, 2402 },
    { 1683, 2414 },{ 1683, 2427 },{ 1684, 2439 },{ 1684, 2451 },
    { 1684, 2464 },{ 1685, 2476 },{ 1685, 2489 },{ 1686, 2501 },
    { 1687, 2514 },{ 1687, 2526 },{ 1688, 2539 },{ 1689, 2551 },
    { 1690, 2564 },{ 1691, 2576 },{ 1692, 2589 },{ 1693, 2601 },
    { 1694, 2614 },{ 1695, 2626 },{ 1697, 2639 },{ 1698, 2651 },
    { 1699, 2664 },{ 1701, 2677 },{ 1702, 2689 },{ 1704, 2702 },
    { 1706, 2715 },{ 1708, 2728 },{ 1709, 2740 },{ 1711, 2753 },
    { 1713, 2766 },{ 1715, 2779 },{ 1717, 2792 },{ 1720, 2805 },
    { 1722, 2818 },{ 1724, 2831 },{ 1726, 2844 },{ 1729, 2857 },
    { 1731, 2870 },{ 1734, 2883 },{ 1737, 2896 },{ 1740, 2909 },
    { 1742, 2923 },{ 1745, 2936 },{ 1748, 2949 },{ 1751, 2963 },
    { 1754, 2976 },{ 1758, 2990 },{ 1761, 3003 },{ 1764, 3017 },
    { 1768, 3030 },{ 1771, 3044 },{ 1775, 3058 },{ 1779, 3072 },
    { 1782, 3086 },{ 1786, 3100 },{ 1790, 3114 },{ 1794, 3128 },
    { 1798, 3142 },{ 1802, 3156 },{ 1807, 3170 },{ 1811, 3184 },
    { 1816, 3199 },{ 1820, 3213 },{ 1825, 3228 },{ 1830, 3243 },
    { 1834, 3257 },{ 1839, 3272 },{ 1844, 3287 },{ 1849, 3302 },
    { 1855, 3317 },{ 1860, 3332 },{ 1865, 3347 },{ 1871, 3362 },
    { 1877, 3378 },{ 1882, 3393 },{ 1888, 3409 },{ 1894, 3424 },
    { 1900, 3440 },{ 1906, 3456 },{ 1913, 3472 },{ 1919, 3488 },
    { 1925, 3504 },{ 1932, 3520 },{ 1939, 3537 },{ 1946, 3553 },
    { 1953, 3570 },{ 1960, 3587 },{ 1967, 3604 },{ 1974, 3621 },
    { 1982, 3638 },{ 1990, 3655 },{ 1997, 3672 },{ 2005, 3690 },
    { 2013, 3708 },{ 2021, 3725 },{ 2030, 3743 },{ 2038, 3761 },
    { 2047, 3780 },{ 2056, 3798 },{ 2053, 3795 },{ 2049, 3790 },
    { 2045, 3784 },{ 2041, 3779 },{ 2037, 3774 },{ 2034, 3768 },
    { 2030, 3763 },{ 2027, 3758 },{ 2023, 3752 },{ 2020, 3747 },
    { 2017, 3742 },{ 2014, 3736 },{ 2011, 3731 },{ 2008, 3726 }
};

const MHW_VEBOX_SETTINGS g_Vebox_Settings =
{
        MHW_MAX_VEBOX_STATES,  //!< uiNumInstances
        MHW_SYNC_SIZE,         //!< uiSyncSize
        MHW_PAGE_SIZE,         //!< uiDndiStateSize
        MHW_PAGE_SIZE,         //!< uiIecpStateSize
        MHW_PAGE_SIZE * 2,     //!< uiGamutStateSize
        MHW_PAGE_SIZE,         //!< uiVertexTableSize
        MHW_PAGE_SIZE,         //!< uiCapturePipeStateSize
        MHW_PAGE_SIZE * 2,     //!< uiGammaCorrectionStateSize
        MHW_PAGE_SIZE * 18     //!< uiHdrStateSize
};

#if (_DEBUG || _RELEASE_INTERNAL)
#define MHW_VEBOX_IS_VEBOX_SPECIFIED_IN_CONFIG(keyval, VDId, shift, mask, bUseVD) \
do\
{\
    int32_t TmpVal = keyval;\
    while (TmpVal != 0) \
    {\
        if (((TmpVal) & (mask)) == (VDId))\
        {\
            bUseVD = true;\
            break;\
        }\
        TmpVal >>= (shift);\
    };\
}while(0)
#endif

enum MHW_VEBOX_INDEX
{
    MHW_VEBOX_INDEX_0 = 0,
    MHW_VEBOX_INDEX_1 = 1,
    MHW_VEBOX_INDEX_2 = 2,
    MHW_VEBOX_INDEX_3 = 3
};

struct _MHW_PAR_T(VEBOX_STATE)
{
    MHW_VEBOX_MODE                   VeboxMode                   = {};
    MHW_VEBOX_CHROMA_SAMPLING        ChromaSampling              = {};
    MHW_VEBOX_3D_LUT                 LUT3D                       = {};
    bool                             bUseVeboxHeapKernelResource = false;
    PMOS_RESOURCE                    pLaceLookUpTables           = nullptr;
    PMOS_RESOURCE                    pVeboxParamSurf             = nullptr;
    PMOS_RESOURCE                    pVebox3DLookUpTables        = nullptr;
    PMOS_RESOURCE                    pVebox1DLookUpTables        = nullptr;
    PMOS_RESOURCE                    DummyIecpResource           = nullptr;
    MHW_MEMORY_OBJECT_CONTROL_PARAMS LaceLookUpTablesSurfCtrl    = {};
    MHW_MEMORY_OBJECT_CONTROL_PARAMS Vebox3DLookUpTablesSurfCtrl = {};
    bool                             bNoUseVeboxHeap             = false;
    bool                             bCmBuffer                   = false;
};

struct _MHW_PAR_T(VEBOX_SURFACE_STATE)
{
    uint32_t SurfaceIdentification                  = 0;
    uint32_t Width                                  = 0;
    uint32_t Height                                 = 0;
    uint32_t HalfPitchForChroma                     = 0;
    uint32_t InterleaveChroma                       = 0;
    uint32_t SurfaceFormat                          = 0;
    uint32_t BayerInputAlignment                    = 0;
    uint32_t BayerPatternOffset                     = 0;
    uint32_t BayerPatternFormat                     = 0;
    uint32_t SurfacePitch                           = 0;
    uint32_t TileMode                               = 0;
    uint32_t XOffsetForU                            = 0;
    uint32_t YOffsetForU                            = 0;
    uint32_t XOffsetForV                            = 0;
    uint32_t YOffsetForV                            = 0;
    uint32_t YOffsetForFrame                        = 0;
    uint32_t XOffsetForFrame                        = 0;
    uint32_t DerivedSurfacePitch                    = 0;
    uint32_t SurfacePitchForSkinScoreOutputSurfaces = 0;
    uint32_t CompressionFormat                      = 0;
};

struct _MHW_PAR_T(VEBOX_TILING_CONVERT)
{
    uint8_t                   InterfaceDescriptorOffset     = 0;
    bool                      GpGpuEnable                   = false;
    uint32_t                  ThreadWidth                   = 0;
    uint32_t                  ThreadHeight                  = 0;
    uint32_t                  ThreadDepth                   = 0;
    uint32_t                  GroupWidth                    = 0;
    uint32_t                  GroupHeight                   = 0;
    uint32_t                  GroupDepth                    = 0;
    uint32_t                  GroupStartingX                = 0;
    uint32_t                  GroupStartingY                = 0;
    uint32_t                  GroupStartingZ                = 0;
    uint32_t                  SLMSize                       = 0;
    uint32_t                  IndirectDataLength            = 0;
    uint32_t                  IndirectDataStartAddress      = 0;
    uint32_t                  BindingTableID                = 0;
    uint32_t                  dwMediaIdOffset               = 0;       //! Offset of the first Media Interface Descriptor (in DSH)
    uint32_t                  iMediaId                      = 0;       //! Media Interface Descriptor #
    uint32_t                  dwKernelOffset                = 0;       //! Kernel offset (in ISH)
    uint32_t                  dwSamplerOffset               = 0;       //! Sampler offset (in DSH)
    uint32_t                  dwSamplerCount                = 0;       //! Sample count
    uint32_t                  dwBindingTableOffset          = 0;       //! Binding table offset (in DSH)
    uint32_t                  iCurbeOffset                  = 0;       //! Curbe offset (in DSH)
    uint32_t                  iCurbeLength                  = 0;       //! Curbe lenght
    bool                      bBarrierEnable                = false;   //! Enable Barrier
    bool                      bGlobalBarrierEnable          = false;   //! Enable Global Barrier (SKL+)
    uint32_t                  dwNumberofThreadsInGPGPUGroup = 0;       //! Number of threads per group
    uint32_t                  dwSharedLocalMemorySize       = 0;       //! Size of SharedLocalMemory (SLM)
    int32_t                   iCrsThdConDataRdLn            = 0;
};

struct _MHW_PAR_T(VEB_DI_IECP)
{
    uint32_t dwEndingX                                               = 0;
    uint32_t dwStartingX                                             = 0;
    uint32_t dwEndingY                                               = 0;
    uint32_t dwStartingY                                             = 0;
    uint32_t dwCurrInputSurfOffset                                   = 0;
    uint32_t dwPrevInputSurfOffset                                   = 0;
    uint32_t dwCurrOutputSurfOffset                                  = 0;
    uint32_t dwStreamID                                              = 0;        // Stream ID for input surface
    uint32_t dwStreamIDOutput                                        = 0;  // Stream ID for output surface

    PMOS_RESOURCE pOsResCurrInput                                    = nullptr;
    PMOS_RESOURCE pOsResPrevInput                                    = nullptr;
    PMOS_RESOURCE pOsResStmmInput                                    = nullptr;
    PMOS_RESOURCE pOsResStmmOutput                                   = nullptr;
    PMOS_RESOURCE pOsResDenoisedCurrOutput                           = nullptr;
    PMOS_RESOURCE pOsResCurrOutput                                   = nullptr;
    PMOS_RESOURCE pOsResPrevOutput                                   = nullptr;
    PMOS_RESOURCE pOsResStatisticsOutput                             = nullptr;
    PMOS_RESOURCE pOsResAlphaOrVignette                              = nullptr;
    PMOS_RESOURCE pOsResLaceOrAceOrRgbHistogram                      = nullptr;
    PMOS_RESOURCE pOsResSkinScoreSurface                             = nullptr;

    MHW_MEMORY_OBJECT_CONTROL_PARAMS CurrInputSurfCtrl               = {};  //can be removed after VPHAL moving to new cmd definition
    MHW_MEMORY_OBJECT_CONTROL_PARAMS PrevInputSurfCtrl               = {};
    MHW_MEMORY_OBJECT_CONTROL_PARAMS StmmInputSurfCtrl               = {};
    MHW_MEMORY_OBJECT_CONTROL_PARAMS StmmOutputSurfCtrl              = {};
    MHW_MEMORY_OBJECT_CONTROL_PARAMS DenoisedCurrOutputSurfCtrl      = {};
    MHW_MEMORY_OBJECT_CONTROL_PARAMS CurrOutputSurfCtrl              = {};
    MHW_MEMORY_OBJECT_CONTROL_PARAMS PrevOutputSurfCtrl              = {};
    MHW_MEMORY_OBJECT_CONTROL_PARAMS StatisticsOutputSurfCtrl        = {};
    MHW_MEMORY_OBJECT_CONTROL_PARAMS AlphaOrVignetteSurfCtrl         = {};
    MHW_MEMORY_OBJECT_CONTROL_PARAMS LaceOrAceOrRgbHistogramSurfCtrl = {};
    MHW_MEMORY_OBJECT_CONTROL_PARAMS SkinScoreSurfaceSurfCtrl        = {};

    MOS_MEMCOMP_STATE CurInputSurfMMCState                           = MOS_MEMCOMP_DISABLED;
};
}  // namespace vebox
}  // namespace mhw

#endif  // __MHW_VEBOX_CMDPAR_H__
