//
//	SubjectAltNameTest.cs - NUnit Test Cases for 
//	Mono.Security.X509.Extensions.SubjectAltName
//
// Authors:
//	David Wolinsky <davidiw@ufl.edu>
//
// Copyright  (C) 2008 
//

//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

using System;
using System.Collections;
using System.Security.Cryptography;
using System.IO;

using Mono.Security;
using Mono.Security.X509;
using Mono.Security.X509.Extensions;

using NUnit.Framework;

namespace MonoTests.Mono.Security.X509.Extensions {

	[TestFixture]
	public class SubjectAltNameTest {

		[Test]
		public void SubjectAltNameGenerator ()
		{
			RSACryptoServiceProvider rsa = new RSACryptoServiceProvider ();
			X509CertificateBuilder x509 = new X509CertificateBuilder ();
			x509.IssuerName = "C=ZA, ST=Western Cape, L=Cape Town, O=Thawte Consulting cc, OU=Certification Services Division, CN=Thawte Server";
			x509.NotAfter = DateTime.MaxValue;
			x509.NotBefore = DateTime.MinValue;
			x509.SubjectName = "C=US, ST=Maryland, L=Pasadena, O=Brent Baccala, OU=FreeSoft, CN=www.freesoft.org";
			x509.SerialNumber = new byte[] {12, 34, 56, 78, 90};
			x509.Version = 3;
			x509.SubjectPublicKey = rsa;

			string[] dns = new string[2];
			dns[0] = "one";
			dns[1] = "two";
			string[] uris = new string[3];
			uris[0] = "one:two://three";
			uris[1] = "Here:I:AM://12345";
			uris[2] = "last:one";
			SubjectAltNameExtension sane = new SubjectAltNameExtension (null, dns, null, uris);
			x509.Extensions.Add (sane);
			byte[] data = x509.Sign (rsa);

			X509Certificate x509_loaded = new X509Certificate (data);
			SubjectAltNameExtension	sane_test = new SubjectAltNameExtension (x509_loaded.Extensions[0]);
			Assert.AreEqual (sane_test.RFC822.Length, 0, "RFC822 count");
			Assert.AreEqual (sane_test.DNSNames.Length, 2, "DNSName count");
			Assert.AreEqual (sane_test.IPAddresses.Length, 0, "IPAddresses count");
			Assert.AreEqual (sane_test.UniformResourceIdentifiers.Length, 3, "URI Count");
			Assert.AreEqual (sane_test.DNSNames[1], "two", "DNSName test");
			Assert.AreEqual (sane_test.UniformResourceIdentifiers[2], "last:one", "URI Test");
		}

		[Test]
		public void ThirdPartyCertificateParse  ()
		{
			byte[] certificate_with_ip_subjectaltname = new byte[] {0x30, 0x82, 0x02, 0x78, 0x30, 0x82, 0x01, 0xE1, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x09, 0x00, 0xEE, 0xF3, 0xC0, 0x32, 0x13, 0x12, 0x66, 0x06, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x78, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0A, 0x53, 0x6F, 0x6D, 0x65, 0x2D, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x18, 0x49, 0x6E, 0x74, 0x65, 0x72, 0x6E, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4C, 0x74, 0x64, 0x31, 0x0D, 0x30, 0x0B, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x13, 0x04, 0x6E, 0x6F, 0x6E, 0x65, 0x31, 0x0D, 0x30, 0x0B, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x04, 0x6E, 0x6F, 0x6E, 0x65, 0x31, 0x13, 0x30, 0x11, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x01, 0x16, 0x04, 0x6E, 0x6F, 0x6E, 0x65, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x38, 0x30, 0x36, 0x31, 0x33, 0x30, 0x34, 0x35, 0x39, 0x34, 0x36, 0x5A, 0x17, 0x0D, 0x30, 0x39, 0x30, 0x36, 0x31, 0x33, 0x30, 0x34, 0x35, 0x39, 0x34, 0x36, 0x5A, 0x30, 0x6F, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0A, 0x53, 0x6F, 0x6D, 0x65, 0x2D, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x18, 0x49, 0x6E, 0x74, 0x65, 0x72, 0x6E, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4C, 0x74, 0x64, 0x31, 0x0A, 0x30, 0x08, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x13, 0x01, 0x61, 0x31, 0x0A, 0x30, 0x08, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x01, 0x61, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x01, 0x16, 0x01, 0x61, 0x30, 0x81, 0x9F, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8D, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xA6, 0xD2, 0x03, 0x5D, 0x91, 0x3D, 0xF3, 0x04, 0x4F, 0xB9, 0xF0, 0xE6, 0x40, 0xD0, 0x4E, 0xDF, 0xF7, 0xCE, 0x35, 0x87, 0xC1, 0x95, 0xEA, 0xFD, 0xDF, 0x44, 0x46, 0x20, 0xE4, 0xAF, 0x69, 0xC8, 0x1C, 0xF1, 0x06, 0x6C, 0x46, 0x20, 0x4D, 0xAA, 0xCC, 0x86, 0x40, 0xBB, 0x79, 0xF8, 0x71, 0x22, 0x87, 0x65, 0xBD, 0x20, 0x1D, 0xAD, 0xC6, 0xB0, 0x7C, 0x17, 0xE6, 0x57, 0xE4, 0x3C, 0x55, 0xD7, 0x7C, 0x8A, 0x98, 0xA2, 0xCD, 0x22, 0x85, 0x0E, 0xA2, 0x90, 0x18, 0x44, 0xA9, 0x7F, 0xA6, 0xD8, 0xDB, 0x6D, 0x09, 0x6D, 0x48, 0x6D, 0xD0, 0xDF, 0x94, 0xBF, 0x3B, 0xE5, 0xDA, 0x5F, 0xA2, 0x6F, 0x3C, 0xE5, 0xCE, 0xF3, 0x53, 0x8B, 0x16, 0x39, 0xDD, 0x3B, 0xC7, 0x0E, 0xA5, 0x75, 0xAA, 0x5A, 0x51, 0xD5, 0x7F, 0x31, 0x44, 0xC8, 0x6A, 0x9B, 0x60, 0xEC, 0xA2, 0xB6, 0x42, 0xCA, 0xA3, 0x43, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x13, 0x30, 0x11, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x1D, 0x11, 0x04, 0x08, 0x30, 0x06, 0x87, 0x04, 0xC0, 0xA8, 0x6F, 0x6F, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x87, 0x23, 0x47, 0x07, 0x99, 0x69, 0x90, 0x8D, 0x65, 0xF9, 0xE4, 0xF3, 0x03, 0xBB, 0x08, 0x67, 0x29, 0x38, 0x0E, 0xF4, 0x2E, 0x23, 0xFF, 0xC2, 0x05, 0x1C, 0x7B, 0xDD, 0xBD, 0xA6, 0x26, 0x46, 0x99, 0x26, 0xC4, 0x8C, 0xD5, 0xFC, 0x4A, 0x39, 0xE6, 0xF5, 0xD7, 0x9F, 0xE5, 0x80, 0x60, 0x01, 0x30, 0x32, 0xC1, 0x86, 0x5C, 0x2E, 0x7F, 0x01, 0xB2, 0xAE, 0x4D, 0x15, 0xBB, 0x9C, 0xB8, 0xC4, 0xF6, 0x18, 0x48, 0x5F, 0xEF, 0x35, 0x78, 0xE5, 0x7F, 0x35, 0x10, 0xA1, 0xDD, 0x8E, 0x69, 0xCA, 0x52, 0x99, 0xBC, 0x89, 0x42, 0x2F, 0xC3, 0xEF, 0xB6, 0xD1, 0x37, 0xE2, 0xF9, 0x68, 0xC1, 0x3C, 0x10, 0x8C, 0xDE, 0x7A, 0xC9, 0x31, 0x3B, 0x4E, 0xAC, 0x44, 0xA1, 0x9F, 0x7C, 0xA7, 0x41, 0x59, 0xE5, 0x77, 0x12, 0xCB, 0xBE, 0xBB, 0x0F, 0x50, 0x5A, 0x14, 0x3B, 0xA7, 0x86, 0x15, 0x5C, 0x61, 0x0A};

			X509Certificate cert = new X509Certificate (certificate_with_ip_subjectaltname);
			SubjectAltNameExtension	sane_test = new SubjectAltNameExtension (cert.Extensions[0]);
			Assert.AreEqual (sane_test.RFC822.Length, 0, "RFC822 count");
			Assert.AreEqual (sane_test.DNSNames.Length, 0, "DNSName count");
			Assert.AreEqual (sane_test.IPAddresses.Length, 1, "IPAddresses count");
			Assert.AreEqual (sane_test.UniformResourceIdentifiers.Length, 0, "URI Count");
			Assert.AreEqual (sane_test.IPAddresses[0], "192.168.111.111", "IPAddress Test");
		}

		static byte [] multiple_entries_cert = new byte [] {
			0x30, 0x82, 0x06, 0x0B, 0x30, 0x82, 0x04, 0xF3, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x01, 0x47, 0x31, 0x5D, 0x40, 0xFA, 0x32, 0x70, 0xE2, 
			0x0F, 0x16, 0x2D, 0xCC, 0xA6, 0x46, 0x0E, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x81, 
			0xB5, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 
			0x0E, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6E, 0x2C, 0x20, 0x49, 0x6E, 0x63, 0x2E, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x04, 0x0B, 
			0x13, 0x16, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6E, 0x20, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x4E, 0x65, 0x74, 0x77, 0x6F, 0x72, 0x6B, 
			0x31, 0x3B, 0x30, 0x39, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x13, 0x32, 0x54, 0x65, 0x72, 0x6D, 0x73, 0x20, 0x6F, 0x66, 0x20, 0x75, 0x73, 0x65, 0x20, 
			0x61, 0x74, 0x20, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3A, 0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E, 0x76, 0x65, 0x72, 0x69, 0x73, 0x69, 0x67, 0x6E, 0x2E, 
			0x63, 0x6F, 0x6D, 0x2F, 0x72, 0x70, 0x61, 0x20, 0x28, 0x63, 0x29, 0x30, 0x39, 0x31, 0x2F, 0x30, 0x2D, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x26, 
			0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6E, 0x20, 0x43, 0x6C, 0x61, 0x73, 0x73, 0x20, 0x33, 0x20, 0x53, 0x65, 0x63, 0x75, 0x72, 0x65, 0x20, 
			0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x20, 0x43, 0x41, 0x20, 0x2D, 0x20, 0x47, 0x32, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x39, 0x31, 0x32, 0x33, 0x30, 
			0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5A, 0x17, 0x0D, 0x31, 0x31, 0x31, 0x32, 0x33, 0x30, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5A, 0x30, 0x81, 
			0xDE, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x44, 0x45, 0x31, 0x1B, 0x30, 0x19, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 
			0x12, 0x42, 0x61, 0x64, 0x65, 0x6E, 0x2D, 0x57, 0x75, 0x65, 0x72, 0x74, 0x74, 0x65, 0x6D, 0x62, 0x65, 0x72, 0x67, 0x31, 0x11, 0x30, 0x0F, 0x06, 
			0x03, 0x55, 0x04, 0x07, 0x14, 0x08, 0x4D, 0x61, 0x6E, 0x6E, 0x68, 0x65, 0x69, 0x6D, 0x31, 0x36, 0x30, 0x34, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x14, 
			0x2D, 0x41, 0x44, 0x47, 0x20, 0x41, 0x70, 0x6F, 0x74, 0x68, 0x65, 0x6B, 0x65, 0x6E, 0x20, 0x44, 0x69, 0x65, 0x6E, 0x73, 0x74, 0x6C, 0x65, 0x69, 
			0x73, 0x74, 0x75, 0x6E, 0x67, 0x73, 0x67, 0x65, 0x73, 0x65, 0x6C, 0x6C, 0x73, 0x63, 0x68, 0x61, 0x66, 0x74, 0x20, 0x6D, 0x62, 0x48, 0x31, 0x0C, 
			0x30, 0x0A, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x14, 0x03, 0x45, 0x44, 0x56, 0x31, 0x33, 0x30, 0x31, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x14, 0x2A, 0x54, 
			0x65, 0x72, 0x6D, 0x73, 0x20, 0x6F, 0x66, 0x20, 0x75, 0x73, 0x65, 0x20, 0x61, 0x74, 0x20, 0x77, 0x77, 0x77, 0x2E, 0x76, 0x65, 0x72, 0x69, 0x73, 
			0x69, 0x67, 0x6E, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x72, 0x70, 0x61, 0x20, 0x28, 0x63, 0x29, 0x30, 0x35, 0x31, 0x24, 0x30, 0x22, 0x06, 0x03, 0x55, 
			0x04, 0x03, 0x14, 0x1B, 0x77, 0x77, 0x77, 0x2E, 0x61, 0x33, 0x30, 0x30, 0x30, 0x2D, 0x66, 0x69, 0x6C, 0x69, 0x61, 0x6C, 0x61, 0x70, 0x6F, 0x74, 
			0x68, 0x65, 0x6B, 0x65, 0x2E, 0x64, 0x65, 0x30, 0x81, 0x9F, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 
			0x00, 0x03, 0x81, 0x8D, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xAE, 0xF3, 0xD4, 0xFE, 0x33, 0x44, 0x0F, 0x68, 0xBB, 0x4C, 0x6D, 0x5F, 
			0xAF, 0xE7, 0xBC, 0x20, 0x05, 0xE3, 0xCB, 0xD6, 0x9D, 0x7A, 0x68, 0x8B, 0x3D, 0x63, 0x9B, 0xED, 0x31, 0x4E, 0x7E, 0xE7, 0x0A, 0x80, 0xF5, 0xDD, 
			0x29, 0xDE, 0x02, 0x71, 0x82, 0xE7, 0x69, 0x02, 0xBC, 0xAC, 0xA6, 0x09, 0xFF, 0x3A, 0x41, 0xCB, 0x3C, 0x9C, 0x68, 0xB0, 0x80, 0xA6, 0xAF, 0x4C, 
			0xED, 0xE7, 0xA7, 0xF1, 0x74, 0x63, 0x3A, 0xF7, 0xC8, 0x66, 0x84, 0x6D, 0xE5, 0x5B, 0x9B, 0x45, 0xBC, 0x78, 0x84, 0x3E, 0x1A, 0x82, 0x84, 0x8C, 
			0xB2, 0x06, 0x64, 0x62, 0xB7, 0xB0, 0xE6, 0x9E, 0x5F, 0xAF, 0x58, 0x93, 0xAC, 0xCE, 0x04, 0x71, 0x2A, 0x11, 0xD9, 0xD5, 0x4E, 0xD0, 0x7F, 0x9E, 
			0xF7, 0x92, 0x5F, 0x83, 0xAE, 0x32, 0x1F, 0x2E, 0x1B, 0xDD, 0x34, 0xE8, 0x28, 0xC8, 0x4D, 0x97, 0x20, 0xBA, 0xE8, 0x6B, 0x02, 0x03, 0x01, 0x00, 
			0x01, 0xA3, 0x82, 0x02, 0x6E, 0x30, 0x82, 0x02, 0x6A, 0x30, 0x81, 0x9A, 0x06, 0x03, 0x55, 0x1D, 0x11, 0x04, 0x81, 0x92, 0x30, 0x81, 0x8F, 0x82, 
			0x21, 0x77, 0x77, 0x77, 0x2E, 0x61, 0x72, 0x7A, 0x6E, 0x65, 0x69, 0x6D, 0x69, 0x74, 0x74, 0x65, 0x6C, 0x2D, 0x62, 0x65, 0x73, 0x74, 0x65, 0x6C, 
			0x6C, 0x63, 0x65, 0x6E, 0x74, 0x65, 0x72, 0x2E, 0x64, 0x65, 0x82, 0x26, 0x77, 0x77, 0x77, 0x2E, 0x78, 0x6E, 0x65, 0x74, 0x2D, 0x61, 0x72, 0x7A, 
			0x6E, 0x65, 0x69, 0x6D, 0x69, 0x74, 0x74, 0x65, 0x6C, 0x2D, 0x62, 0x65, 0x73, 0x74, 0x65, 0x6C, 0x6C, 0x63, 0x65, 0x6E, 0x74, 0x65, 0x72, 0x2E, 
			0x64, 0x65, 0x82, 0x25, 0x77, 0x77, 0x77, 0x2E, 0x61, 0x64, 0x67, 0x2D, 0x61, 0x72, 0x7A, 0x6E, 0x65, 0x69, 0x6D, 0x69, 0x74, 0x74, 0x65, 0x6C, 
			0x2D, 0x62, 0x65, 0x73, 0x74, 0x65, 0x6C, 0x6C, 0x63, 0x65, 0x6E, 0x74, 0x65, 0x72, 0x2E, 0x64, 0x65, 0x82, 0x1B, 0x77, 0x77, 0x77, 0x2E, 0x61, 
			0x33, 0x30, 0x30, 0x30, 0x2D, 0x66, 0x69, 0x6C, 0x69, 0x61, 0x6C, 0x61, 0x70, 0x6F, 0x74, 0x68, 0x65, 0x6B, 0x65, 0x2E, 0x64, 0x65, 0x30, 0x09, 
			0x06, 0x03, 0x55, 0x1D, 0x13, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0B, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x04, 0x04, 0x03, 0x02, 0x05, 0xA0, 0x30, 0x45, 
			0x06, 0x03, 0x55, 0x1D, 0x1F, 0x04, 0x3E, 0x30, 0x3C, 0x30, 0x3A, 0xA0, 0x38, 0xA0, 0x36, 0x86, 0x34, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F, 0x2F, 
			0x53, 0x56, 0x52, 0x53, 0x65, 0x63, 0x75, 0x72, 0x65, 0x2D, 0x47, 0x32, 0x2D, 0x63, 0x72, 0x6C, 0x2E, 0x76, 0x65, 0x72, 0x69, 0x73, 0x69, 0x67, 
			0x6E, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x53, 0x56, 0x52, 0x53, 0x65, 0x63, 0x75, 0x72, 0x65, 0x47, 0x32, 0x2E, 0x63, 0x72, 0x6C, 0x30, 0x44, 0x06, 
			0x03, 0x55, 0x1D, 0x20, 0x04, 0x3D, 0x30, 0x3B, 0x30, 0x39, 0x06, 0x0B, 0x60, 0x86, 0x48, 0x01, 0x86, 0xF8, 0x45, 0x01, 0x07, 0x17, 0x03, 0x30, 
			0x2A, 0x30, 0x28, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x01, 0x16, 0x1C, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3A, 0x2F, 0x2F, 0x77, 
			0x77, 0x77, 0x2E, 0x76, 0x65, 0x72, 0x69, 0x73, 0x69, 0x67, 0x6E, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x72, 0x70, 0x61, 0x30, 0x1D, 0x06, 0x03, 0x55, 
			0x1D, 0x25, 0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 
			0x03, 0x02, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xA5, 0xEF, 0x0B, 0x11, 0xCE, 0xC0, 0x41, 0x03, 0xA3, 
			0x4A, 0x65, 0x90, 0x48, 0xB2, 0x1C, 0xE0, 0x57, 0x2D, 0x7D, 0x47, 0x30, 0x76, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 
			0x6A, 0x30, 0x68, 0x30, 0x24, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x86, 0x18, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F, 0x2F, 
			0x6F, 0x63, 0x73, 0x70, 0x2E, 0x76, 0x65, 0x72, 0x69, 0x73, 0x69, 0x67, 0x6E, 0x2E, 0x63, 0x6F, 0x6D, 0x30, 0x40, 0x06, 0x08, 0x2B, 0x06, 0x01, 
			0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x34, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F, 0x2F, 0x53, 0x56, 0x52, 0x53, 0x65, 0x63, 0x75, 0x72, 0x65, 0x2D, 
			0x47, 0x32, 0x2D, 0x61, 0x69, 0x61, 0x2E, 0x76, 0x65, 0x72, 0x69, 0x73, 0x69, 0x67, 0x6E, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x53, 0x56, 0x52, 0x53, 
			0x65, 0x63, 0x75, 0x72, 0x65, 0x47, 0x32, 0x2E, 0x63, 0x65, 0x72, 0x30, 0x6E, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x0C, 0x04, 
			0x62, 0x30, 0x60, 0xA1, 0x5E, 0xA0, 0x5C, 0x30, 0x5A, 0x30, 0x58, 0x30, 0x56, 0x16, 0x09, 0x69, 0x6D, 0x61, 0x67, 0x65, 0x2F, 0x67, 0x69, 0x66, 
			0x30, 0x21, 0x30, 0x1F, 0x30, 0x07, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x04, 0x14, 0x4B, 0x6B, 0xB9, 0x28, 0x96, 0x06, 0x0C, 0xBB, 0xD0, 
			0x52, 0x38, 0x9B, 0x29, 0xAC, 0x4B, 0x07, 0x8B, 0x21, 0x05, 0x18, 0x30, 0x26, 0x16, 0x24, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F, 0x2F, 0x6C, 0x6F, 
			0x67, 0x6F, 0x2E, 0x76, 0x65, 0x72, 0x69, 0x73, 0x69, 0x67, 0x6E, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x76, 0x73, 0x6C, 0x6F, 0x67, 0x6F, 0x31, 0x2E, 
			0x67, 0x69, 0x66, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x2F, 
			0x1E, 0x71, 0x21, 0xD9, 0xCE, 0xBB, 0x16, 0xEA, 0xA2, 0xD9, 0x5B, 0xA6, 0x0E, 0x1B, 0x9A, 0x32, 0xC2, 0xFA, 0x38, 0xF5, 0x3F, 0xC7, 0xEE, 0xFC, 
			0xD7, 0xBE, 0x92, 0x14, 0xAA, 0xBE, 0x94, 0x63, 0xE6, 0xCE, 0x46, 0x9D, 0x28, 0x11, 0x6E, 0x8D, 0x64, 0xDB, 0x4B, 0x7D, 0x03, 0xD2, 0xE7, 0x6B, 
			0x23, 0xA4, 0xD2, 0x67, 0xFE, 0x1B, 0xFC, 0x7E, 0x2C, 0xAA, 0xB6, 0x1F, 0x36, 0xD5, 0x65, 0x47, 0xCE, 0xC1, 0x4C, 0xD2, 0x83, 0xBA, 0x1A, 0x8E, 
			0x39, 0x74, 0x0E, 0xCC, 0x29, 0x1A, 0x91, 0x09, 0x32, 0x5F, 0x01, 0x14, 0x7A, 0xC8, 0x9C, 0x14, 0x36, 0x78, 0x5F, 0xB0, 0x5E, 0x07, 0xFD, 0x34, 
			0xD1, 0x93, 0xD6, 0x79, 0xA0, 0xC1, 0xA0, 0xF2, 0xD7, 0xF0, 0x5E, 0xF7, 0x57, 0xB0, 0xE7, 0x75, 0x60, 0xC5, 0xE6, 0x4F, 0xA9, 0x78, 0x72, 0x47, 
			0x72, 0x8F, 0x8A, 0x6B, 0xCD, 0x3D, 0x0D, 0x56, 0xB1, 0x85, 0x5B, 0x2E, 0x1F, 0xF1, 0x00, 0x86, 0x83, 0x72, 0xD4, 0x80, 0x71, 0x1A, 0xBB, 0x08, 
			0xCC, 0x6B, 0x67, 0xDA, 0x9B, 0xCB, 0x99, 0xDD, 0xAF, 0x27, 0x0F, 0xCE, 0x30, 0x02, 0x69, 0x74, 0x64, 0x3C, 0x68, 0xB5, 0xFF, 0x60, 0x5F, 0x1D, 
			0x6A, 0xBB, 0xF5, 0x0D, 0xC5, 0x96, 0x85, 0xDA, 0x36, 0x4E, 0xF3, 0x84, 0x97, 0x62, 0x9E, 0x18, 0x6E, 0xA9, 0x8D, 0xFD, 0x54, 0x9E, 0x39, 0xFA, 
			0x4A, 0x38, 0xF6, 0xF1, 0x1C, 0x63, 0x5D, 0x09, 0x09, 0xC4, 0x18, 0xA9, 0x8C, 0x96, 0x75, 0xEB, 0x0D, 0x91, 0xA5, 0xE8, 0x7E, 0x08, 0x38, 0x49, 
			0x77, 0x10, 0x05, 0xF0, 0xD9, 0x57, 0x5A, 0xE2, 0xA0, 0x0C, 0x10, 0x03, 0x46, 0xA2, 0xD1, 0x6A, 0x12, 0x21, 0x81, 0x24, 0x35, 0x94, 0x51, 0x93, 
			0x2E, 0x5D, 0x91, 0x2D, 0xAE, 0xB6, 0xBE, 0xFA, 0x26, 0x12, 0xB5, 0x38, 0x9A, 0xAD, 0x6E
		};

		[Test]
		public void MultipleEntriesInExtension ()
		{
			X509Certificate cert = new X509Certificate (multiple_entries_cert);
			SubjectAltNameExtension sane = new SubjectAltNameExtension (cert.Extensions ["2.5.29.17"]);
			Assert.AreEqual (4, sane.DNSNames.Length, "Count");
			Assert.AreEqual ("www.arzneimittel-bestellcenter.de", sane.DNSNames [0], "0");
			Assert.AreEqual ("www.xnet-arzneimittel-bestellcenter.de", sane.DNSNames [1], "1");
			Assert.AreEqual ("www.adg-arzneimittel-bestellcenter.de", sane.DNSNames [2], "2");
			Assert.AreEqual ("www.a3000-filialapotheke.de", sane.DNSNames [3], "3");
		}

		[Test]
		public void MultipleEntriesInExtension_AsASN1 ()
		{
			X509Certificate cert = new X509Certificate (multiple_entries_cert);
			SubjectAltNameExtension sane = new SubjectAltNameExtension (cert.Extensions ["2.5.29.17"].ASN1);
			Assert.AreEqual ("www.arzneimittel-bestellcenter.de", sane.DNSNames [0], "0");
			Assert.AreEqual ("www.xnet-arzneimittel-bestellcenter.de", sane.DNSNames [1], "1");
			Assert.AreEqual ("www.adg-arzneimittel-bestellcenter.de", sane.DNSNames [2], "2");
			Assert.AreEqual ("www.a3000-filialapotheke.de", sane.DNSNames [3], "3");
		}
	}
}
