///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file DataChannel.h
 * \brief Contains the definition of the AtomViz::DataChannel class.
 */

#ifndef __DATA_CHANNEL_H
#define __DATA_CHANNEL_H

#include <atomviz/AtomViz.h>

#include <core/reference/RefTarget.h>
#include <core/gui/properties/PropertiesEditor.h>
#include <core/scene/ObjectNode.h>
#include <core/viewport/Viewport.h>
#include <base/linalg/Tensor.h>

namespace AtomViz {

class AtomsObject;	// defined in AtomsObject.h

/**
 * \brief A single data channel contained in an AtomsObject.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT DataChannel : public RefTarget
{
	Q_OBJECT
public:

	/// \brief Identifiers for the standard data channels used in an
	///        AtomsObject.
	enum DataChannelIdentifier {
		UserDataChannel = 0,	//< This is reserved for non-standard data channels
		AtomTypeChannel = -1,
		PositionChannel = -2,
		SelectionChannel = -3,
		ColorChannel = -4,
		DisplacementChannel = -5,
		PotentialEnergyChannel = -6,
		KineticEnergyChannel = -7,
		TotalEnergyChannel = -8,
		VelocityChannel = -9,
		RadiusChannel = -10,
		ClusterChannel = -11,
		CoordinationChannel = -12,
		CNATypeChannel = -13,
		AtomIndexChannel = -14,
		StressTensorChannel = -15,
		StrainTensorChannel = -16,
		DeformationGradientChannel = -17,
		OrientationChannel = -18,
		ForceChannel = -19,
		MassChannel = -20,
		PeriodicImageChannel = -21,
		TransparencyChannel = -22,
		BondsChannel = -23,
	};
	Q_ENUMS(DataChannelIdentifier)

public:

	/// \brief Serialization constructor that creates an empty data channel.
	///
	/// \note This constructor is only used when a data channel is loaded from a scene file.
	///       It must not be used to create a new data channel.
	explicit DataChannel(bool isLoading = false);

	/// \brief Constructor that creates a non-standard (user) data channel object.
	/// \param dataType Specifies the data type (integer, floating-point, ...) of the per-atom elements
	///                 in the new data channel. The data type is specified as identifier according to the
	///                 Qt meta type system.
	/// \param dataTypeSize The size of the data type given by \a dataType in bytes.
	///                     This is necessary because the Qt type system has no function to query
	///                     the size of the type programmaticaly.
	/// \param componentCount The number of components per atom of type \a dataType.
	DataChannel(int dataType, size_t dataTypeSize, size_t componentCount);

	/// \brief Constructor that creates a standard data channel.
	/// \param which Specifies which standard data channel should be created.
	///              This must not be DataChannelIdentifier::UserDataChannel.
	/// \param componentCount The component count if this type of data channel
	///                       has a variable component count; otherwise 0.
	///
	/// Data type, component count and name are automatically chosen by this
	/// constructor.
	DataChannel(DataChannelIdentifier which, size_t componentCount = 0);

	/// \brief Gets the channels's name.
	/// \return The name of channel as shown to the user.
	/// \sa setName()
	const QString& name() const { return _name; }

	/// \brief Sets the channel's name.
	/// \param name The new name string.
	/// \undoable
	/// \sa name()
	void setName(const QString& name);

	/// \brief Returns the name string used by default for the given standard data channel.
	/// \param which Any of the standard channel identifier except DataChannelIdentifier::UserDataChannel.
	/// \return The name string used for the given standard data channel by default.
	static QString standardChannelName(DataChannelIdentifier which);

	/// Returns the data type used by the given standard data channel.
	static int standardChannelType(DataChannelIdentifier which);
	/// \brief Returns the number of vector components per atom used by the given standard data channel.
	/// \param which The standard channel type for which the number of components should be returned.
	/// \return The number of fixed components or 0 if this kind of data channel has a variable number of components.
	static size_t standardChannelComponentCount(DataChannelIdentifier which);
	/// \brief Returns the list of component names for the given standard channel.
	/// \param which The standard channel type for which the component names should be returned.
	/// \param componentCount Optional number of actual components if the standard data channel has a variable number of components.
	static QStringList standardChannelComponentNames(DataChannelIdentifier which, size_t componentCount = 0);
	/// Returns a list with the names and identifiers of all defined standard data channels.
	static QMap<QString, DataChannelIdentifier> standardChannelList();

	/// \brief Returns the number of atoms in this data channel.
	/// \return The total number of data elements in this channel divided by the
	///         number of elements per atom.
	///
	/// This is always equal to the number of atoms in the AtomsObject.
	/// \sa AtomsObject::atomsCount()
	/// \sa setSize()
	size_t size() const { return _numAtoms; }

	/// \brief Resizes the data channel.
	/// \param newSize The new number of atoms.
	///
	/// If this data channel is part of an AtomsObject that the method
	/// AtomsObject::setAtomsCount() instead of this method to resize
	/// all data channels at once and to keep the channel sizes in sync.
	///
	/// \sa AtomsObject::setAtomsCount()
	/// \sa size()
	void setSize(size_t newSize) {
		OVITO_ASSERT_MSG(channelUsageCount() == 0, "DataChannel::setSize()", "The size of the data channel may only be changed when it is not part of an AtomsObject.");
		resize(newSize);
	}

	/// \brief Returns the identifier of this data channel.
	/// \return The identifier that specifies whether the channel is
	///         one of the standard channels or a custom channel.
	///
	/// This number is used to identify the channel within the
	/// AtomsObject it belongs to.
	DataChannelIdentifier id() const { return _id; }

	/// \brief Returns the data type of the channel.
	/// \return The identifier of the data type used for the elements stored in
	///         this channel according to the Qt meta type system.
	int type() const { return _type; }

	/// \brief Returns the number of bytes per value.
	/// \return Number of bytes used to store a single value of the data type
	///         specified by type().
	size_t dataTypeSize() const { return _dataTypeSize; }

	/// \brief Returns the number of bytes used per atom.
	/// \return The size of the channel's data type multiplied by the component count.
	size_t perAtomSize() const { return _perAtomSize; }

	/// \brief Returns the number of array elements per atom.
	/// \return The number of atomic data values stored per atom in this channel.
	/// \sa componentNames()
	/// \sa setComponentCount
	size_t componentCount() const { return _componentCount; }

	/// \brief Changes the number of components per atom.
	/// \param count The new number of atomic data values stored per atom in this channel.
	/// \sa componentCount()
	///
	/// \note Calling this function will destroy all data stored in the channel.
	void setComponentCount(size_t count);

	/// \brief Returns the human-readable names for the components stored per atom.
	/// \return The names of the vector components if this channel contains more than one value per atom.
	///         If this is only a single valued channel then an empty list is returned by this method because
	///         then no naming is necessary.
	const QStringList& componentNames() const { return _componentNames; }

	/// Returns a read-only pointer to the raw elements in the data channel.
	const char* constData() const {
		return _dataBuffer.constData();
	}

	/// Returns a read-only pointer to the first integer element in the data channel.
	/// This method may only be used if this channel is a channel of data type integer.
	const int* constDataInt() const {
		OVITO_ASSERT(type() == qMetaTypeId<int>());
		return reinterpret_cast<const int*>(_dataBuffer.constData());
	}

	/// Returns a read-only pointer to the first float element in the data channel.
	/// This method may only be used if this channel is a channel of data type float.
	const FloatType* constDataFloat() const {
		OVITO_ASSERT(type() == qMetaTypeId<FloatType>());
		return reinterpret_cast<const FloatType*>(_dataBuffer.constData());
	}

	/// Returns a read-only pointer to the first vector element in the data channel.
	/// This method may only be used if this channel is a channel of data type Vector3 or a FloatType channel with 3 components.
	const Vector3* constDataVector3() const {
		OVITO_ASSERT(type() == qMetaTypeId<Vector3>() || (type() == qMetaTypeId<FloatType>() && componentCount() == 3));
		return reinterpret_cast<const Vector3*>(_dataBuffer.constData());
	}

	/// Returns a read-only pointer to the first point element in the data channel.
	/// This method may only be used if this channel is a channel of data type Point3 or a FloatType channel with 3 components.
	const Point3* constDataPoint3() const {
		OVITO_ASSERT(type() == qMetaTypeId<Point3>() || (type() == qMetaTypeId<FloatType>() && componentCount() == 3));
		return reinterpret_cast<const Point3*>(_dataBuffer.constData());
	}

	/// Returns a read-only pointer to the first tensor element in the data channel.
	/// This method may only be used if this channel is a channel of data type Tensor2 or a FloatType channel with 9 components.
	const Tensor2* constDataTensor2() const {
		OVITO_ASSERT(type() == qMetaTypeId<Tensor2>() || (type() == qMetaTypeId<FloatType>() && componentCount() == 9));
		return reinterpret_cast<const Tensor2*>(_dataBuffer.constData());
	}

	/// Returns a read-only pointer to the first symmetric tensor element in the data channel.
	/// This method may only be used if this channel is a channel of data type SymmetricTensor2 or a FloatType channel with 6 components.
	const SymmetricTensor2* constDataSymmetricTensor2() const {
		OVITO_ASSERT(type() == qMetaTypeId<SymmetricTensor2>() || (type() == qMetaTypeId<FloatType>() && componentCount() == 6));
		return reinterpret_cast<const SymmetricTensor2*>(_dataBuffer.constData());
	}

	/// Returns a read-only pointer to the first quaternion element in the data channel.
	/// This method may only be used if this channel is a channel of data type Quaternion or a FloatType channel with 4 components.
	const Quaternion* constDataQuaternion() const {
		OVITO_ASSERT(type() == qMetaTypeId<Quaternion>() || (type() == qMetaTypeId<FloatType>() && componentCount() == 4));
		return reinterpret_cast<const Quaternion*>(_dataBuffer.constData());
	}

	/// Returns a read-write pointer to the raw elements in the data channel.
	char* data() {
		return _dataBuffer.data();
	}

	/// Returns a read-write pointer to the first integer element in the data channel.
	/// This method may only be used if this channel is a channel of data type integer.
	int* dataInt() {
		OVITO_ASSERT(type() == qMetaTypeId<int>());
		return reinterpret_cast<int*>(_dataBuffer.data());
	}

	/// Returns a read-write pointer to the first float element in the data channel.
	/// This method may only be used if this channel is a channel of data type float.
	FloatType* dataFloat() {
		OVITO_ASSERT(type() == qMetaTypeId<FloatType>());
		return reinterpret_cast<FloatType*>(_dataBuffer.data());
	}

	/// Returns a read-write pointer to the first vector element in the data channel.
	/// This method may only be used if this channel is a channel of data type Vector3 or a FloatType channel with 3 components.
	Vector3* dataVector3() {
		OVITO_ASSERT(type() == qMetaTypeId<Vector3>() || (type() == qMetaTypeId<FloatType>() && componentCount() == 3));
		return reinterpret_cast<Vector3*>(_dataBuffer.data());
	}

	/// Returns a read-write pointer to the first point element in the data channel.
	/// This method may only be used if this channel is a channel of data type Point3 or a FloatType channel with 3 components.
	Point3* dataPoint3() {
		OVITO_ASSERT(type() == qMetaTypeId<Point3>() || (type() == qMetaTypeId<FloatType>() && componentCount() == 3));
		return reinterpret_cast<Point3*>(_dataBuffer.data());
	}

	/// Returns a read-write pointer to the first tensor element in the data channel.
	/// This method may only be used if this channel is a channel of data type Tensor2 or a FloatType channel with 9 components.
	Tensor2* dataTensor2() {
		OVITO_ASSERT(type() == qMetaTypeId<Tensor2>() || (type() == qMetaTypeId<FloatType>() && componentCount() == 9));
		return reinterpret_cast<Tensor2*>(_dataBuffer.data());
	}

	/// Returns a read-write pointer to the first symmetric tensor element in the data channel.
	/// This method may only be used if this channel is a channel of data type SymmetricTensor2 or a FloatType channel with 6 components.
	SymmetricTensor2* dataSymmetricTensor2() {
		OVITO_ASSERT(type() == qMetaTypeId<SymmetricTensor2>() || (type() == qMetaTypeId<FloatType>() && componentCount() == 6));
		return reinterpret_cast<SymmetricTensor2*>(_dataBuffer.data());
	}

	/// Returns a read-write pointer to the first quaternion element in the data channel.
	/// This method may only be used if this channel is a channel of data type Quaternion or a FloatType channel with 4 components.
	Quaternion* dataQuaternion() {
		OVITO_ASSERT(type() == qMetaTypeId<Quaternion>() || (type() == qMetaTypeId<FloatType>() && componentCount() == 4));
		return reinterpret_cast<Quaternion*>(_dataBuffer.data());
	}

	/// Returns an integer element at the given index (if this is an integer data channel).
	int getInt(size_t atomIndex) const {
		OVITO_ASSERT(atomIndex < size() && componentCount() == 1);
		return constDataInt()[atomIndex];
	}

	/// Returns a float element at the given index (if this is a float data channel).
	FloatType getFloat(size_t atomIndex) const {
		OVITO_ASSERT(atomIndex < size() && componentCount() == 1);
		return constDataFloat()[atomIndex];
	}

	/// Returns an integer element at the given index (if this is an integer data channel).
	int getIntComponent(size_t atomIndex, size_t componentIndex) const {
		OVITO_ASSERT(atomIndex < size() && componentIndex < componentCount());
		return constDataInt()[atomIndex*componentCount() + componentIndex];
	}

	/// Returns a float element at the given index (if this is a float data channel).
	FloatType getFloatComponent(size_t atomIndex, size_t componentIndex) const {
		OVITO_ASSERT(atomIndex < size() && componentIndex < componentCount());
		return constDataFloat()[atomIndex*componentCount() + componentIndex];
	}

	/// Returns a Vector3 element at the given index (if this is a vector data channel).
	const Vector3& getVector3(size_t atomIndex) const {
		OVITO_ASSERT(atomIndex < size());
		return constDataVector3()[atomIndex];
	}

	/// Returns a Point3 element at the given index (if this is a point data channel).
	const Point3& getPoint3(size_t atomIndex) const {
		OVITO_ASSERT(atomIndex < size());
		return constDataPoint3()[atomIndex];
	}

	/// Returns a Tensor2 element stored for the given atom.
	const Tensor2& getTensor2(size_t atomIndex) const {
		OVITO_ASSERT(atomIndex < size());
		return constDataTensor2()[atomIndex];
	}

	/// Returns a SymmetricTensor2 element stored for the given atom.
	const SymmetricTensor2& getSymmetricTensor2(size_t atomIndex) const {
		OVITO_ASSERT(atomIndex < size());
		return constDataSymmetricTensor2()[atomIndex];
	}

	/// Returns a Quaternion element stored for the given atom.
	const Quaternion& getQuaternion(size_t atomIndex) const {
		OVITO_ASSERT(atomIndex < size());
		return constDataQuaternion()[atomIndex];
	}

	/// Sets the value of an integer element at the given index (if this is an integer data channel).
	void setInt(size_t atomIndex, int newValue) {
		OVITO_ASSERT(atomIndex < size());
		dataInt()[atomIndex] = newValue;
	}

	/// Sets the value of a float element at the given index (if this is a float data channel).
	void setFloat(size_t atomIndex, FloatType newValue) {
		OVITO_ASSERT(atomIndex < size());
		dataFloat()[atomIndex] = newValue;
	}

	/// Sets the value of an integer element at the given index (if this is an integer data channel).
	void setIntComponent(size_t atomIndex, size_t componentIndex, int newValue) {
		OVITO_ASSERT(atomIndex < size() && componentIndex < componentCount());
		dataInt()[atomIndex*componentCount() + componentIndex] = newValue;
	}

	/// Sets the value of a float element at the given index (if this is a float data channel).
	void setFloatComponent(size_t atomIndex, size_t componentIndex, FloatType newValue) {
		OVITO_ASSERT(atomIndex < size() && componentIndex < componentCount());
		dataFloat()[atomIndex*componentCount() + componentIndex] = newValue;
	}

	/// Sets the value of a Vector3 element at the given index (if this is a vector data channel).
	void setVector3(size_t atomIndex, const Vector3& newValue) {
		OVITO_ASSERT(atomIndex < size());
		dataVector3()[atomIndex] = newValue;
	}

	/// Sets the value of a Point3 element at the given index (if this is a point data channel).
	void setPoint3(size_t atomIndex, const Point3& newValue) {
		OVITO_ASSERT(atomIndex < size());
		dataPoint3()[atomIndex] = newValue;
	}

	/// Sets the value of a Tensor2 element for the given atom.
	void setTensor2(size_t atomIndex, const Tensor2& newValue) {
		OVITO_ASSERT(atomIndex < size());
		dataTensor2()[atomIndex] = newValue;
	}

	/// Sets the value of a SymmetricTensor2 element for the given atom.
	void setSymmetricTensor2(size_t atomIndex, const SymmetricTensor2& newValue) {
		OVITO_ASSERT(atomIndex < size());
		dataSymmetricTensor2()[atomIndex] = newValue;
	}

	/// Sets the value of a Quaternion element for the given atom.
	void setQuaternion(size_t atomIndex, const Quaternion& newValue) {
		OVITO_ASSERT(atomIndex < size());
		dataQuaternion()[atomIndex] = newValue;
	}

	/// Returns the number of AtomsObjects that have a reference to this data channel.
	int channelUsageCount() const;

	/// Returns the visibility flag for this data channel.
	bool isVisible() const { return _isVisible; }

	/// Sets the visibility flag for this data channel.
	/// If the UndoManager is currently recording then this
	/// method will register a operation record to make the property
	/// change undoable.
	void setVisible(bool visible);

	/// \brief Renders the channel' contents in a viewport.
	/// \param time The current animation time.
	/// \param vp The viewport into which the channel should be rendered.
	/// \param atoms The AtomsObject to which this DataChannel belongs to.
	/// \param contextNode The scene object the AtomsObject belongs to.
	virtual void render(TimeTicks time, Viewport* vp, AtomsObject* atoms, ObjectNode* contextNode) {}

	/// \brief Renders the channel's contents in high-quality mode to an offscreen buffer.
	/// \param time The current animation time.
	/// \param atoms The AtomsObject to which this DataChannel belongs to.
	/// \param view Describes the projection parameters.
	/// \param contextNode The scene object the AtomsObject belongs to.
	/// \param imageWidth The width of the image buffer in pixels.
	/// \param imageHeight The height of the image buffer in pixels.
	/// \param glcontext The OpenGL rendering context.
	virtual void renderHQ(TimeTicks time, AtomsObject* atoms, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext) {}

	/// \brief Returns the bounding box of the channel's geometry when rendered in the viewports.
	/// \param time The current animation time.
	/// \param atoms The AtomsObject to which this DataChannel belongs to.
	/// \param contextNode The scene object the AtomsObject belongs to.
	/// \param validityInterval This is used to return the validity interval during which the bounding box doesn't change.
	/// \return The bounding box of the rendered geometry or an empty box if the channel is not rendered in the viewports.
	virtual Box3 boundingBox(TimeTicks time, AtomsObject* atoms, ObjectNode* contextNode, TimeInterval& validityInterval) { return Box3(); }

	/// \brief Lets the channel clear all its internal caches.
	///
	/// This method is automatically invoked for each DataChannel by the AtomsObject::invalidate()
	/// method. It informs the data channel that the AtomsObject it belongs to has
	/// changed in some way.
	virtual void clearCaches() {}

	/// \brief Returns whether the per-atom data is saved along with the scene.
	/// \return \c true if data is stored in the scene file; \c false if the data resides in an external file.
	bool serializeData() const { return _serializeData; }

	/// \brief Sets whether the per-atom data is saved along with the scene.
	/// \param on \c true if data should be stored in the scene file; \c false if the data resides in an external file.
	/// \undoable
	void setSerializeData(bool on) { _serializeData = on; }

	// From RefTarget class:

	/// Returns the title of this object.
	virtual QString schematicTitle() { return _name; }

public:

	Q_PROPERTY(QString name READ name WRITE setName)
	Q_PROPERTY(bool isVisible READ isVisible WRITE setVisible)
	Q_PROPERTY(bool serializeData READ serializeData WRITE setSerializeData)
	Q_PROPERTY(size_t size READ size)
	Q_PROPERTY(DataChannelIdentifier id READ id)
	Q_PROPERTY(int type READ type)

protected:

	/// Saves the class' contents to the given stream.
	virtual void saveToStream(ObjectSaveStream& stream);
	/// Loads the class' contents from the given stream.
	virtual void loadFromStream(ObjectLoadStream& stream);
	/// Creates a copy of this object.
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);

	/// Resizes the array to the given size. This is an internal method
	/// used by the AtomsObject to resize all its data channels when the
	/// number of atoms changes.
	void resize(size_t newSize);

	/// Copies the contents from the given source channel into this channel.
	/// Atoms for which the bit in the given mask is set are skipped.
	virtual void filterCopy(DataChannel* source, const dynamic_bitset<>& mask);

	/// The identifier of this data channel. It is unique within the
	/// AtomsObject this data channel belongs to.
	DataChannelIdentifier _id;

	/// The name of this data channel.
	QString _name;

	/// The data type of the channel.
	int _type;

	/// The number of bytes per data type value.
	size_t _dataTypeSize;

	/// The number of elements in the data channel.
	/// This is always equal to the number of atoms in the AtomsObject this
	/// data channel belongs to.
	size_t _numAtoms;

	/// The number of bytes per element.
	/// This is the size of the channel's data type multiplied by the component count.
	size_t _perAtomSize;

	/// The number of array elements per atom.
	size_t _componentCount;

	/// The names of the vector components if this channel contains more than one value per atom.
	QStringList _componentNames;

	/// The internal data array that holds the elements.
	QByteArray _dataBuffer;

	/// This flag controls the visibility of the data channel.
	bool _isVisible;

	/// Controls whether the data is saved along with the scene.
	PropertyField<bool> _serializeData;

	friend class AtomsObject;
	friend class DeleteAtomsKernel;

private:

	DECLARE_SERIALIZABLE_PLUGIN_CLASS(DataChannel)
	DECLARE_PROPERTY_FIELD(_serializeData)
};

/// A list of pointers to DataChannel objects.
typedef QVector<DataChannel*> DataChannelList;

/**
 * \brief An offline reference to a DataChannel in an AtomsObject
 *
 * This small helper class can be used to store a reference to a
 * particular data channel of an AtomsObject.
 *
 * This helper class does not store a direct pointer to the channel but
 * only the identifier and/or name of the channel. This information is then
 * used to look up the channel on demand using AtomsObject::lookupDataChannel().
 *
 * \author Alexander Stukowski
 */
class DataChannelReference
{
public:

	/// \brief Default constructor.
	DataChannelReference() : _id(DataChannel::UserDataChannel) {}
	/// \brief Constructor for references to standard data channel.
	DataChannelReference(DataChannel::DataChannelIdentifier id) : _id(id), _name(DataChannel::standardChannelName(id)) {}
	/// \brief Constructor for references to a data channel.
	DataChannelReference(DataChannel::DataChannelIdentifier id, const QString& name) : _id(id), _name(name) {}
	/// \brief Constructor for references to custom data channels.
	DataChannelReference(const QString& name) : _id(DataChannel::UserDataChannel), _name(name) {}
	/// \brief Constructor for references to an existing data channels.
	DataChannelReference(DataChannel* channel) : _id(channel->id()), _name(channel->name()) {}

	/// \brief Gets the identifier of the referenced channel.
	/// \return The channel identifier.
	DataChannel::DataChannelIdentifier id() const { return _id; }

	/// \brief Sets the referenced channel.
	void setId(DataChannel::DataChannelIdentifier id) {
		_id = id;
		if(id != DataChannel::UserDataChannel)
			_name = DataChannel::standardChannelName(id);
	}

	/// \brief Gets the human-readable name of the referenced channel.
	/// \return The channel name.
	const QString& name() const { return _name; }

	/// \brief Compares two data channel references for equality.
	bool operator==(const DataChannelReference& other) const {
		if(id() != other.id()) return false;
		if(id() != DataChannel::UserDataChannel) return true;
		return name() == other.name();
	}

	/// \brief Returns whether this reference object does not point to a DataChannel.
	bool isNull() const { return id() == DataChannel::UserDataChannel && name().isEmpty(); }

private:

	/// The identifier of the channel.
	DataChannel::DataChannelIdentifier _id;

	/// The human-readable name of the channel.
	/// It is only used for custom non-standard data channels.
	QString _name;
};

/// Writes a DataChannelReference to an output stream.
inline SaveStream& operator<<(SaveStream& stream, const DataChannelReference& r)
{
	stream.writeEnum(r.id());
	stream << r.name();
	return stream;
}

/// Reads a DataChannelReference from an input stream.
inline LoadStream& operator>>(LoadStream& stream, DataChannelReference& r)
{
	DataChannel::DataChannelIdentifier id;
	QString name;
	stream.readEnum(id);
	stream >> name;
	if(id != DataChannel::UserDataChannel)
		r = DataChannelReference(id);
	else
		r = DataChannelReference(name);
	return stream;
}

/**
 * \brief A properties editor for the DataChannel class.
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT DataChannelEditor : public PropertiesEditor
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

private:
	Q_OBJECT
	DECLARE_PLUGIN_CLASS(DataChannelEditor)
};

};	// End of namespace AtomViz

Q_DECLARE_METATYPE(AtomViz::DataChannelReference)

#endif // __DATA_CHANNEL_H
