/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2004 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef VERSION
#define VERSION "UNKNOWN VERSION"
#endif

#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <librevenge-stream/librevenge-stream.h>
#include <libmwaw/libmwaw.hxx>
#include <librvngabw/librvngabw.hxx>

#include "ABWStringDocumentHandler.hxx"

class ABWOutputFileHelper
{
public:
	ABWOutputFileHelper(const char *outFileName) : mOutFileName(outFileName) {}
	~ABWOutputFileHelper() {}

	bool convertTextDocument(librevenge::RVNGInputStream &input)
	{
		writerperfectrvngabw::ABWStringDocumentHandler contentHandler;
		librvngabw::ABWTextGenerator collector(&contentHandler);
		collector.registerEmbeddedImageHandler("image/mwaw-odg", &handleEmbeddedMWAWGraphicImage);
		collector.registerEmbeddedObjectHandler("image/mwaw-ods", &handleEmbeddedMWAWSpreadsheetObject);
		if (MWAWDocument::MWAW_R_OK != MWAWDocument::parse(&input, &collector))
			return false;
		if (mOutFileName)
		{
			FILE *fhandle = fopen(mOutFileName, "wb");
			if (!fhandle)
				return false;
			fprintf(fhandle, "%s\n", contentHandler.getData().cstr());
			fclose(fhandle);
		}
		else
			printf("%s\n", contentHandler.getData().cstr());
		return true;
	}

	static bool isSupportedFormat(librevenge::RVNGInputStream &input)
	{
		MWAWDocument::Type type;
		MWAWDocument::Kind kind;
		MWAWDocument::Confidence confidence = MWAWDocument::isFileFormatSupported(&input, type, kind);
		if (confidence != MWAWDocument::MWAW_C_EXCELLENT)
		{
			fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Mac Classic document.\n");
			return false;
		}
		if (kind != MWAWDocument::MWAW_K_TEXT)
		{
			fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Mac Classic text document.\n");
			return false;
		}

		return true;
	}
private:
	static bool handleEmbeddedMWAWGraphicImage(const librevenge::RVNGBinaryData &data, librevenge::RVNGBinaryData &output)
	{
		librevenge::RVNGStringVector svgOutput;
		librevenge::RVNGSVGDrawingGenerator generator(svgOutput, "");
		bool result = MWAWDocument::decodeGraphic(data, &generator);
		if (!result || svgOutput.empty() || svgOutput[0].empty())
			return false;

		output.clear();
		const char *svgHeader = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>\n"
		                        "<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.1//EN\""
		                        " \"http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd\">\n";
		output.append((unsigned char *)svgHeader, strlen(svgHeader));
		output.append((unsigned char *)svgOutput[0].cstr(), strlen(svgOutput[0].cstr()));
		return true;
	}
	static bool handleEmbeddedMWAWSpreadsheetObject(const librevenge::RVNGBinaryData &data, librvngabw::ABWGenerator &generator)
	{
		librvngabw::ABWSpreadsheetToTableGenerator exporter(generator);
		return MWAWDocument::decodeSpreadsheet(data, &exporter);
	}
	//! the ouput file name
	char const *mOutFileName;
};

int printUsage(char const *name)
{
	fprintf(stderr, "Usage: %s [OPTION] <Mac Document> [file.abw]\n", name);
	fprintf(stderr, "\ttries to convert a Mac Document in a abw file.\n");
	fprintf(stderr, "\tIf given the result is stored in file.abw, \n");
	fprintf(stderr, "\tif not, the result is sent to stdout.\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "Options:\n");
	fprintf(stderr, "\t-h:                Shows this help message\n");
	fprintf(stderr, "\t-v:                Output %s version \n", name);
	fprintf(stderr, "\n");
	fprintf(stderr, "Note:\n");
	fprintf(stderr, "\tyou can find the list of recognized formats at\n");
	fprintf(stderr, "\thttp://sourceforge.net/p/libmwaw/wiki/Home/\n");
	fprintf(stderr, "\t(at least the list of formats recognized by the lattest version of libmwaw)\n");

	return 1;
}

int printVersion(char const *name)
{
	printf("%s %s\n", name, VERSION);
	return 0;
}


int main(int argc, char *argv[])
{
	bool printHelp=false;
	int ch;

	while ((ch = getopt(argc, argv, "hvx")) != -1)
	{
		switch (ch)
		{
		case 'v':
			printVersion(argv[0]);
			return 0;
		case 'h':
		default:
			printHelp=true;
			break;
		}
	}
	if ((argc != 1+optind && argc != 2+optind) || printHelp)
	{
		printUsage(argv[0]);
		return -1;
	}

	char const *szInputFile = argv[optind];
	char const *szOutFile = (argc == 2+optind) ? argv[optind+1] : 0;
	librevenge::RVNGFileStream input(szInputFile);
	if (!ABWOutputFileHelper::isSupportedFormat(input))
		return 1;
	ABWOutputFileHelper helper(szOutFile);
	bool ok=helper.convertTextDocument(input);
	if (!ok)
	{
		fprintf(stderr, "ERROR : Couldn't convert the document\n");
		return 1;
	}

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
