/*
 *  Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "TemplateAST_p.h"

#include <float.h>

#include <GTLCore/Macros_p.h>
#include <GTLCore/StdTypes.h>
#include <GTLCore/Type.h>

#include "GenerationContext_p.h"
#include "Debug.h"

using namespace OpenCTL;
using namespace OpenCTL::TemplateAST;

Node::~Node() {
}

GTLCore::String Node::typeToString(const GTLCore::Type* _type) {
  switch(_type->dataType()) {
    case GTLCore::Type::UNSIGNED_INTEGER8:
    case GTLCore::Type::UNSIGNED_INTEGER16:
    case GTLCore::Type::UNSIGNED_INTEGER32:
      return " unsigned int ";
    case GTLCore::Type::INTEGER8:
    case GTLCore::Type::INTEGER16:
    case GTLCore::Type::INTEGER32:
      return " int ";
    case GTLCore::Type::FLOAT32:
      return " float ";
    case GTLCore::Type::FLOAT16:
      return " half ";
    default:
      GTL_ABORT("Unsupported: " << _type);
  }
}

GTLCore::String Node::typeMax(const GTLCore::Type* _type) {
  switch(_type->dataType()) {
    case GTLCore::Type::UNSIGNED_INTEGER8:
      return GTLCore::String::number(UINT8_MAX);
    case GTLCore::Type::UNSIGNED_INTEGER16:
      return GTLCore::String::number(UINT16_MAX);
    case GTLCore::Type::UNSIGNED_INTEGER32:
      return GTLCore::String::number((unsigned int)(UINT32_MAX));
    case GTLCore::Type::INTEGER8:
      return GTLCore::String::number(INT8_MAX);
    case GTLCore::Type::INTEGER16:
      return GTLCore::String::number(INT16_MAX);
    case GTLCore::Type::INTEGER32:
      return GTLCore::String::number(INT32_MAX);
    case GTLCore::Type::FLOAT32:
      return GTLCore::String::number(FLT_MAX);
    case GTLCore::Type::FLOAT16:
      return GTLCore::String::number(FLT_MAX);
    default:
      GTL_ABORT("Unsupported: " << _type);
  }
}

GTLCore::String Node::typeMin(const GTLCore::Type* _type) {
  switch(_type->dataType()) {
    case GTLCore::Type::UNSIGNED_INTEGER8:
      return GTLCore::String::number(0);
    case GTLCore::Type::UNSIGNED_INTEGER16:
      return GTLCore::String::number(0);
    case GTLCore::Type::UNSIGNED_INTEGER32:
      return GTLCore::String::number(0);
    case GTLCore::Type::INTEGER8:
      return GTLCore::String::number(INT8_MIN);
    case GTLCore::Type::INTEGER16:
      return GTLCore::String::number(INT16_MIN);
    case GTLCore::Type::INTEGER32:
      return GTLCore::String::number(int(INT32_MIN));
    case GTLCore::Type::FLOAT32:
      return GTLCore::String::number(FLT_MIN);
    case GTLCore::Type::FLOAT16:
      return GTLCore::String::number(FLT_MIN);
    default:
      GTL_ABORT("Unsupported: " << _type);
  }
}
GTLCore::String Node::typeUnit(const GTLCore::Type* _type) {
  switch(_type->dataType()) {
    case GTLCore::Type::UNSIGNED_INTEGER8:
      return GTLCore::String::number(UINT8_MAX);
    case GTLCore::Type::UNSIGNED_INTEGER16:
      return GTLCore::String::number(UINT16_MAX);
    case GTLCore::Type::UNSIGNED_INTEGER32:
      return GTLCore::String::number((unsigned int)(UINT32_MAX));
    case GTLCore::Type::INTEGER8:
      return GTLCore::String::number(INT8_MAX);
    case GTLCore::Type::INTEGER16:
      return GTLCore::String::number(INT16_MAX);
    case GTLCore::Type::INTEGER32:
      return GTLCore::String::number(INT32_MAX);
    case GTLCore::Type::FLOAT32:
      return GTLCore::String::number(1.0);
    case GTLCore::Type::FLOAT16:
      return GTLCore::String::number(1.0);
    default:
      GTL_ABORT("Unsupported: " << _type);
  }
}

NodesList::NodesList( const std::list<Node*>& _nodes) : m_nodes(_nodes)
{
}

NodesList::~NodesList()
{
}

void NodesList::generate(TemplateGenerationContext* _context)
{
  foreach(Node* node, m_nodes)
  {
    node->generate(_context);
  }
}

TextNode::TextNode(const GTLCore::String& _text) : m_text(_text)
{
}

TextNode::~TextNode()
{
}

void TextNode::generate(TemplateGenerationContext* _context)
{
  _context->append(m_text);
}

OperationNode::OperationNode( const GTLCore::String& _name, unsigned int _inputs, const GTLCore::String& _arg, const GTLCore::String& _optArg) : m_name(_name), m_inputs(_inputs), m_arg(_arg), m_optArg(_optArg)
{
}

OperationNode::~OperationNode()
{
}

void OperationNode::generate(TemplateGenerationContext* _context)
{
  GTLCore::String header = "void " + m_name + "(";
  std::vector<GTLCore::String> inHeaders(m_inputs - 1);
  GTLCore::String outHeader;
  
  const GTLCore::PixelDescription& pd = _context->pixelDescription();
  
  for(std::size_t i = 0; i < pd.channels(); ++i)
  {
    const GTLCore::Type* type = pd.channelTypes()[i];
    GTLCore::String string_type = typeToString(type);
    GTLCore::String string_num = GTLCore::String::number((unsigned int)i);
    header += string_type + " in_" + string_num + ", ";
    for( int j = 1; j < m_inputs; ++j)
    {
      inHeaders[j - 1] += string_type + " in_" + string_num + "_" + GTLCore::String::number(j) + ", ";
    }
    outHeader += "output " + string_type + " out_" + string_num;
    if(i != pd.channels() - 1 )
    {
      outHeader += ", ";
    }
  }
  for( int j = 1; j < m_inputs; ++j)
  {
    header += inHeaders[j-1];
  }
  header += m_arg;
  header += outHeader + m_optArg + " )";
  _context->append(header);
}

AllChannelsNode::AllChannelsNode( NodesList* _nodesList, WhichChannel _whichChannel ) : m_nodesList(_nodesList), m_whichChannel(_whichChannel) {
}

AllChannelsNode::~AllChannelsNode() {
  delete m_nodesList;
}

void AllChannelsNode::generate(TemplateGenerationContext* _context) {
  const GTLCore::PixelDescription& pd = _context->pixelDescription();
  for(int i = 0; i < int(pd.channels()); ++i)
  {
    if( m_whichChannel == AllChannel
        or ( m_whichChannel == ColorChannel and i != pd.alphaPos() )
        or ( m_whichChannel == AlphaChannel and i == pd.alphaPos() ) )
    {
      _context->startLocalContext("_" + GTLCore::String::number(i), i);
      m_nodesList->generate(_context);
      _context->endLocalContext();
    }
  }
}

void AlphaNode::generate(TemplateGenerationContext* _context) {
  int idx = _context->pixelDescription().alphaPos();
  if( idx == -1) {
    _context->append("1.0");
  } else {
    _context->append( "in_" + GTLCore::String::number(idx) );
  }
}

void AlphaMaxNode::generate(TemplateGenerationContext* _context) {
  int idx = _context->pixelDescription().alphaPos();
  if( idx == -1) {
    _context->append( GTLCore::String::number(FLT_MAX) );
  } else {
    _context->append( typeMax( _context->pixelDescription().channelTypes()[idx] ) );
  }
}

void AlphaMinNode::generate(TemplateGenerationContext* _context) {
  int idx = _context->pixelDescription().alphaPos();
  if( idx == -1) {
    _context->append( GTLCore::String::number(FLT_MIN) );
  } else {
    _context->append( typeMin( _context->pixelDescription().channelTypes()[idx] ) );
  }
}

void AlphaUnitNode::generate(TemplateGenerationContext* _context) {
  int idx = _context->pixelDescription().alphaPos();
  if( idx == -1) {
    _context->append("1.0");
  } else {
    _context->append( typeUnit( _context->pixelDescription().channelTypes()[idx] ) );
  }
}

InOutNode::InOutNode(Which _which, Quantity _quantity, unsigned int idx) : m_idx( idx == 0 ? "" : "_" + GTLCore::String::number(idx)), m_quantity(_quantity) {
  switch(_which)
  {
    case In:
      m_string = "in";
      break;
    case Out:
      m_string = "out";
      break;
  }
}

void InOutNode::generate(TemplateGenerationContext* _context) {
  switch(m_quantity)
  {
    case One:
      _context->append(m_string + _context->suffix() + m_idx);
      break;
    case All:
    {
      for(std::size_t i = 0; i < _context->pixelDescription().channels(); ++ i)
      {
        _context->append(m_string + "_" + GTLCore::String::number((unsigned int)i) + m_idx);
        if( i != _context->pixelDescription().channels() - 1)
        {
          _context->append(",");
        }
      }
    }
      break;
  }
}

VarNode::VarNode(const GTLCore::String& _name ) : m_name(_name) {
}

void VarNode::generate(TemplateGenerationContext* _context) {
  GTLCore::String realname = m_name + _context->suffix();
  _context->append(realname);
}

void ChannelMaxNode::generate(TemplateGenerationContext* _context) {
  _context->append(typeMax(_context->currentChannelType()));
}

void ChannelMinNode::generate(TemplateGenerationContext* _context) {
  _context->append(typeMin(_context->currentChannelType()));
}

void ChannelUnitNode::generate(TemplateGenerationContext* _context) {
  _context->append(typeUnit(_context->currentChannelType()));
}

void ChannelTypeNode::generate(TemplateGenerationContext* _context) {
  _context->append(typeToString(_context->currentChannelType()));
}

NamedMaxNode::NamedMaxNode(const GTLCore::String& _name) : m_name(_name) {
}

void NamedMaxNode::generate(TemplateGenerationContext* _context) {
  _context->append(typeMax(_context->namedType(m_name)));
}

NamedMinNode::NamedMinNode(const GTLCore::String& _name) : m_name(_name) {
}

void NamedMinNode::generate(TemplateGenerationContext* _context) {
  _context->append(typeMin(_context->namedType(m_name)));
}

NamedUnitNode::NamedUnitNode(const GTLCore::String& _name) : m_name(_name) {
}

void NamedUnitNode::generate(TemplateGenerationContext* _context) {
  _context->append(typeUnit(_context->namedType(m_name)));
}

NamedTypeNode::NamedTypeNode(const GTLCore::String& _name) : m_name(_name) {
}

void NamedTypeNode::generate(TemplateGenerationContext* _context) {
  _context->append(typeToString(_context->namedType(m_name)));
}
