/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _OPENCTL_DEBUG_THREAD_H_
#define _OPENCTL_DEBUG_THREAD_H_

#ifdef OPENGTL_ENABLE_DEBUG_THREAD

#include <algorithm>

#include <llvm/Support/MutexGuard.h>
#include <llvm/System/Mutex.h>
#include <llvm/System/Atomic.h>

#include "Debug.h"
#include "StdTypes.h"

struct Guard {
  Guard() : value(0)
  {
  }
  void increment()
  {
    llvm::sys::AtomicIncrement(&value);
  }
  void decrement()
  {
    llvm::sys::AtomicDecrement(&value);
  }
  llvm::sys::cas_flag value;
  llvm::sys::Mutex mutexWrite;
  llvm::sys::Mutex mutex;
  std::list<int> readers;
};

/**
 * @internal
 * Do not use this class directly, use the macro TEST_GUARD_READ.
 * 
 * This class test that no other thread is trying to write the value.
 */
struct TestAcquireMutexRead {
  TestAcquireMutexRead(Guard* _guard) : guard(_guard)
  {
    llvm::MutexGuard g(guard->mutex);
    // Check if this thread is already reading
    int id = int64_t(pthread_self()) >> 10;
    alreadyReading = std::find(guard->readers.begin(), guard->readers.end(), id) != guard->readers.end();
    if(alreadyReading) return;
    
    // Check that no thread is writing
    bool v = guard->mutexWrite.tryacquire();
    if(v)
    {
      guard->mutexWrite.release();
    } else {
      GTL_ABORT("Class is been written by an other thread.");
    }
    guard->increment();
    guard->readers.push_back(id);
  }
  ~TestAcquireMutexRead()
  {
    if(not alreadyReading)
    {
      int id = int64_t(pthread_self()) >> 10;
      llvm::MutexGuard g(guard->mutex);
      guard->decrement();
      guard->readers.remove(id);
    }
  }
  Guard* guard;
  bool alreadyReading;
};

/**
 * @internal
 * Do not use this class directly, use the macro TEST_GUARD_WRITE.
 * 
 * This class try to acquire the write right, and test that no other thread is trying to access the value.
 */
struct TestAcquireMutexWrite : TestAcquireMutexRead {
  TestAcquireMutexWrite(Guard* _guard) : TestAcquireMutexRead(_guard)
  {
    llvm::MutexGuard g(_guard->mutex);
    bool v = guard->mutexWrite.tryacquire();
    if(not v)
    {
      GTL_ABORT("Class is been accessed by an other thread.");
    }
    if(guard->value > 1)
    {
      GTL_ABORT("Class is been readed by an other thread.");
    }
  }
  ~TestAcquireMutexWrite()
  {
    llvm::MutexGuard g(guard->mutex);
    guard->mutexWrite.release();
  }
};

/**
 * Add the guard parameters to a class.
 */
#define TEST_GUARD_VARIABLE  Guard guard;
/**
 * Add this to a function that read.
 */
#define TEST_GUARD_READ(d_ptr) TestAcquireMutexRead testAcquireMutex(&d_ptr->guard);
/**
 * Add this to a function that write.
 */
#define TEST_GUARD_WRITE(d_ptr) TestAcquireMutexWrite testAcquireWrite(&d_ptr->guard);

#else

#define TEST_GUARD_VARIABLE
#define TEST_GUARD_READ(d_ptr)
#define TEST_GUARD_WRITE(d_ptr)

#endif

#endif
