/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ModuleData_p.h"

#include <sstream>

#include <llvm/Module.h>
#include <llvm/PassManager.h>
#include <llvm/Target/TargetMachine.h>
#include <llvm/Analysis/Verifier.h>
#include <llvm/Transforms/Utils/Cloning.h>
#include <llvm/Linker.h>
#ifdef LLVM_27_OR_28
#include <llvm/System/Path.h>
#else
#include <llvm/Support/Path.h>
#endif

#include "Debug.h"
#include "Parameter.h"
#include "Type.h"
#include "Function_p.h"
#include "Function.h"
#include "TypesManager.h"
#include "Macros_p.h"

using namespace GTLCore;

ModuleData::ModuleData( llvm::Module* llvmModule) : m_llvmModule(llvmModule), m_llvmLinkedModule(0), m_typesManager(new TypesManager)
{
  GTL_ASSERT(llvmModule);
}

ModuleData::~ModuleData()
{
  for( std::map<ScopedName, std::list<Function*>* >::iterator it = m_functions.begin();
       it != m_functions.end(); ++it)
  {
    foreach( Function* function, *it->second )
    {
      delete function;
    }
    delete it->second;
  }
  delete m_llvmLinkedModule;
  delete m_llvmModule;
  delete m_typesManager;
}

bool ModuleData::appendFunction(const ScopedName& name, Function* function)
{
  GTL_DEBUG(name);
  if( m_functions.find(name) == m_functions.end())
  {
    std::list<Function*>* functions = new std::list<Function*>();
    functions->push_back(function);
    m_functions[name] = functions;
  } else {
    // TODO might be worth to check that the function doesn't exist already
    m_functions[name]->push_back(function);
  }
  return true;
}

const std::list<Function*>* ModuleData::function(const String& _currentNameSpace, const String& _name) const
{
  GTL_DEBUG( _currentNameSpace << "::" << _name );
  for( std::map<ScopedName, std::list<Function*>*>::const_iterator it = m_functions.begin();
       it != m_functions.end(); ++it)
  {
    GTL_DEBUG( it->first );
    if( (it->first.nameSpace() == "" or it->first.nameSpace() == _currentNameSpace )
        and it->first.name() == _name )
    {
      GTL_DEBUG( it->second );
      return it->second;
    }
  }
  return 0;
}

const std::list<Function*>* ModuleData::function(const ScopedName& name) const
{
  std::map<ScopedName, std::list<Function*>* >::const_iterator it = m_functions.find(name);
  if( it == m_functions.end())
  {
    return 0;
  } else {
    return it->second;
  }
}

std::list<Function*> ModuleData::functions()
{
  std::list<Function*> functions;
  GTL_DEBUG( m_functions.size());
  for( std::map<ScopedName, std::list<Function*>*>::iterator it = m_functions.begin();
       it != m_functions.end(); ++it)
  {
    foreach(Function* function, *it->second)
    {
      functions.push_back(function);
    }
  }
  return functions;
}

const GTLCore::Function* ModuleData::function(const GTLCore::String& _nameSpace, const GTLCore::String& name, std::vector<GTLCore::Parameter>& arguments) const
{
  GTLCore::Function* bestFunction = 0;
  const std::list<GTLCore::Function*>* functions = function(_nameSpace, name);
  int bestLossLessConversionCount = arguments.size() + 1;
  int bestLossConversionCount = arguments.size() + 1;
  foreach(GTLCore::Function* function, *functions)
  {
    if( arguments.size() <= function->parameters().size() and arguments.size() >= function->d->data->minimumParameters() )
    {
      int currentLossLessConversionCount = 0;
      int currentLossConversionCount = 0;
      bool good = true;
      for(std::size_t i = 0;  i < arguments.size(); ++i)
      {
        const GTLCore::Parameter& arg_w = arguments[i];
        const GTLCore::Parameter& arg_f = function->parameters()[i];
        if( arg_w.isOutput() and not arg_f.isOutput() )
        {
          good = false;
          break;
        } 
        if( arg_w.type() != arg_f.type() )
        {
          switch( arg_w.type()->dataType() )
          {
            case Type::INTEGER8:
            case Type::UNSIGNED_INTEGER8:
            case Type::INTEGER16:
            case Type::UNSIGNED_INTEGER16:
            case Type::INTEGER32:
            case Type::UNSIGNED_INTEGER32:
              if(arg_f.type()->bitsSize() < arg_w.type()->bitsSize() )
              {
                ++currentLossConversionCount;
              } else {
                ++currentLossLessConversionCount;
              }
              break;
            case Type::FLOAT32:
            case Type::FLOAT16:
              if(arg_f.type()->isFloatingPoint())
              {
                ++currentLossLessConversionCount;
              } else {
                ++currentLossConversionCount;
              }
              break;
            default:
              std::cout << *arg_w.type() << std::endl;
              good = false;
              break;
          }
          if(not good) break;
        }
      }
      if( good )
      {
        if( not bestFunction or currentLossLessConversionCount < bestLossLessConversionCount
            or ( currentLossLessConversionCount == bestLossLessConversionCount
                 and ( currentLossLessConversionCount < bestLossLessConversionCount
                       or ( currentLossLessConversionCount == bestLossLessConversionCount
                            and ( bestFunction->parameters().size() < function->parameters().size() ) ) ) ) )
        {
          bestFunction = function;
          bestLossConversionCount = currentLossConversionCount;
          bestLossLessConversionCount = currentLossLessConversionCount;
        }
      }
    }
  }
  return bestFunction;
}

bool ModuleData::appendConstant(const GTLCore::ScopedName& name, const GTLCore::Type* type, const GTLCore::Value& val)
{
  if (m_constants.find(name) == m_constants.end())
  {
    m_constants[name] = type;
    m_constantsValue[name] = val;
    return true;
  } else {
    return false;
  }
}

const std::map<GTLCore::ScopedName, const GTLCore::Type*>& ModuleData::constants() const
{
  return m_constants;
}

const std::map<ScopedName, GTLCore::Value>& ModuleData::constantsValue() const
{
  return m_constantsValue;
}

void ModuleData::linkWith( const ModuleData* _module )
{
  foreach( const llvm::Module* mod, _module->m_linkModuleWith )
  {
    linkWith(mod);
  }
  linkWith(_module->llvmModule());
}

void ModuleData::linkWith( const llvm::Module* _module )
{
  foreach( const llvm::Module* mod, m_linkModuleWith )
  {
    if( mod == _module ) return;
  }
  m_linkModuleWith.push_back( _module );
}

void ModuleData::doLink()
{
  std::list<GTLCore::String> hide;
  GTL_ASSERT( not m_llvmLinkedModule );
  m_llvmLinkedModule = m_llvmModule;
  m_llvmModule = llvm::CloneModule( m_llvmModule );
  llvm::Linker linker("", m_llvmLinkedModule);
  std::string errorMessage;
  foreach( const llvm::Module* mod, m_linkModuleWith )
  {
    llvm::Module* clone = llvm::CloneModule( mod );
    
    for(llvm::Module::iterator it = clone->getFunctionList().begin();
        it != clone->getFunctionList().end(); ++it)
    {
      GTLCore::String name(it->getName());
      if(name.isEmpty())
      {
        it->setLinkage(llvm::GlobalValue::InternalLinkage);
      } else {
        hide.push_back(name);
      }
    }
    for(llvm::Module::global_iterator it = clone->getGlobalList().begin();
        it != clone->getGlobalList().end(); ++it)
    {
      GTLCore::String name(it->getName());
      if(name.isEmpty())
      {
        it->setLinkage(llvm::GlobalValue::InternalLinkage);
      } else {
        hide.push_back(name);
      }
    }
    linker.LinkInModule( clone, &errorMessage );
    
    GTL_DEBUG("Linking error: " << errorMessage );
    delete clone;
  }
  foreach( const GTLCore::String& mod, m_linkModuleWithArchives )
  {
    bool v = false;
    linker.LinkInArchive( llvm::sys::Path( (const std::string&) mod), v);
  }
  
  linker.releaseModule();
  
  // Hide symbols
  foreach(GTLCore::String name, hide)
  {
    llvm::GlobalValue* value = m_llvmLinkedModule->getNamedValue((const std::string&)name);
    if(value and not value->isDeclaration())
    {
      value->setLinkage(llvm::GlobalValue::InternalLinkage);
    }
  }
}

void ModuleData::hideAllSymbolsBut(std::list<GTLCore::String> _keepVisible)
{
  for(llvm::Module::iterator it = m_llvmLinkedModule->getFunctionList().begin();
      it != m_llvmLinkedModule->getFunctionList().end(); ++it)
  {
    if(not it->isDeclaration() and std::find(_keepVisible.begin(), _keepVisible.end(), GTLCore::String(it->getName())) == _keepVisible.end() )
    {
      it->setLinkage(llvm::GlobalValue::InternalLinkage);
    }
  }
  for(llvm::Module::global_iterator it = m_llvmLinkedModule->getGlobalList().begin();
      it != m_llvmLinkedModule->getGlobalList().end(); ++it)
  {
    if(not it->isDeclaration() and std::find(_keepVisible.begin(), _keepVisible.end(), GTLCore::String(it->getName())) == _keepVisible.end() )
    {
      it->setLinkage(llvm::GlobalValue::InternalLinkage);
    }
  }
}
