/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

// C++ Headers
#include <iostream>
#include <fstream>
#include <cstdlib>

// GTLCore Headers
#include <GTLCore/Region.h>

// GTLImageIO Headers
#include <GTLImageIO/ImageDC.h>
#include <GTLImageIO/ImageDCRegistry.h>
#include <GTLImageIO/Version.h>

void printVersion()
{
  std::cout << GTLImageIO::LibraryShortName() << " - " << GTLImageIO::LibraryName() << " - " << GTLImageIO::LibraryVersionString() << std::endl;
  std::cout << GTLImageIO::LibraryCopyright() << std::endl;
}
void printHelp()
{
  std::cout << "Usage : gtlconvert fileNameInput fileNameOutput" << std::endl;
  std::cout << std::endl;
  std::cout << "Options : " << std::endl;
  std::cout << std::endl;
  std::cout << "  -h --help               print this message" << std::endl;
  std::cout << "  -v --version            print the version information" << std::endl;
}

#define ARG_IS(a,b) GTLCore::String(argv[ai]) == a or GTLCore::String(argv[ai]) == b

int main(int argc, char** argv)
{
  GTLCore::String fileName = "";
  for(int ai = 1; ai < argc; ai++)
  {
    if(ARG_IS("-h","--help"))
    {
      printHelp();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-v","--version"))
    {
      printVersion();
      return EXIT_SUCCESS;
    } else {
      break;
    }
  }
  if( argc != 3)
  {
    printHelp();
  } else {
    GTLCore::String inputFileName = argv[1]; 
    GTLCore::String outputFileName = argv[2];
    const GTLImageIO::ImageDC* decoder = GTLImageIO::ImageDCRegistry::instance()->decoder( inputFileName );
    if(not decoder )
    {
      std::cerr << "No decoder for " << inputFileName << std::endl;
      return EXIT_FAILURE;
    }
    const GTLImageIO::ImageDC* encoder = GTLImageIO::ImageDCRegistry::instance()->encoder( outputFileName );
    if(not decoder )
    {
      std::cerr << "No encoder for " << outputFileName << std::endl;
      return EXIT_FAILURE;
    }
    GTLCore::String errMsg;
    GTLCore::RegionI region;
    GTLCore::AbstractImage* image = decoder->decode( inputFileName, &region, &errMsg );
    if( not image )
    {
      std::cerr << "Error while decoding " << inputFileName << " : " << errMsg << std::endl;
      return EXIT_FAILURE;
    }
    if(not encoder->encode( image, region, outputFileName, 0, &errMsg ))
    {
      std::cerr << "Error while encoding " << outputFileName << " : " << errMsg << std::endl;
      return EXIT_FAILURE;
    }
  }
  return EXIT_SUCCESS;
}
