///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/SubObjectParameterUI.h>
#include <core/gui/properties/BooleanPropertyUI.h>

#include "AtomsObjectEditor.h"
#include <atomviz/atoms/datachannels/PositionDataChannel.h>
#include <atomviz/atoms/datachannels/AtomTypeDataChannel.h>

namespace AtomViz {

IMPLEMENT_PLUGIN_CLASS(AtomsObjectEditor, PropertiesEditor)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void AtomsObjectEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create the main rollout.
	QWidget* mainPanel = createRollout(tr("Atoms"), rolloutParams, "atomviz.objects.atoms_object");
	{
		QVBoxLayout* layout = new QVBoxLayout(mainPanel);
		layout->setContentsMargins(4,4,4,4);
		layout->setSpacing(0);

		QGroupBox* displayGroupBox = new QGroupBox(tr("Display"));
		layout->addWidget(displayGroupBox);
		QVBoxLayout* sublayout = new QVBoxLayout(displayGroupBox);
		sublayout->setContentsMargins(0,0,0,0);

		_positionChannelEditor = new PositionDataChannelEditor();
		_positionChannelEditor->initialize(container(), RolloutInsertionParameters().insertInto(displayGroupBox));

		QGroupBox* atomtypesGroupBox = new QGroupBox(tr("Atom types"));
		layout->addWidget(atomtypesGroupBox);
		sublayout = new QVBoxLayout(atomtypesGroupBox);
		sublayout->setContentsMargins(0,0,0,0);

		_atomTypeChannelEditor = new AtomTypeDataChannelEditor();
		_atomTypeChannelEditor->initialize(container(), RolloutInsertionParameters().insertInto(atomtypesGroupBox));
	}

	// Create the data channels rollout.
	QWidget* dataChannelsPanel = createRollout(tr("Data channels"), rolloutParams.after(mainPanel).collapse());
	{
		QVBoxLayout* layout = new QVBoxLayout(dataChannelsPanel);
		layout->setContentsMargins(4,4,4,4);
		layout->setSpacing(0);

		RefTargetListParameterUI* dataChannelListUI = new RefTargetListParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObject, _dataChannels), RolloutInsertionParameters(dataChannelsPanel).animate());
		layout->addWidget(dataChannelListUI->listWidget());
	}

	// Open a sub-editor for the SimulationCell sub-object.
	SubObjectParameterUI* subEditorUI = new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObject, _simulationCell), rolloutParams.before(dataChannelsPanel).collapse());

	// Let the system call our handler every time the edited object changes.
	connect(this, SIGNAL(contentsChanged(RefTarget*)), this, SLOT(onEditContentsChanged(RefTarget*)));
}


/******************************************************************************
* Is called by the system when a the AtomsObject changed.
******************************************************************************/
void AtomsObjectEditor::onEditContentsChanged(RefTarget* newObject)
{
	AtomsObject* atoms = static_object_cast<AtomsObject>(newObject);

	DataChannel* positionChannel = NULL;
	DataChannel* atomTypeChannel = NULL;
	if(atoms) {
		positionChannel = atoms->getStandardDataChannel(DataChannel::PositionChannel);
		atomTypeChannel = atoms->getStandardDataChannel(DataChannel::AtomTypeChannel);
	}

	if(positionChannel != _positionChannelEditor->editObject())
		_positionChannelEditor->setEditObject(positionChannel);
	if(atomTypeChannel != _atomTypeChannelEditor->editObject())
		_atomTypeChannelEditor->setEditObject(atomTypeChannel);
}

};	// End of namespace AtomViz
