use crate::{
    parse::{expand_attribute_value, PropertyClass},
    types::dom::{Node, NodeData},
};
use html5ever::{
    local_name,
    serialize::{self, Serialize, TraversalScope},
    Attribute, LocalName, QualName,
};
use log::warn;
use std::{
    cell::RefCell,
    convert::TryFrom,
    fmt,
    io::{self, BufWriter, Write},
    rc::Rc,
};
use url::Url;

#[cfg(test)]
use html5ever::{parse_document, tendril::TendrilSink, ParseOpts};

#[cfg(test)]
use crate::types::dom::RcDom;

use super::ignored_elements;

/// A Facade over `Node` to invoke some common operations.
pub struct Element {
    pub attributes: RefCell<Vec<Attribute>>,
    pub name: String,
    pub node: Rc<Node>,
}

impl fmt::Debug for Element {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let html = extract_html(
            Rc::clone(&self.node),
            "http://indieweb.org".parse().unwrap(),
        )
        .map_err(|_e| std::fmt::Error {})?;
        f.debug_struct("Element")
            .field("attributes", &self.attributes)
            .field("name", &self.name)
            .field("node", &html)
            .finish()
    }
}

impl TryFrom<Rc<Node>> for Element {
    type Error = crate::Error;

    fn try_from(value: Rc<Node>) -> Result<Self, Self::Error> {
        if let NodeData::Element {
            ref attrs,
            ref name,
            ..
        } = value.data
        {
            Ok(Self {
                attributes: RefCell::clone(attrs),
                name: name.local.to_string(),
                node: Rc::clone(&value),
            })
        } else {
            Err(crate::Error::NotAnElement)
        }
    }
}

impl Element {
    /// Resolves an attribute's value; requires it to be provided.
    pub(crate) fn attribute(&self, attr_name: &str) -> Option<String> {
        self.maybe_attribute(attr_name)
            .filter(|attribute_value| !attribute_value.is_empty())
    }

    /// Obtains the value of an attribute with no requirement for a value.
    pub(crate) fn maybe_attribute(&self, attr_name: &str) -> Option<String> {
        attribute_value(RefCell::clone(&self.attributes), attr_name)
    }

    /// Determines if an attribute exists.
    pub(crate) fn has_attribute(&self, attr_name: &str) -> bool {
        self.maybe_attribute(attr_name).is_some()
    }

    /// Composes a list of all of the CSS class names on this element.
    pub(crate) fn classes(&self) -> Vec<String> {
        self.attribute("class")
            .map(|class_str| {
                class_str
                    .split_ascii_whitespace()
                    .map(|s| s.to_owned())
                    .collect::<Vec<_>>()
            })
            .unwrap_or_default()
    }

    /// Checks if this element has the provided class.
    pub(crate) fn has_class(&self, class_name: &str) -> bool {
        self.classes().contains(&class_name.to_owned())
    }

    /// Obtains all of the [property classes][PropertyClass] defined on this element.
    pub(crate) fn property_classes(&self) -> Option<Vec<PropertyClass>> {
        self.attribute("class")
            .map(PropertyClass::list_from_string)
            .filter(|list| !list.is_empty())
    }

    /// Confirms if any property class has been defined on this element.
    pub(crate) fn is_property_element(&self) -> bool {
        self.property_classes().is_some()
    }

    /// Determines if this element is showing any h- (Microformat item) properties.
    pub(crate) fn is_microformat_item(&self) -> bool {
        self.property_classes()
            .unwrap_or_default()
            .iter()
            .any(|p| p.is_root())
    }

    /// Obtains all of the root classes on this element.
    pub(crate) fn microformat_item_types(&self) -> Vec<crate::types::Class> {
        PropertyClass::extract_root_classes(self.property_classes().unwrap_or_default())
    }
}

fn attribute_value(attrs: RefCell<Vec<Attribute>>, name: &str) -> Option<String> {
    attrs
        .borrow()
        .iter()
        .find(|attr| &attr.name.local.to_ascii_lowercase() == name)
        .map(|attr| attr.value.to_string())
}

fn extract_html(node: Rc<Node>, base_url: Url) -> Result<String, crate::Error> {
    let mut serializer = HtmlExtractionSerializer::new(base_url);
    node.serialize(&mut serializer, TraversalScope::IncludeNode)
        .map_err(crate::Error::IO)
        .and_then(|()| serializer.to_string())
}

#[test]
fn extract_html_test() {
    crate::test::enable_logging();
    let html = "<html><head></head><body>wow</body></html>";
    let dom_result = parse_document(RcDom::default(), ParseOpts::default())
        .from_utf8()
        .read_from(&mut html.as_bytes());

    assert!(dom_result.is_ok());

    let dom = dom_result.unwrap().document;

    let obtained_html = extract_html_from_children(dom, "https://example.com".parse().unwrap());

    assert_eq!(obtained_html, Ok(html.to_owned()));
}

pub(crate) fn extract_html_from_children(
    node: Rc<Node>,
    base_url: Url,
) -> Result<String, crate::Error> {
    node.children
        .borrow()
        .iter()
        .try_fold(String::default(), |mut acc, node| {
            match extract_html(Rc::clone(node), base_url.clone()) {
                Ok(html) => {
                    acc.push_str(&html);
                    Ok(acc)
                }
                Err(e) => Err(e),
            }
        })
}

#[derive(Default)]
struct ElemInfo {
    html_name: Option<LocalName>,
    ignore_children: bool,
}
struct HtmlExtractionSerializer {
    pub writer: BufWriter<Vec<u8>>,
    base_url: Url,
    stack: Vec<ElemInfo>,
}

fn tagname(name: &QualName) -> LocalName {
    name.local.clone()
}

impl HtmlExtractionSerializer {
    pub fn new(url: Url) -> Self {
        let writer = BufWriter::new(Vec::default());
        Self {
            writer,
            base_url: url,
            stack: vec![ElemInfo {
                html_name: None,
                ignore_children: false,
            }],
        }
    }

    pub fn to_string(&self) -> Result<String, crate::Error> {
        String::from_utf8(self.writer.buffer().to_vec()).map_err(crate::Error::Utf8)
    }

    fn parent(&mut self) -> &mut ElemInfo {
        if self.stack.is_empty() {
            warn!("ElemInfo stack empty, creating new parent");
            self.stack.push(Default::default());
        }
        self.stack.last_mut().unwrap()
    }

    // NOTE: Should we allow for escaping non UTF-8 content (emoji, etc)?
    fn write_escaped(&mut self, text: &str, attr_mode: bool) -> io::Result<()> {
        for c in text.chars() {
            match c {
                '&' => self.writer.write_all(b"&amp;"),
                '\u{00A0}' => self.writer.write_all(b"&nbsp;"),
                '"' if attr_mode => self.writer.write_all(b"&quot;"),
                '<' if !attr_mode => self.writer.write_all(b"&lt;"),
                '>' if !attr_mode => self.writer.write_all(b"&gt;"),
                c => self.writer.write_fmt(format_args!("{}", c)),
            }?;
        }
        Ok(())
    }
}

impl serialize::Serializer for HtmlExtractionSerializer {
    fn start_elem<'a, AttrIter>(
        &mut self,
        name: html5ever::QualName,
        attrs: AttrIter,
    ) -> io::Result<()>
    where
        AttrIter: Iterator<Item = serialize::AttrRef<'a>>,
    {
        if self.parent().ignore_children {
            self.stack.push(ElemInfo {
                html_name: Some(name.local.clone()),
                ignore_children: true,
            });
            return Ok(());
        }

        self.writer.write_all(b"<")?;
        self.writer.write_all(tagname(&name).as_bytes())?;
        for (name, value) in attrs {
            self.writer.write_all(b" ")?;

            self.writer.write_all(name.local.as_bytes())?;
            self.writer.write_all(b"=\"")?;

            let actual_value =
                expand_attribute_value(name.local.to_string().as_str(), value, &self.base_url);

            self.write_escaped(&actual_value, true)?;
            self.writer.write_all(b"\"")?;
        }
        self.writer.write_all(b">")?;

        let ignore_children = matches!(
            name.local,
            local_name!("area")
                | local_name!("base")
                | local_name!("basefont")
                | local_name!("bgsound")
                | local_name!("br")
                | local_name!("col")
                | local_name!("embed")
                | local_name!("frame")
                | local_name!("hr")
                | local_name!("img")
                | local_name!("input")
                | local_name!("keygen")
                | local_name!("link")
                | local_name!("meta")
                | local_name!("param")
                | local_name!("source")
                | local_name!("track")
                | local_name!("wbr")
        );

        self.stack.push(ElemInfo {
            html_name: Some(name.local.clone()),
            ignore_children,
        });

        Ok(())
    }

    fn end_elem(&mut self, name: html5ever::QualName) -> io::Result<()> {
        let info = match self.stack.pop() {
            Some(info) => info,
            _ => panic!("no ElemInfo"),
        };
        if info.ignore_children {
            return Ok(());
        }

        self.writer.write_all(b"</")?;
        self.writer.write_all(tagname(&name).as_bytes())?;
        self.writer.write_all(b">")
    }

    fn write_text(&mut self, text: &str) -> io::Result<()> {
        match self.parent().html_name {
            Some(local_name!("iframe"))
            | Some(local_name!("noembed"))
            | Some(local_name!("noframes"))
            | Some(local_name!("plaintext")) => Ok(()),
            _ => self.write_escaped(text, false),
        }
    }

    fn write_comment(&mut self, _text: &str) -> io::Result<()> {
        Ok(())
    }

    fn write_doctype(&mut self, _name: &str) -> io::Result<()> {
        Ok(())
    }

    fn write_processing_instruction(&mut self, _target: &str, _data: &str) -> io::Result<()> {
        Ok(())
    }
}

struct PlainTextExtractionSerializer {
    pub writer: BufWriter<Vec<u8>>,
    base_url: Url,
    ignore_text: bool,
    swap_img_with_src: bool,
}

impl PlainTextExtractionSerializer {
    pub fn new(base_url: Url, swap_img_with_src: bool) -> Self {
        let writer = BufWriter::new(Vec::default());
        Self {
            writer,
            ignore_text: false,
            swap_img_with_src,
            base_url,
        }
    }

    pub fn to_string(&self) -> Result<String, crate::Error> {
        String::from_utf8(self.writer.buffer().to_vec()).map_err(crate::Error::Utf8)
    }
}

impl serialize::Serializer for PlainTextExtractionSerializer {
    fn start_elem<'a, AttrIter>(
        &mut self,
        name: html5ever::QualName,
        mut attrs: AttrIter,
    ) -> io::Result<()>
    where
        AttrIter: Iterator<Item = serialize::AttrRef<'a>>,
    {
        if ignored_elements().contains(&name.local.to_ascii_lowercase().to_string()) {
            self.ignore_text = true;
            Ok(())
        } else if name.local.to_ascii_lowercase().to_string().as_str() == "img" {
            let lookup_attrs = if self.swap_img_with_src {
                vec!["src", "alt"]
            } else {
                vec!["alt"]
            };
            let attr_opt = attrs
                .find(|(name, _)| {
                    lookup_attrs.contains(&name.local.to_ascii_lowercase().to_string().as_str())
                })
                .map(|(name, value)| {
                    if &name.local == "src" {
                        let value = expand_attribute_value(
                            name.local.to_string().as_str(),
                            value,
                            &self.base_url,
                        )
                        .trim()
                        .to_string();

                        if !value.is_empty() {
                            format!(" {value} ")
                        } else {
                            Default::default()
                        }
                    } else {
                        value.trim().to_string()
                    }
                })
                .filter(|v| {
                    if self.swap_img_with_src {
                        true
                    } else {
                        !v.is_empty()
                    }
                });

            if let Some(attr_string) = attr_opt {
                self.write_text(&attr_string)
            } else {
                Ok(())
            }
        } else {
            Ok(())
        }
    }

    fn end_elem(&mut self, _name: html5ever::QualName) -> io::Result<()> {
        self.ignore_text = false;
        Ok(())
    }

    fn write_text(&mut self, text: &str) -> io::Result<()> {
        if !self.ignore_text {
            self.writer.write(text.as_bytes()).and(Ok(()))
        } else {
            Ok(())
        }
    }

    fn write_comment(&mut self, _text: &str) -> io::Result<()> {
        Ok(())
    }

    fn write_doctype(&mut self, _name: &str) -> io::Result<()> {
        Ok(())
    }

    fn write_processing_instruction(&mut self, _target: &str, _data: &str) -> io::Result<()> {
        Ok(())
    }
}

pub fn extract_only_text(parent_node: Rc<Node>, base_url: Url) -> Result<String, crate::Error> {
    let mut serializer = PlainTextExtractionSerializer::new(base_url, false);
    parent_node
        .serialize(&mut serializer, TraversalScope::IncludeNode)
        .map_err(crate::Error::IO)
        .and_then(|()| serializer.to_string())
}

pub fn extract_text(parent_node: Rc<Node>, base_url: Url) -> Result<String, crate::Error> {
    let mut serializer = PlainTextExtractionSerializer::new(base_url, true);
    parent_node
        .serialize(&mut serializer, TraversalScope::IncludeNode)
        .map_err(crate::Error::IO)
        .and_then(|()| serializer.to_string())
}

#[test]
fn extract_text_test() {
    crate::test::enable_logging();
    let html = "<html><head></head><body>Sometimes, I want to just <b>jump</b> and <em>run</em>.<script>No</script><style>body { background: black; }</style><template>nice</template></body></html>";
    let dom_result = parse_document(RcDom::default(), ParseOpts::default())
        .from_utf8()
        .read_from(&mut html.as_bytes());

    assert!(dom_result.is_ok());

    let dom = dom_result.unwrap().document;

    let obtained_text = dom
        .children
        .borrow()
        .iter()
        .filter_map(|child| {
            extract_only_text(Rc::clone(child), "https://indieweb.org".parse().unwrap()).ok()
        })
        .collect::<Vec<_>>()
        .join("");

    assert_eq!(
        obtained_text,
        "Sometimes, I want to just jump and run.".to_string()
    );
}
