//=- LoongArchISelDAGToDAG.h - A dag to dag inst selector for LoongArch ---===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//
//
// This file defines an instruction selector for the LoongArch target.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_LIB_TARGET_LOONGARCH_LOONGARCHISELDAGTODAG_H
#define LLVM_LIB_TARGET_LOONGARCH_LOONGARCHISELDAGTODAG_H

#include "LoongArch.h"
#include "LoongArchSelectionDAGInfo.h"
#include "LoongArchTargetMachine.h"
#include "llvm/CodeGen/SelectionDAGISel.h"

// LoongArch-specific code to select LoongArch machine instructions for
// SelectionDAG operations.
namespace llvm {
class LoongArchDAGToDAGISel : public SelectionDAGISel {
  const LoongArchSubtarget *Subtarget = nullptr;

public:
  LoongArchDAGToDAGISel() = delete;

  explicit LoongArchDAGToDAGISel(LoongArchTargetMachine &TM,
                                 CodeGenOptLevel OptLevel)
      : SelectionDAGISel(TM, OptLevel) {}

  bool runOnMachineFunction(MachineFunction &MF) override {
    Subtarget = &MF.getSubtarget<LoongArchSubtarget>();
    return SelectionDAGISel::runOnMachineFunction(MF);
  }

  void Select(SDNode *Node) override;

  bool SelectInlineAsmMemoryOperand(const SDValue &Op,
                                    InlineAsm::ConstraintCode ConstraintID,
                                    std::vector<SDValue> &OutOps) override;

  bool SelectBaseAddr(SDValue Addr, SDValue &Base);
  bool SelectAddrConstant(SDValue Addr, SDValue &Base, SDValue &Offset);
  bool selectNonFIBaseAddr(SDValue Addr, SDValue &Base);
  bool SelectAddrRegImm12(SDValue Addr, SDValue &Base, SDValue &Offset);

  bool selectShiftMask(SDValue N, unsigned ShiftWidth, SDValue &ShAmt);
  bool selectShiftMaskGRLen(SDValue N, SDValue &ShAmt) {
    return selectShiftMask(N, Subtarget->getGRLen(), ShAmt);
  }
  bool selectShiftMask32(SDValue N, SDValue &ShAmt) {
    return selectShiftMask(N, 32, ShAmt);
  }

  bool selectSExti32(SDValue N, SDValue &Val);
  bool selectZExti32(SDValue N, SDValue &Val);

  bool selectVSplat(SDNode *N, APInt &Imm, unsigned MinSizeInBits) const;

  template <unsigned ImmSize, bool IsSigned = false>
  bool selectVSplatImm(SDValue N, SDValue &SplatVal);

  bool selectVSplatUimmInvPow2(SDValue N, SDValue &SplatImm) const;
  bool selectVSplatUimmPow2(SDValue N, SDValue &SplatImm) const;

  // Return the LoongArch branch opcode that matches the given DAG integer
  // condition code. The CondCode must be one of those supported by the
  // LoongArch ISA (see translateSetCCForBranch).
  static unsigned getBranchOpcForIntCC(ISD::CondCode CC) {
    switch (CC) {
    default:
      llvm_unreachable("Unsupported CondCode");
    case ISD::SETEQ:
      return LoongArch::BEQ;
    case ISD::SETNE:
      return LoongArch::BNE;
    case ISD::SETLT:
      return LoongArch::BLT;
    case ISD::SETGE:
      return LoongArch::BGE;
    case ISD::SETULT:
      return LoongArch::BLTU;
    case ISD::SETUGE:
      return LoongArch::BGEU;
    }
  }

// Include the pieces autogenerated from the target description.
#include "LoongArchGenDAGISel.inc"
};

class LoongArchDAGToDAGISelLegacy : public SelectionDAGISelLegacy {
public:
  static char ID;
  explicit LoongArchDAGToDAGISelLegacy(LoongArchTargetMachine &TM,
                                       CodeGenOptLevel OptLevel);
};

} // end namespace llvm

#endif // LLVM_LIB_TARGET_LOONGARCH_LOONGARCHISELDAGTODAG_H
