/*------------------------------------------------------------------------------
 *
 * Copyright (c) 2011-2025, EURid vzw. All rights reserved.
 * The YADIFA TM software product is provided under the BSD 3-clause license:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *        * Redistributions of source code must retain the above copyright
 *          notice, this list of conditions and the following disclaimer.
 *        * Redistributions in binary form must reproduce the above copyright
 *          notice, this list of conditions and the following disclaimer in the
 *          documentation and/or other materials provided with the distribution.
 *        * Neither the name of EURid nor the names of its contributors may be
 *          used to endorse or promote products derived from this software
 *          without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 *----------------------------------------------------------------------------*/

/**-----------------------------------------------------------------------------
 * @defgroup query_ex Database top-level query function
 * @ingroup dnsdb
 * @brief Database top-level query function
 *
 *  Database top-level query function
 *
 * @{
 *----------------------------------------------------------------------------*/

#pragma once

#include "zdb_lock.h"

#include <dnsdb/zdb_query_to_wire_context.h>

/**
 * Initialises the context for a query to the database
 *
 * This call locks the database structure.
 * Any call to this MUST be followed by a call to zdb_query_to_wire_finalize ASAP.
 *
 * @param context the context to initialise
 * @param mesg the message
 * @param db the database
 *
 * @return the status of the message (probably useless)
 */

static inline void zdb_query_to_wire_context_init(zdb_query_to_wire_context_t *context, dns_message_t *mesg, zdb_t *db)
{
    dns_packet_writer_init_append_to_message(&context->pw, mesg);
    context->mesg = mesg;
    context->fqdn = dns_message_get_canonised_fqdn(mesg);
#if DNSCORE_HAS_RRL_SUPPORT
    context->fqdn_label = NULL;
#endif
    context->flags = ~0;
    context->record_type = dns_message_get_query_type(mesg);
    context->db = db;
    memset(&context->answer_count, 0, (const uint8_t *)&context->ns_rrsets[0] - (const uint8_t *)&context->answer_count);
    zdb_lock(db, ZDB_MUTEX_READER);
}

/**
 * Builds a dns_message_t answer for a query.
 *
 * Typical usage:
 *
 *  zdb_query_to_wire_context_t context;
 *  zdb_query_to_wire_context_init(&context, mesg);
 *  zdb_query_to_wire(database, &context);
 *  zdb_query_to_wire_finalize(&context);   // absolutely mandatory after a call to zdb_query_to_wire_context_init
 *
 *  At this point the message is ready.
 *  TSIG signature could be the next step before answering.
 *
 * @param db the zone database
 * @param context the context to query for
 * @return the query status
 */

finger_print zdb_query_to_wire(zdb_query_to_wire_context_t *context);

/**
 * Releases resources and database an zone locks associated to the context.
 * Must ALWAYS be called to conclude a call to zdb_query_to_wire_context_init
 *
 * @param context
 */

void         zdb_query_to_wire_finalize(zdb_query_to_wire_context_t *context);

/** @} */
