/***************************************************************************
 *   Copyright (C) 2009 by Kai Dombrowe <just89@gmx.de>                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .        *
 ***************************************************************************/
 
#include <KDebug>
#include <QGraphicsSceneDragDropEvent>
#include <KConfig>
#include <KConfigGroup>
#include <QFile>
#include <KUrl>
#include <QApplication>
#include <Plasma/PopupApplet>
#include <QGraphicsView>

#include "listwidget.h"
#include "listwidgetitem.h"



ListWidget::ListWidget(QGraphicsWidget *parent)
    : QGraphicsWidget(parent),
    m_dragStartPos(-1, -1),
    m_dragStarted(false)
{

    setAcceptDrops(true);
    setAcceptHoverEvents(true);
    m_layout = new QGraphicsLinearLayout(Qt::Vertical);
    setLayout(m_layout);
    
}


ListWidget::~ListWidget()
{


}


void ListWidget::addItem(ListWidgetItem *item)
{

    m_layout->addItem(item);
    m_items.append(item);
    connect(item, SIGNAL(selected()), this, SLOT(deselectItem()));
    item->installEventFilter(this);
    
}


void ListWidget::insertItem(const int &index, ListWidgetItem *item)
{

    m_layout->insertItem(index, item);
    m_items.insert(index, item);
    connect(item, SIGNAL(selected()), this, SLOT(deselectItem()));
    item->installEventFilter(this);
    
}


void ListWidget::removeItem(ListWidgetItem *item)
{
 
    m_layout->removeItem(item);
    m_items.removeAll(item);
    item->disconnect();
    delete item;
    
}


QList<ListWidgetItem*> ListWidget::items() const
{

    return m_items;

}     


ListWidgetItem *ListWidget::selectedItem() const
{

    for (int i = 0; i < m_items.size(); i++) {
        if (m_items.at(i)->isSelected()) {
            return m_items.at(i);
        }
    }
    kDebug() << "no selected item";
    return 0;

}


void ListWidget::deselectItem()
{

    ListWidgetItem *selected = static_cast<ListWidgetItem*>(sender());
    if (!selected)
        return;
    foreach (ListWidgetItem *item, m_items) {
        if (item == selected)
            continue;
        item->setSelected(false);
    }

}


void ListWidget::dragEnterEvent(QGraphicsSceneDragDropEvent *event)
{

    QGraphicsWidget::dragEnterEvent(event);
    event->setAccepted(event->mimeData()->hasFormat("text/plain"));
    
}


void ListWidget::dragMoveEvent(QGraphicsSceneDragDropEvent *event)
{

    QGraphicsWidget::dragMoveEvent(event);
    
}


void ListWidget::dragLeaveEvent(QGraphicsSceneDragDropEvent *event)
{

    QGraphicsWidget::dragLeaveEvent(event);

}


void ListWidget::dropEvent(QGraphicsSceneDragDropEvent *event)
{

    kDebug() << event->mimeData()->formats();
    if (event->mimeData()->hasFormat("text/plain")) {
        QString file = KUrl(event->mimeData()->data("text/plain")).toLocalFile();
        if (!QFile(file).exists()) {
            event->ignore();
            kDebug() << "file not found";
            return;
        }
        KConfig cfg(file);
        KConfigGroup desktop(&cfg, "Desktop Entry");
        if (!desktop.readEntry("Categories", "").split(";").contains("Game")) {
            kDebug() << "no game";
            event->ignore();
            return;
        }
        ListWidgetItem *item = new ListWidgetItem(this);
        item->setTitle(desktop.readEntry("Name", "No Title"));
        item->setIcon(desktop.readEntry("Icon", ""));
        item->setCommand(desktop.readEntry("exec", ""));
        int index = dropIndexFromPoint(event->scenePos());
        if (index == -1) {
            addItem(item);
        } else {
            insertItem(index, item);
        }
        emit droped(item);
        event->accept();
    } else if (event->mimeData()->hasFormat("INTERNAL")) {
        int befor = event->mimeData()->data("Index").toInt();
        int after = dropIndexFromPoint(event->scenePos());
        if (after == -1) {
            after = m_items.size()+1;
        } else if (after > befor) {
            after--;
        }
        m_items.move(befor, after);
        while (m_layout->count() != 0) {
            m_layout->removeAt(0);
        }
        foreach (ListWidgetItem *item, m_items) {
            m_layout->addItem(item);
        }
        event->accept();
    } else {
        kDebug() << "ignor drop";
        event->ignore();
    }

}


bool ListWidget::eventFilter(QObject *obj, QEvent *event)
{

    Q_UNUSED(obj);
    switch (event->type()) {
        case QEvent::GraphicsSceneDrop: {
            dropEvent(static_cast<QGraphicsSceneDragDropEvent*>(event));
            return true;
        }
        default: return false;
    }

}


int ListWidget::dropIndexFromPoint(const QPointF pos)
{

    foreach (ListWidgetItem *item, m_items) {
        if (!item->mapRectToScene(item->contentsRect()).contains(pos)) {
            continue;
        } else {
            if (pos.y() > item->mapRectToScene(item->contentsRect()).center().y()) {
                return m_items.indexOf(item)+1;
            } else {
                return m_items.indexOf(item);
            }
        }
    }
    return -1;

}
     

int ListWidget::indexOf(ListWidgetItem *item) const
{

    return m_items.indexOf(item);

}


#include "listwidget.moc"